program PREPROC;

{ Program will generate the data required by the solids package from
            the user created file.
  HISTORY OF MODIFICATIONS:
    Version 1.0   (date unknown)
    Version 1.1   (September 1990) - Added support for large memory model.
    Version 1.2   (September 1991) - Added option to read old SURFMODL files,
                                     with re-use option.
}
const
   Prepvsn: string[5] = '1.20b';                  { version number }
   Lastupd: string[20] = '29 September 1991';     { date of last update }

{$DEFINE  BIGMEM  }   { Used for large memory model                         }

const
    MAXVAR = 20;                     { maximum # of numeric inputs on a line }
    MAXOUTLN = 200;                  { maximum # of outline nodes }
    NOPTIONS = 5;                    { options available 0..NOPTIONS }
    WORKINGVERS = 2;                 { current # of working versions }
    MAXMATL = 50;                    { maximum # of materials }
{$ifdef BIGMEM}
const MAXNODES = 3000;      { maximum # of nodes in the entire solid }
      MAXCONNECT = 12000;   { maximum # of connections in entire solid }
      MAXSURF = 4000;       { maximum # of surfaces in entire solid }
                            { (MAXSURF = MAXCONNECT / 3) }
{$else}
const MAXNODES = 1024;      { maximum # of nodes in the entire solid }
      MAXCONNECT = 4096;    { maximum # of connections in entire solid }
      MAXSURF = 1365;       { maximum # of surfaces in entire solid }
                            { (MAXSURF = MAXCONNECT / 3) }
{$endif}

const Matl_reuse: boolean = FALSE; { re-use material numbers across datafiles? }

type
    vartype = array[1..MAXVAR] of real;
    text80 = string[80];
    vector = array[1..3] of real;

{$ifdef BIGMEM}
  { A note on the BIGMEM definition:  Everything included under this
    section is a trick designed to overcome the memory limitations
    imposed by Turbo Pascal version 3.x and below.  Since TP limits
    all variable storage to one segment (64K), the following pointer
    definitions overcome this by storing the major SURFMODL arrays
    in the heap space.
  }
      heaparray1 = record World : array[1..MAXNODES] of vector;
                   end;
      hptr1 = ^heaparray1;
      heaparray2 = record Connect :array[1..MAXCONNECT] of integer;
                   end;
      hptr2 = ^heaparray2;
      heaparray3 = record Matl : array[1..MAXSURF] of integer;
                   end;
      hptr3 = ^heaparray3;
{$endif}

{$ifdef BIGMEM}
var   ptra : hptr1;   { World }
      ptrb : hptr2;   { Connect }
      ptrc : hptr3;   { Matl }
{$ELSE}
    World:            array[1..MAXNODES] of vector; { vertex coords }
    Connect:          array[1..MAXCONNECT] of integer;  { connection array }
    Matl:             array[1..MAXSURF] of integer; { matl# of each surface }
{$endif}

var
    Filename:         text80;                       { name of file to read }
    Infile:           text;                         { file to read }
    Nnodes:           integer;                      { number of nodes }
    Nsurf:            integer;                      { number of surfaces }
    Nmatl:            integer;                      { # materials }
    Maxvert:          integer;                      { max # of vert input }
    Realmaxsurf:      integer;                      { max #surfaces, based on }
                                                    { Maxvert and MAXCONNECT }
    Nsides:           integer;                      { #sides of surfaces used }
    Mat:              integer;                      { material# }
    Line_num:         integer;                      { line # of error }
    Num:              integer;                      { #inputted vals on line }
    Realvar:          vartype;                      { input array }
    Comment:          text80;                       { comment at end of line }
    Flpurpose:        string[127];                  { title for screen }
    Continue:         boolean;                      { flag end of input }
    In_version:       integer;                      { input version number }
    Cmmd:             integer;                      { preprocessor selection# }
    R1, R2, R3:       array[1..MAXMATL] of real;    { matl reflect constants }
    Color:            array[1..MAXMATL] of integer; { material color number }
    Ambient:          array[1..MAXMATL] of real;    { ambient light intensity }
    Checkmax:         boolean;                      { flag to check Maxvert }
    memerr :          boolean;                      { True if mem error }
    Inparm:           integer;                      { parm# of input file }
    Outparm:          integer;                      { parm# of output file }

{ Two important functions for decoding the Connect array: }

function KONNEC (Surf, Vert: integer): integer;
{ Decode the Connect array to yield the connection data: Vertex Vert of
surface Surf. This function returns an index to the global world arrays
(i.e., a node number) }
begin
{$ifdef BIGMEM}
with ptrb^ do
begin
{$endif}
  Konnec := Connect[(Surf-1) * Maxvert + Vert];
{$ifdef BIGMEM}
end; {with}
{$endif}
end; { function Konnec }

function NVERT (Surf: integer): integer;
{ Count the number of (non-zero) vertices in surface Surf. }
var Vert: integer;
begin
  for Vert := 1 to Maxvert do
    if konnec (Surf, Vert) = 0 then begin
      Nvert := Vert - 1;
      exit;
    end;
  Nvert := Maxvert;
end; { function Nvert }

procedure USAGE;
begin
  writeln ('usage: PREPROC [/R] infile outfile');
  writeln ('  where /R causes material numbers to be re-used.');
  halt;
end;

{ include files for preprocessor }
{$I PREINIT.INC }                                 { procedure INITIAL }
{$I OINREAL.INC }                                 { procedure INREAL }
{$I SCSHROT.INC }                                 { procedures SCALENODES, }
                                                  { SHIFTNODES, ROTATENODES }
{$I REVOLVE.INC }                                 { procedure REVOLVE }
{$I PREREAD.INC }                                 { procedure READFILE }
{$I EXTRUDE.INC }                                 { procedure EXTRUDE }
{$I PLANE.INC }                                   { procedure PLANE }
{$I PATCH.INC }                                   { procedure PATCH }
{$I SURFFILE.INC }                                { procedure SURFFILE }
{$I WRITEFIL.INC }                                { procedure WRITEFILE }

begin
{$ifdef BIGMEM }
  { Allocate memory from heap }
   memerr := false;

   new (ptra);
   if ptra = nil then
     memerr := true;

   new (ptrb);
   if ptrb = nil then
     memerr := true;

   new (ptrc);
   if ptrc = nil then
     memerr := true;

   if memerr then begin
     writeln ('You have run out of memory, you must do one of:');
     writeln ('  -- Increase your available memory');
     writeln ('  -- Decrease the array dimensions in PREPROC and recompile');
     writeln ('  -- Run the smaller version of PREPROC.');
     writeln;
     halt(1);
   end;
{$endif} {BIGMEM}

  writeln ('PREPROC ', Prepvsn, ' (', Lastupd, ')');
  initial;
  Filename := Paramstr(Inparm);
  openfile (Filename, Infile);
  readln (Infile, Flpurpose);
  Line_num := 2;

{ Read the version number }
  Num := inreal (Infile, Realvar, Comment, Line_num, FALSE);
  if (Num <> 1) or (Realvar[1] < 1) or (Realvar[1] > WORKINGVERS) then begin
    writeln ('Error in version number');
    if (Num <> 1) then
      writeln ('Expecting one numeric input.')
    else
      writeln ('Current versions are between 1 and ',WORKINGVERS,' only');
    close (Infile);
    halt;
  end; { if Num }
  In_version := round(Realvar[1]);

{ Read the control line }
  Line_num := Line_num + 1;
  Num := inreal (Infile, Realvar, Comment, Line_num, FALSE);
  if (Num <> 3) or (Realvar[1] > MAXMATL) or (Realvar[2] < 4) or
      (Realvar[2] > MAXNODES) or (Realvar[3] < 1) or
      (Realvar[3] > 2) then begin
    writeln ('Bad control line input (line ',Line_num,')');
    if (Num <> 3) then
      writeln ('Expecting 3 numeric inputs');
    if (Realvar[1] > MAXMATL) then
      writeln ('Number of materials (',Nmatl,') exceeds maximum (',
        MAXMATL,')');
    if (Realvar[2] < 4) then
      writeln ('Must have at least 4 vertices');
    if (Realvar[2] > MAXNODES) then
      writeln ('Maximum vertices must be less than ',MAXNODES);
    if (Realvar[3] < 1) or (Realvar[3] > 2) then
      writeln ('Nsides must be 1 or 2');
    close (Infile);
    halt;
  end; { if Num }
  Nmatl := round(Realvar[1]);
  Maxvert := round(Realvar[2]);
  Nsides := round(Realvar[3]);
  Realmaxsurf := MAXCONNECT div Maxvert;

{ Read the material data }
  for Mat := 1 to Nmatl do begin
    Line_num := Line_num + 1;
    Num := inreal (Infile, Realvar, Comment, Line_num, FALSE);
    if (Num <> 5) then begin
      writeln ('Bad input: Reading data for material #',Mat,' (line ',
           Line_num,')');
      close (Infile);
      halt;
    end;
    R1[Mat] := Realvar[1];
    R2[Mat] := Realvar[2];
    R3[Mat] := Realvar[3];
    Color[Mat] := round(Realvar[4]);
    Ambient[Mat] := Realvar[5];
  end;  {for Mat}

  repeat
    Line_num := Line_num + 1;
    Num := inreal (Infile, Realvar, Comment, Line_num, FALSE);
    if (Num <> 1) or (Realvar[1] < 0) or (Realvar[1] > NOPTIONS) then begin
      writeln ('Bad input for solid type code (line ',Line_num,')');
      if (Num <> 1) then
        writeln ('Expecting one numeric input')
      else
        writeln ('Option code must be between 0 and ',NOPTIONS:1);
      close (Infile);
      halt;
    end; { if Num }

    Cmmd := round(Realvar[1]);
    case Cmmd of
      0:  begin
            writeln ('PREPROC: Writing file (',Nnodes,' nodes and ', Nsurf,
                ' surfaces)');
            Continue := FALSE;
          end;
      1:  surffile;
      2:  revolve;
      3:  extrude;
      4:  plane;
      5:  patch;
    end;
  until (not Continue);

  close (Infile);
  if (Nmatl < 1) then begin
    writeln ('Error in input file: no materials specified.');
    usage;
  end;

{ Write the new file to disk }
  writefile (Paramstr(Outparm));

end. { program PREPROC }
