/********************************************************
*                REXXBox 
*              A CD Player for REXX
*
* It supports playplist, pause, resume, track display
* author and title display, error detection
*
* It supports the following command line options
*
*   Prints a wide variety of information about the CD.
*   1. The artist and CD title
*   2. The songs on the disc, their length and their order in the playlist
*   3. The current volume, and output type
*   4. A menu of choices during playback
*
* /O for output.  Lets you chose how the audio will be heard.
*   Possible options are:                                                    
*   1. Headphones (front of CD or audio connector)                           
*   2. Sound card (digital music transfer to your sound card)--CPU intensive!
*
* /P to change a playlist.  Allows you to change the order songs are played
*
* /H for help.    Will give you help using the REXXbox
*
* /C for color.   Turns off colors (ANSI mode).  In case you are color blind
*                or have a buggy display driver.                             
*
* /S for quiet.  Will display absolutely no instructions. Useful for batch mode
*********************************************************************************/

PlaylistExists = 'FALSE'
volume    = 100
outputset = 'FALSE'
outputdir = 'headphones'
env = 'OS2ENVIRONMENT'
ansisupport = 'ON'
animation = 1
delay = 1

SIGNAL ON ERROR /* When commands fail, call "error" routine. */

/*  load system utility functions to access ANSI and cursor functions */

call rxfuncadd sysloadfuncs, rexxutil, sysloadfuncs
call rxfuncadd SysTextScreenSize, rexxutil, SysTextScreenSize
call rxfuncadd SysCurPos, rexxutil, SysCurPos

/* Call routine to determine if we are running in a session with ANSI */
/* support (i.e. supports colors, cursor control etc.)                */

ansisupport = TestANSI()
if ansisupport = 'ON' then
   parse value SysTextScreenSize() with screenrow screencol

parse arg arg1 arg2 arg3 arg4 arg5

/* If the user asks for help, display options and then quit */

If (arg1 = '/?' | arg1 = '/h' | arg1 = '/?') THEN
   do
     DisplayHelpOptions()
   End
Else
  Do
  verbose = FALSE
  output  = FALSE
  playlistchange = FALSE
  End

/* we support up to five simultaneous options -- see  DisplayHelpOptions */

rc = ParseArgs(arg1 )
rc = ParseArgs(arg2 )
rc = ParseArgs(arg3 )
rc = ParseArgs(arg4 )
rc = ParseArgs(arg5 )

/* Load multimedia functions */

rc = RXFUNCADD('mciRxInit','MCIAPI','mciRxInit')
InitRC = mciRxInit()

/* Open the CD */

rc = mciRxSendString("open cdaudio alias supercd wait", 'Retst', '0', '0')

/* The following routine makes sure a CD is in the CD-ROM drive */
/* REXXbox can not function without some type of a CD           */

rc = EnsureCDInDrive('TRUE')

/* Obtain the unique product ID associated with the CD */

rc = mciRxSendString("info supercd id wait", cdIndentifier, '0', '0')
call sysloadfuncs

/* Get the directory where the REXXBOX.INI file is located */

rc = GetHome()

/* Now, check to see if the product id for this CD is stored in the INI file */
/* If we haven't seen this CD before, we won't have the information in the   */
/* INI file                                                                  */

cdkey = SysIni(mmhome, cdIndentifier, 'Artist')

IF cdkey \= 'ERROR:' THEN
  do
    /* This CD's description is in the INI file */
    /* Find out the number of tracks, artist and title */

    NumTracks = SysIni(mmhome, cdIndentifier, 'NumTracks' )
    if verbose = 'TRUE' THEN
      do
          /* Display the information in color if ANSI support is available */

          if ansisupport = 'ON' then
            do
             call charOut ,  LEFT('Artist is:', 19 )      || "1B"x || "[36;40m" || SysIni(mmhome, cdIndentifier, 'Artist') || "1B"x || "[0m"
             say ''
             call charOut ,  LEFT('Title is: ', 19 )      || "1B"x || "[36;40m" || SysIni(mmhome, cdIndentifier, 'Title') || "1B"x || "[0m"
             say ''
             call charOut ,  LEFT('Number of tracks:', 19)|| "1B"x || "[36;40m" || NumTracks || "1B"x || "[0m"
             say ''
             say ''
            END
         else /* we must display plain/jane text descriptions */
           do
             say  LEFT('Artist is:', 19 )       SysIni(mmhome, cdIndentifier, 'Artist')
             say  LEFT('Title is: ', 19 )       SysIni(mmhome, cdIndentifier, 'Title')
             say  LEFT('Number of tracks:', 19) NumTracks
           END
      END

    /* Setup default playlist */
    /* The playlist is the order that the music will be played */
    /* Unless the user specifies otherwise, we will play it sequentially */

    do loop = 0 to NumTracks
       Playlist.loop = loop
    END

    maxnamelen = 0
    /* Now, load the names associated with each track on the CD */
    /* The user gave us this information the first time the CD was inserted */

    do loop = 1 to NumTracks
      TrackKey = 'TrackName:' loop
      TrackName.loop = SysIni(mmhome, cdIndentifier, TrackKey );

      if LENGTH(TrackName.loop ) > maxnamelen  THEN
         maxnamelen = LENGTH(TrackName.loop )
    END

    /* Figure out if the caller has a playlist setup */

    PlaylistExists = SysIni(mmhome, cdIndentifier, 'Playlist' )

    if PlaylistExists = 'ERROR:' THEN
      PlaylistExists = FALSE
    ELSE
      do
         NumSongs = SysIni(mmhome, cdIndentifier, 'PlaylistTracks')
         /* If the playlist exists, but we can't figure out the number of songs */
         /* then, tell the user they have a corrupt REXXBOX.INI file.           */

         if NumSongs = 'ERROR:' THEN
            do
              say 'Configuration error--rexxbox.ini is corrupt'
            END
         ELSE
            /* Load the order of the playlist based on the total # of songs */

            do loop = 1 to NumSongs
              TrackKey = 'PlayTrackName:' loop
              Playlist.loop = SysIni(mmhome, cdIndentifier, TrackKey)
            END

      END

   volume = SysIni(mmhome, cdIndentifier, 'Volume' )

   /* if there is a bad ini value or if its not there--try to recover */

   if  volume = 'ERROR:' THEN
     volume = 100

   /* The user can have the output routed to either:
   *    the headphones (front panel/audio connector)
   *    or the sound card (via digital transfer)
   *
   * Figure out which setting the user chose last time we ran.
   */

   outputdir = SysIni(mmhome, cdIndentifier, 'outputdir' )
   if  outputdir = 'ERROR:' THEN
     do
     outputdir = 'headphones'
     say 'error getting output direction from REXXBOX.INI file'
     END

  END
ELSE /* We've never seen this CD before--find out artist/title info from user */
  do
    say 'Unknown CD--which artist recorded this CD?'
    pull artist.0
    say 'What is the title of the CD?'
    pull artist.1

    /* Place this information in the REXXBOX.INI file */

    iniRc = SysIni(mmhome, cdIndentifier, 'Artist', artist.0 )
    iniRc = SysIni(mmhome, cdIndentifier, 'Title',  artist.1 )

    /* Figure out how many tracks are on the CD */

    rc = mciRxSendString("status supercd number of tracks wait", 'NumTracks', '0', '0')

    if rc <> 0 THEN
      do
      say 'error getting number of tracks'

      /* Default to 1 track */

      iniRc = SysIni(mmhome, cdIndentifier, 'NumTracks',  1 )
      END
    ELSE
      do
      /* Save the total number of tracks */

      iniRc = SysIni(mmhome, cdIndentifier, 'NumTracks',  NumTracks )

      maxnamelen = 0

        /* Find out the name of each track */

        do loop = 1 to NumTracks
          say 'Enter name for track: 'loop ': '
          pull TrackName.loop
          TrackKey = 'TrackName:' loop
          iniRc = SysIni(mmhome, cdIndentifier, TrackKey,  TrackName.loop )

          if LENGTH(TrackName.loop ) > maxnamelen  THEN
             maxnamelen = LENGTH(TrackName.loop )
        END
      END

    /* Indicate that we haven't created a playlist yet */

    iniRc = SysIni(mmhome, cdIndentifier, 'Playlist',  'FALSE' )

    /* Inform the user of all the information they just gave us */

    say 'Artist is: 'SysIni(mmhome, cdIndentifier, 'Artist')
    say 'Title is: 'SysIni(mmhome, cdIndentifier, 'Title')
    say 'Number of tracks: 'SysIni(mmhome, cdIndentifier, 'NumTracks' )
    do loop = 1 to NumTracks
        TrackKey = 'TrackName:' loop
        say 'Track 'loop' is: ' SysIni(mmhome, cdIndentifier, TrackKey )
    END

  END

/* Determine the basic capabilities of the device we are using */

rc = BasicCaps()

/* Retrieve the playlist the user wants to hear */

rc = PlaylistSetup()

/* Figure out which output device they wish to use */

rc = DetermineOutput()

/* Figure out the length (IN TIME) for each track on the CD */

rc = GetTrackInfo()

/* Play the CD--updated the display etc. */

rc = PlayMusic()

/* Before we quit, save current settings (such as volume) */
/* so that when the user runs REXXbox again, all of their */
/* customization remains in effect                        */

rc = SaveSettings()

/* Cleanup and open devices and unload multimedia */

rc = mciRxSendString("close supercd wait", 'Retst', '0', '0')
rc = mciRxExit()                /* Tell the DLL we're going away */

if verbose = 'TRUE' THEN
   say 'Thank you for using the REXXbox'
exit

/***************************************************************
*
* The Basic capabilities function queries the CD device and
* determine some basic capabilities (i.e. does the device
* support volume, can it stream, can it prevent software ejects?
***************************************************************/

BasicCaps:


rc = mciRxSendString("capability supercd can eject wait", 'Retst', '0', '0')
If (Retst='TRUE') Then
  SoftEject = TRUE
ELSE
  SoftEject = FALSE

rc = mciRxSendString("capability supercd can lockeject wait", 'Retst', '0', '0')
If (Retst='TRUE') Then
  LockEject = TRUE
ELSE
  LockEject = FALSE

/* Many older or lower-end CD's don't support volume control-check and   */
/* find out what hardware capabilities are available.                    */

rc = mciRxSendString("capability supercd can setvolume wait", 'Retst', '0', '0')
If (Retst='TRUE') Then
  VolumeCap = TRUE
ELSE
  VolumeCap = FALSE

/* Digital Transfer lets your CD send data directly from the CD to your  */
/* sound card without ANY loss in quality.  However, this option is very */
/* CPU intensive--use with care                                          */

rc = mciRxSendString("capability supercd can stream wait", 'Retst', '0', '0')
If (Retst='TRUE') Then
  Stream = TRUE
ELSE
  Stream = FALSE

return 0

/***************************************************************
* Playlist setup determine the sequence of tracks that the user
* wants to hear.
***************************************************************/

PlaylistSetup:

/* If this is not the first time a CD has been inserted, PlaylistExists will be true */
/* In this case, we don't want to ask the user unless they use the /p option on the  */
/* the command line                                                                  */

If PlaylistExists = 'TRUE' THEN
  do
    /* if the /p option was used, they want to change the playlist */

    if playlistchange = 'TRUE' THEN
      do
        rc = PlaylistInput()
      END

  END
ELSE /* Create a playlist--first time we've seen this CD */
  do
    say 'A playlist allows you to control the order in which songs play'
    say 'Do you wish to setup a playlist (Y/N)'
    pull answer
    if answer = 'y' | answer = 'Y' THEN
      do
        rc = PlaylistInput()
      END /* caller wants to create playlist */
    else
      do
        /* Choose defaults for the playlist */
        do loop = 1 to NumTracks
          song = Playlist.loop
          say 'Song number 'loop' will be track ' TrackName.song
        END

      END

  END

return 0


/***************************************************************
* PlaylistInput retrieves the order of tracks that the user
* wants to hear.  It will also save the information to an
* ini file.
***************************************************************/

PlaylistInput:

        say 'How many songs in the playlist?'
        pull NumSongs

        Say 'Here are the tracks available on the CD'
        do loop = 1 to NumTracks
          say 'Track 'loop' is: ' TrackName.loop
        END

        do loop = 1 to NumSongs
          say 'Song number 'loop' will be track number?'

          /* need error checking */

          pull Playlist.loop
        END

        do loop = 1 to NumSongs
          song = Playlist.loop
          say 'Song number 'loop' will be track number ' TrackName.song
        END

        Say 'Save playlist (Y/N)?'
        pull answer

        if answer = 'y' | answer = 'Y' THEN
           do
             iniRc = SysIni(mmhome, cdIndentifier, 'Playlist',  'TRUE' )

             iniRc = SysIni(mmhome, cdIndentifier, 'PlaylistTracks', NumSongs )
             do loop = 1 to NumSongs
               TrackKey = 'PlayTrackName:' loop
               iniRc = SysIni(mmhome, cdIndentifier, TrackKey, Playlist.loop )
             END

           END

        return 0

/***************************************************************
* DetermineOutput lets the user determine where the output of
* the music will be heard.  By default, the music will be heard
* throught the front panel and also the audio connector.
* However, the user can route the output directly to the sound
* card.
*
* This function is ONLY available if the user uses the /O
* option on the command line.
***************************************************************/

DetermineOutput:


if output  = 'TRUE' THEN
  do
    say 'Where do you wish to hear the music (H)eadphones or (S)ound Card?'
    say 'Please enter H for headphones and S for sound card: '
    pull destination
    if destination = 's' | destination = 'S' THEN
      do
         if stream = 'FALSE' THEN
           do
              say "Sorry, but your hardware doesn't support streaming"
           END
         ELSE
           do
             rc = mciRxSendString("connector supercd enable type cd stream wait", 'Retst', '0', '0')
             /* TODO:  need error code check */

             if verbose = 'TRUE' THEN
               do
                 say 'Enabling streaming mode.  Return Code: ' mciRxGetErrorString(rc, 'errstr')
                 outputdir = 'digital'
                 outputset = 'TRUE'
               END
             ELSE
               do
               END

           END
      END
    /* Enabling the headphones */
    ELSE
      do
         rc = mciRxSendString("connector supercd enable type headphones wait", 'Retst', '0', '0')
         if verbose = 'TRUE' THEN
           do
             say 'Enabling headphones.  Return Code: ' mciRxGetErrorString(rc, 'errstr')
             outputdir = 'headphones'
             outputset = 'TRUE'
           END
         ELSE
           do
           END


      END
  END
  return 0

/***************************************************************
* GetTrackInfo will retrieve all of the information that is associated
* with a given track.
* This includes the tracks length in time (milliseconds).
***************************************************************/

GetTrackInfo:

  /* we are going to loop through each track and query the device */
  /* to determine how long each track (or song) will play         */

  /* Set to TMSF format to get time info based on the requested track */

  rc = mciRxSendString("set supercd time format tmsf wait", 'Retst', '0', '0')
  trackstring = 'status supercd length track'

  /* Only display this info if the /v command line options is passed */

  if verbose = 'TRUE' THEN
     do
     /* If ANSI, then display in color */

     if ansisupport = 'ON' then
        do
          call charOut , "1B"x || "[32;40m" || LEFT('Title', maxnamelen + 3)  || ' '
          call charOut , "1B"x || "[31;40m" || LEFT('Length', 10 )  || ' '
          call charOut , "1B"x || "[36;40m" || LEFT('Order', 20)
          say""
          call charOut , "1B"x || "[32;40m" || LEFT('-----', maxnamelen + 3)  || ' '
          call charOut , "1B"x || "[31;40m" || LEFT('------', 10 )   || ' '
          call charOut , "1B"x || "[36;40m" || LEFT('-----', 20)
          call charOut , "1B"x || "[0m"
          say""
        END
     else
        do
          say LEFT('Title', maxnamelen + 3) LEFT('Length', 10 ) LEFT('Order', 20)
          say LEFT('-----', maxnamelen + 3) LEFT('------', 10 ) LEFT('-----', 20)
        END
     END

  do loop = 1 to NumTracks

      trackcommand = trackstring loop ' wait '

      /* ask for the length (IN TIME) of each track */
      /* The system will return an answer in TMSF   */
      /* which looks like: M:S:F                    */
      /* we must parse the answer to obtain the     */
      /* pieces we want.                            */

      rc = mciRxSendString(trackcommand, 'len', '0', '0')
      PARSE VALUE len WITH minutes ':' seconds ':' frames
      TrackLength.loop = ((minutes * 60) + seconds )

      /* find out where this song is in the playlist */
      foundplaytrack = 'FALSE'
      search = 1
      /* If the user choose verbose display, then we will indicate */
      /* where this particular song will play in the playlist      */

      do while ( ( search <= NumSongs ) & ( foundplaytrack = 'FALSE' ) )
         if ( loop = Playlist.search ) THEN
            do
              foundplaytrack = 'TRUE'
            END
         else
           do
             search = search + 1
           END
      end /* do */
      if ( foundplaytrack \= 'TRUE' ) THEN
         search = 0

      time = minutes':'seconds
      if verbose = 'TRUE' THEN
         do
            if ( search > 0 ) then
               do
               /* If ANSI, then display in color */
               if ansisupport = 'ON' then
                  do
                 call charOut , "1B"x || "[32;40m" || LEFT( TrackName.loop, maxnamelen + 3) || ' '
                 call charOut , "1B"x || "[31;40m" || LEFT(time, 10) || ' '
                 call charOut , "1B"x || "[36;40m" || search
                 call charOut , "1B"x || "[0m"
                 say ""
                  END
               else
                  do
                  say LEFT( TrackName.loop, maxnamelen + 3) LEFT(time, 10) search
                  END
               END
            else
               do
               /* If ANSI, then display in color */
               if ansisupport = 'ON' then
                  do
                 call charOut , "1B"x || "[32;40m" || LEFT( TrackName.loop, maxnamelen + 3) || ' '
                 call charOut , "1B"x || "[31;40m" || LEFT(time, 10)
                 call charOut , "1B"x || "[0m"
                 say ""
                  END
               else
                  do
                  say LEFT( TrackName.loop, maxnamelen + 3) LEFT(time, 10)
                  END
               END
         END

  End
  if verbose = 'TRUE' THEN
    say ''
  return 0


/***************************************************************
* PlayMusic will play the playlist that the user has created for
* the CD.  It will play each track in the playlist sequentially
* and supports Fast Forward, Rewind, volume adjustments, Pause
* and resume.
***************************************************************/

PlayMusic:

/* Jump to the first track */
 quit      = FALSE
 trackquit = FALSE




 volup     = FALSE
 voldown   = FALSE
 forward   = FALSE
 rewind    = FALSE
 paused    = FALSE

 /* Display where the user will be hearing the audio output from the CD */

 if outputset = 'FALSE'  THEN
    do
      if verbose = 'TRUE' THEN
        /* If ANSI, then display in color */
        if ansisupport = 'ON' then
          do
            call charOut , "Output: "
          END
    if outputdir = 'digital' then
      do
      rc = mciRxSendString("connector supercd enable type cd stream wait", 'Retst', '0', '0')
      if verbose = 'TRUE' THEN
        /* If ANSI, then display in color */
        if ansisupport = 'ON' then
          do
            call charOut , "1B"x || "[32;40m" || "digital transfer" || "1B"x || "[0m"
            say ""
          END
        else
          do
           say 'OUTPUT: digital transfer'
          END
      END
    else
      do
      rc = mciRxSendString("connector supercd enable type headphones wait", 'Retst', '0', '0')
      if verbose = 'TRUE' THEN
        /* If ANSI, then display in color */
        if ansisupport = 'ON' then
          do
            call charOut , "1B"x || "[32;40m" || "headphones" || "1B"x || "[0m"
            say ""
          END
        else
          do
           say 'OUTPUT: headphones'
          END
      END
    END

 outputset = 'TRUE'

 /* Unless we are in quiet mode, then display the current volume setting */

 if (quiet <> 'TRUE') THEN
    do
      /* If ANSI, then display in color */
      if ansisupport = 'ON' then
        do
          call charOut , "Volume: "
          rc = DrawVolumeBar();
          say ""
        END
      else
        do
          say 'Volume :'volume
        END
    END

 /* Prepare basic strings which will be reused in the loops below */

 volumestring = 'set supercd audio volume '
 playstring = 'play supercd from '

 /* Let the user know the menu options we will make available */

 rc = DisplayMenu()

 /* Set the volume to the last setting the user had on the CD */

 volumecommand =  volumestring volume ' wait'
 rc = mciRxSendString(volumecommand, 'Retst', '0', '0')


 rc = mciRxSendString("set supercd time format tmsf wait", 'Retst', '0', '0')
 quitloop = 0

 /* Until the user hits 'Q' to quit, keep playing the CD */

 do while quit <> 'TRUE'
    /* Start with the first item in the playlist FOREVER */

    playloop = 1

    /* loop through each song in the playlist */

    do while ( ( playloop <= NumSongs ) & ( quit = FALSE ) )
       trackplaying = Playlist.playloop

       if debug = 'TRUE' THEN
         say 'Playing track 'trackplaying' named ' TrackName.trackplaying

       /* calculate time offsets from the beginning of the track                  */
       /* use milliseconds since its easier to determine time offsets in this     */
       /* rather than converting a track offset into a time unit                  */

       playcommand = playstring trackplaying':0:0:0'
       rc = mciRxSendString(playcommand, 'Retst', '0', '0')

       cdtime = 0

       rc = ClearLine()
       /*
       * Play the current song until we either reach the end of the
       * track OR the user hits a key which causes us to stop playing
       * the track (i.e rewind, forward or quit )
       */

       do while ( (cdtime < TrackLength.trackplaying) & (trackquit = FALSE) )

          /* See if the user has hit a key, if so, process it */
          rc = GetPlayInput()

          /* Display the current time we are playing */

          rc = DisplayTime()

          rc = mciRxSendString("status supercd position wait", 'currenttime', '0', '0')

          /*
          * An example of defensive programming--if the user ejects the
          * CD while it is playing, the call above will fail.
          * we will use the mciRxGetErrorString call to display an English
          * error message for the return code, and wait until the
          * user corrects the problem before continuing
          */

          if ( rc <> 0 ) then
             do

             errorrc = mciRxGetErrorString(rc, 'errstr')
             /* If ANSI, then display in color and bold */
             if ansisupport = 'ON' then
               do
                /* clear the line */

                call charOut , "1B"x || "[K" || "1B"x || "[1m"
                call charOut ,  "Error: "
                call charOut ,  "1B"x || "[0m" || errstr " please correct"
               END
             else
               do
                say "Error rc" errstr
               END
             /* Wait till problem is corrected */

             rc = EnsureCDInDrive( 'FALSE' )

             /* Restart playback--after the error the driver WILL not */
             /* automatically restart playback */

             playcommand = playstring trackplaying':'minutes':'seconds':0'
             rc = mciRxSendString(playcommand, 'Retst', '0', '0')
             rc = mciRxSendString("status supercd position wait", 'currenttime', '0', '0')

             /* Clear out the error information etc */

             rc = CharOut(, D2C(27) || "[6n")
             pull errorlocation
             row = substr( errorlocation, 3, 2 )
             location = '['row';1H'
             rc = CharOut(, D2C(27) || location )
             call charOut , "1B"x || "[K" || "1B"x || "[1m"

             END

          PARSE VALUE currenttime WITH tracks ':' minutes ':' seconds ':' frames

          /* translate into milliseconds */
          cdtime = ((minutes * 60) + seconds )

          /* if the user wants to decrease the volume AND */
          /* the hardware is capable of volume changes    */
          /* then decrease it */

          if ( ( voldown = 'TRUE')  & ( VolumeCap = 'TRUE') ) THEN
            do
               volume = volume - 4
               IF volume <= 0 THEN
                 volume = 0
               volumecommand =  volumestring volume ' wait'
               rc = mciRxSendString(volumecommand, 'Retst', '0', '0')
            voldown = FALSE
            rc = UpdateVolume();
            END

          /* if the user wants to increase the volume AND */
          /* the hardware is capable of volume changes    */
          /* then increase it */

          if ( (volup = 'TRUE') & ( VolumeCap = 'TRUE') ) THEN
            do
              volume = volume + 4
              IF volume > 100 THEN
                volume = 100
              volumecommand =  volumestring volume ' wait'
              rc = mciRxSendString(volumecommand, 'Retst', '0', '0')
              volup = FALSE
              rc = UpdateVolume();
            END


       END /* do while we haven't played past the end of the current track */

       trackquit = FALSE
       /* To rewind--we perform a trick, simply track 1 from our current position */
       /* in the playlist, if that puts us past position 1, wrap around to the end*/
       /* of the list */

       if ( rewind = 'TRUE' ) THEN
         do
         playloop = playloop - 1
         if ( playloop <= 0 ) THEN
           playloop =  NumSongs
         END
       ELSE
         /* advance one track in the playlist */

         playloop = playloop + 1
    END /* loop through the playlist */

 END /* while the user hasn't quit */

    return 0


/***************************************************************
* GetPlayInput will allow the user to either quit or modify playback
* volume or position.
***************************************************************/

GetPlayInput:


   /* Check to see if the user has pressed a key */

   if CHARS(STDIN) > 0 THEN
     do
     keyin = SysGetKey('NOECHO')
     SELECT
        when keyin = 'q' | keyin = 'Q' THEN
          do
            trackquit = TRUE
            quit = TRUE
          END
        when keyin = 'f' | keyin = 'F' THEN
          do
            trackquit = TRUE
            forward = TRUE
          END
        when keyin = 'w' | keyin = 'W' THEN
          do
            trackquit = TRUE

            rewind = TRUE
          END
        when keyin = 'u' | keyin = 'U' THEN
          do
            trackquit = FALSE
            volup = TRUE
          END
        when keyin = 'd' | keyin = 'D' THEN
          do
            trackquit = FALSE
            voldown = TRUE
          END
        when keyin = 'p' | keyin = 'P' THEN
          do
            trackquit = FALSE
            rc = mciRxSendString("pause supercd wait", 'Retst', '0', '0')
            paused = TRUE
          END
        when keyin = 'r' | keyin = 'R' THEN
          do
            if ( paused = 'TRUE' ) THEN
               do
                 rc = mciRxSendString("resume supercd wait", 'Retst', '0', '0')
                 paused = FALSE
               END
            trackquit = FALSE
          END
     otherwise
       if debug = 'TRUE' THEN
         say 'Invalid key--try again!'
     END
   END /* if there are characters available */

 return 0

/***************************************************************
* GetHome will retrieve the location of the multimedia directory.
* We will store our INI files in this directory
***************************************************************/

GetHome:

/* Get the multimedia home directory to store ini files */
/* allows program to run from any drive                 */
mmhome = value('mmbase',,env)
if mmhome = '' THEN
  mmhome = 'c:\mmos2\rexxbox.ini'
ELSE
  do

   PARSE VALUE mmhome WITH mmhome ';'
   mmhome = mmhome'\rexxbox.ini'
  END
return 0

/***************************************************************
* SaveSettings will save user options (such as volume and
* audio output location) to the ini file.
*
* Allows settings to survive across sessions.
***************************************************************/

SaveSettings:
    iniRc = SysIni(mmhome, cdIndentifier, 'Volume', volume )
    iniRc = SysIni(mmhome, cdIndentifier, 'outputdir', outputdir )
return 0


/***************************************************************
* EnsureCDInDrive makes sure that an audio CD is in the CD ROM
* it will not continue until the user inserts one.
***************************************************************/

EnsureCDInDrive:
   arg DisplayTxt
rc = mciRxSendString("status supercd media present wait", 'Retst', '0', '0')
If Retst <> 'TRUE'  then
     do
     /* Only display a warning message if the caller wants us to */

     if DisplayTxt = 'TRUE' then
       do
         say 'Must have CD inserted to run this command file'
         say 'Please insert'
       END
     rc = mciRxSendString("status supercd media present wait", 'Retst', '0', '0')

       do while Retst <> 'TRUE'
         rc = mciRxSendString("status supercd media present wait", 'Retst', '0', '0')
       End
     End

return 0


/***************************************************************
* DisplayTime updates the screen with the current time.
***************************************************************/

DisplayTime:
 playanimate = ''

 /* If ANSI, then display an animated playback icon                  */
 /* The only way to turn this off is to use /q from the command line */

 /* The animated cursor moves the '>' character accross 8 spaces to  */
 /* simulate a graphic PM button                                     */

 if ( (quiet <> 'TRUE') & (ansisupport = 'ON')) THEN
    do
    call charOut , "1B"x || "[s"
    /* Display playback animation */

    if ( paused = 'TRUE' )then
       playanimate = 'PAUSED  '
    else
      do
        do loop = 1 to 8
           if ( animation = loop ) THEN
             do
               playanimate = playanimate'>'
             END
           else
             do
               playanimate = playanimate' '
             END
        END
        if ( delay = 5 ) THEN
           do
             animation = animation + 1
             if ( animation > 8 ) THEN
                animation = 1
             delay = 1
           END
        else
          delay = delay + 1
      END

    call charOut ,playanimate || " " || TrackName.trackplaying"  " || "1B"x || "[31;40m"  || minutes ':' seconds || "1B"x || "[0m" || "1B"x || "[u"
    END
return 0

/***************************************************************
* Update volume will update the current volume setting.
***************************************************************/

UpdateVolume:

 /* If ANSI, then display a volume slider */
 /* it simply is a space bar in red       */
 if ( (quiet <> 'TRUE') & (ansisupport = 'ON')) THEN
    do

      rc = CharOut(, D2C(27) || "[6n")
      pull tracklocation
      row = substr( tracklocation, 3, 2 )
      col = substr( tracklocation , 6, 2 )


      /* Work around weird ANSI screen report */
      if ( row = screenrow ) THEN
        do
          volrow = row - 3
          row = row - 1
        END
      else
        do
          volrow = row - 2
        END
      volcol = 9
      vollocation = '['volrow';'volcol'H'
      rc = CharOut(, D2C(27) || vollocation )
      rc = DrawVolumeBar();

      location = '['row';'col'H'
      rc = CharOut(, D2C(27) || location )

    END
return 0

DrawVolumeBar:
      /* display volume bar in red */

      /* If ANSI, then we can draw the bar */
      if ansisupport = 'ON' then
         do
         call charOut , "| "
         call charOut ,"1B"x || "[31;41m"
         volloop = 0
         VolumeDisplay = ''
         do while volloop <= volume
            VolumeDisplay = VolumeDisplay' '
            volloop = volloop + 4
         end /* do */

         call charOut , VolumeDisplay
         VolumeDisplay = ''
         volloop = volume
         do while volloop <= 100
            VolumeDisplay = VolumeDisplay' '
            volloop = volloop + 4
         end /* do */
         VolumeDisplay = VolumeDisplay'|'
         call charOut ,"1B"x || "[0m"
         call charOut , VolumeDisplay
         /* return to default colors */

      END
return 0


ClearLine:

  /* If ANSI, then clearing the line is possible */
  if ansisupport = 'ON' then
    do
    rc = CharOut(, D2C(27) || "[6n")
    pull clearlocation
    clrrow = substr( clearlocation, 3, 2 )
    if ( clrrow = screenrow ) THEN
       do
       clrrow = clrrow - 1
       END
    clrline = '['clrrow';1H'
    rc = CharOut(, D2C(27) || clrline )

    call charOut , "1B"x || "[K"
    END
return 0

DisplayMenu:
 if quiet <> 'TRUE' THEN
    do
      /* If ANSI, then display a color menu with highlights etc */
      if ansisupport = 'ON' then
         do
         call charOut , "1B"x || "[31;47m" || "P" || "1B"x || "[30;47m" || "ause  "
         call charOut , "1B"x || "[31;47m" || "R" || "1B"x || "[30;47m" || "esume  "
         call charOut , "1B"x || "[31;47m" || "F" || "1B"x || "[30;47m" || "orward  "
         call charOut , "re" || "1B"x || "[31;47m" || "W" || "1B"x || "[30;47m" || "ind  "
         call charOut , "volume" || "1B"x || "[31;47m" || "U" || "1B"x || "[30;47m" || "p  "
         call charOut , "volume" || "1B"x || "[31;47m" || "D" || "1B"x || "[30;47m" || "own  "
         call charOut , "1B"x || "[31;47m" || "Q" || "1B"x || "[30;47m" || "uit " || "1B"x || "[0m"
         say ""
         END
      else
        do
          say 'Choose one of the options below:'
          say '(P)ause'
          say '(R)esume '
          say '(F)orward '
          say 'Re(w)ind'
          say 'Volume (U)p'
          say 'Volume (D)own'
          say '(Q)uit'
        END /* if not ansi */

    END

 return 0

/* TestANSI checks to see if the cursor can move left or right */
/* if it can, then we assume ANSI functions are available.     */
/* if not, the user will not get colors, dynamic display etc.  */

TestANSI:
   answer = 'OFF'
   leftcursor = SysCurPos()
   /* try to move right one column */

   call charOut , "1B"x || "[1C"

   /* check to see if our position changed */

   rightcursor = SysCurPos()
   if leftcursor <> rightcursor THEN
     do
      /* move left to restore our previous position */

      call CharOut , "1B"x || "[D"
      answer = 'ON'
     END

 RETURN answer


ParseArgs:
   arg argument
     SELECT
       WHEN (argument = '/c' | argument = '/C') THEN
         do
           ansisupport = 'OFF'
         END
       WHEN (argument = '/p' | argument = '/P' ) THEN
         do
          playlistchange = TRUE
         END

       WHEN (argument = '/o' | argument = '/O') THEN
         do
         output  = TRUE
         END

       WHEN (argument = '/d' | argument = '/D') THEN
         do
           say 'Debug mode on'
           debug = TRUE
         END

       WHEN (argument = '/s' | argument = '/S') THEN
         do
           quiet = 'TRUE'
         END
       WHEN (argument = '/v' | argument = '/V' ) THEN
         do
           if ansisupport = 'ON' then
              do
              parse value SysTextScreenSize() with screenrow screencol
              call charOut ,"1B"x || "[1m"
              call charOut , Center('The REXXbox--a REXX based CD player', screencol )
              call charOut ,"1B"x || "[0m"
              say ""
              END
           else
              do
              say 'The REXXbox--a REXX based CD player'
              END
           verbose = TRUE
         END
     OTHERWISE
       do
          if ( LENGTH(argument ) > 0 ) THEN
            do
              say argument' is an invalid parameter'
              DisplayHelpOptions()
            END
       END
     END
return 0


DisplayHelpOptions:

   say 'The REXXbox is a FREEWARE REXX-based multimedia CD player'
   say 'It is case insensitive and supports the following options :'
   say ''
   say '/V for verbose mode.'
   say '   Prints a wide variety of information about the CD.'
   say '   1. The artist and CD title'
   say '   2. The songs on the disc, their length and their order in the playlist'
   say '   3. The current volume, and output type'
   say '   4. A menu of choices during playback'
   say ''
   say '/O for output.  Lets you chose how the audio will be heard.'
   say '   Possible options are:'
   say '   1. Headphones (front of CD or audio connector)'
   say '   2. Sound card (digital music transfer to your sound card)--CPU intensive!'
   say ''
   say '/P to change a playlist.  Allows you to change the order songs are played'
   say ''
   say '/H for help.    Will give you help using the REXXbox'
   say ''
   say '/C for color.   Turns off colors (ANSI mode).  In case you are color blind'
   say '                or have a buggy display driver.'
   say ''
   say '/S for quiet.  Will display absolutely no instructions. Useful for batch mode'
   say ''
   exit 0



error:
   say 'Error' ErrRC 'at line' sigl ', sourceline:' sourceline(sigl)
   say 'Terminating'
   pull wait
   rc = mciRxSendString("close supercd wait", 'Retst', '0', '0')
   rc = mciRxExit()               /* Tell the DLL we're going away */
   exit ErrRC                /* exit, tell caller things went poorly */

halt:
/*
 * Close all device alias's, in case we previously killed
 * this batch file in the same process.
 */
   say 'Terminal Error!!!!'
   exitrc = mciRxSendString("close supercd wait", 'Retst', '0', '0')
   pull wait
   exitrc = mciRxExit()               /* Tell the DLL we're going away */
exit rc

