
title CTIMER4 -- 'PROFILER TIMER' MODIFIED FOR MSC / TC
page 75,132

;Original source:   BYTE Jan/87 pg. 157,  Byron Sheppard
;Mike Dumdei, 6 Holly Lane, Texarkana TX 75503

comment |====================================================================

void timer_on(char dispflag);
long timer_off(void);

These function's intended use is timing execution time of a section of code
very accurately.  The code these functions were derived from was written by
Byron Sheppard and appeared in BYTE magazine.  Enhancements to the original
code -- other than the C interface -- include self-calibration, preservation
of the interrupt flag, and a lot of simplification.  Their accuracy is based
on the accuracy of the crystal clock source driving the timer chip on the
machine they are ran on ( should be very accurate ).  A +-2 usec error is
probably a reasonable margin assuming an accurate clock.  Usage:

   timer_on(0);  /* (flag = 0) == do not use internal display routine */
   some_function();
   some_code;
   elapsed_usecs = timer_off();

If timer_on is passed a 1, the routines will write the elapsed time to the
screen using a DOS function.  You will notice that code that runs over a
few usecs will return varying times (+-200 usec) from one run to the next.
That is due to differing numbers of timer interrupts occurring during
execution of the code.  The maximum time the timer may run without calling
timer_off is slightly over 1 hour.  If the internal display function
is not needed, it can be easily removed.
============================================================================|

BIOSDATA SEGMENT AT 40H
        ORG     006ch
BIOSLoTick LABEL WORD
BIOSDATA ENDS

IFNDEF  model
        .MODEL  small, c
ELSE
  %     .MODEL  model, c
ENDIF
        .CODE

        include casm.inc

        .DATA

adjval          DW      0       ;adjust factor due to call overhead
tickstart       DW      0       ;beginning value of BIOS tick
cal_flag        DB      0       ;flag set if adj factor has been calculated
display         DB      0       ;type of output flag

msg             DB              0dh, 0ah, 'Elapsed time = '
msgsec          DB              '000 secs '
msgmsec         DB              '000 msec '
msgusec         DB              '000 usec    $'

        .CODE

timer_on PROC, DispFlag:BYTE
        xor     ax,ax
        cmp     cal_flag,al     ;check if calibration has been done
        je      __calibrate     ;run calibration routine if 1st time called
cal_done::
        mov     bl,DispFlag
        mov     display,bl      ;save output flag
;-- initialize counter 0 of 8253 timer
        mov     al,00110100b    ;select counter 0, LSB/MSB, mode 2 binary
        out     43H,al          ;mode register for 8253
        xor     al,al           ;clear timer so 1st tick is full count
        out     40H,al          ;LSB,               ; 40H = timer reg 0
        out     40H,al          ;MSB, write LSB/MSB of counter register
;-- get current tick BIOS time-of-day
        mov     ax,BIOSDATA     ;seg address of BIOS data
        mov     es,ax
        mov     ax,es:BIOSLoTick ;get low word of BIOS tick variable
        mov     tickstart,ax    ;save the beginning tick count
        ret                     ;back to caller
timer_on ENDP

;-- This procedure calculates an adjustment value to account for the
;   elapsed time due to the call to timer_on and timer_off.  This is
;   done so that only the elapsed time of the test code is reported.
__calibrate PROC
        mov     cal_flag,1      ;set the calibrate flag
        pushf
        push    ax              ;AX = display flag == 0, no display
        cli                     ;disable interrupts
        call    timer_on
        add     sp,2            ;take display flag arg off the stack
IF @CODESIZE
        call FAR PTR timer_off
ELSE
        call    timer_off       ;turn the timer back off
ENDIF
        popf                    ;restore original interrupt status
        mov     adjval,ax       ;save adjustment factor
        jmp s   cal_done        ;back to timer_on routine
__calibrate ENDP

timer_off PROC
        pushf                   ;save original interrupt status
        cli                     ;clear interrupts
        xor     al,al
        out     43H,al          ;latch timer for read
        in      al,40H          ;read LSB of timer
        mov     dl,al
        in      al,40H          ;read MSB of timer
        mov     dh,al           ;DX = MSB/LSB of timer count
        mov     al,00110110b
        out     43H,al
        xor     al,al
        out     40H,al
        out     40H,al          ;reset timer chip to original mode
        mov     ax,BIOSDATA
        mov     es,ax
        mov     cx,es:BIOSLoTick ;get current BIOS tick count
        sub     cx,tickstart
        popf                    ;restore original interrupt status
        call    NEAR PTR __convert ;convert ticks/counts to usec
        cmp     display,0
        je      @F              ;done if not displaying
        call    NEAR PTR __scrn_display ;else display on the screen
@@:
        ret                     ;back to caller
timer_off ENDP

;-- convert and screen display were made subroutines in order to
;   support a timer_read function.  I later decided to not put in a
;   timer_read but left the code broke out anyway.

__convert PROC NEAR
        neg     dx              ;invert DX since count is down counter
        mov     ax,8381
        mul     dx              ;multiply times 838.096 ns per count
        mov     bx,10000
        div     bx              ;AX = time in usec
        shr     bx,1            ;BX = 5000
        cmp     dx,bx           ;check in need to round up
        jb      @F              ;jmp if not
        inc     ax              ;else round
@@:
        xchg    ax,cx           ;AX = BIOS tick, CX = counter usecs
        push    ax              ;save ticks
        shr     ax,1            ;AX = ticks / 2
        add     cx,ax           ;add 1 usec for every 2 ticks -- see (5) below
        pop     ax              ;restore ticks
        mov     bx,54925
        mul     bx              ;ticks * 54.925 41(5 abv = 41 rounded up) msec
        add     ax,cx
        adc     dx,0            ;DX:AX = total usecs before ajustment
        or      dx,dx
        jnz     @F              ;jmp if lots of usecs
        cmp     ax,adjval
        ja      @F              ;jmp if more usecs than adjustment value
        mov     ax,adjval       ;else set usecs equal to adj value
@@:
        sub     ax,adjval       ;adjust for function call overhead
        sbb     dx,0            ;DX:AX = adjusted total usecs
        ret                     ;back to caller
__convert ENDP

__scrn_display PROC NEAR
        push    dx
        push    ax              ;else save return value on stack
        mov     bx,1000
        div     bx              ;DX = usecs, AX = msecs
        mov     cx,dx           ;CX = usecs
        xor     dx,dx
        div     bx              ;DX = msecs, AX = secs
        mov     bx,OFST msgsec[2]
        call    NEAR PTR __hex2ascii ;convert secs value to ASCII
        mov     ax,dx
        mov     bx,OFST msgmsec[2]
        call    NEAR PTR __hex2ascii ;convert msecs value to ASCII
        mov     ax,cx
        mov     bx,OFST msgusec[2]
        call    NEAR PTR __hex2ascii ;convert usecs value to ASCII
        mov     dx,OFST msg
        mov     ah,9
        int     21h             ;display elapsed time
        pop     ax
        pop     dx              ;restore return value
off_exit:
        ret                     ;back to caller
__scrn_display ENDP

__hex2ascii PROC NEAR
        push    cx
        push    dx
        mov     cl,10           ;divide by 10
        mov     ch,3            ;3 digits
@@:
        div     cl
        add     ah,'0'          ;remainder + 30h = converted digit
        mov     [bx],ah         ;store ASCII digit
        xor     ah,ah
        dec     bx              ;working 1's up to 100's so backup
        dec     ch              ;one less digit
        jnz     @B              ;loop if not finished
        pop     dx
        pop     cx
        ret                     ;back to caller
__hex2ascii ENDP

        END

