/***********************************************************************/
/* REXX.C - REXX interface routines.                                   */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1996 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3849 7731
 * Bellara                       http://www.gu.edu.au/gwis/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: rexx.c 2.1 1995/06/24 16:31:04 MH Rel MH $
*/

/*--------------------------- global data -----------------------------*/
/* these are here because EMX include REXX definitions from within os2.h */
/*---------------------------------------------------------------------*/
#  define INCL_RXSUBCOM       /* Subcommand handler values */
#  define INCL_RXSHV          /* Shared variable support */
#  define INCL_RXSYSEXIT      /* System exit routines */
#  define INCL_DOSPROCESS     /* Process/Thread Info */
#  define INCL_RXFUNC         /* External functions */

#include <the.h>
#include <proto.h>

LINE *rexxout_first_line=NULL;
LINE *rexxout_last_line=NULL;
LINE *rexxout_curr=NULL;
LINETYPE rexxout_number_lines=0L;

#if defined(HAVE_PROTO) || defined(USE_REXXIMC)

#include <query.h>
#include <rexx.h>

#define BUF_SIZE 512

#if defined(USE_REXX6000)
LONG THE_Commands(PRXSTRING, PUSHORT,PRXSTRING);
LONG THE_Exit_Handler(LONG, LONG, PEXIT);
USHORT THE_Function_Handler(PSZ,ULONG,PRXSTRING,PSZ,PRXSTRING);
#else
RexxSubcomHandler THE_Commands;
RexxExitHandler THE_Exit_Handler;
RexxFunctionHandler THE_Function_Handler;
#endif


#if defined(HAVE_PROTO)
static RXSTRING *get_rexx_variable(CHARTYPE *,RXSTRING *,short);
static short valid_target_function(ULONG, RXSTRING []);
static short run_os_function(ULONG, RXSTRING []);
static int run_os_command(CHARTYPE *,CHARTYPE *,CHARTYPE *,CHARTYPE *);
static CHARTYPE *MakeAscii(RXSTRING *);
static char *get_a_line(FILE *, char *, int *, int *);
static short set_rexx_variables_from_file(char *,CHARTYPE *);
#else
static RXSTRING *get_rexx_variable();
static short valid_target_function();
static short run_os_function();
static int run_os_command();
static CHARTYPE *MakeAscii();
static char *get_a_line();
static short set_rexx_variables_from_file();
#endif

static LINETYPE captured_lines;
static bool rexx_halted;

struct function_item
 {
  CHARTYPE *name;                      /* name of item */
  short item_number;                  /* unique number for item */
  short item_index;                  /* index value for the whole item */
  CHARTYPE query;                         /* valid query response ? */
 };
typedef struct function_item FUNCTION_ITEM;

 static FUNCTION_ITEM function_item[] =
  {
   {(CHARTYPE *)"ALT.0",ITEM_ALT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"ALT.1",ITEM_ALT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"ALT.2",ITEM_ALT,2,QUERY_FUNCTION},
   {(CHARTYPE *)"ARBCHAR.0",ITEM_ARBCHAR,0,QUERY_FUNCTION},
   {(CHARTYPE *)"ARBCHAR.1",ITEM_ARBCHAR,1,QUERY_FUNCTION},
   {(CHARTYPE *)"ARBCHAR.2",ITEM_ARBCHAR,2,QUERY_FUNCTION},
   {(CHARTYPE *)"AUTOSAVE.0",ITEM_AUTOSAVE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"AUTOSAVE.1",ITEM_AUTOSAVE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BACKUP.0",ITEM_BACKUP,0,QUERY_FUNCTION},
   {(CHARTYPE *)"BACKUP.1",ITEM_BACKUP,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BEEP.0",ITEM_BEEP,0,QUERY_FUNCTION},
   {(CHARTYPE *)"BEEP.1",ITEM_BEEP,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.0",ITEM_BLOCK,0,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.1",ITEM_BLOCK,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.2",ITEM_BLOCK,2,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.3",ITEM_BLOCK,3,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.4",ITEM_BLOCK,4,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.5",ITEM_BLOCK,5,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK.6",ITEM_BLOCK,6,QUERY_FUNCTION},
   {(CHARTYPE *)"CASE.0",ITEM_CASE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CASE.1",ITEM_CASE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CASE.2",ITEM_CASE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"CASE.3",ITEM_CASE,3,QUERY_FUNCTION},
   {(CHARTYPE *)"CASE.4",ITEM_CASE,4,QUERY_FUNCTION},
   {(CHARTYPE *)"CLEARSCREEN.0",ITEM_CLEARSCREEN,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CLEARSCREEN.1",ITEM_CLEARSCREEN,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CLOCK.0",ITEM_CLOCK,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CLOCK.1",ITEM_CLOCK,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CMDARROWS.0",ITEM_CMDARROWS,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CMDARROWS.1",ITEM_CMDARROWS,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CMDLINE.0",ITEM_CMDLINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CMDLINE.1",ITEM_CMDLINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CMDLINE.2",ITEM_CMDLINE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"CMDLINE.3",ITEM_CMDLINE,3,QUERY_FUNCTION},
   {(CHARTYPE *)"COLUMN.0",ITEM_COLUMN,0,QUERY_FUNCTION},
   {(CHARTYPE *)"COLUMN.1",ITEM_COLUMN,1,QUERY_FUNCTION},
   {(CHARTYPE *)"COMPAT.0",ITEM_COMPAT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"COMPAT.1",ITEM_COMPAT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CURLINE.0",ITEM_CURLINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CURLINE.1",ITEM_CURLINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CURLINE.2",ITEM_CURLINE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"CURLINE.3",ITEM_CURLINE,3,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.0",ITEM_CURSOR,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.1",ITEM_CURSOR,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.2",ITEM_CURSOR,2,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.3",ITEM_CURSOR,3,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.4",ITEM_CURSOR,4,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.5",ITEM_CURSOR,5,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.6",ITEM_CURSOR,6,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.7",ITEM_CURSOR,7,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSOR.8",ITEM_CURSOR,8,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSORSTAY.0",ITEM_CURSORSTAY,0,QUERY_FUNCTION},
   {(CHARTYPE *)"CURSORSTAY.1",ITEM_CURSORSTAY,1,QUERY_FUNCTION},
   {(CHARTYPE *)"DIRFILEID.0",ITEM_DIRFILEID,0,QUERY_FUNCTION},
   {(CHARTYPE *)"DIRFILEID.1",ITEM_DIRFILEID,1,QUERY_FUNCTION},
   {(CHARTYPE *)"DIRFILEID.2",ITEM_DIRFILEID,2,QUERY_FUNCTION},
   {(CHARTYPE *)"DISPLAY.0",ITEM_DISPLAY,0,QUERY_FUNCTION},
   {(CHARTYPE *)"DISPLAY.1",ITEM_DISPLAY,1,QUERY_FUNCTION},
   {(CHARTYPE *)"DISPLAY.2",ITEM_DISPLAY,2,QUERY_FUNCTION},
   {(CHARTYPE *)"EOF.0",ITEM_EOF,0,QUERY_FUNCTION},
   {(CHARTYPE *)"EOF.1",ITEM_EOF,1,QUERY_FUNCTION},
   {(CHARTYPE *)"EOLOUT.0",ITEM_EOLOUT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"EOLOUT.1",ITEM_EOLOUT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"ETMODE.0",ITEM_ETMODE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"ETMODE.1",ITEM_ETMODE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"FNAME.0",ITEM_FNAME,0,QUERY_FUNCTION},
   {(CHARTYPE *)"FNAME.1",ITEM_FNAME,1,QUERY_FUNCTION},
   {(CHARTYPE *)"FPATH.0",ITEM_FPATH,0,QUERY_FUNCTION},
   {(CHARTYPE *)"FPATH.1",ITEM_FPATH,1,QUERY_FUNCTION},
   {(CHARTYPE *)"FTYPE.0",ITEM_FTYPE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"FTYPE.1",ITEM_FTYPE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"GETENV.0",ITEM_GETENV,0,QUERY_FUNCTION},
   {(CHARTYPE *)"GETENV.1",ITEM_GETENV,1,QUERY_FUNCTION},
   {(CHARTYPE *)"HEX.0",ITEM_HEX,0,QUERY_FUNCTION},
   {(CHARTYPE *)"HEX.1",ITEM_HEX,1,QUERY_FUNCTION},
   {(CHARTYPE *)"HEXDISPLAY.0",ITEM_HEXDISPLAY,0,QUERY_FUNCTION},
   {(CHARTYPE *)"HEXDISPLAY.1",ITEM_HEXDISPLAY,1,QUERY_FUNCTION},
   {(CHARTYPE *)"HEXSHOW.0",ITEM_HEXSHOW,0,QUERY_FUNCTION},
   {(CHARTYPE *)"HEXSHOW.1",ITEM_HEXSHOW,1,QUERY_FUNCTION},
   {(CHARTYPE *)"HEXSHOW.2",ITEM_HEXSHOW,2,QUERY_FUNCTION},
   {(CHARTYPE *)"IDLINE.0",ITEM_IDLINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"IDLINE.1",ITEM_IDLINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"IMPMACRO.0",ITEM_IMPMACRO,0,QUERY_FUNCTION},
   {(CHARTYPE *)"IMPMACRO.1",ITEM_IMPMACRO,1,QUERY_FUNCTION},
   {(CHARTYPE *)"IMPMOS.0",ITEM_IMPOS,0,QUERY_FUNCTION},
   {(CHARTYPE *)"IMPMOS.1",ITEM_IMPOS,1,QUERY_FUNCTION},
   {(CHARTYPE *)"INSERTMODE.0",ITEM_INSERTMODE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"INSERTMODE.1",ITEM_INSERTMODE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LASTMSG.0",ITEM_LASTMSG,0,QUERY_FUNCTION},
   {(CHARTYPE *)"LASTMSG.1",ITEM_LASTMSG,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LASTRC.0",ITEM_LASTRC,0,QUERY_FUNCTION},
   {(CHARTYPE *)"LASTRC.1",ITEM_LASTRC,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LENGTH.0",ITEM_LENGTH,0,QUERY_FUNCTION},
   {(CHARTYPE *)"LENGTH.1",ITEM_LENGTH,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LINE.0",ITEM_LINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"LINE.1",ITEM_LINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LINEND.0",ITEM_LINEND,0,QUERY_FUNCTION},
   {(CHARTYPE *)"LINEND.1",ITEM_LINEND,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LINEND.2",ITEM_LINEND,2,QUERY_FUNCTION},
   {(CHARTYPE *)"MACROEXT.0",ITEM_MACROEXT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"MACROEXT.1",ITEM_MACROEXT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"MACROPATH.0",ITEM_MACROPATH,0,QUERY_FUNCTION},
   {(CHARTYPE *)"MACROPATH.1",ITEM_MACROPATH,1,QUERY_FUNCTION},
   {(CHARTYPE *)"MARGINS.0",ITEM_MARGINS,0,QUERY_FUNCTION},
   {(CHARTYPE *)"MARGINS.1",ITEM_MARGINS,1,QUERY_FUNCTION},
   {(CHARTYPE *)"MARGINS.2",ITEM_MARGINS,2,QUERY_FUNCTION},
   {(CHARTYPE *)"MARGINS.3",ITEM_MARGINS,3,QUERY_FUNCTION},
   {(CHARTYPE *)"MONITOR.0",ITEM_MONITOR,0,QUERY_FUNCTION},
   {(CHARTYPE *)"MONITOR.1",ITEM_MONITOR,1,QUERY_FUNCTION},
   {(CHARTYPE *)"MONITOR.2",ITEM_MONITOR,2,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGLINE.0",ITEM_MSGLINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGLINE.1",ITEM_MSGLINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGLINE.2",ITEM_MSGLINE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGLINE.3",ITEM_MSGLINE,3,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGLINE.4",ITEM_MSGLINE,4,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGMODE.0",ITEM_MSGMODE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"MSGMODE.1",ITEM_MSGMODE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"NBFILE.0",ITEM_NBFILE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"NBFILE.1",ITEM_NBFILE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"NEWLINES.0",ITEM_NEWLINES,0,QUERY_FUNCTION},
   {(CHARTYPE *)"NEWLINES.1",ITEM_NEWLINES,1,QUERY_FUNCTION},
   {(CHARTYPE *)"NONDISP.0",ITEM_NONDISP,0,QUERY_FUNCTION},
   {(CHARTYPE *)"NONDISP.1",ITEM_NONDISP,1,QUERY_FUNCTION},
   {(CHARTYPE *)"NUMBER.0",ITEM_NUMBER,0,QUERY_FUNCTION},
   {(CHARTYPE *)"NUMBER.1",ITEM_NUMBER,1,QUERY_FUNCTION},
   {(CHARTYPE *)"POSITION.0",ITEM_POSITION,0,QUERY_FUNCTION},
   {(CHARTYPE *)"POSITION.1",ITEM_POSITION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"POSITION.2",ITEM_POSITION,2,QUERY_FUNCTION},
   {(CHARTYPE *)"POSITION.3",ITEM_POSITION,3,QUERY_FUNCTION},
   {(CHARTYPE *)"PREFIX.0",ITEM_PREFIX,0,QUERY_FUNCTION},
   {(CHARTYPE *)"PREFIX.1",ITEM_PREFIX,1,QUERY_FUNCTION},
   {(CHARTYPE *)"PREFIX.2",ITEM_PREFIX,2,QUERY_FUNCTION},
   {(CHARTYPE *)"PRINTER.0",ITEM_PRINTER,0,QUERY_FUNCTION},
   {(CHARTYPE *)"PRINTER.1",ITEM_PRINTER,1,QUERY_FUNCTION},
   {(CHARTYPE *)"REPROFILE.0",ITEM_REPROFILE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"REPROFILE.1",ITEM_REPROFILE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"RESERVED.0",ITEM_RESERVED,0,QUERY_FUNCTION},
   {(CHARTYPE *)"RESERVED.1",ITEM_RESERVED,1,QUERY_FUNCTION},
   {(CHARTYPE *)"REXXOUTPUT.0",ITEM_REXXOUTPUT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"REXXOUTPUT.1",ITEM_REXXOUTPUT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"REXXOUTPUT.2",ITEM_REXXOUTPUT,2,QUERY_FUNCTION},
   {(CHARTYPE *)"SCALE.0",ITEM_SCALE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"SCALE.1",ITEM_SCALE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"SCALE.2",ITEM_SCALE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"SCOPE.0",ITEM_SCOPE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"SCOPE.1",ITEM_SCOPE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"SCREEN.0",ITEM_SCREEN,0,QUERY_FUNCTION},
   {(CHARTYPE *)"SCREEN.1",ITEM_SCREEN,1,QUERY_FUNCTION},
   {(CHARTYPE *)"SCREEN.2",ITEM_SCREEN,2,QUERY_FUNCTION},
   {(CHARTYPE *)"SELECT.0",ITEM_SELECT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"SELECT.1",ITEM_SELECT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"SELECT.2",ITEM_SELECT,2,QUERY_FUNCTION},
   {(CHARTYPE *)"SHADOW.0",ITEM_SHADOW,0,QUERY_FUNCTION},
   {(CHARTYPE *)"SHADOW.1",ITEM_SHADOW,1,QUERY_FUNCTION},
   {(CHARTYPE *)"SIZE.0",ITEM_SIZE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"SIZE.1",ITEM_SIZE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"STATUSLINE.0",ITEM_STATUSLINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"STATUSLINE.1",ITEM_STATUSLINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"STAY.0",ITEM_STAY,0,QUERY_FUNCTION},
   {(CHARTYPE *)"STAY.1",ITEM_STAY,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TABKEY.0",ITEM_TABKEY,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TABKEY.1",ITEM_TABKEY,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TABKEY.2",ITEM_TABKEY,2,QUERY_FUNCTION},
   {(CHARTYPE *)"TABLINE.0",ITEM_TABLINE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TABLINE.1",ITEM_TABLINE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TABLINE.2",ITEM_TABLINE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"TABS.0",ITEM_TABS,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TABS.1",ITEM_TABS,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TABSIN.0",ITEM_TABSIN,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TABSIN.1",ITEM_TABSIN,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TABSIN.2",ITEM_TABSIN,2,QUERY_FUNCTION},
   {(CHARTYPE *)"TABSOUT.0",ITEM_TABSOUT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TABSOUT.1",ITEM_TABSOUT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TABSOUT.2",ITEM_TABSOUT,2,QUERY_FUNCTION},
   {(CHARTYPE *)"TERMINAL.0",ITEM_TERMINAL,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TERMINAL.1",ITEM_TERMINAL,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TOF.0",ITEM_TOF,0,QUERY_FUNCTION},
   {(CHARTYPE *)"TOF.1",ITEM_TOF,1,QUERY_FUNCTION},
   {(CHARTYPE *)"VERIFY.0",ITEM_VERIFY,0,QUERY_FUNCTION},
   {(CHARTYPE *)"VERIFY.1",ITEM_VERIFY,1,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSHIFT.0",ITEM_VERSHIFT,0,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSHIFT.1",ITEM_VERSHIFT,1,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSION.0",ITEM_VERSION,0,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSION.1",ITEM_VERSION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSION.2",ITEM_VERSION,2,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSION.3",ITEM_VERSION,3,QUERY_FUNCTION},
   {(CHARTYPE *)"VERSION.4",ITEM_VERSION,4,QUERY_FUNCTION},
   {(CHARTYPE *)"WIDTH.0",ITEM_WIDTH,0,QUERY_FUNCTION},
   {(CHARTYPE *)"WIDTH.1",ITEM_WIDTH,1,QUERY_FUNCTION},
   {(CHARTYPE *)"WORDWRAP.0",ITEM_WORDWRAP,0,QUERY_FUNCTION},
   {(CHARTYPE *)"WORDWRAP.1",ITEM_WORDWRAP,1,QUERY_FUNCTION},
   {(CHARTYPE *)"ZONE.0",ITEM_ZONE,0,QUERY_FUNCTION},
   {(CHARTYPE *)"ZONE.1",ITEM_ZONE,1,QUERY_FUNCTION},
   {(CHARTYPE *)"ZONE.2",ITEM_ZONE,2,QUERY_FUNCTION},
   {(CHARTYPE *)"AFTER",ITEM_AFTER_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BATCH",ITEM_BATCH_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BEFORE",ITEM_BEFORE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BLANK",ITEM_BLANK_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BLOCK",ITEM_BLOCK_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"BOTTOMEDGE",ITEM_BOTTOMEDGE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"COMMAND",ITEM_COMMAND_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"CURRENT",ITEM_CURRENT_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"DIR",ITEM_DIR_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"END",ITEM_END_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"EOF",ITEM_EOF_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"FIRST",ITEM_FIRST_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"FOCUSEOF",ITEM_FOCUSEOF_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"FOCUSTOF",ITEM_FOCUSTOF_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"INBLOCK",ITEM_INBLOCK_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"INCOMMAND",ITEM_INCOMMAND_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"INITIAL",ITEM_INITIAL_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"INPREFIX",ITEM_INPREFIX_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"LEFTEDGE",ITEM_LEFTEDGE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"MODIFIABLE",ITEM_MODIFIABLE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"RIGHTEDGE",ITEM_RIGHTEDGE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"SPACECHAR",ITEM_SPACECHAR_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TOF",ITEM_TOF_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"TOPEDGE",ITEM_TOPEDGE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"VERONE",ITEM_VERONE_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"VALID_TARGET",ITEM_VALID_TARGET_FUNCTION,1,QUERY_FUNCTION},
   {(CHARTYPE *)"RUN_OS",ITEM_RUN_OS_FUNCTION,1,QUERY_FUNCTION},
   {NULL,0,0,0},
  };

/***********************************************************************/
#if defined(USE_REXX6000)
LONG THE_Commands
#else
ULONG THE_Commands
#endif
#if defined(HAVE_PROTO)
  (PRXSTRING Command,    /* Command string passed from the caller    */
   PUSHORT   Flags,      /* pointer to short for return of flags     */
   PRXSTRING Retstr)     /* pointer to RXSTRING for RC return        */
#else
  (Command,Flags,Retstr)
   PRXSTRING Command;    /* Command string passed from the caller    */
   PUSHORT   Flags;      /* pointer to short for return of flags     */
   PRXSTRING Retstr;     /* pointer to RXSTRING for RC return        */
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 SHVBLOCK shv;
 RXSTRING argstr;
/*--------------------------- processing ------------------------------*/
 if (allocate_temp_space(Command->strlength,TEMP_TEMP_CMD) != RC_OK)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
    *Flags = RXSUBCOM_ERROR;             /* raise an error condition   */
    sprintf((DEFCHAR *)Retstr->strptr, "%d", rc);   /* format return code string  */
                                         /* and set the correct length */
    Retstr->strlength = strlen((DEFCHAR *)Retstr->strptr);
    return 0L;                           /* processing completed       */
   }
 memcpy(temp_cmd,Command->strptr,Command->strlength);
 temp_cmd[Command->strlength] = '\0';

 if (strcmp("XXYYZZ",(DEFCHAR *)temp_cmd) == 0)
   {
    shv.shvnext=NULL;                                /* only one block */
    shv.shvcode=RXSHV_NEXTV;                             /* direct set */

    argstr.strptr=NULL;
    argstr.strlength=0;
    shv.shvname=argstr;
    rc = 0;
    while(rc != 2)
     {
      rc = RexxVariablePool(&shv);            /* get the next variable */
      if (rc != 2)
        {
         sprintf((DEFCHAR *)temp_cmd,"i <%s> <%s> <%ld>",
             shv.shvname.strptr,shv.shvvalue.strptr,shv.shvvalue.strlength);
         command_line(temp_cmd,COMMAND_ONLY_FALSE);
        }
     }
   }
 else
   rc = command_line(temp_cmd,COMMAND_ONLY_FALSE);
 if (rc < 0)
    *Flags = RXSUBCOM_ERROR;             /* raise an error condition   */
 else
    *Flags = RXSUBCOM_OK;                /* not found is not an error  */

 sprintf((DEFCHAR *)Retstr->strptr, "%d", rc); /* format return code string  */
                                         /* and set the correct length */
 Retstr->strlength = strlen((DEFCHAR *)Retstr->strptr);
 return 0L;                              /* processing completed       */
}
/***********************************************************************/
LONG THE_Exit_Handler
#if defined(HAVE_PROTO)
     (LONG  ExitNumber,    /* code defining the exit function    */
      LONG  Subfunction,   /* code defining the exit subfunction */
      PEXIT ParmBlock)     /* function dependent control block   */
#else
     (ExitNumber,Subfunction,ParmBlock)
      LONG  ExitNumber;    /* code defining the exit function    */
      LONG  Subfunction;   /* code defining the exit subfunction */
      PEXIT ParmBlock;     /* function dependent control block   */
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
#ifndef XCURSES
 extern WINDOW *foot;
 extern bool batch_only;
#endif
 extern CHARTYPE rexx_filename[10];
 extern CHARTYPE rexx_pathname[MAX_FILE_NAME+1];
 extern LINETYPE CAPREXXMAXx;
 extern bool CAPREXXOUTx;
 extern bool rexx_output;
/*--------------------------- local data ------------------------------*/
 RXSIOTRC_PARM *trc_parm = (RXSIOTRC_PARM *)ParmBlock;
 LONG rc=0L;
 char rexxout_temp[60];
 VIEW_DETAILS *found_view=NULL;
/*--------------------------- processing ------------------------------*/
 if (Subfunction != RXSIOSAY   /* ignore all but RXSIOSAY and RXSIOTRC */
 &&  Subfunction != RXSIOTRC)
    return(RXEXIT_NOT_HANDLED);
/*---------------------------------------------------------------------*/
/* If this is the first time this exit handler is called, set up the   */
/* handling of the result; either open the capture file, or set the    */
/* terminal out of curses mode so scrolling etc. work.                 */
/*---------------------------------------------------------------------*/
 if (!rexx_output)
   {
    rexx_output = TRUE;
    if (CAPREXXOUTx)
      {
/*---------------------------------------------------------------------*/
/* Free up the existing linked list (if any)                           */
/*---------------------------------------------------------------------*/
       rexxout_first_line = rexxout_last_line = lll_free(rexxout_first_line);
       rexxout_number_lines = 0L;
       if ((found_view = find_file(rexx_pathname,rexx_filename)) != (VIEW_DETAILS *)NULL)
         {
          found_view->file_for_view->first_line = found_view->file_for_view->last_line = NULL;
          found_view->file_for_view->number_lines = 0L;
         }
/*---------------------------------------------------------------------*/
/* first_line is set to "Top of File"                                  */
/*---------------------------------------------------------------------*/
       if ((rexxout_first_line = add_line(rexxout_first_line,NULL,TOP_OF_FILE,
            strlen((DEFCHAR *)TOP_OF_FILE),0)) == NULL)
          rc = RXEXIT_RAISE_ERROR;
/*---------------------------------------------------------------------*/
/* last line is set to "Bottom of File"                                */
/*---------------------------------------------------------------------*/
       if ((rexxout_last_line = add_line(rexxout_first_line,rexxout_first_line,BOTTOM_OF_FILE,
            strlen((DEFCHAR *)BOTTOM_OF_FILE),0)) == NULL)
          rc = RXEXIT_RAISE_ERROR;
       rexxout_curr = rexxout_first_line;
       if (found_view != (VIEW_DETAILS *)NULL)
         {
          found_view->file_for_view->first_line = rexxout_first_line;
          found_view->file_for_view->last_line = rexxout_last_line;
         }
      }
#ifndef XCURSES
    else
      {
       if (!batch_only)
         {
          wmove(foot,0,COLS-1);
          wrefresh(foot);
          suspend_curses();
         }
       printf("\n");                       /* scroll the screen 1 line */
       fflush(stdout);
      }
#endif
   }
/*---------------------------------------------------------------------*/
/* If the REXX interpreter has been halted by line limit exceeded, just*/
/* return to the interpreter indicating that THE is hadnling the output*/
/* of messages. This is done to stop the "clutter" that comes back as  */
/* the interpreter tries to tell us that it is stopping.               */
/*---------------------------------------------------------------------*/
 if (rexx_halted)
    return(RXEXIT_HANDLED);
/*---------------------------------------------------------------------*/
/* If we are capturing the rexx output, print the string to the file.  */
/*---------------------------------------------------------------------*/
 if (CAPREXXOUTx)
   {
    rexxout_number_lines++;
    if ((rexxout_curr = add_line(rexxout_first_line,rexxout_curr,
                                 (CHARTYPE *)trc_parm->rxsio_string.strptr,
                                 (trc_parm->rxsio_string.strlength==(-1))?0:trc_parm->rxsio_string.strlength,0)) == NULL)
       rc = RXEXIT_RAISE_ERROR;
    else
       rc = RXEXIT_HANDLED;
   }
 else
    rc = RXEXIT_NOT_HANDLED;
/*---------------------------------------------------------------------*/
/* If the number of lines processed exceeds the line limit, display our*/
/* own message telling what has happened and exit with                 */
/* RXEXIT_RAISE_ERROR. This tells the interpreter that it is to stop.  */
/*---------------------------------------------------------------------*/
 if (++captured_lines > CAPREXXMAXx)
   {
    if (CAPREXXOUTx)
      {
       rexxout_number_lines++;
       sprintf(rexxout_temp,"THE: REXX macro halted - line limit (%ld) exceeded",CAPREXXMAXx);
       rexxout_curr = add_line(rexxout_first_line,rexxout_curr,
                               (CHARTYPE *)rexxout_temp,
                               strlen((DEFCHAR *)rexxout_temp),0);
      }
    else
       printf("THE: REXX macro halted - line limit (%ld) exceeded\n",CAPREXXMAXx);
    rc = RXEXIT_RAISE_ERROR;
    rexx_halted = TRUE;
   }
 return(rc);
}
/***********************************************************************/
#if defined(USE_REXX6000)
USHORT THE_Function_Handler
#else
ULONG THE_Function_Handler
#endif
#if defined(HAVE_PROTO)
#  if defined(EMX)
     (PCSZ   FunctionName,  /* name of function */
#  else
     (PSZ   FunctionName,  /* name of function */
#  endif
      ULONG Argc,          /* number of arguments    */
      RXSTRING Argv[],     /* array of arguments in RXSTRINGs */
#  if defined(EMX)
      PCSZ   QueueName,     /* name of queue */
#  else
      PSZ   QueueName,     /* name of queue */
#  endif
      PRXSTRING Retstr)    /* return string   */
#else
     (FunctionName,Argc,Argv,QueueName,Retstr)
      PSZ   FunctionName;  /* name of function */
      ULONG Argc;          /* number of arguments    */
      RXSTRING Argv[];     /* array of arguments in RXSTRINGs */
      PSZ    QueueName;     /* name of queue */
      PRXSTRING Retstr;    /* return string   */
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[MAX_VARIABLES_RETURNED];
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
/*---------------------------------------------------------------------*/
/* Find the external function name in the array. Error if not found.   */
/*---------------------------------------------------------------------*/
 for (i=0;function_item[i].name != NULL;i++)
   {
    if (memcmpi((CHARTYPE *)FunctionName,function_item[i].name,strlen((DEFCHAR *)function_item[i].name))==0)
      {
       switch(function_item[i].item_number)
         {
          case ITEM_VALID_TARGET_FUNCTION:
               valid_target_function(Argc,Argv);
               break;
          case ITEM_RUN_OS_FUNCTION:
               run_os_function(Argc,Argv);
               break;
          default:
               if (Argv == NULL)
                  (void)get_item_values(function_item[i].item_number,
                        (CHARTYPE *)"",function_item[i].query,(LINETYPE)Argc,NULL,0L);
               else
                  (void)get_item_values(function_item[i].item_number,
                        (CHARTYPE *)"",function_item[i].query,(LINETYPE)Argc,
                        (CHARTYPE *)Argv[0].strptr,(LINETYPE)Argv[0].strlength);
               break;
         }
       if (item_values[function_item[i].item_index].len > 256)
         {
#ifdef USE_REXX6000
          if ((Retstr->strptr = (PUCHAR)(*the_malloc)(item_values[function_item[i].item_index].len)) == NULL)
#else
          if ((Retstr->strptr = (PSZ)(*the_malloc)(item_values[function_item[i].item_index].len)) == NULL)
#endif
            {
             display_error(30,(CHARTYPE *)"",FALSE);
             return(1);
            }
         }
       memcpy(Retstr->strptr,item_values[function_item[i].item_index].value,
                             item_values[function_item[i].item_index].len);
       Retstr->strlength = item_values[function_item[i].item_index].len;
       return(0);
      }
   }
 return(1);                                    /* fatal error for REXX */
}
/***********************************************************************/
short initialise_rexx
#if defined(HAVE_PROTO)
      (void)
#else
      ()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 APIRET rc;
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    initialise_rexx");
#endif

#if defined(MSWIN)
 if (RexxThread(GetCurrentTask(),THREAD_ATTACH) != THREAD_ATTACH_AOK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
#endif

#if defined(USE_REXX6000)
 rc = RexxRegisterSubcom((PSZ)"THE",
                         (PRXSUBCOM)&THE_Commands,
                         (PUCHAR)NULL);
#else
 rc = RexxRegisterSubcomExe((PSZ)"THE",
                           (PFN)THE_Commands,
                           (PUCHAR)NULL);
#endif

 if (rc != RXSUBCOM_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return((short)rc);
   }

#if !defined(USE_REXX6000)
 rc = RexxRegisterExitExe((PSZ)"THE_EXIT",
                          (PFN)THE_Exit_Handler,
                          (PUCHAR)NULL);
 if (rc != RXEXIT_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return((short)rc);
   }
#endif

 for (i=0;function_item[i].name != NULL;i++)
   {
#if defined(USE_REXX6000)
    rc = RexxRegisterFunction((PSZ)function_item[i].name,
                              (PRXFUNC)&THE_Function_Handler,NULL);
#else
    rc = RexxRegisterFunctionExe((PSZ)function_item[i].name,
                                 (PFN)THE_Function_Handler);
#endif
    if (rc != RXFUNC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return((short)rc);
      }
   }

#ifdef TRACE
 trace_return();
#endif
 return((short)rc);
}
/***********************************************************************/
short finalise_rexx
#if defined(HAVE_PROTO)
    (void)
#else
    ()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 APIRET rc;
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    finalise_rexx");
#endif

#if defined(USE_REXX6000)
 rc = RexxDeregisterSubcom((PSZ)"THE");
#else
 rc = RexxDeregisterSubcom((PSZ)"THE",(PSZ)NULL);
 rc = RexxDeregisterExit((PSZ)"THE_EXIT",(PSZ)NULL);
#endif
 for (i=0;function_item[i].name != NULL;i++)
   {
    rc = RexxDeregisterFunction((PSZ)function_item[i].name);
   }

#if defined(MSWIN)
 (void)RexxThread(GetCurrentTask(),THREAD_DETACH);
#endif

#ifdef TRACE
 trace_return();
#endif
 return((short)rc);
}
/***********************************************************************/
short execute_macro_file
#if defined(HAVE_PROTO)
      (CHARTYPE *filename,CHARTYPE *params)
#else
      (filename,params)
       CHARTYPE *filename;
       CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool batch_only;
 extern bool error_on_screen;
#ifndef XCURSES
 extern CHARTYPE number_of_files;
#endif
 extern CHARTYPE rexx_filename[10];
 extern CHARTYPE rexx_pathname[MAX_FILE_NAME+1];
 extern CHARTYPE rexx_macro_name[MAX_FILE_NAME+1];
 extern CHARTYPE *temp_cmd;
 extern bool CAPREXXOUTx;
 extern bool rexx_output;
/*--------------------------- local data ------------------------------*/
#if defined(USE_REGINA) || defined(USE_REXX6000)
 LONG rexxrc=0L;
#else
 USHORT rexxrc=0L;
#endif
 RXSTRING retstr;
 RXSTRING argstr;
 APIRET rc=0;
 CHAR retbuf[250];
 LONG num_params=0L;
 CHARTYPE *rexx_args=NULL;
 RXSYSEXIT exit_list[2];                /* system exit list           */
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    execute_macro_file");
#endif
/*---------------------------------------------------------------------*/
/* Determine how many parameters are to be passed to the interpreter.  */
/* Only 0 or 1 are valid values.                                       */
/*---------------------------------------------------------------------*/
 if (params == NULL
 || strcmp((DEFCHAR *)params,"") == 0)
   {
    num_params = 0;
    MAKERXSTRING(argstr,"",0);
   }
 else
   {
    num_params = 1;
    if ((rexx_args = (CHARTYPE *)(*the_malloc)(strlen((DEFCHAR *)params)+1)) == (CHARTYPE *)NULL)
      {
       display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY);
      }
    strcpy((DEFCHAR *)rexx_args,(DEFCHAR *)params);
    MAKERXSTRING(argstr,(DEFCHAR *)rexx_args,strlen((DEFCHAR *)rexx_args));
   }

 MAKERXSTRING(retstr,retbuf,sizeof(retbuf));

 strcpy((DEFCHAR *)rexx_macro_name,(DEFCHAR *)filename);
/*---------------------------------------------------------------------*/
/* Set up pointer to REXX Exit Handler.                                */
/*---------------------------------------------------------------------*/
#if defined(USE_REXX6000)
 exit_list[0].sysexit_func = THE_Exit_Handler;
#else
 exit_list[0].sysexit_name = "THE_EXIT";
#endif
 exit_list[0].sysexit_code = RXSIO;
 exit_list[1].sysexit_code = RXENDLST;

 captured_lines = 0L;
 rexx_output = FALSE;
 rexx_halted = FALSE;
/*---------------------------------------------------------------------*/
/* Call the REXX interpreter.                                          */
/*---------------------------------------------------------------------*/
 rc = RexxStart((LONG)num_params,
               (PRXSTRING)&argstr,
               (PSZ)filename,
               (PRXSTRING)NULL,
               (PSZ)"THE",
               (LONG)RXCOMMAND,
               (PRXSYSEXIT)exit_list,
#if defined(USE_OS2REXX)
               (PSHORT)&rexxrc,
#else
               (PLONG)&rexxrc,
#endif
               (PRXSTRING)&retstr);

/*---------------------------------------------------------------------*/
/* Edit the captured file or clean up after REXX output displays.      */
/*---------------------------------------------------------------------*/
 if (rexx_output)
   {
    rexx_output = FALSE;
    if (CAPREXXOUTx)
      {
       strcpy((DEFCHAR *)temp_cmd,(DEFCHAR *)rexx_pathname);
       strcat((DEFCHAR *)temp_cmd,(DEFCHAR *)rexx_filename);
       Xedit(temp_cmd);
      }
    else
      {
       if (batch_only)
          error_on_screen = TRUE;
#ifndef XCURSES
       else
/*---------------------------------------------------------------------*/
/* Pause for operator intervention and restore the screen to the       */
/* current screen if there are still file(s) in the ring.              */
/*---------------------------------------------------------------------*/
         {
          printf("\n%s",HIT_ANY_KEY);
          fflush(stdout);
          resume_curses();
          (void)my_getch(stdscr);
          if (number_of_files > 0)
             restore_THE();
         }
#endif
      }
   }

 if (rexx_args != NULL)
    (*the_free)(rexx_args);
#ifdef TRACE
 trace_return();
#endif
 return((short)rc);
}
/***********************************************************************/
short set_rexx_variable
#if defined(HAVE_PROTO)
      (CHARTYPE *name,CHARTYPE *value,short value_length,short suffix)
#else
      (name,value,value_length,suffix)
       CHARTYPE *name;
       CHARTYPE *value;
       short value_length;
       short suffix;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 SHVBLOCK shv;
 CHAR variable_name[50];
 short rc=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    set_rexx_variable");
#endif

 shv.shvnext=NULL;                                   /* only one block */
#if defined(USE_REGINA)
 shv.shvcode=RXSHV_SYSET; /* for Regina (no direct set) use symbolic set */
#else
 shv.shvcode=RXSHV_SET;               /* ... for others use direct set */
#endif
/*---------------------------------------------------------------------*/
/* This calls the RexxVariablePool() function for each value. This is  */
/* not the most efficient way of doing this.                           */
/*---------------------------------------------------------------------*/
 sprintf(variable_name,"%s.%-d",name,suffix);
 (void)make_upper((CHARTYPE *)variable_name);/* make variable name uppercase */
/*---------------------------------------------------------------------*/
/* Now (attempt to) set the REXX variable                              */
/* Add name/value to SHVBLOCK                                          */
/*---------------------------------------------------------------------*/
 MAKERXSTRING(shv.shvname, variable_name, strlen(variable_name));
 MAKERXSTRING(shv.shvvalue,(DEFCHAR *)value,value_length);
/*---------------------------------------------------------------------*/
/* One or both of these is needed, too <sigh>                          */
/*---------------------------------------------------------------------*/
 shv.shvnamelen=strlen(variable_name);
 shv.shvvaluelen=value_length;

#if defined(USE_OS2REXX) || defined(USE_REXX6000)
 rc=(short)RexxVariablePool(&shv);              /* Set the REXX variable */
#else
 rc = RexxVariablePool(&shv);                 /* Set the REXX variable */
/* rc = RXSHV_OK;*/
#endif
 if (rc != RXSHV_OK 
 &&  rc != RXSHV_NEWV)
   {
    display_error(25,(CHARTYPE *)"",FALSE);
    rc = RC_SYSTEM_ERROR;
   }
 else
    rc = RC_OK;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
static RXSTRING *get_rexx_variable
#if defined(HAVE_PROTO)
      (CHARTYPE *name,RXSTRING *value,short suffix)
#else
      (name,value,suffix)
       CHARTYPE *name;
       RXSTRING *value;
       short suffix;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 static SHVBLOCK shv;
 CHAR variable_name[50];
 short rc=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    get_rexx_variable");
#endif
 shv.shvnext=NULL;                                   /* only one block */
#if defined(USE_REGINA)
 shv.shvcode=RXSHV_SYFET; /* for Regina (no direct set) use symbolic set */
#else
 shv.shvcode=RXSHV_FETCH;             /* ... for others use direct set */
#endif
/*---------------------------------------------------------------------*/
/* This calls the RexxVariablePool() function for each value. This is  */
/* not the most efficient way of doing this.                           */
/*---------------------------------------------------------------------*/
 sprintf(variable_name,"%s.%-d",name,suffix);
 (void)make_upper((CHARTYPE *)variable_name);/* make variable name uppercase */
/*---------------------------------------------------------------------*/
/* Now (attempt to) get the REXX variable                              */
/* Set shv.shvvalue to NULL to force interpreter to allocate memory.   */
/*---------------------------------------------------------------------*/
 MAKERXSTRING(shv.shvname, variable_name, strlen(variable_name));
 shv.shvvalue.strptr = NULL;
 shv.shvvalue.strlength = 0;
/*---------------------------------------------------------------------*/
/* One or both of these is needed, too <sigh>                          */
/*---------------------------------------------------------------------*/
 shv.shvnamelen=strlen(variable_name);
 shv.shvvaluelen=0;
#if defined(USE_OS2REXX) || defined(USE_REXX6000)
 rc=(short)RexxVariablePool(&shv);              /* Set the REXX variable */
#else
 rc = RexxVariablePool(&shv);                 /* Set the REXX variable */
/* rc = RXSHV_OK;*/
#endif
 if (rc == RXSHV_OK)
   {
#ifdef USE_REXX6000
    value->strptr = (PUCHAR)(*the_malloc)((sizeof(char)*shv.shvvalue.strlength)+1);
#else
    value->strptr = (char *)(*the_malloc)((sizeof(char)*shv.shvvalue.strlength)+1);
#endif
    if (value->strptr != NULL)
      {
       value->strlength = shv.shvvalue.strlength;
       memcpy(value->strptr,shv.shvvalue.strptr,value->strlength);
       *(value->strptr+value->strlength) = '\0';
      }
#if defined(__IBMC__)
    DosFreeMem(shv.shvvalue.strptr);
#else
    (*the_free)(shv.shvvalue.strptr);
#endif
   }
#ifdef TRACE
 trace_return();
#endif
 return(value);
}
/***********************************************************************/
static short valid_target_function
#if defined(HAVE_PROTO)
      (ULONG Argc,RXSTRING Argv[])
#else
     (Argc,Argv)
      ULONG Argc;
      RXSTRING Argv[];
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_prefix_macro;
 extern LINETYPE prefix_current_line;
 extern VALUE item_values[MAX_VARIABLES_RETURNED];
 extern CHARTYPE *target_buffer;
 extern short target_buffer_len;
/*--------------------------- local data ------------------------------*/
 static TARGET target={NULL,0L,0L,NULL,0,0,FALSE};
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 LINETYPE true_line=0L;
 short rc=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    valid_target_function");
#endif
 switch(1)
   {
    case 1:
         if (Argc < 1 || Argc > 2)  /* incorrect no of arguments - error */
           {
            item_values[1].value = (CHARTYPE *)"ERROR";
            item_values[1].len = 5;
            break;
           }
         if (Argc == 2)
            target_type = target_type | TARGET_SPARE;
/* allocate sufficient space for the spare string and 2 longs */
         if (target_buffer == NULL)
           {
            target_buffer = (CHARTYPE *)(*the_malloc)(sizeof(CHARTYPE)*(Argv[0].strlength+30));
            target_buffer_len = Argv[0].strlength+30;
           }
         else
           {
            if (target_buffer_len < Argv[0].strlength+30)
              {
               target_buffer = (CHARTYPE *)(*the_realloc)(target_buffer,sizeof(CHARTYPE)*(Argv[0].strlength+30));
               target_buffer_len = Argv[0].strlength+30;
              }
           }
         if (target_buffer == (CHARTYPE *)NULL)
           {
            item_values[1].value = (CHARTYPE *)"ERROR";
            item_values[1].len = 5;
            free_target(&target);
            break;
           }
         memcpy(target_buffer,Argv[0].strptr,Argv[0].strlength);
         *(target_buffer+Argv[0].strlength) = '\0';
         if (in_prefix_macro)
            true_line = prefix_current_line;
         else
            true_line = get_true_line();

         initialise_target(&target);
         rc = validate_target(target_buffer,&target,target_type,true_line,FALSE,FALSE);
         if (rc == RC_TARGET_NOT_FOUND)
           {
            item_values[1].value = (CHARTYPE *)"NOTFOUND";
            item_values[1].len = 8;
            free_target(&target);
            break;
           }
         if (rc != RC_OK)
           {
            item_values[1].value = (CHARTYPE *)"ERROR";
            item_values[1].len = 5;
            free_target(&target);
            break;
           }
         if (Argc == 2
         &&  target.spare != (-1))
            sprintf((DEFCHAR *)target_buffer,"%ld %ld %s",target.true_line,target.num_lines,
                                        target.rt[target.spare].string);
         else
             sprintf((DEFCHAR *)target_buffer,"%ld %ld",target.true_line,target.num_lines);
         item_values[1].value = target_buffer;
         item_values[1].len = strlen((DEFCHAR *)target_buffer);
         free_target(&target);
         break;
   }
 item_values[0].value = (CHARTYPE *)"1";
 item_values[0].len = 1;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
static short run_os_function
#if defined(HAVE_PROTO)
      (ULONG Argc,RXSTRING Argv[])
#else
     (Argc,Argv)
      ULONG Argc;
      RXSTRING Argv[];
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[MAX_VARIABLES_RETURNED];
/*--------------------------- local data ------------------------------*/
 int rc=0;
 static CHARTYPE num0[5];                  /* large enough for 1000+rc */
 CHARTYPE *cmd=NULL,*instem=NULL,*outstem=NULL,*errstem=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    run_os_function");
#endif
 switch(Argc)
   {
    case 0:
         sprintf((DEFCHAR *)num0,"%d",RC_INVALID_OPERAND+1000);
         break;
    case 4:
         if ((errstem = (CHARTYPE *)MakeAscii(&Argv[3])) == NULL)
           {
            sprintf((DEFCHAR *)num0,"%d",RC_OUT_OF_MEMORY+1000);
            break;
           }
    case 3:
         if ((outstem = (CHARTYPE *)MakeAscii(&Argv[2])) == NULL)
           {
            sprintf((DEFCHAR *)num0,"%d",RC_OUT_OF_MEMORY+1000);
            break;
           }
    case 2:
         if ((instem = (CHARTYPE *)MakeAscii(&Argv[1])) == NULL)
           {
            sprintf((DEFCHAR *)num0,"%d",RC_OUT_OF_MEMORY+1000);
            break;
           }
    case 1:
         if ((cmd = (CHARTYPE *)MakeAscii(&Argv[0])) == NULL)
           {
            sprintf((DEFCHAR *)num0,"%d",RC_OUT_OF_MEMORY+1000);
            break;
           }
         rc = run_os_command(cmd,instem,outstem,errstem);
         sprintf((DEFCHAR *)num0,"%d",rc);
         break;
    default:
         sprintf((DEFCHAR *)num0,"%d",RC_INVALID_OPERAND+1000);
         break;
   }
 item_values[1].value = num0;
 item_values[1].len = strlen((DEFCHAR *)num0);
 item_values[0].value = (CHARTYPE *)"1";
 item_values[0].len = 1;
 if (cmd) (*the_free)(cmd);
 if (instem) (*the_free)(instem);
 if (outstem) (*the_free)(outstem);
 if (errstem) (*the_free)(errstem);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
static int run_os_command
#if defined(HAVE_PROTO)
      (CHARTYPE *cmd,CHARTYPE *instem,CHARTYPE *outstem,CHARTYPE *errstem)
#else
      (cmd,instem,outstem,errstem)
       CHARTYPE *cmd;
       CHARTYPE *instem;
       CHARTYPE *outstem;
       CHARTYPE *errstem;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 RXSTRING tmpstr;
 bool in=TRUE,out=TRUE,err=TRUE;
 bool out_and_err_same=FALSE;
 int inlen=0,outlen=0,errlen=0;
 int i=0;
 FILE *infp=NULL;
 char *infile="",*outfile="",*errfile="";
 long innum=0L;
 int rc=0,rcode=0;
 int save_stdin=0,save_stdout=0,save_stderr=0;
 int infd=0,outfd=0,errfd=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    run_os_command");
#endif
/*---------------------------------------------------------------------*/
/* Determine if we are redirecting stdin, stdout or both and if the    */
/* values passed as stem variables end in '.'.                         */
/*---------------------------------------------------------------------*/
 if (instem == NULL
 ||  strcmp((DEFCHAR *)instem,"") == 0)
    in = FALSE;
 else
   {
    inlen = strlen((DEFCHAR *)instem);
    if (*(instem+inlen-1) == '.')
       *(instem+inlen-1) = '\0';
    else
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND+1000);
      }
   }
 if (outstem == NULL
 ||  strcmp((DEFCHAR *)outstem,"") == 0)
    out = FALSE;
 else
   {
    outlen = strlen((DEFCHAR *)outstem);
    if (*(outstem+outlen-1) == '.')
       *(outstem+outlen-1) = '\0';
    else
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND+1000);
      }
   }
 if (errstem == NULL
 ||  strcmp((DEFCHAR *)errstem,"") == 0)
    err = FALSE;
 else
   {
    errlen = strlen((DEFCHAR *)errstem);
    if (*(errstem+errlen-1) == '.')
       *(errstem+errlen-1) = '\0';
    else
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND+1000);
      }
   }
/*---------------------------------------------------------------------*/
/* Ensure that stdin stem is different to both stdout and stderr stems.*/
/*---------------------------------------------------------------------*/
 if (in)
   {
    if (out
    &&  strcmp((DEFCHAR *)instem,(DEFCHAR *)outstem) == 0)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND+1000);
      }
    if (err
    &&  strcmp((DEFCHAR *)instem,(DEFCHAR *)errstem) == 0)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND+1000);
      }
   }
/*---------------------------------------------------------------------*/
/* An extra check here to determine if stdout and stderr are to be     */
/* redirected to the same place.                                       */
/*---------------------------------------------------------------------*/
 if (out && err)
   {
    if (strcmp((DEFCHAR *)outstem,(DEFCHAR *)errstem)==0)
       out_and_err_same = TRUE;
   }
/*---------------------------------------------------------------------*/
/* If redirecting stdin, get the value of instem.0 to determine how    */
/* many variables to get...                                            */
/*---------------------------------------------------------------------*/
 if (in)
   {
    tmpstr.strptr = NULL;
    (void)get_rexx_variable(instem,&tmpstr,0);
    if (tmpstr.strptr == NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_SYSTEM_ERROR+1000);
      }
    if (!valid_positive_integer((CHARTYPE *)tmpstr.strptr))
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND+1000);
      }
    innum = atol((DEFCHAR *)tmpstr.strptr);
    (*the_free)(tmpstr.strptr);
/*---------------------------------------------------------------------*/
/* Write the contents of the stdin stem to a temporary file...         */
/*---------------------------------------------------------------------*/
    infile = (char *)(*the_malloc)(L_tmpnam);
    if (infile == NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY+1000);
      }
    if ((infile = tmpnam(infile)) == NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_ACCESS_DENIED+1000);
      }
#if defined(GO32) || defined(EMX)
/*---------------------------------------------------------------------*/
/* For djgpp and emx convert all \ to / for internal file handling     */
/* functions.                                                          */
/*---------------------------------------------------------------------*/
    strtrans(infile,'\\','/');
#endif
    if ((infp = fopen(infile,"w")) == NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_ACCESS_DENIED+1000);
      }
    for (i=0;i<innum;i++)
      {
       tmpstr.strptr = NULL;
       (void)get_rexx_variable(instem,&tmpstr,i+1);
       if (tmpstr.strptr == NULL)
         {
#ifdef TRACE
          trace_return();
#endif
          return(RC_SYSTEM_ERROR+1000);
         }
       fputs((DEFCHAR *)tmpstr.strptr,infp);
       fputs("\n",infp);
       (*the_free)(tmpstr.strptr);
      }
    if (fclose(infp))
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_ACCESS_DENIED+1000);
      }
   }
/*---------------------------------------------------------------------*/
/* If redirecting stdout, create the name of a temporary file for the  */
/* output.                                                             */
/*---------------------------------------------------------------------*/
 if (out)
   {
    outfile = (char *)(*the_malloc)(L_tmpnam);
    if (outfile == NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY+1000);
      }
    if ((outfile = tmpnam(outfile)) == NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_ACCESS_DENIED+1000);
      }
   }
/*---------------------------------------------------------------------*/
/* If redirecting stderr, create the name of a temporary file for the  */
/* output.                                                             */
/*---------------------------------------------------------------------*/
 if (err)
   {
    if (out_and_err_same)
      {
       errfile = outfile;
      }
    else
      {
       errfile = (char *)(*the_malloc)(L_tmpnam);
       if (errfile == NULL)
         {
#ifdef TRACE
          trace_return();
#endif
          return(RC_OUT_OF_MEMORY+1000);
         }
       if ((errfile = tmpnam(errfile)) == NULL)
         {
#ifdef TRACE
          trace_return();
#endif
          return(RC_ACCESS_DENIED+1000);
         }
      }
   }
/*---------------------------------------------------------------------*/
/* Save file ids for stdin, stdout and stderr, then reassign them to   */
/* the appropriate temporary files.                                    */
/*---------------------------------------------------------------------*/
 if (in) save_stdin = dup(fileno(stdin));
 if (out) save_stdout = dup(fileno(stdout));
 if (err) save_stderr = dup(fileno(stderr));
 if (in)
   {
    if ((infd = open(infile,O_RDONLY)) == (-1))
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_ACCESS_DENIED+1000);
      }
   }
 if (out)
   {
#if defined(UNIX)
    if ((outfd = open(outfile,O_RDWR|O_CREAT|O_TRUNC)) == (-1))
#else
    if ((outfd = open(outfile,O_RDWR|O_CREAT|O_TRUNC,S_IWRITE|S_IREAD)) == (-1))
#endif
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_ACCESS_DENIED+1000);
      }
   }
 if (out_and_err_same)
    errfd = outfd;
 else
   {
    if (err)
      {
#if defined(UNIX)
       if ((errfd = open(errfile,O_RDWR|O_CREAT|O_TRUNC)) == (-1))
#else
       if ((errfd = open(errfile,O_RDWR|O_CREAT|O_TRUNC,S_IWRITE|S_IREAD)) == (-1))
#endif
         {
#ifdef TRACE
          trace_return();
#endif
          return(RC_ACCESS_DENIED+1000);
         }
      }
   }
#if defined(UNIX)
  if (out)
     chmod(outfile,S_IWUSR|S_IRUSR);
  if (!out_and_err_same)
     if (err)
        chmod(errfile,S_IWUSR|S_IRUSR);
#endif
 if (in)  dup2(infd,fileno(stdin));
 if (out) dup2(outfd,fileno(stdout));
 if (err) dup2(errfd,fileno(stderr));
 if (in)  close(infd);
 if (out) close(outfd);
 if (!out_and_err_same)
    if (err) close(errfd);
/*---------------------------------------------------------------------*/
/* Execute the OS command supplied.                                    */
/*---------------------------------------------------------------------*/
 rcode = system((DEFCHAR *)cmd);
/*---------------------------------------------------------------------*/
/* Put all file ids back the way they were...                          */
/*---------------------------------------------------------------------*/
 if (in)  dup2(save_stdin,fileno(stdin));
 if (out) dup2(save_stdout,fileno(stdout));
 if (err) dup2(save_stderr,fileno(stderr));
 if (in)  close(save_stdin);
 if (out) close(save_stdout);
 if (err) close(save_stderr);
/*---------------------------------------------------------------------*/
/* If redirecting stdout, we now have to read the file and set a REXX  */
/* variable for each line read.                                        */
/*---------------------------------------------------------------------*/
 if (out)
    rc = set_rexx_variables_from_file(outfile,outstem);
 if (err)
   {
    if (!out_and_err_same)
       rc = set_rexx_variables_from_file(errfile,errstem);
   }
/*---------------------------------------------------------------------*/
/* Delete the temporary file(s) and free up any memory.                */
/*---------------------------------------------------------------------*/
 if (in)
   {
    unlink(infile);
    (*the_free)(infile);
   }
 if (out)
   {
    unlink(outfile);
    (*the_free)(outfile);
   }
 if (err)
   {
    if (!out_and_err_same)
      {
       unlink(errfile);
       (*the_free)(errfile);
      }
   }
/*---------------------------------------------------------------------*/
/* Return with, hopefully, return code from system() command.          */
/*---------------------------------------------------------------------*/
#ifdef TRACE
 trace_return();
#endif
 if (rc)
    return(rc+1000);
 else
    return(rcode);
}
/***********************************************************************/
static CHARTYPE *MakeAscii
#if defined(HAVE_PROTO)
      (RXSTRING *rxstring)
#else
     (rxstring)
      RXSTRING *rxstring;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 CHARTYPE *string=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    MakeAscii");
#endif

 string = (CHARTYPE *)(*the_malloc)((sizeof(CHARTYPE)*rxstring->strlength)+1);
 if (string != NULL)
   {
    memcpy(string,rxstring->strptr,rxstring->strlength);
    *(string+(rxstring->strlength)) = '\0';
   }
#ifdef TRACE
 trace_return();
#endif
 return(string);
}
/***********************************************************************/
static char *get_a_line
#if defined(HAVE_PROTO)
      (FILE *fp,char *string,int *length,int *rcode)
#else
      (fp,string,length,rcode)
      FILE *fp;
      char *string;
      int *length;
      int *rcode;
#endif
/***********************************************************************/
{
 int ch;
 static int bufs = 1;
 register int i=0;
/*---------------------------------------------------------------------*/
/* Allocate the first block of memory.                                 */
/*---------------------------------------------------------------------*/
 if ((string = (char *)(*the_malloc)(BUF_SIZE+1)) == NULL)
   {
    *rcode = RC_OUT_OF_MEMORY;
    return(NULL);
   }
 while(1)
   {
/*---------------------------------------------------------------------*/
/* Read a character from the stream...                                 */
/*---------------------------------------------------------------------*/
    if ((ch = fgetc(fp)) == EOF)
      {
/*---------------------------------------------------------------------*/
/* If EOF is reached, check that it really is end of file.             */
/*---------------------------------------------------------------------*/
       if (feof(fp))
         {
          *length = i;
          *rcode = RC_TOF_EOF_REACHED;
          return(string);
         }
      }
/*---------------------------------------------------------------------*/
/* If end of line is reached, nul terminate string and return.         */
/*---------------------------------------------------------------------*/
    if ((char)ch == '\n')
      {
       *(string+i) = '\0';
       break;
      }
/*---------------------------------------------------------------------*/
/* All other characters, copy to string.                               */
/*---------------------------------------------------------------------*/
    *(string+i++) = (char)ch;
/*---------------------------------------------------------------------*/
/* If we have got to the end of the allocated memory, realloc some more*/
/*---------------------------------------------------------------------*/
    if (i == BUF_SIZE*bufs)
      {
       if ((string = (char *)realloc(string,(BUF_SIZE*(++bufs))+1)) == NULL)
         {
          *rcode = RC_OUT_OF_MEMORY;
          return(NULL);
         }
      }
   }
/*---------------------------------------------------------------------*/
/* Return a line read from the temporary file.                         */
/*---------------------------------------------------------------------*/
 *length = i;
 *rcode = 0;
 return(string);
}
/***********************************************************************/
static short set_rexx_variables_from_file
#if defined(HAVE_PROTO)
      (char *filename,CHARTYPE *stem)
#else
      (filename,stem)
       char *filename;
       CHARTYPE *stem;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 FILE *fp=NULL;
 register int i=0;
 char *string=NULL;
 int length=0,rcode=0,rc=0;
 char tmpnum[10];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    set_rexx_variables_from_file");
#endif
 if ((fp = fopen(filename,"r")) == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_ACCESS_DENIED);
   }
 for (i=0;;i++)
   {
    string = get_a_line(fp,string,&length,&rcode);
    if (rcode == RC_OUT_OF_MEMORY)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rcode);
      }
    if (rcode == RC_TOF_EOF_REACHED
    && length == 0)
      {
       (*the_free)(string);
       break;
      }
    rc = set_rexx_variable(stem,(CHARTYPE *)string,strlen(string),i+1);
    (*the_free)(string);
    if (rcode == RC_TOF_EOF_REACHED)
       break;
   }
 sprintf(tmpnum,"%d",i);
 rc = set_rexx_variable(stem,(CHARTYPE *)tmpnum,strlen(tmpnum),0);
 if (fclose(fp))
    rc = RC_ACCESS_DENIED;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
#else
/*---------------------------------------------------------------------*/
/* The following are dummy routines to enable the compliation of THE   */
/* with a nonANSI compiler.                                            */
/*---------------------------------------------------------------------*/
/***********************************************************************/
short initialise_rexx()
/***********************************************************************/
{
 return(RC_INVALID_ENVIRON);                        /* force an error */
}
/***********************************************************************/
short finalise_rexx()
/***********************************************************************/
{
 return(RC_OK);
}
/***********************************************************************/
short execute_macro_file(filename,params)
CHARTYPE *filename;
CHARTYPE *params;
/***********************************************************************/
{
 return(RC_OK);
}
/***********************************************************************/
short set_rexx_variable(name,value,value_length,suffix)
CHARTYPE *name;
CHARTYPE *value;
short value_length;
short suffix;
/***********************************************************************/
{
 return(RC_OK);
}
#endif
