/*
 *   MDVXHELP - A utility to create DV/X Help Files
 *
 *   Copyright 1992 by Daniel J. Bodoh
 *
 *   File: mdvxhelp.c
 *
 *   Compiled with Turbo C 2.0
 *
 *   History:
 *      28-MAY-92   DJB   Created
 *      29-MAY-92   DJB   Modified to replace single \n\r's with spaces
 *                        BUG FIX: buffer index not rezerod in process_section()
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <dir.h>
#include <string.h>
#include <mem.h>

#include "mdvxhelp.h"


/* Globals */
unsigned long curoff=0;     /* current offset into output file */
Header header={0,0};        /* File header */
Tag *first_tag=NULL;        /* Head of tag list */
Dir_elem *directory;
char outfile[MAXPATH];     /* out file name */
                            /* Head of directory list */
FILE *ofp;                  /* output file */

char ifile[MAXFILES][MAXPATH];      /* input filenames */
int numfiles=0;

int main(int argc, char *argv[]) {
    int i;
    char ext[MAXEXT];
    char tmp[MAXPATH];
    Tag *t;
    int xpm;
    FILE *fp;

    if (argc<3) {
        /* print out help screen */
        printf("MDVXHELP %s Copyright 1992 by Daniel J. Bodoh\n",VERSION);
        printf("\nusage: mdvxhelp <output-file> <section-file ...> [<icon-file ...>]\n");
        printf("     or: mdvxhelp <output-file> @<list-file>\n");

        exit(0);
    }

    strcpy(outfile,argv[1]);
    /* open the output file */
    if ((ofp=fopen(outfile,"wb"))==NULL) {
        fprintf(stderr,"Cannot %s for output\n",argv[1]);
        exit(1);
    }

    if (argv[2][0]=='@') {
        /* list file */
        if ((fp=fopen(argv[2]+1,"r"))==NULL) {
            err_exit("Could not open list file\n");
        }
        i=0;
        while (fgets(tmp,sizeof tmp,fp)!=NULL) {
            sscanf(tmp,"%s",ifile[i]);
            if (i++>=MAXFILES) {
                fprintf(stderr,"Warning: more than %d input files; extra ignored\n",MAXFILES);
                i=MAXFILES;
                break;
            }
        }
        numfiles=i;
        fclose(fp);
    }
    else {
        /* read files from command line */
        numfiles = argc-2;
        if (numfiles>MAXFILES) {
            numfiles=MAXFILES;
            fprintf(stderr,"Warning: more than %d input files; extra ignored\n",MAXFILES);
        }
        for (i=0;i<numfiles;i++) {
            strcpy(ifile[i],argv[i+2]);
        }
    }


    /* create directory array */
    header.dir_size=(numfiles+3)*sizeof(Dir_elem);
    directory = (Dir_elem *)malloc(header.dir_size);
    if (directory==NULL) err_exit("Out of memory");
    memset(directory,0,header.dir_size);

    /* write the header immediately; update later */
    if (fwrite(&header,sizeof header,(size_t)1,ofp)<1) {
        err_exit("Error writing to file");
    }
    curoff += sizeof header;

    xpm=0;
    for (i=0;i<numfiles;i++) {
        /* fill a directory entry */
        /* what's the extension? */
        fnsplit(ifile[i],NULL,NULL,NULL,ext);
        strupr(ext);
        if (strcmp(ext,".XPM")==0) {
            if (!xpm) {
                /* before the icons, write out the tags */
                /* create BKCOVER dir entry */
                create_dir_entry("BKCOVER",curoff,i+1);
                /* write out tags */
                fputs("/CCblue/\n\r",ofp);
                curoff += 10;
                t = first_tag;
                while (t!=NULL) {
                    fputs(t->tagstr,ofp);
                    curoff += strlen(t->tagstr);
                    t=t->flink;
                }
                fputc(0x1A,ofp);
                curoff += 1;
                xpm=1;
            }
            create_dir_entry(ifile[i],curoff,i+2);
            process_xpm(ifile[i]);
        }
        else {
            create_dir_entry(ifile[i],curoff,i+1);
            process_section(ifile[i]);
        }
    }

    /* put file together */
    if (!xpm) {
        /* if no icons were written, we have to write the tags */
        /* create BKCOVER dir entry */
        create_dir_entry("BKCOVER",curoff,numfiles+1);
        /* write out tags */
        fputs("/CCblue/\r\n",ofp);
        curoff += 10;
        t = first_tag;
        while (t!=NULL) {
            fputs(t->tagstr,ofp);
            curoff += strlen(t->tagstr);
            t=t->flink;
        }
        fputc(0x1A,ofp);
        curoff += 1;
    }

    header.dir_off=curoff;

    create_dir_entry(NULL,curoff,0);
    /* write the directory */
    if (fwrite(directory,header.dir_size,1,ofp)<1) {
        err_exit("Error writing to file");
    }
    rewind(ofp);
    /* update header */
    if (fwrite(&header,sizeof header,(size_t)1,ofp)<1) {
        err_exit("Error writing to file");
    }


    fclose(ofp);
    exit(0);
}

void err_exit(char *msg) {
    /* Writes msg to stderr; deletes output file; exits */
    fprintf(stderr,"%s\n",msg);
    fclose(ofp);
    unlink(outfile);
    exit(1);
}

void process_section(char *file) {
    /* Reads in a section file; processes it and writes to outfile */
    FILE *fp;
    char msg[80];
    static char ibuf[BUFFER_SIZE];
    static char obuf[BUFFER_SIZE];
    size_t cnt;
    long i, j;
    char h[2];

    if ((fp=fopen(file,"rb"))==NULL) {
        sprintf(msg,"Cannot open %s",file);
        err_exit(msg);
    }

    printf("Adding section %s...",file);

    h[0]=fgetc(fp);
    if (h[0]=='$') {
        /* the first and second lines contains a tag.  Let's read it in */
        fgets(ibuf,sizeof ibuf,fp);
        fgets(obuf,sizeof obuf,fp);
        strcat(ibuf,obuf);
        create_tag(ibuf);
        printf("Tag attached...\n");
    }
    else {
        ungetc(h[0],fp);
        printf("\n");
    }

    cnt = fread(ibuf, 1, sizeof ibuf, fp);

    h[0]=h[1]=0;

    while (cnt!=0) {
        j=0;
        for (i=0;i<cnt;i++) {
            /* skip single \r\n's but don't skip double...*/
            if (ibuf[i]=='\r' || ibuf[i]=='\n') {
                if ((h[0]=='\r' && h[1]=='\n')||(h[0]=='\n' && h[1]=='\r')) {
                    obuf[j++]='\n';
                    obuf[j++]='\r';
                    i++;    /* jump over next char, assume is \r or \n */
                    h[0]=h[1]=0;
                }
                else {
                    h[0]=h[1];
                    h[1]=ibuf[i];
                }
            }
            else {
                /* replace \n\r with a space */
                if ((h[0]=='\r' && h[1]=='\n')||(h[0]=='\n' && h[1]=='\r')) {
                    obuf[j++]=' ';
                }
                h[0]=h[1];
                h[1]=ibuf[i];
                obuf[j++]=ibuf[i];
            }

        }
        if (fwrite(obuf,1,j,ofp)<j) {
            err_exit("Error writing to file");
        }
        curoff += j;
        cnt = fread(ibuf,1, sizeof ibuf, fp);
    }
    fputc(0x1A,ofp); /* section separator */
    curoff += 1;
    fclose(fp);
}

void process_xpm(char *file) {
    /* Reads in an XPM and writes to outfile */
    FILE *fp;
    char msg[80];
    static char ibuf[BUFFER_SIZE];
    size_t cnt;

    if ((fp=fopen(file,"rb"))==NULL) {
        sprintf(msg,"Cannot open %s",file);
        err_exit(msg);
    }

    printf("Adding icon %s...\n",file);

    cnt = fread(ibuf, 1, sizeof ibuf, fp);

    while (cnt!=0) {
        if (fwrite(ibuf,1,cnt,ofp)<cnt) {
            err_exit("Error writing to file");
        }
        curoff += cnt;
        cnt = fread(ibuf, 1, sizeof ibuf, fp);
    }
    fclose(fp);
}

void create_dir_entry(char *path, long offset, int i) {
    char file[MAXFILE], ext[MAXEXT];

    fnsplit(path,NULL,NULL,file,ext);
    switch(i) {
        case 0:
            /* this is directory */
            strcpy(directory[i].sect_name,"directory");
            break;
        case 1:
            /* this is FRCOVER */
            strcpy(directory[i].sect_name,"FRCOVER");
            break;
        case 2:
            strcpy(directory[i].sect_name,"CONTENTS");
            break;
        default:
            strcpy(directory[i].sect_name,file);
            strcat(directory[i].sect_name,ext);
            break;
    }

    directory[i].offset=offset;
}

void create_tag(char *tagtxt) {
    Tag *p;

    /* traverse the current list */
    p = first_tag;

    if (p!=NULL) {
        while (p->flink!=NULL) p=p->flink;
        if ((p->flink=(Tag *)malloc(sizeof(Dir_elem)))==NULL) {
            err_exit("Out of memory");
        }
        p=p->flink;
    }
    else {
        if ((first_tag=(Tag *)malloc(sizeof(Dir_elem)))==NULL) {
            err_exit("Out of memory");
        }
        p=first_tag;
    }
    if ((p->tagstr=(char *)malloc(strlen(tagtxt)+1))==NULL) {
        err_exit("Out of memory");
    }
    strcpy(p->tagstr,tagtxt);
    p->flink=NULL;
}
