/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * gencode.c
 *
 * codegen basics 
 */
#include <stdio.h>                     
#include <memory.h>
#include <string.h>
#include "utype.h"
#include "umem.h"
#include "asm.h"
#include "input.h"
#include "gencode.h"
#include "extern.h"
#include "sections.h"
#include "macros.h"
#include "expr.h"

extern int pass;
extern HASHREC **hashtable;
extern FILE *listfile;
extern int level;
extern int lineno[];
extern int macrolevel;
extern int prm_procrevision;
extern BOOL ifskip;
extern uint allocsize;
extern int sectionnum;

extern long basealign ;
extern int usedrevision ;
extern LIST *undeflist;
extern int opcodesize;
extern BYTE pcbuf[BUFLEN];
extern BYTE *outputbuf;
extern long org;
extern void (* optable[])(); /* NOPROTOTYPE */
extern int curfile;
extern int prm_debug;
extern int level;
extern int lineno[];
extern LIST *incfiles;
extern EXPRESSION *lnsect;
extern char *curname;
extern SECTIONREC *cursection;
extern linename[];

EXPRESSION *entrypoint;
char filesused[256];
static int lastfixline = 0,lastfixfile = 0;

void gencodeinit(void)
{
	int i;
	for (i=0; i <256; i++)
		filesused[i] = 0;
	entrypoint = 0;
}
void UndefError(char *name , BOOL flag)
{
	char buf[BUFLEN];
	sprintf(buf,"Undefined Symbol %s",name);
	Error(buf);
}
void badsize(void)
{
  Error("Specified size is bad");
}
void badexpression(void)
{
	Error("Undefinable expression");
}
void unimplemented(void)
{
  Error("Unimplemented opcode");
}
void invproc(void)
{
  Error("Invalid for this processor type");
}
void validproc(int level)
{
  if (level > usedrevision)
 	  usedrevision = level;
}
void proctype(int level)
{
  if (level > prm_procrevision)
		invproc();
	else
		validproc(level);
}

void missingop(void)
{
	Error("Missing Operand");
}
void illop(void)
{
	Error("Bad operand");
}
void extraop(void)
{
	Error("Extra operand");
}
void badbyte(int size)
{
  if (size == TBYTE)
		badsize();
}
void badfloat(int size)
{
	if (size  && size != TBYTE && size != TWORD && size != TLONG)
		badsize();
}
void constexpect(void)
{
  Error("Constant expected");
}
void badconst(void)
{
  Error("Bad constant size");
}
void brarange(void)
{
  Error("Relative branch out of range");
}
/* see if is numeric (possibly global) */
BOOL numericExpression(EXPRESSION *exp)
{
  if (exp->ismacro || exp->isopcode || exp->isoperand || exp->ismovemreg || exp->isfp) {
		Error("Invalid type in expression");
		return(FALSE);
	}
	return(TRUE);
}
/* gen fixup data */
void linkadr(EXPRESSION*data, int size, BOOL rel)
{
	long vsize=size;
	if (data->swapper != 0xffff)
		vsize = ((vsize << 16) + (data->swapper & 0xffff));
  if (!data->isdef && !data->name) {
		badexpression();
	}
	else
		if (pass == 1) {
			if (!data->isdef) {
				LDATA *ldata = AllocateMemory(sizeof(LDATA));
				ldata->size = size;
				ldata->name = AllocateMemory(strlen(data->name)+1);
				ldata->relofs = org;
				strcpy(ldata->name, data->name);
				AppendToList(&undeflist, ldata);
			}
		}
		else {
			if ((data->section == sectionnum) && !rel && !data->relmode && !data->relpc)
				registersamefixup(data,vsize);
			else
				registerfixup(data,vsize, rel);
	  }
}
/* these next few routines get overriden a lot */
/* put 16 bits, no fixup */
void putuword(unsigned val)
{
  pcbuf[opcodesize] = (BYTE)(val >> 8);
  pcbuf[opcodesize+1] = (BYTE)(val & 255);
	opcodesize+=2;
}
/* put 32 bits, with fixup */
void putdword(EXPRESSION *data, BOOL rel)
{
	long temp = doswap(data->swapper, data->x.value);
	if (data->isextern || data->relmode)
		temp = 0;
	if (data->size && (data->size != TLONG))
		badsize();
  pcbuf[opcodesize+0] = (BYTE)(temp >> 24);
  pcbuf[opcodesize+1] = (BYTE)((temp >>16) & 255);
  pcbuf[opcodesize+2] = (BYTE)((temp >> 8) & 255);
  pcbuf[opcodesize+3] = (BYTE)(temp & 255);
	linkadr(data,TLONG,rel);
	opcodesize+=4;
}
/* put 16 bits, with fixups */
void putword(EXPRESSION *data,BOOL rel)
{
	long temp = doswap(data->swapper, data->x.value);
	if (data->isextern || data->relmode)
		temp = 0;
	if (data->size && (data->size != TWORD))
		badsize();
  pcbuf[opcodesize] = (BYTE)(temp >> 8);
  pcbuf[opcodesize+1] = (BYTE)(temp & 255);
	linkadr(data,TWORD,rel);
	opcodesize+=2;
}
/* put a byte, with fixups */
void putbyte(EXPRESSION *data,BOOL rel)
{
	long temp = doswap(data->swapper, data->x.value);
	if (data->isextern || data->relmode)
		temp = 0;
	pcbuf[opcodesize] = 0;
  pcbuf[opcodesize+1] = (BYTE) temp;
  linkadr(data,TBYTE,rel);
	opcodesize+=2;
}
/* test for word value */
BOOL isword(EXPRESSION *data)
{
  long addr= data->x.value;
  if (data->size == TWORD)
    return(TRUE);
	if (data->isfp) {
		IllegalFloat();
		return(TRUE);
	}
	if (data->swapper && ((data->swapper & 0xff00) == 0xff00))
		return(TRUE);
  if (data->isdef && !data->size && (!data->section ||sectionnum == data->section))
  	return ((addr == (addr & 0x7fff)) || ((addr&0xffff8000L) == 0x8000)||((addr&0xffff8000L) == 0xffff8000L));
	return(FALSE);
}
/* test for byte value */
BOOL isbyte(EXPRESSION *data)
{
  long addr= data->x.value;
  if (data->size == TBYTE)
    return(TRUE);
	if (data->isfp) {
		IllegalFloat();
		return(TRUE);
	}
	if (data->swapper && ((data->swapper & 0xfff0) == 0xfff0))
		return(TRUE);
  if (data->isdef && !data->size && (!data->islabel || !data->section ||sectionnum == data->section))
  	return ((addr == (addr & 0x7f)) || ((addr&0xffffff80L) == 0x80)|| ((addr&0xffffff80L) == 0xff80)||((addr&0xffffff80L) == 0xffffff80L));
	return(FALSE);
}
/* test for signed word value */
BOOL iswordsigned(EXPRESSION *data)
{
  long addr= data->x.value;
  if (data->size == TWORD)
    return(TRUE);
	if (data->isfp) {
		IllegalFloat();
		return(TRUE);
	}
  if (data->isdef && !data->size && (!data->islabel || !data->section ||sectionnum == data->section))
  	return ((addr == (addr & 0x7fff)) || ((addr&0xffff8000L) == 0xffff8000L));
	return(FALSE);
}
/* test for byte signed value */
BOOL isbytesigned(EXPRESSION *data)
{
  long addr= data->x.value;
  if (data->size == TBYTE)
    return(TRUE);
	if (data->isfp) {
		IllegalFloat();
		return(TRUE);
	}
  if (data->isdef && !data->size && (!data->section ||sectionnum == data->section))
  	return ((addr == (addr & 0x7f)) || ((addr&0xffffff80L) == 0xffffff80L));
	return(FALSE);
}
/* parser makes an operand with this */
OPERANDDATA *makeoperand(int areg, int type, EXPRESSION *basedisp, BASE *basereg, EXPRESSION * outerdisp)
{
  OPERANDDATA *rv = AllocateMemory(sizeof(OPERANDDATA));
  rv->type = type;
  rv->reg = areg;
  rv->basereg = basereg;
  rv->basedisp = basedisp;
  rv->outerdisp = outerdisp;
	return(rv);
}
/* 680x0 base value */	
BASE *makebase(int reg, int adr_data, int type, int scale)
{
  BASE *rv = AllocateMemory(sizeof(BASE));
  rv->reg = reg;
  rv->adrtype = adr_data;
	rv->type = type;
  rv->size = scale;
	return(rv);
}
/* basic routines */
void setorg(long pos)
{
	/* obsolete */
  org = pos;
}
void doend(EXPRESSION *pos)
{
	while (macrolevel)
		CloseMacro();
	while(level >= 0)
		closeFile();
	
	if (pos &&!pos->islabel)
		Error("Invalid entry point");

  entrypoint = pos;
}
long currentorg(void)
{
  return(org);
}

/* delete an expression expressions */
void delexpr(EXPRESSION *exp1, EXPRESSION *exp2)
{
  if (exp1 && exp1->isintermed) {
		if (exp1->name)
			DeallocateMemory(exp1->name);
		DeallocateMemory(exp1);
	}
  if (exp2 && exp2->isintermed) {
		if (exp2->name)
			DeallocateMemory(exp2->name);
		DeallocateMemory(exp2);
	}
}
/* validate expressions */
BOOL checkAllExpr(OPERANDDATA *op)
{
	BOOL rv = TRUE;
	if (op) {
		if (op->basedisp)
			rv = rv && (op->basedisp->isfp || numericExpression(op->basedisp) );
		if (op->outerdisp)
			rv = rv && (numericExpression(op->outerdisp));
	}
	return(rv);
}
/*
 * delete an operand
 */
void deloperand(OPERANDDATA *op)
{
  if (!op)
		return;
	delexpr(op->basedisp,op->outerdisp);
	if (op->basereg)
		DeallocateMemory(op->basereg);
	DeallocateMemory(op);
}
/* align keyword */
void xalign(EXPRESSION *exp) 
{
	if (exp->isdef && numericExpression(exp) && outputbuf && exp->x.value >= 0) {
		if (exp->x.value >1) {
			long t = org % exp->x.value;
		
			if (t)
				for (t = exp->x.value - t; t>0; t--)
					outputbuf[org++] = 0;
			if ((exp->x.value > basealign && exp->x.value % basealign)
					|| (exp->x.value < basealign && basealign % exp->x.value))
				basealign *= exp->x.value;
			else
				if (basealign < exp->x.value)
					basealign = exp->x.value;
		}
	}
	else
		Error("Invalid align statement");
}
/* move the genned code to secion memory and output lineno info */
void copytomem(void)
{
	LIST *i = incfiles;
	/* output lineno info */
	if (prm_debug && pass ==  2 && cursection) {
		int newsection = sectionnum;
		int neworg = org;
		SECTIONREC *lastsect = cursection;
		switchnamedsection(linename);
		if (opcodesize && (lastfixline != lineno[level] || lastfixfile != curfile)) {
			lastfixline = lineno[level];
			lastfixfile = curfile;
			filesused[curfile] = 1;
			if (org+7 > allocsize)
				sectionalloc();
			outputbuf[org] = curfile;
			*((short*)&outputbuf[org+1]) = lineno[level];
			*((long*)&outputbuf[org+3]) = 0;
			org += 7;
			lnfixup(neworg,org-4,newsection);
		}
		cursection->highest = org;
		switchnamedsection(lastsect->name);
	}

	/* move the data */
  if (!outputbuf) {
		Error("Not in a region, can't emit code");
	}
	else {
		if (!ifskip) {
			if (org +opcodesize > allocsize)
				sectionalloc();
			memcpy(&outputbuf[org],pcbuf,opcodesize);
			org += opcodesize;
		}
	}
	opcodesize = 0;
}