/*                               fixclock.c
 *
 * This program corrects the system time upon boot up. The system time is
 * normally obtained from the cmos clock. This clock is crystal controlled
 * but may exhibit an error of several minutes per month anyway. A knowledge
 * of this error allows it to be compensated for and corrected.
 *
 * The program is designed to be called from the autoexec.bat file upon boot
 * up. An example of the proper entry in the .bat file is as follows:
 *
 *          fixclock tz=cst6cdt 23:13:45 10-25-88 -213 L
 *
 * The first parameter sets the timezone. It consists of tz= followed by a
 * three letter abbreviation of the standard timezone: pst, mst, cst, or est.
 * This is followed by a number indicating the number of hours between the
 * local time and GMT, followed by the abbreviation for daylight savings time,
 * pdt, mdt, cdt, or edt if it is observed in your area. If it is not observed
 * then don't append it. For instance, in parts of Indiana you would have a
 * timezone parameter that looks like this: tz=cst6
 *
 * The next two entries are the time and date the cmos clock was properly set.
 * These values are obtained from the clock statistics produced by the program
 * clockerr, when the q parameter is used.
 *
 * The next entry is also obtained from the clock statistics report and is a
 * signed integer representing the number of seconds gained or lost by the
 * cmos clock per month. If the clock looses time then it is negative.
 *
 * The last entry on the command line determines whether the system will
 * display local time or GMT. If you wish local time use an L here. For GMT
 * use a G. If you choose the local option then the system will always display
 * the local time and will correct for daylight savings time when it is in
 * effect without any action on your part.
 */

#include <stdlib.h>
#include <time.h>
#include <stdio.h>
#include <ctype.h>
#include <dos.h>

#define  MONTH  2419200.0                   /* Seconds in a month (28 days) */
#define  LCL    1
#define  GMT    0

void    main(int argc, char *argv[]);
void    set_time(time_t *time, int local);
time_t  clock_started(char *argv[]);

void  main(int argc, char *argv[])
   {
   int      i;
   time_t   error, cmos_time, new_time;
   double   elapsed;

   /*
    * Check for proper number of command line arguments (5)
    */
   if(argc != 6)
       {
       puts("Incorrect Number Of Parameters...");
       exit(0);
       }

   /*
    * The case of the first argument doesn't matter.
    */
   for(i=0; *(argv[1]+i); i++)
      *(argv[1]+i)=toupper( *(argv[1]+i) );

   /*
    * Put the time zone info which is the first argument, into the
    * program environment. Next, set the global timezone variables
    * to the specified values.
    */
   putenv(argv[1]);
   tzset();

   /*
    * time(&cmos_time) returns the time kept by the cmos clock, this is
    * converted to seconds since the epoch. clock_started(argv) returns
    * the time when the cmos clock was set, converted to seconds since
    * the epoch, the time when the cmos clock was set is obtained from
    * entries on the command line. The clock error is also obtained from
    * the command line. new_time represents the corrected time in seconds
    * since the epoch and is used to set the system time.
    */
   elapsed = time(&cmos_time) - clock_started(argv);
   error = atoi(argv[4]);
   new_time = cmos_time - elapsed * error/MONTH;
   switch(toupper(*argv[5]))
      {
      case 'L':
		/*
		 * Sets the correct local time as requested by the "L"
		 * on the command line.
		 */
		set_time(&new_time, LCL);
		break;

      case 'G':
		/*
		 * Sets the correct Greenwich Mean Time by adding the
		 * value of the timezone variable to the corrected time.
		 */
		new_time += timezone;
		set_time(&new_time, GMT);
      }
   exit(0);
   }

time_t clock_started(char *argv[])            /* clock value when first set */
   {

   /*
    * Command line values representing the time at which the cmos clock was
    * set are retrieved and converted to integer values to fill the structure
    * systime. This structure is then passed to mktime() so that the return
    * value is a variable of type time_t.
    */
   struct tm   systime;
   systime.tm_hour  =  atoi(argv[2]);
   systime.tm_min   =  atoi(argv[2]+3);
   systime.tm_sec   =  atoi(argv[2]+6);
   systime.tm_mon   =  atoi(argv[3])-1;    /* MSOFT definition idiosyncracy */
   systime.tm_mday  =  atoi(argv[3]+3);
   systime.tm_year  =  atoi(argv[3]+6);
   systime.tm_isdst =  0;           /* Command line value is standard time. */
   return mktime(&systime);
   }

void  set_time(time_t *time, int local)     /* Set the correct system time. */
   {
   struct tm         *time_date, *dst;
   struct dostime_t  to_dostime;
   struct dosdate_t  to_dosdate;

   /*
    * Convert the correct time to a structure of type tm, then examine
    * dst->is_dst to see if daylight savings time is in effect and observed
    * and to see if local time format is desired. If so then add an hour to
    * the correct time which otherwise is standard time.
    */
   dst = localtime(time);
   if(dst->tm_isdst && daylight && local)
      *time = *time + 3600;                 /* DST and local time in effect */

   /*
    * The corrected time is converted to a structure of type tm. The elements
    * of this structure are used to assign values to the structure to_dostime
    * and to_dosdate. These two structures are then used to set the system
    * date and time.
    */
   time_date = localtime(time);
   to_dostime.hour = time_date->tm_hour;
   to_dostime.minute = time_date->tm_min;
   to_dostime.second = time_date->tm_sec;
   to_dostime.hsecond = 50;                         /* In lieu of rounding. */
   _dos_settime(&to_dostime);
   to_dosdate.day = time_date->tm_mday;
   to_dosdate.month = time_date->tm_mon + 1;
   to_dosdate.year = time_date->tm_year + 1900;
   to_dosdate.dayofweek = time_date->tm_wday;
   _dos_setdate(&to_dosdate);
   }
