
/*============================================================================

	CD2FILE.C
	Creates CDROM-images as DOS-file.
  This file is part of the

  PseudoCD package (version 02)
  Copyright (C) C.Kulms, 1997

  This is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this software; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

============================================================================*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#include <dir.h>
#include <process.h>

#include "imaghead.h"		/* header of image file */
#include "cdrom.h"			/* CDROM related stuff */

const char	_acWarn[] = "\nNOTE THAT DATA ON CD IS SUBJECT TO COPYRIGHT LAW.\n"
	"It is shown whether the data on a particular CD has been flagged as 'digital\n"
	"copy prohibited', by displaying a 'YES' or 'NO' under the banner 'copy'. The\n"
	"appropriate permissions should be sought before you create an image of a CD\n"
	"that contains one or more tracks flagged 'NO'.\n\n";

t_IMAGEHEADER	_tImageHeader;	/* head of image file */
char					_acHeaderFill[HEADERHDSECS * BPHDS - sizeof(t_IMAGEHEADER)];

#define NUMCDSECS	16	/* number of cd-sectors per read */
char _acBuffer[NUMCDSECS * BPCDS];	/* buffer for read and IOCTL-requests */

/*
	GetCD
	Gets basic information about CDROM.
	return: 1, exits if no CDROM in drive
*/
int GetCD( char cCD )
{
int		iIndex,	/* index counter */
			iCDno;	/* drive number of CDROM drive */

	iCDno = cCD - 'A';

	/* check if disk present in drive */
	MSCDEX_IoctlRequest( iCDno, IOCTL_DEVSTAT, _acBuffer );
	if( ((t_DEVSTATUS *)&_acBuffer[1])->uNoDisk )
	{
		fprintf( stderr, "**** ERROR: no CDROM in CDROM-drive %c. ****\n", cCD );
		exit( -1 );
	}

	/* get volume size */
	MSCDEX_IoctlRequest( iCDno, IOCTL_VOLSIZE, _acBuffer );
	_tImageHeader.ulVolSize = *(unsigned long *)&_acBuffer[1];

	/* get track range */
	MSCDEX_IoctlRequest( iCDno, IOCTL_DISKINFO, _acBuffer );
	_tImageHeader.ucLowTrack = _acBuffer[1];
	_tImageHeader.ucHighTrack = _acBuffer[2];
	_tImageHeader.tLeadOut = *(t_REDBOOK *)&_acBuffer[3];

	/* calculate total sizes */
	_tImageHeader.ulTotSize = (unsigned long)
						((unsigned)_tImageHeader.tLeadOut.cMinute * 60
						+ (unsigned)_tImageHeader.tLeadOut.cSecond) * 75
						+ (unsigned long)_tImageHeader.tLeadOut.cFrame - 150;
	_tImageHeader.ulTotHdSecs = _tImageHeader.ulTotSize << SPCDSsh;
	_tImageHeader.ulTotBytes = _tImageHeader.ulTotSize << BPCDSsh;

	/* get track infos */
	for( iIndex = _tImageHeader.ucLowTrack - 1; iIndex < _tImageHeader.ucHighTrack; iIndex++ )
	{
		_acBuffer[1] = (char)iIndex + 1; /* I count them from 0 */
		MSCDEX_IoctlRequest( iCDno, IOCTL_TRACKINFO, _acBuffer );
		_tImageHeader.tTrackInfo[iIndex].tStartAdr = *(t_REDBOOK *)&_acBuffer[2];
		_tImageHeader.tTrackInfo[iIndex].ucCtrl = _acBuffer[6];
	}

	/* report */
	fprintf( stdout, "---- CDROM %c: ----\n", cCD );
	fputs( "track  copy  type   start\n", stdout );
	for( iIndex = _tImageHeader.ucLowTrack - 1; iIndex < _tImageHeader.ucHighTrack; iIndex++ )
	{
		fprintf( stdout, "  %3d  %s   %s  %02d:%02d,%02d\n", iIndex + 1,
							(_tImageHeader.tTrackInfo[iIndex].ucCtrl & 0x20) ? "YES" : "NO ",
							_tImageHeader.tTrackInfo[iIndex].ucCtrl & 0x40 ? "DATA " : "AUDIO",
							_tImageHeader.tTrackInfo[iIndex].tStartAdr.cMinute,
							_tImageHeader.tTrackInfo[iIndex].tStartAdr.cSecond,
							_tImageHeader.tTrackInfo[iIndex].tStartAdr.cFrame );
	}
	fprintf( stdout, "total size: %lu bytes (%lu cd-sectors)\n",
					_tImageHeader.ulTotBytes, _tImageHeader.ulTotSize );

	return( 1 );	
}

/*
	CopyCD
	Writes image-header and 1:1 sector copy of CDROM into file.
	return: 1, exits on error
*/
int CopyCD( char cCD, const char *pcFileName )
{
FILE	*ptFile;					/* target file */
unsigned long	ulSector;	/* sector counter */
int		iCDno;						/* drive number of CDROM drive */

	iCDno = cCD - 'A';
	fprintf( stdout, "source cdrom is %c, target file is '%s'\n\n", cCD, pcFileName );

	/* open file */
	if( !(ptFile = fopen( pcFileName, "wb" )) )
		perror( pcFileName ), exit( -1 );

	/* write header */
	if( fwrite( &_tImageHeader, BPHDS, HEADERHDSECS, ptFile ) != HEADERHDSECS )
		perror( pcFileName ), exit( -1 );

	/* copy cd-sectors */
	for( ulSector = 0; ulSector + NUMCDSECS <= _tImageHeader.ulTotSize; ulSector += NUMCDSECS )
	{
		fprintf( stderr, "%lu\r", ulSector );
		ReadCDSectors( iCDno, _acBuffer, ulSector, NUMCDSECS );
		if( fwrite( _acBuffer, BPCDS, NUMCDSECS, ptFile ) != NUMCDSECS )
			perror( pcFileName ), exit( -1 );
	}
	if( ulSector < _tImageHeader.ulTotSize )
	{
		fprintf( stderr, "%lu\r", ulSector );
		ReadCDSectors( iCDno, _acBuffer, ulSector, _tImageHeader.ulTotSize - ulSector );
		if( fwrite( _acBuffer, BPCDS, _tImageHeader.ulTotSize - ulSector, ptFile )
				!= _tImageHeader.ulTotSize - ulSector )
			perror( pcFileName ), exit( -1 );
	}

	fclose( ptFile );
	return( 1 );
}

/*
	main
	Glues together thee functions above.
*/
int main( int iArgc, char *apcArgv[] )
{
struct dfree	tDFree;	/* data about free space on disk */
char	acCDs[27];			/* MSCDEX's list of CDROM drives */
int		iDrive;					/* target drive */

	fputs( "CD2FILE v02  Copyright (C) 1997 C.Kulms\n"
				"CD2FILE comes with ABSOLUTELY NO WARRANTY; this is free software, and you are\n"
				"welcome to redistribute it under the conditions of the GNU General Public\n"
				"License; see file 'gnu.gpl' for details.\n\n", stderr );

	if( iArgc < 2 || *apcArgv[1] == '/' || *apcArgv[1] == '?' )
	{
		fputs( "Creates CDROM-image as file. Insert source-CDROM in drive and\n", stderr );
		fputs( "- to view track-list use:\n"
						"  CD2FILE cdrom-drive\n", stderr );
		fputs( "- to create an image-file use:\n"
						"  CD2FILE cdrom-drive [target-drive:][path]filename [comment]\n",
						stderr );
		fputs( _acWarn, stderr );
		fputs( "Note: CDROM images have to be continuous - so it's a good idea to run\n"
						"  DEFRAG.EXE disk: /F\nbefore you create an image.\n", stderr );
		exit( 0 );
	}

	/* first check if MSCDEX is installed */
	if( !MSCDEX_NumberOfDrives() )
	{
		fputs( "**** ERROR: MSCDEX is not installed. ****\n", stderr );
		exit( -1 );
	}
	/* check if given disk-letters is CDROM-drive */
	*apcArgv[1] = toupper( *apcArgv[1] );
	MSCDEX_GetCDDrives( (char far *)&acCDs );
	acCDs[26] = 0;
	if( *apcArgv[1] < 'C' || !strchr( acCDs, *apcArgv[1] - 'A' ) )
	{
		fprintf( stderr, "**** ERROR: drive %c: is not a CDROM-drive. ****\n", *apcArgv[1] );
		exit( -1 );
	}

	/* get the cdrom parameters */
	GetCD( *apcArgv[1] );
	fputs( _acWarn, stdout );
	if( iArgc < 3 )
		return( 0 );

	/* check space on disk */
	if( apcArgv[2][1] == ':' )
		iDrive = toupper( *apcArgv[2] ) - '@';
	else
		iDrive = getdisk();
	getdfree( iDrive, &tDFree );
	if( tDFree.df_sclus == (unsigned)-1 )
	{
		fprintf( stderr, "**** ERROR: Unable to get size of free space on drive %c. ****\n",
							iDrive + '@' );
		exit( -1 );
	}
	if( (unsigned long)tDFree.df_avail
			* (unsigned long)tDFree.df_sclus
			< _tImageHeader.ulTotHdSecs + HEADERHDSECS )
	{
		fputs( "**** ERROR: not enough space on target drive. ****\n", stderr );
		exit( -1 );
	}

	/* setup data */
	memcpy( _tImageHeader.acSignature, SIGNATURE, SIGNLEN );
	if( iArgc > 3 )
		strncpy( _tImageHeader.acImgComment, apcArgv[3], IMGCOMSIZE );
	_tImageHeader.acImgComment[strlen( _tImageHeader.acImgComment )] = '$';

	/* create image */
	CopyCD( *apcArgv[1], apcArgv[2] );

	/* start child */
	if( execlp( "FILE4DRV.EXE", "FILE4DRV.EXE", apcArgv[2], NULL ) )
	{
		fputs( "**** ERROR: cannot execute FILE4DRV.EXE ****\n", stderr );
		exit( -1 );
	}

	return( 0 );
}

/* end of file 'CD2FILE.C' */

