/*
**                ---  fifotest.c ---
**
**  FIFOTEST calculate the size of TX FIFO by counting TX interrupts.
**
**  IMPORTANT
**    If running under Win 3.1, exit Windows before running.
**    If running under Win 95 or Win NT, you MUST boot to DOS before running.
*/

#ifdef DPMI
#include "windows.h"
#include "use_dpmi.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <dos.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include "pcl4c.h"

#define ESC 0x1b
#define CR  0x0d
#define LF  0x0a

#define PC 1
#define DB 2
#define BB 3
#define CC 4

/*** Global Variables ***/

#define TEST_SIZE 63

int ComPort = COM1;
int BaudCode = Baud19200;

int MultiUART = 0;
int MultiIRQ = 0;
int MultiStatus = 0;

void AllDone(void);
void CheckAlloc(int,char *);
int  ErrorCheck(int);
int  AllocSeg(int);
long TimeMark;
float WaitSecs;

static int  RxSeg = 0;
static int  TxSeg = 0;

char *Ptr[4] = {"SMALL","COMPACT","MEDIUM","LARGE"};

static long BaudRate[10] =  {300,600,1200,2400,4800,9600,
                              19200,38400,57600,115200};
/* standard PC port configuration */

static int StdAdrPC[4] = {0x3f8,0x2f8,0x3e8,0x2e8};
static int StdIrqPC[4] = {4,3,4,3};
static char *UartType[4] = {"8250/16450", "16550", "16650", "16750"};

static int Selection = 0;  /* PC, DB, or BB */

void main(int argc, char *argv[])
{int Port;
 int TestLength  = 1024;
 float CharsPerInt;
 int TxBase;
 char *P;
 int ComLimit = 0;
 char c;
 int Version;
 int Model;
 int i, n, rc;
 if(argc!=3)
  {printf("Usage: fifotest {pc|db|bb|cc} port\n");
   exit(1);
  }
 P = argv[1];
 if((strcmp(P,"pc")==0)||(strcmp(P,"PC")==0)) Selection = PC;
 if((strcmp(P,"db")==0)||(strcmp(P,"DB")==0)) Selection = DB;
 if((strcmp(P,"bb")==0)||(strcmp(P,"BB")==0)) Selection = BB;
 if((strcmp(P,"cc")==0)||(strcmp(P,"CC")==0)) Selection = CC;
 if(Selection==0)
   {puts("Must specify 'PC', 'DB' or 'BB' as 1st argument");
    puts("EG:  SELFTEST PC 1 4");
    exit(1);
   }

 ComPort = atoi(argv[2]) -1;
 printf("Port      = COM%d\n",1+ComPort);
 if(Selection==CC)
   {
#if 1
    /*** Custom Configuration: COM4 uses IRQ10 ***/
    printf("Custom configuration: COM4(IRQ10)\n");
    SioIRQ(COM4,IRQ10);
#else
    /*** Custom Configuration: 4 port board ***/
    printf("[ Custom configuration: 0x280,288,290,298: IRQ10,15,10,15]\n");
    MultiUART = 0x280; MultiIRQ = IRQ10; MultiStatus = 0;
    SioUART(COM1,0x280); SioIRQ(COM1,IRQ10);
    SioUART(COM2,0x288); SioIRQ(COM2,IRQ15);
    SioUART(COM3,0x290); SioIRQ(COM3,IRQ10);
    SioUART(COM4,0x298); SioIRQ(COM4,IRQ15);
#endif
   }
 if(Selection==DB)
   {/*** Custom Configuration: DigiBoard PC/8 ***/
    MultiUART = 0x180; MultiIRQ = IRQ5; MultiStatus = 0x1C0;
    printf("[ Configuring DigiBoard as COM1-COM8 (IRQ%d) @ 0x%x ]\n",
       MultiIRQ,MultiUART);
    SioPorts(1+ComLimit,COM1,MultiStatus,DIGIBOARD);
    for(Port=COM1;Port<=ComLimit;Port++)
      {/* set DigiBoard UART addresses */
       ErrorCheck( SioUART(Port,MultiUART+8*Port) );
       /* set DigiBoard IRQ */
       ErrorCheck( SioIRQ(Port,MultiIRQ) );
      }
   }
 if(Selection==BB)
   {/*** Custom Configuration: BOCA BB2016 ***/
    MultiUART = 0x100; MultiIRQ = IRQ10; MultiStatus = MultiUART + 7;
    printf("[ Configuring BOCA Board as COM1-COM16 (IRQ%d) @ 0x%x ]\n",
       MultiIRQ,MultiUART);
    SioPorts(1+ComLimit,COM1,MultiStatus,BOCABOARD);
    for(Port=COM1;Port<=ComLimit;Port++)
      {/* set BOCA Board UART addresses */
       ErrorCheck( SioUART(Port,MultiUART+8*Port) );
       /* set BOCA Board IRQ */
       ErrorCheck( SioIRQ(Port,MultiIRQ) );
      }
   }
 if(Selection==PC)
   {/*** Standard Configuration: 4 port card ***/
    puts("[ Configuring for PC ]");
    for(i=COM1;i<=COM4;i++)
      {SioUART(i,StdAdrPC[i]);
       SioIRQ(i,StdIrqPC[i]);
      }
   }
 CheckAlloc(RxSeg = AllocSeg(1024), "RX");
 ErrorCheck( SioRxBuf(ComPort,RxSeg,Size1024) );
 if(SioInfo('I'))
   {CheckAlloc(TxSeg = AllocSeg(1024), "TX");
    ErrorCheck( SioTxBuf(ComPort,TxSeg,Size1024) );
   }


 ErrorCheck( SioParms(ComPort,NoParity,OneStopBit,WordLength8) );
 ErrorCheck( SioReset(ComPort,BaudCode) );
 printf("COM%d reset\n",1+ComPort);
 SioRxClear(ComPort);

 printf("***\n");
 printf("*** FIFOTEST 6.1\n");
 printf("***\n");
 Version = SioInfo('V');
 Model = SioInfo('M');
 printf("*** Lib Ver : %d.%d\n",Version/16,Version%16);
 printf("***   Model : %s \n", Ptr[Model&3] );
 printf("*** TX Intr : ");
 if(SioInfo('I')) puts("Enabled.");
 else puts("Disabled.");
 printf("***    DPMI : ");
 if(SioInfo('P')) printf("YES\n");
 else printf("NO\n");
 /* display port info */
 printf("***    COM%d : ",1+ComPort);
 if(Selection==PC) printf("Adr=%3x IRQ=%d\n",StdAdrPC[ComPort],StdIrqPC[ComPort]);
 else printf("Adr=%3xh IRQ=%d Status=%xh\n",MultiUART+8*(ComPort-COM1),MultiIRQ,MultiStatus);

 n = 0x03 & SioFIFO(ComPort,3);
 printf("***    FIFO : %s\n", UartType[n] );
 if(n==0)
   {printf("\nNo FIFO to test!\n");
    AllDone();
    exit(1);
   }
 printf("***\n");

 printf("\nTesting COM%d FIFO by counting TX interrupts.\n",ComPort+1);

 SioBaud(ComPort,BaudCode);
 SioFIFO(ComPort,3);

 SioRxClear(ComPort);
 SioPutc(ComPort,'A');
 if(SioGetc(ComPort,5) > 0)
   {printf("Incoming serial data. Disconnect RS232 cable!\n");
    AllDone();
    exit(1);
   }

 WaitSecs = 0.25 + (float)TestLength / ( (float)BaudRate[BaudCode] / 11.0) ;
 printf("WaitSecs = %3.1f\n", WaitSecs);

 TxBase = SioInfo('T');
 for(i=0;i<TestLength;i++) SioPutc(ComPort,'$');

 /* wait long enough for all characters to be sent */
 TimeMark = SioTimer() + (int)(18.2*WaitSecs);
 while(SioTimer()<TimeMark);

 i = SioInfo('T');
 printf("  %3d TX interrupts on %d outgoing bytes: ", i-TxBase,TestLength);
 if(i==TxBase)
   {printf("No TX interrupts!\n");
   }
 else
  {CharsPerInt = (float)TestLength / (float)(i-TxBase);
   if(CharsPerInt>1.0)
     printf("%3.1f characters per interrupt\n",CharsPerInt);
   else puts("TX FIFO is NOT operational [or not 16550/16750 UART]");
   /* PCL4C stuffs one less than maximum FIFO size */
   /* 15 for 16550, 31 for 16650, and 63 for 16750 */
   n = (int) CharsPerInt;
   if((n>=14)&&(n<=17)) printf("   16 byte FIFO detected!\n");
   if((n>=30)&&(n<=34)) printf("   32 byte FIFO detected!\n");
   if((n>=62)&&(n<=66)) printf("   64 byte FIFO detected!\n");
  }

 printf("\n");
 AllDone();
 exit(0);
} /* end main */

int ErrorCheck(int Code)
{/* trap PCL error codes */
 if(Code<0)
     {SioError(Code);
      AllDone();
      exit(1);
     }
 return(0);
} /* end ErrorCheck */

#ifdef DPMI

int AllocSeg(int Size)
{long hMem;
 int  Selector = 0;
 hMem = (long) GlobalDosAlloc( (long)Size );
 if(hMem)
   {/* get selector */
    Selector = LOWORD(hMem);
    GlobalPageLock(Selector);
    return Selector;
   }
 else return 0;
}

#else

int AllocSeg(int Size)
{int Seg;
 char far *Ptr;
 /* allocate far heap */
 Ptr = (char far *) _fmalloc(Size+16);
 if(Ptr==NULL) return 0;
 /* SEG:0 points to buffer */
 Seg = FP_SEG(Ptr) + ((FP_OFF(Ptr)+15)>>4);
 return Seg;
}

#endif

void AllDone(void)
{printf("Shutting down COM%d\n",1+ComPort);
 SioDone(ComPort);
#ifdef DPMI
 printf("Freeing DOS memory: ");
 /* free DOS memory */
 if(RxSeg)
    {printf("RX ");
     GlobalPageUnlock(RxSeg);
     GlobalDosFree(RxSeg);
    }
 if(TxSeg)
    {printf("TX ");
     GlobalPageUnlock(TxSeg);
     GlobalDosFree(TxSeg);
    }
 #endif
 exit(0);
}

void CheckAlloc(int Seg,char *Msg)
{if(Seg==0)
  {printf("Cannot allocate memory for %s\n",Msg);
   AllDone();
   exit(1);
  }
}



