/*******  Listing 5  ***************************  APPS.C  ************
*  apps.c -- Applications using OBJECT CODE EXPLORER
*  Author:   Thomas E. Siering, 1991. Copyruight notice in Listing 1
* 
*  Compiler: Microsoft C 6.0, Turbo C++ 1.0
* 
*  Compile time switches:
*    PRN - listings use only characters avaialable on printers
*    CON - listings use only characters available on the console
* 
*  Links with: svc.c and ox.c. This module contains the main driver
* 
*  Usage: apps file.obj
* 
*  Generates file.dmp and file.anl, showing the contents of file.obj
***********************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "ox.h"


/* Define the following DOS system constants for non MS C compilers */
#ifndef _MAX_PATH
#define _MAX_PATH	260	/* max. length of full pathname */
#endif
#ifndef _MAX_FNAME
#define _MAX_FNAME	256	/* max. length of file name component */
#endif

#if defined(__TURBOC__)
#define SplitPath fnsplit          /* Turbo C */
#else
#define SplitPath _splitpath       /* MS C    */
#endif


PUBLIC FILE *AnalysisFH;
PUBLIC char *obj;

PRIVATE char ProgramName[] = { "ox" };


PRIVATE void Init(int argc, char *argv[], char ObjFN[], char **ws,
        long *ObjSize);
PRIVATE void Dump(char *ObjP, int FileLength, char *ObjName);
PRIVATE void Analyze(char Obj[], long ObjLength, char *ObjName);


PUBLIC void main(int argc, char *argv[]);

void main(int argc, char *argv[])
{
    long FileLength;
    char ObjFN[_MAX_FNAME];

    Init(argc, argv, ObjFN, &obj, &FileLength);
    Dump(obj, (int) FileLength, ObjFN);
    if (argv[2][1] != 'd' && argv[2][1] != 'D')
        Analyze(obj, FileLength, ObjFN);
    free(obj);
}

/*----------------------------------------------
*   Init  --  Initialize various mundane things
*--------------------------------------------**/
PRIVATE void Init(int argc, char *argv[], char ObjFNRoot[], char **ws,
        long *ObjSize)
{
    char ObjFileName[_MAX_PATH];
    char Dummy[_MAX_PATH];
    char FileExt[5];        /* allow for leading . and trailing \0 */

    if (argc < 2) {
        fprintf(stderr, "CALL: %s FN.OBJ [options]\n", ProgramName);
        fprintf(stderr, "Options: /d -- Dump only\n");
        exit(-1);
    }

    strcpy(ObjFileName, argv[1]);
    strcpy(ObjFNRoot, argv[1]);   /* for remainder of program, w/out ext. */

    /* Did the user enter an extension w/ file name? */
    SplitPath(ObjFileName, Dummy, Dummy, Dummy, FileExt);
    if (FileExt[0] == '.' && FileExt[1] != '\0')  /* has an extension */
        ObjFNRoot[strrchr(ObjFNRoot, '.') - ObjFNRoot] = '\0';
    else
        strcat(ObjFileName, ".OBJ\0");  /* OBJ ext needed to open file */

    InstObjExp(ObjFileName, ws, ObjSize);
}

#define  PRN           /* for output ctl: enable PRN -OR- CON */
/* #define  CON */
#ifdef PRN
   #define CHKCHAR  !isprint
#else
   #define CHKCHAR  iscntrl
#endif

#define  MAXLN    16
#define  ASCIICOL 62

/*-----------------------------------------
*   Dump --  formatted dump of OBJ records
*----------------------------------------*/
PRIVATE void Dump(char *ObjP, int  FileLength, char *ObjName)
{
    FILE *DumpFH;
    int  LocCtr = 0;
    char ObjRecName[OMFNAMELENGTH];
    char DumpFN[_MAX_PATH];
    int  RecLength;
    int  PrintLength;

    strcpy(DumpFN, ObjName);
    strcat(DumpFN, ".DMP");

    if ((DumpFH = fopen(DumpFN, "w")) == NULL) {
        fprintf(stderr, "\aCould Not Open File %s.\n", DumpFN);
        exit(0xFF);
    }

    while (FileLength) {
        RecLength = *((int *) (ObjP + 1)) + 3;
        if (GetObjRecordName(*ObjP, ObjRecName) != OK) {
            Output(Warning, DumpFH,
                    "\aNon-MS-Documented Record %02X in file %s.\n",
                    *ObjP & 0x00FF, ObjName);
            Output(Message, DumpFH, "      **%02X**:\n", *ObjP & 0x00FF);
        }
        else
            Output(Message, DumpFH, "      %s:\n", ObjRecName);

        while (RecLength) {
            PrintLength = (RecLength > MAXLN) ? MAXLN : RecLength;
            PrintObjDumpLine(ObjP, DumpFH, LocCtr, PrintLength);
            RecLength -= PrintLength;
            FileLength   -= PrintLength;
            LocCtr += PrintLength;
            ObjP    += PrintLength;
        }
    }
    fclose(DumpFH);
}

/*----------------------------------------------------------
*   PrintObjDumpLine -- print one line of formatted OBJ dump
*----------------------------------------------------------*/
PUBLIC void PrintObjDumpLine(char *src, FILE *dest, int LocCtr, 
                             int PrintLength)
{
    int i;
    static char line[100];
    char *lptr = line;
    char *aptr = line + ASCIICOL;

    memset(line, ' ', sizeof(line));
    lptr += sprintf(lptr, "%04X    ", LocCtr);

    for (i = 0; i < PrintLength; i++, src++) {
        if (i == 8) {
            lptr += 3;
            aptr += 2;
        }
        sprintf(lptr, "%02X", *src & 0x00FF);
        *aptr++ = (CHKCHAR(*src)) ? (char) '.' : *src;
        lptr += 2;
        *lptr++ = ' ';
    }
    sprintf(aptr, "\n\0");
    fputs(line, dest);
}

/*-----------------------------------------------------
*   Analyze -- Analyze the OBJ module records in detail
*----------------------------------------------------*/
PRIVATE void Analyze(
    char Obj[],                 /* OBJ file in memory  */
    long ObjLength,             /* PrintLength of OBJ  */
    char *ObjName)              /* name of OBJ         */

{
    char *RecordP = Obj;
    unsigned char RecordType = '\0';
    unsigned int  RecordLength;
    char AnalysisFN[_MAX_PATH];
    char RecordName[8];

    strcpy(AnalysisFN, ObjName);
    strcat(AnalysisFN, ".ANL");

    if ((AnalysisFH = fopen(AnalysisFN, "w")) == NULL) {
        fprintf(stderr, "\aCould Not Open File %s.\n", AnalysisFN);
        exit(-1);
    }

    while (ObjLength > 0L) {
        RecordType = *RecordP;
        if (GetObjRecordName(RecordType, RecordName) == OK)
            Output(Message, AnalysisFH, "\n%04X   --   %s\n",
                    GetRecordRelPos(RecordP), RecordName);
        RecordP++;
        RecordLength = *(int *) RecordP;
        RecordP += 2;            /* advance ptr past type/PrintLength */
        --RecordLength;          /* discount CHKSM */
        ObjLength -= 4L;         /* discount CHKSM/type/PrintLength */

        switch(RecordType) {
            case BLKDEF :        /* 0x7A */
                DoBLKDEF();
                break;
            case BLKEND :        /* 0x7C : end-of-program-block indicator */
                DoBLKEND();
                break;
            case THEADR :        /* 0x80 : Translator Header Record -- T  */
                DoTHEADR(RecordP);
                break;
            case LHEADR :        /* 0x82 : Translator Header Record -- L  */
                DoLHEADR(RecordP);
                break;
            case COMENT :        /* 0x88 : Comment Record */
                DoCOMENT(RecordP, RecordLength);
                break;
            case MODEND :        /* 0x8A : Module End Record */
                DoMODEND(RecordP);
                break;
            case EXTDEF :        /* 0x8C : External Definitions Record */
                DoEXTDEF(RecordP, RecordLength);
                break;
            case TYPDEF :        /* 0x8E : Type Definitions Rec (obsolete)*/
                DoTYPDEF();
                break;
            case PUBDEF :        /* 0x90 : Public Definitions Record */
                DoPUBDEF(RecordP, RecordLength);
                break;
            case LINNUM :        /* 0x94 : Line Numbers Record */
                DoLINNUM(RecordP, RecordLength);
                break;
            case LNAMES :        /* 0x96 : Names List Record */
                DoLNAMES(RecordP, RecordLength);
                break;
            case SEGDEF :        /* 0x98 : Segment Definition Record */
                DoSEGDEF(RecordP);
                break;
            case GRPDEF :        /* 0x9A : Group Definition Record  */
                DoGRPDEF(RecordP, RecordLength);
                break;
            case FIXUPP :        /* 0x9C : References Fixup Record  */
                DoFIXUPP(RecordP, RecordLength);
                break;
            case LEDATA :        /* 0xA0 : Enumerated Data Record   */
                DoLEDATA(RecordP, RecordLength);
                break;
            case LIDATA :        /* 0xA2 : Iterated Data Record     */
                DoLIDATA(RecordP, RecordLength);
                break;
            case COMDEF :        /* 0xA0 : Communal Names Definition Rec */
                DoCOMDEF(RecordP, RecordLength);
                break;
            case BAKPAT:         /* 0xB2 : MS Extension: Quick-C backpatch */
                DoBAKPAT(RecordP, RecordLength);
                break;
            case LEXTDEF:        /* 0xB4 : MS Extension: local EXTDEF */
                DoEXTDEF(RecordP, RecordLength);
                break;
            case LPUBDEF:       /* 0xB6 : MS Extension: local PUBDEF */
                DoPUBDEF(RecordP, RecordLength);
                break;
            case LCOMDEF:       /* 0xB8 : MS Extension: local COMDEF */
                DoCOMDEF(RecordP, RecordLength);
                break;

            default:            /* not recognized, undocumented, or error */
                Output(Warning, AnalysisFH,
                        "Non-MS-Documented Record %02X at offset %Xx\n",
                        RecordType & 0x00FF, (RecordP - Obj));
                break;
        }
        ObjLength -= (long) RecordLength;
        RecordP += RecordLength;
        ++RecordP;                         /* skip CHKSM        */
    }

    if (ObjLength > 0L)     /* Any leftovers?  This should NOT happen! */
        Output(Error, AnalysisFH,
                "OBJ module record not recognized. Terminating\n");

    fclose(AnalysisFH);
}