
------------------------------------------------------------
--        Name: Alden Dima
--      E-mail: dimaaa@seas.gwu.edu
--      School: The George Washington University
--              School of Engineering and Applied Science
--              Washington, D.C.
--       Class: CSci 298 - Independent Study
--     Project: Ada Curses Binding and Textual User Interface
--        File: adatui.adb 
--        Date: 12/24/95 
-- Description: Package body for AdaTUI, the Ada 95 version 
--              of the Textual User Interface (TUI) version
--              1.02 created by P. J. Kunst and included as 
--              sample C code with PDCurses version 2.2.
--              Implements a basic user interface that includes
--              functions for pull-down menus and input boxes.
--      Usages: Implemented as a parameter-less generic
--              package to work with Ada's pointer scoping
--              rules.  To use AdaTUI in an application, 
--              instantiate a version of this package in 
--              same scope where menu-driven procedures are
--              declared. 
--   Revisions: 5/31/96 - AAD - Rewrote Ada-Curses binding to
--              enhance portability and maintainability.  Made
--              necessary changes in AdaTUI/TUIDemo to
--              accomodate new binding.  Made minor changes
--              to correct a problem using access types with
--              unconstrained arrays discovered by GNAT 3.03.
--              Eliminated several unused variables.
------------------------------------------------------------

with pdcurses;
use type pdcurses.bool;

with ada.characters.handling;
with ada.strings.fixed;
with bstrings; 
with calendar;
with integer_io;
with interfaces.c; use type interfaces.c.int;
with text_io;
with twiddle_unsigned; use twiddle_unsigned; 

package body AdaTUI is

--
-- Static
--
   TITLECOLOR       : constant :=          1;  -- tui.c:36
   MAINMENUCOLOR    : constant := 16#800002#;  -- tui.c:37
   MAINMENUREVCOLOR : constant := 16#A00003#;  -- tui.c:38
   SUBMENUCOLOR     : constant := 16#800004#;  -- tui.c:39
   SUBMENUREVCOLOR  : constant := 16#A00005#;  -- tui.c:40
   BODYCOLOR        : constant :=          6;  -- tui.c:41
   STATUSCOLOR      : constant := 16#800007#;  -- tui.c:42
   INPUTBOXCOLOR    : constant :=	   8;  -- tui.c:43
   EDITBOXCOLOR     : constant := 16#A00009#;  -- tui.c:44
   th               : constant :=          1;  -- tui.c:46
   mh               : constant :=          1;  -- tui.c:47
   sh               : constant :=          2;  -- tui.c:48

--
-- Using magic numbers for screen height and width for now...
--

   bh : constant := 25 - th - mh - sh - 1;
   bw : constant := 80;
  
   screen_window : pdcurses.A_WINDOW_T;
   title_window  : pdcurses.A_WINDOW_T;
   main_window   : pdcurses.A_WINDOW_T;
   body_window   : pdcurses.A_WINDOW_T;
   status_window : pdcurses.A_WINDOW_T;
   next_y        : integer;
   next_x        : integer;
   key           : integer := pdcurses.ERR;
   ch            : integer := pdcurses.ERR;
   quit          : boolean := FALSE;
   in_pdcurses     : boolean := FALSE;


   procedure make_charv (
      source  : in     bstrings.bounded_string;
      target  :    out c.charv ) 
   is separate;


   function pad_string (
      str    : string;
      length : integer ) 
      return   bstrings.bounded_string
   is separate;


   function prepad_string (
      str    : string;
      length : natural )
      return   bstrings.bounded_string
   is separate;


   procedure rmline (
      window : pdcurses.A_WINDOW_T;
      nr     : integer ) 
   is separate;


   procedure init_color is separate;


   procedure set_color (
      window : pdcurses.A_WINDOW_T;
      color  : integer ) 
   is separate;


   procedure color_box (  
      window : pdcurses.A_WINDOW_T;
      color  : integer;
      hasbox : boolean )
   is separate;


   procedure idle is separate;


   procedure menu_dimension (
      a_menu  : a_menu_t;
      lines   : access integer;
      cols    : access integer )
   is separate;


   procedure set_menu_position ( y, x : integer ) is
   begin
      next_y := y;
      next_x := x;
   end set_menu_position;


   procedure get_menu_position ( y, x : access integer) is
   begin
      y.all := next_y;
      x.all := next_x;
   end get_menu_position;


   function hotkey ( str : string ) return character is separate;


   procedure repaint_menu ( 
      menu_window : pdcurses.A_WINDOW_T; 
      a_menu : a_menu_t ) 
   is separate;


   procedure repaint_main_menu ( 
      width  : integer;
      a_menu : a_menu_t ) 
   is separate;


   procedure main_help is
   begin
      status_message (
	 msg => "Use arrow keys and Enter to select (Alt-X to quit)");
   end main_help;


   procedure hide_cursor is
      void : c.signed_int;
   begin
      void := pdcurses.curs_set (0);
   end hide_cursor;


   procedure normal_cursor is
      void : c.signed_int;
   begin
      void := pdcurses.curs_set (1);
   end normal_cursor;


   procedure insert_cursor is
      void : c.signed_int;
   begin
      void := pdcurses.curs_set (2);
   end insert_cursor;


   procedure main_menu (mp : a_menu_t) is separate;


   procedure clean_up is
      void : c.signed_int;
   begin
      if in_pdcurses then
	 void := pdcurses.delwin ( title_window );
	 void := pdcurses.delwin ( main_window );
	 void := pdcurses.delwin ( body_window );
	 void := pdcurses.delwin ( status_window );
	 normal_cursor;
	 void := pdcurses.endwin;
	 in_pdcurses := FALSE;
      end if;
   end clean_up;

--
-- External 
--

   procedure clear_body_win is
      void : c.signed_int;
   begin
      void := pdcurses.werase (body_window);
      void := pdcurses.wmove  (body_window, 0, 0);
   end clear_body_win;


   procedure refresh_body_win is
      void : c.signed_int;
   begin
      void := pdcurses.wrefresh(body_window);
   end refresh_body_win;


   function body_length return integer is
   begin
      return integer(pdcurses.getmaxy(body_window));
   end body_length;


   function body_win return pdcurses.A_WINDOW_T is
   begin
      return body_window;
   end body_win;


   procedure rmerror is
   begin
      rmline (
	 window => status_window,
	 nr     =>    0 ); 
   end rmerror;


   procedure rmstatus is
   begin
      rmline (
	 window => status_window,
	 nr     =>    1 ); 
   end rmstatus;


   procedure title_message  ( msg : string ) is separate;


   procedure body_message   ( msg : string ) is separate;


   procedure error_message  ( msg : string ) is separate;


   procedure status_message ( msg : string ) is separate;


   function key_pressed return boolean is 
   begin
      if ch = pdcurses.ERR then
	 ch := integer ( pdcurses.wgetch ( body_window ) );
      end if;

      return ( ch /= pdcurses.ERR );
   end key_pressed;


   function get_key return integer is
      result : integer;
   begin
      result := ch;
      ch   := pdcurses.ERR;
      quit := ( result = pdcurses.ALT_X );
      return result;
   end get_key;


   procedure flush_keys is
      void : integer;
   begin
      while key_pressed loop
      	void := get_key;
      end loop;
   end flush_keys;


   function wait_for_key return integer is
   begin
      flush_keys;
      while not key_pressed loop
	 idle;
      end loop;
      return get_key;
   end wait_for_key;


   procedure do_exit is
   begin
      quit := TRUE;
   end do_exit;


   procedure do_menu ( mp: a_menu_t ) is separate;


   procedure start_menu ( 
      mp    : a_menu_t;
      title : string  )
   is separate;


--
-- another static function
--
   procedure repaint_edit_box (
      a_window : pdcurses.A_WINDOW_T;
      x        : integer;
      buffer   : string ) 
   is separate;


   procedure win_edit_string (
      window :        pdcurses.A_WINDOW_T;
      buffer : in out bstrings.bounded_string;
      field  : in     integer;
      key    :    out integer )
   is separate;


   function win_input_box (
      a_window  : pdcurses.A_WINDOW_T;
      num_lines : integer;
      num_cols  : integer )
      return pdcurses.A_WINDOW_T 
   is separate;


   procedure get_strings (
      items : in out input_items; 
      field : in     integer;
      key   :    out integer )
   is separate;


--
-- The following are defined as macros in tui.h
--

   procedure edit_string ( 
      buffer : in out bstrings.bounded_string;
      field  : in     integer;
      key    :    out integer ) is
   begin
      win_edit_string (
	 window => screen_window, 
	 buffer => buffer, 
	 field  => field,
	 key    => key );
   end edit_string;


   procedure move_edit_string (
      y      : in     integer;
      x      : in     integer;
      buffer : in out bstrings.bounded_string;
      field  : in     integer;
      key    :    out integer )

   is
      temp_y : c.signed_int;
      temp_x : c.signed_int;

   begin
      temp_y := c.signed_int(y);
      temp_x := c.signed_int(x);

      if pdcurses.move ( temp_y, temp_x) = pdcurses.ERR then
	 key := integer ( pdcurses.ERR );
	 return; 
      else
	 edit_string (
	    buffer => buffer,
	    field  => field,
	    key    => key );

	 return;
      end if;
   end move_edit_string;


   procedure move_win_edit_string (
      window :        pdcurses.A_WINDOW_T;
      y      : in     integer;
      x      : in     integer;
      buffer : in out bstrings.bounded_string;
      field  : in     integer;
      key    :    out integer )
      
   is
      temp_y : c.signed_int;
      temp_x : c.signed_int;

   begin
      temp_y := c.signed_int(y);
      temp_x := c.signed_int(x);

      if pdcurses.wmove ( window, temp_y, temp_x ) = pdcurses.ERR then
	 key := integer(pdcurses.ERR);
	 return;
      else
	 win_edit_string (
	    window => window,
	    buffer => buffer,
	    field  => field,
	    key    => key );

	 return;
      end if;
   end move_win_edit_string;


   function input_box (
      lines : integer;
      cols  : integer)
      return  pdcurses.A_WINDOW_T is
   begin
      return win_input_box ( screen_window, lines, cols );
   end input_box;


   function move_input_box (
      y     : integer;
      x     : integer;
      lines : integer;
      cols  : integer)
      return  pdcurses.A_WINDOW_T 

   is
      temp_y : c.signed_int;
      temp_x : c.signed_int;

   begin
      temp_y := c.signed_int(y);
      temp_x := c.signed_int(x);

      if pdcurses.move ( temp_y, temp_x ) = pdcurses.ERR then
	 return screen_window; -- ???
      else
	 return input_box ( lines, cols );
      end if;
   end move_input_box;


   function win_move_input_box (
      win    : pdcurses.A_WINDOW_T;
      y      : integer;
      x      : integer;
      lines  : integer;
      cols   : integer)
      return   pdcurses.A_WINDOW_T

   is
      temp_y : c.signed_int;
      temp_x : c.signed_int;

   begin
      temp_y := c.signed_int(y);
      temp_x := c.signed_int(x);
      if pdcurses.wmove ( win, temp_y, temp_x ) = pdcurses.ERR then
	 return win;
      else
	 return win_input_box ( win, lines, cols );
      end if;
   end win_move_input_box;

--
-- Allows tuidemo.show_file to access body_window
--
   procedure body_window_addch ( item : character ) is 
      temp_int  : integer;
      temp_char : pdcurses.chtype;
      void      : c.signed_int;
   begin
      temp_int := character'pos(item);
      temp_char := pdcurses.chtype(temp_int);
      void := pdcurses.waddch ( body_window, temp_char );
   end body_window_addch;

end AdaTUI;
