/* strtol.c -- convert a string to a long
   Copyright (C) 1989, 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Author:
	Mike Rendell			Department of Computer Science
	michael@garfield.mun.edu	Memorial University of Newfoundland
	..!uunet!garfield!michael	St. John's, Nfld., Canada
	(709) 737-4550			A1C 5S7
*/

#include <ctype.h>

#define	okdigit(c, b)	(isdigit(c) && \
		((c) < (b) + '0' || (b) == 1 && (c) == 1) ||\
		(b) > 9 && ((c) | 0x20) >= 'a' && ((c) | 0x20) < (b) - 10 + 'a')
#define	todigit(c, b)	(isdigit(c) ? (c) - '0' : ((c) | 0x20) - 'a' + 10)

/* Parse a number in STR and return the result.  If E is not null
   it will point to the last character parsed; if no valid number was
   found, then E will point to STR.  Numbers are parsed as follows:
 	- if BASE is <= 0 then the following rules are used:
 		prefix			assumed base
 		 0			8
 		 0[oO]			8
 		 [1-9]			10
 		 0[dDtT]		10
 		 0[xX]			16
 		 0[bB]			2
 	- if BASE is > 0 then the string is assumed to be in base BASE
 	(leading 0x,0o,0b,0t,0d's are ignored if BASE is 16,8,2,10,10
 	respectivly).
   Any number may start with a + or -.  This should be compatable with
   SysV strtol except for the addition of 0o,0d,0t and 0b.  */

long
strtol (str, e, base)
     char *str;
     char **e;
     int base;
{
  long rval = 0;
  int sign = 1;
  int gotnum = 0;
  char *s = str;

  while (isspace (*s))
    s++;

  if (*s == '-')
    {
      sign = -1;
      s++;
    }
  else if (*s == '+')
    s++;

  if (base <= 0 || base > 36)
    {
      if (*s == '0')
	switch (s[1])
	  {
	  case 'd':
	  case 'D':
	  case 't':
	  case 'T':
	    s += 2;
	    while (isdigit (*s))
	      {
		rval = rval * 10 + *s++ - '0';
		gotnum++;
	      }
	    break;

	  case 'x':
	  case 'X':
	    s += 2;
	    for (; isxdigit (*s); s++, gotnum++)
	      rval = (rval << 4) + (isdigit (*s) ?
				    *s - '0'
				    : (*s | 0x20) - 'a' + 10);
	    break;

	  case 'b':
	  case 'B':
	    s += 2;
	    while (*s == '0' || *s == '1')
	      {
		rval = (rval << 1) | *s++ & 1;
		gotnum++;
	      }
	    break;

	  case 'o':
	  case 'O':
	    s++;
	  default:		/* an octal number */
	    while (*++s >= '0' && *s <= '9')
	      {
		rval = (rval << 3) + *s - '0';
		gotnum++;
	      }
	  }
      else			/* must be base 10 */
	while (isdigit (*s))
	  {
	    rval = rval * 10 + *s++ - '0';
	    gotnum++;
	  }
    }
  else
    {				/* use base */
      if (*s == '0' &&
	  (base == 16 && (s[1] | 0x20) == 'x'
	   || base == 8 && (s[1] | 0x20) == 'o'
	   || base == 2 && (s[1] | 0x20) == 'b'
	   || base == 10 && ((s[1] | 0x20) == 'd'
			     || (s[1] | 0x20) == 't')))
	s += 2;
      for (; okdigit (*s, base); s++)
	{
	  rval = rval * base + todigit (*s, base);
	  gotnum++;
	}
    }

  /* Mark where we left off */
  if (e != (char **) 0)
    *e = gotnum ? s : str;

  return rval * sign;
}
