/* STAT.C - reimplementation of stat() for MS-DOS Microsoft C
 ******************************************************************************
 *
 *	int stat(const char *filename, struct stat *statbuf)
 *
 ******************************************************************************
 * edit history is at the end of the file
 ******************************************************************************
 * Copyright 1995 by the Summer Institute of Linguistics, Inc.
 *
 * This file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; see the file COPYING.LIB.  If
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave,
 * Cambridge, MA 02139, USA.
 */
#include <stdlib.h>		/* For getenv() */
#include <string.h>
#include <errno.h>		/* For errno */
#include <sys/types.h>
#include <sys/stat.h>
#include <dos.h>		/* For _dos_getdiskfree(), ... */
#include <direct.h>
#include <time.h>

#define NUL '\0'
/*
 *  BDOS function codes
 */
#define SEARCHFIRST	 0x1100		/* search for first entry */
#define	SETDTA		 0x1A00		/* set disk transfer address */
#define GETDTA		 0x2F00		/* get disk transfer address */

#define ALL_FILES_DIRS	(_A_RDONLY | _A_HIDDEN | _A_SYSTEM | _A_SUBDIR)

#pragma pack(1)
struct dos_fcb		/* MS-DOS File Control Block */
    {
    unsigned char drive_number;
    char          filename[8];
    char          extension[3];
    unsigned      current_block;
    unsigned      record_size;
    unsigned long file_size;
    unsigned      write_date;
    unsigned      write_time;
    unsigned char reserved[8];
    unsigned char current_record;
    unsigned long relative_record;
    };
struct dos_extfcb	/* MS-DOS Extended File Control Block */
    {
    unsigned char flag_byte;		/* always set to 0xFF */
    unsigned char reserved[5];
    unsigned char attribute;
    struct dos_fcb fcb;
    };
struct dos_searchfcb	/* MS-DOS unopened File Control Block */
    {
    unsigned char drive_number;
    char          filename[8];
    char          extension[3];
    unsigned char reserve;
    unsigned      entry_number;
    unsigned      directory_cluster;
    unsigned char reserved[4];
    unsigned char real_drive;
    };
struct dos_dirent	/* MS-DOS Directory Entry */
    {
    char          filename[8];
    char          extension[3];
    unsigned char attributes;
    unsigned char reserved[10];
    /*
     *  Bitfields for write time:	Bitfields for write date:
     *  15-11	hours (0-23)		15-9    year - 1980
     *  10-5	minutes			8-5     month
     *  4-0	seconds/2		4-0     day
     */
    unsigned      write_time;
    unsigned      write_date;
    unsigned      first_cluster;
    unsigned long file_size;
    };
#pragma pack()

/*****************************************************************************
 * NAME
 *    stat
 * ARGUMENTS
 *    filename - pathname of a file
 *    statbuf  - struct for holding file status information
 * DESCRIPTION
 *    implementation of stat() for MS-DOS that uses the starting cluster number
 *    as the st_ino field value, and the REAL drive number (regardless of JOIN)
 *    as the st_dev field value
 *    also, treat "con", "nul", and "prn" as character special devices, and
 *    handle root directories okay
 * RETURN VALUE
 *    0 for success, -1 for error
 */
int stat(const char *filename, struct stat *statbuf)
{
static struct dos_extfcb wild_fcb, read_fcb;
unsigned save_dx, save_ds;
union _REGS inr, outr;
struct _SREGS segr;
char far *pf;		/* need both segment and offset for system calls */
unsigned newdrive, olddrive;
char oldcwd[_MAX_PATH], newcwd[_MAX_PATH];
char name[40];			/* massive overkill ([13] should be enough) */
struct dos_dirent *dosdir;
struct dos_searchfcb *search;
int len;
const char *pathname;
char *p;
int retval;
struct tm tm_buf;

if ((filename == (char *)NULL) || (statbuf == (struct stat *)NULL))
    {
    errno = ENOENT;		/* no wildcards allowed */
    return( -1 );
    }
if (	(strchr(filename,'?') != (char *)NULL) ||
	(strchr(filename,'*') != (char *)NULL) ||
	(strchr(filename,'[') != (char *)NULL) )
    {
    errno = ENOENT;		/* no wildcards allowed */
    return( -1 );
    }
memset(statbuf, 0, sizeof(struct stat));
if ((_stricmp(filename, "con") == 0) || (_stricmp(filename, "con:") == 0))
    {
    statbuf->st_dev = 1;
    statbuf->st_rdev = 1;
special_device:
    statbuf->st_ino   = 1;
    statbuf->st_mode  = _S_IFCHR | 0666;
    statbuf->st_nlink = 1;
    statbuf->st_uid   = 1;
    statbuf->st_gid   = 1;
    statbuf->st_size  = 0;
    statbuf->st_atime = time((time_t *)NULL);
    statbuf->st_mtime = statbuf->st_atime;
    statbuf->st_ctime = statbuf->st_atime;
    return( 0 );
    }
if ((_stricmp(filename, "nul") == 0) || (_stricmp(filename, "nul:") == 0))
    {
    statbuf->st_dev = 2;
    statbuf->st_rdev = 2;
    goto special_device;
    }
if ((_stricmp(filename, "prn") == 0) || (_stricmp(filename, "prn:") == 0))
    {
    statbuf->st_dev = 3;
    statbuf->st_rdev = 3;
    goto special_device;
    }
/*
 *  see if we need to change drives or directories
 */
if (	(_getdcwd(0, oldcwd, _MAX_PATH) == (char *)NULL) ||
	(_fullpath(newcwd, filename, _MAX_PATH) == (char *)NULL) )
    {
    errno = ENOENT;
    return(-1);
    }
for ( p = oldcwd ; *p ; ++p )
    {
    if (*p == '\\')
	*p = '/';
    }
strlwr(oldcwd);
for ( p = newcwd ; *p ; ++p )
    {
    if (*p == '\\')
	*p = '/';
    }
strlwr(newcwd);
p = strrchr(newcwd, '/');
if (p != (char *)NULL)
    {
    if (strcmp(newcwd+1, ":/") == 0)
	strcpy(name, ".");
    else
	{
	strncpy(name, p+1, 40);
	name[39] = '\0';
	if (p == newcwd+2)
	    newcwd[3] = '\0';
	else
	    *p = '\0';
	}
    }
else
    {					/* should never happen! */
    strncpy(name, newcwd, 40);
    name[39] = '\0';
    }
olddrive = oldcwd[0] & 0x1F;
newdrive = newcwd[0] & 0x1F;
if (olddrive != newdrive)
    {
    retval = _chdrive(newdrive);
    if ((retval == -1) || (_getdrive() != newdrive))
	{
	errno = ENOENT;
	return(-1);
	}
    if (_getdcwd(newdrive, oldcwd, _MAX_PATH) == (char *)NULL)
	{
	errno = ENOENT;
	if (olddrive != newdrive)
	    _chdrive(olddrive);
	return(-1);
	}
    for ( p = oldcwd ; *p ; ++p )
	{
	if (*p == '\\')
	    *p = '/';
	}
    strlwr(oldcwd);
    }
if (strcmp(oldcwd, newcwd) != 0)
    {
    if (_chdir(newcwd) == -1)
	{
	errno = ENOENT;
	if (olddrive != newdrive)
	    _chdrive(olddrive);
	return(-1);
	}
    }
/*
 *  save the current DTA value and set a new one
 */
inr.x.ax = GETDTA;
_intdosx( &inr, &outr, &segr );
save_dx = outr.x.dx;
save_ds = segr.ds;
inr.x.ax = SETDTA;
pf = (char far *)&read_fcb;
inr.x.dx = FP_OFF(pf);
segr.ds  = FP_SEG(pf);
_intdosx( &inr, &outr, &segr );
/*
 *  there's some serious MS-DOS hacking needed to get the REAL drive number
 *  (regardless of JOINs), and the starting cluster number of the file for
 *  a foolproof (but phony) i-node value that won't ever fool "mv"
 */
memset(&wild_fcb, 0, sizeof(struct dos_extfcb));
wild_fcb.flag_byte = 0xFF;
wild_fcb.attribute = ALL_FILES_DIRS;
memset(wild_fcb.fcb.filename, ' ', 8);
memset(wild_fcb.fcb.extension, ' ', 3);
if (*name == NUL)
    wild_fcb.fcb.filename[0] = '.';
else if ((strcmp(name, ".") == 0) || (strcmp(name, "..") == 0))
    memcpy(wild_fcb.fcb.filename, name, strlen(name));
else
    {
    p = strchr(name, '.');
    if (p != (char *)NULL)
	len = p - name;
    else
	len = strlen(name);
    memcpy(wild_fcb.fcb.filename, name, len);
    if (p != (char *)NULL)
	{
	++p;
	len = strlen(p);
	memcpy(wild_fcb.fcb.extension, p, len);
	}
    }
inr.x.ax = SEARCHFIRST;
pf = (char far *)&wild_fcb;
inr.x.dx = FP_OFF(pf);
segr.ds  = FP_SEG(pf);
_intdosx( &inr, &outr, &segr );
if (outr.h.al == 0xFF)
    {
    if (    (strcmp(newcwd+1, ":/") == 0) &&
	    ((*name == NUL) || (strcmp(name, ".") == 0)) )
	{
	statbuf->st_dev = newdrive - 1;
	statbuf->st_ino = 0;
	statbuf->st_mode = _S_IFDIR | 0777;
	statbuf->st_nlink = 1;
	statbuf->st_uid = 1;
	statbuf->st_gid = 1;
	statbuf->st_rdev = newdrive - 1;
	statbuf->st_size = 0L;
	statbuf->st_atime = 315554400;	/* Tue Jan 01 00:00:00 1980 */
	statbuf->st_mtime = statbuf->st_atime;
	statbuf->st_ctime = statbuf->st_atime;
	retval = 0;
	}
    else
	{
	errno = ENOENT;
	retval = -1;
	}
    }
else
    {
    p = (char *)&read_fcb.fcb;
    dosdir = (struct dos_dirent *)(p+1);
    search = (struct dos_searchfcb *)&wild_fcb.fcb;

    statbuf->st_dev = search->real_drive - 1;
    statbuf->st_ino = dosdir->first_cluster;
    statbuf->st_mode = 0444;
    if (dosdir->attributes & _A_SUBDIR)
	statbuf->st_mode |= _S_IFDIR | 0111;
    else
	{
	statbuf->st_mode |= _S_IFREG;
	p = strrchr(filename, '.');
	if (	(p != (char *)NULL) &&
		(   (strcmp(p, ".exe") == 0) ||
		    (strcmp(p, ".com") == 0) ||
		    (strcmp(p, ".bat") == 0) ) )
	    statbuf->st_mode |= 0111;
	}
    if ((dosdir->attributes & _A_RDONLY) == 0)
	statbuf->st_mode |= 0222;
    statbuf->st_nlink = 1;
    statbuf->st_uid = 1;
    statbuf->st_gid = 1;
    statbuf->st_rdev = newdrive - 1;
    statbuf->st_size = dosdir->file_size;
    tm_buf.tm_sec  = (dosdir->write_time & 0x1F) * 2;
    tm_buf.tm_min  = (dosdir->write_time >> 5) & 0x3F;
    tm_buf.tm_hour = (dosdir->write_time >> 11) & 0x1F;
    tm_buf.tm_mday = dosdir->write_date & 0x1F;
    tm_buf.tm_mon  = ((dosdir->write_date >> 5) & 0x0F) - 1;
    tm_buf.tm_year = ((dosdir->write_date >> 9) & 0x7F) + 80;
    tm_buf.tm_wday  = 0;
    tm_buf.tm_yday  = 0;
    tm_buf.tm_isdst = 0;
    statbuf->st_atime = mktime(&tm_buf);
    if (tm_buf.tm_isdst)
	statbuf->st_atime -= 3600;
    statbuf->st_mtime = statbuf->st_atime;
    statbuf->st_ctime = statbuf->st_atime;
    retval = 0;
    }
/*
 *  restore the original directory and drive
 */
if (strcmp(oldcwd, newcwd) != 0)
    _chdir(oldcwd);
if (olddrive != newdrive)
    _chdrive(olddrive);
/*
 *  restore the original DTA value
 */
inr.x.ax = SETDTA;
inr.x.dx = save_dx;
segr.ds  = save_ds;
_intdosx( &inr, &outr, &segr );
return( retval );
}

/******************************************************************************
 * EDIT HISTORY
 ******************************************************************************
 * 20-Feb-95	SRMc - write stat() function for MS-DOS that fills in "inode"
 *			and "dev" fields uniquely for each file/directory
 */
