// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#ifndef DeviceClass
#define DeviceClass

#include <Lib3d/internals/Debuggable.H>
#include <Lib3d/internals/Array.H>
#include <Lib3d/Matrix.H>
#include <std.h>

// This should be broad enough to cover most framebuffer devices.

class Device : public Debuggable
{
public:
    virtual ~Device();

    uint getDepth()     const { return depth; }
    uint getWidth()     const { return width; }
    uint getHeight()    const { return height; }
    uint getPixelSize() const { return pixelSize; }
    uint getRowWidth()  const { return rowSize; }

    const char *getName() const { return "Device"; }
    uchar *getBuffer() { return frameBuf; }

    void setDirty( uint, uint, uint, uint );
    void clearBuffer( uint background = 0 );

    virtual bool requestResize( uint &requestX, uint &requestY );
    virtual void notifyResize();
    virtual void notifyExpose( uint, uint, uint, uint );

    virtual void swapBuffers() = 0;
    virtual uint allocateColour( uint red, uint green, uint blue ) = 0;

    enum {
	MouseCapability       = 0x01,
	KeyboardCapability    = 0x02,
    };

    static Device *create(uint height, 
			  uint width, 
			  uint min_depth, 
			  uint flags = 0);
    
    // Hmm...
    virtual void enableMouseCapability();
    virtual void enableKeyboardCapability();
    virtual void disableMouseCapability();
    virtual void disableKeyboardCapability();

    virtual void processPendingEvents();
    int getMouseXRel() { int i = mouseXRel; mouseXRel = 0; return i; }
    int getMouseYRel() { int i = mouseYRel; mouseYRel = 0; return i; }

protected:
    Device( Exemplar, uint capabilities = 0); 
    Device( uint width, uint height, uint min_depth );
    Device();			   

    virtual Device *clone( uint height, uint width, uint min_depth ) = 0;

    virtual int  estimateSpeed() const = 0;
    virtual bool initialize() = 0;
    void setSize( uint w, uint h ); 
    ostream &print( ostream &out ) const;
    
    bool isActive() const { return active; }

    static void registerChildClass( Device * );
    static int compare( const Device **, const Device ** );


private:
    static PointerArray<Device> *children;

protected:
    bool active;
    uchar *frameBuf;		// Assumed to be in X/Y order, (0,0)==top left.
    uint pixelSize;
    uint rowSize;
    uint width;
    uint height;
    uint depth;
    uint xmin, xmax;
    uint ymin, ymax;
    uint oldxmin, oldxmax;
    uint oldymin, oldymax;

    uint capabilities;		// Supported IO methods.
    uint enabled;		// Enabled IO methods.

    int mouseXRel;
    int mouseYRel;

public:
    static Exemplar exemplar;
};

inline void 
Device::setDirty(uint xn, uint yn, uint xx, uint yx )
{
    if_debug {
	debug() << "Setting dirty x:"<<xn<<"-"<<xx
	        << " y:"<<yn<<"-"<<yx
		<< endlog;
    }

    if (xmin > xn) xmin = xn;
    if (xmax < xx) xmax = xx;
    if (ymin > yn) ymin = yn;
    if (ymax < yx) ymax = yx;
}

#endif


