#include <stdio.h>
#include <dos.h>
#include <malloc.h>
#include <memory.h>
#include <time.h>

#include "mcontrol.h"
#include "msnd_dsp.h"

// 12/17/92  -  Changed for compatibility with Borland compiler
//              Added new make files for Borlands MAKE utility
//
// 3/12/93   -  Fixed Board Initialization code
//              UploadDSPcode and put in shared RAM initialization

// This file contains the functions and definitions used to directly
// Access the multisound card

#ifndef TRUE
#define TRUE  1
#define FALSE 0
#endif

#define MPU_WAIT_COUNTER 50
//;- mpu-401 equates

#define MPU_TX_DATA_RDY   0x040     //0 = mpu tx empty
                                    //1 = mpu tx full
#define MPU_RX_DATA_RDY   0x080     //0 = mpu rx data available (full)
                                    //1 = mpu rx empty

#define MPU_RESET         0x0FF
#define MPU_ACK           0x0FE
#define MPU_UART_MODE     0x03F
#define MPU_VERSION       0x0AC

#define MIDI_ACTIVE_SENSING 0x0FE
#define MIDI_RES_SEND_TO_IS 0x0F9
#define MIDI_RES_SEND_TO_EO 0x0FD

// MPU equates.
#define HP_MPU_DATA   0x0
#define HP_MPU_STAT   0x1
#define HP_MPU_CMD    0x1

#define MPUSR_RX_BIT  (1<<7)
#define MPUSR_TX_BIT  (1<<6)


#define MAX_FIFO_READ  3    //max number to try to read out of fifo after reset

//      *************************************************
//      *            Global Variable Definitions                                          *
//      *************************************************

void( __interrupt __far *nDSPSaveVect )();
void( __interrupt __far *nMPUSaveVect )();


BYTE bIrq, bMem;
WORD wBASEIO;
WORD wCFGIO;
int  nIRQValue;

WORD wMPUIO = 0;
WORD nMPUIRQ = 5;
WORD wCDIO0 = 0;
WORD wCDIO1 = 0;
WORD nCDIRQ = 11;
WORD wJOYIO = 0;

BYTE bMPUInStarted = 0;
DWORD dwMPUInterruptCounter = 0;


BYTE bDspIsrRunning = 0;
BYTE bMpuIsrRunning = 0;
WORD *pwHostQueue , *pwHostHead , *pwHostTail;
struct SMA0_CommonData far *SMA;
struct JobQueueStruct far *DAPQ , *DARQ , *MODQ , *MIDQ , *DSPQ;
struct DAQueueDataStruct far *CurDAQD;
struct DAQueueDataStruct far *CurDARQD;
WORD far *pwDSPQData , *pwMIDQData , *pwMODQData;
BYTE bCurrBank;
BYTE far *pbDSPinit;
BYTE far *pbDSPperm;
BYTE bCurrentMidiPatch;

BYTE bCurrMOP = HWINIT_MOP; // init values so it gets set at least once.
BYTE bCurrMIP = HWINIT_MOP;


#define MPU_BUFFER_SIZE      2048

BYTE *pMpuDataBuffer = NULL;
BYTE *pMpuDataHead = NULL;
BYTE *pMpuDataTail = NULL;
BYTE *pMpuDataEnd = NULL;


void _interrupt _far DSPInterrupt();
void _interrupt _far MPUInterrupt();

// external variables defined in dspcode.obj

extern BYTE     far bDspInitStart;
extern WORD     far wDspInitSize;
extern BYTE     far bDspPermStart;
extern WORD     far wDspPermSize;
extern WORD     far wDspPermWords;

#define FAIL goto Fail

#define TIMER_MS 2381       // 10 Millisecond on the PC high resoluition timer.



// some defines for the PNP configuraiton routines.
typedef DWORD CONFIGRET;
#define CR_SUCCESS 0
#define CR_FAILURE -1

// data structures for PNP configuration routines.
typedef struct tagLDDATA {
  BYTE bLDNumber;
//  BOOL fActive;
  WORD wIOAddr0;
  WORD wIOAddr1;
  WORD wIRQ;
  WORD wRAMAddr;
} LDDATA, *PLDDATA ;

// PNP register and function definitions.
// function returns
#define   CFGOP_NOERR            0x00
#define   CFGOP_TIMEOUT          0x01
#define   CFGOP_BADVERIFY        0x03
#define   CFGOP_BADFILENAME      0x04
#define   CFGOP_NOHARDWARE       0x05
#define   CFGOP_NOREADPORT       0x00
#define   CFGOP_CANTWRITE        0x06
#define   CFGOP_NOMEM            0x07

// Configuration register definitions
// device regs
#define   IREG_LOGDEVICE    0x07
#define   IREG_ACTIVATE     0x30
#define   LD_ACTIVATE       0x01
#define   LD_DISACTIVATE    0x00
#define   IREG_EECONTROL    0x3F

// memory window regs
#define   IREG_MEMBASEHI    0x40
#define   IREG_MEMBASELO    0x41
#define   IREG_MEMCONTROL   0x42
#define   IREG_MEMRANGEHI   0x43
#define   IREG_MEMRANGELO   0x44
#define   MEMTYPE_8BIT      0x00
#define   MEMTYPE_16BIT     0x02
#define   MEMTYPE_RANGE     0x00
#define   MEMTYPE_HIADDR    0x01

// I/O regs
#define   IREG_IO0_BASEHI   0x60
#define   IREG_IO0_BASELO   0x61
#define   IREG_IO1_BASEHI   0x62
#define   IREG_IO1_BASELO   0x63

// IRQ regs
#define   IREG_IRQ_NUMBER   0x70
#define   IREG_IRQ_TYPE     0x71
#define   IRQTYPE_HIGH      0x02
#define   IRQTYPE_LOW       0x00
#define   IRQTYPE_LEVEL     0x01
#define   IRQTYPE_EDGE      0x00

// forward declarations.
CONFIGRET WriteLogicalDevice( LDDATA *ld, WORD wPortAddress);
CONFIGRET ConfigureRAM(BYTE bLogDev, WORD wRAMAddress, WORD wPortAddress);
CONFIGRET ConfigureIRQ(BYTE bLogDev, WORD wIRQ, WORD wPortAddress);
CONFIGRET ConfigureIO1(BYTE bLogDev, WORD wIOAddress, WORD wPortAddress);
CONFIGRET ConfigureIO0(BYTE bLogDev, WORD wIOAddress, WORD wPortAddress);
CONFIGRET ActivateLogicalDevice( BYTE bLogDev, WORD wPortAddress);
CONFIGRET Config_Hardware(void);
CONFIGRET CheckDSPHardware();
CONFIGRET CheckDSPMem();
CONFIGRET VerifyHWExists(WORD wPortAddress);
BYTE ReadCFGRegister(WORD wRegister, WORD wPortAddress);


#define INT1 {_asm int 1}
#define INT3 {_asm int 3}



/************************************************************************
 *
 *    function : void InitializeSMA( void )
 *
 *    params   : none
 *
 *    operation: This function initializes the common data in the SMA,
 *                           and initializes the hosts pointers to that area.
 *
 *    returns  : none
 *
 * **********************************************************************/
void InitializeSMA( void )
{
    WORD wInker;
    int n;
    struct DAQueueDataStruct far *lpDAQ;


    if (Config_Hardware() != CR_SUCCESS)
    {
        printf("Initialize SMA failed\n");
        exit(0);
    }

    if (CheckDSPHardware() != CR_SUCCESS)
        exit(0);





    outp( ( wBASEIO + HP_BLKS ) , HPBLKSEL_0 );
    for( wInker = 0 ; wInker < 0x8000 ; wInker++ ) {
        *( pMEM.p + wInker ) = 0x00;
    }

    outp( ( wBASEIO + HP_BLKS ) , HPBLKSEL_1 );
    for( wInker = 0 ; wInker < 0x8000 ; wInker++ ) {
        *( pMEM.p + wInker ) = 0x00;
    }

    outp( ( wBASEIO + HP_BLKS ) , HPBLKSEL_0 );

    DAPQ = ( struct JobQueueStruct far * )( pMEM.p + DAPQ_OFFSET );
    DARQ = ( struct JobQueueStruct far * )( pMEM.p + DARQ_OFFSET );
    MODQ = ( struct JobQueueStruct far * )( pMEM.p + MODQ_OFFSET );
    MIDQ = ( struct JobQueueStruct far * )( pMEM.p + MIDQ_OFFSET );
    DSPQ = ( struct JobQueueStruct far * )( pMEM.p + DSPQ_OFFSET );

    SMA = ( struct SMA0_CommonData far * )( pMEM.p + SMA_STRUCT_START );

    CurDAQD = ( struct DAQueueDataStruct far * )( pMEM.p + DAPQ_DATA_BUFF );
    CurDARQD = ( struct DAQueueDataStruct far * )( pMEM.p + DARQ_DATA_BUFF );


    // initialize the data queue structures.
    for( n= 0, lpDAQ = CurDAQD ; n < 3 ; n++, lpDAQ) 
    {
	    lpDAQ->wStart      = PCTODSP_BASED( (DWORD)( DAP_BUFF_SIZE * n) );
	    lpDAQ->wSize       = DAP_BUFF_SIZE;
	    lpDAQ->wFormat     = 1;
	    lpDAQ->wSampleSize = 16;
	    lpDAQ->wChannels   = 1;
	    lpDAQ->wSampleRate = 44100;
	    lpDAQ->wIntMsg     = HIMT_PLAY_DONE * 0x100 + n;
	    lpDAQ->wFlags      = n + 1;
    }

    for( n= 0, lpDAQ = CurDARQD ; n < 3 ; n++, lpDAQ)
    {

        lpDAQ->wStart      = PCTODSP_BASED( (DWORD)( DAR_BUFF_SIZE * n) ) + 0x4000;
        lpDAQ->wSize       = DAR_BUFF_SIZE;
        lpDAQ->wFormat     = 1;
        lpDAQ->wSampleSize = 16;
        lpDAQ->wChannels   = 2;        
        lpDAQ->wSampleRate = 44100;
        lpDAQ->wIntMsg     = HIMT_RECORD_DONE * 0x100 + n;
        lpDAQ->wFlags      = n + 1;

    }
    
    pwDSPQData = (WORD far *)( pMEM.p + DSPQ_DATA_BUFF );
    pwMODQData = (WORD far *)( pMEM.p + MODQ_DATA_BUFF );
    pwMIDQData = (WORD far *)( pMEM.p + MIDQ_DATA_BUFF );


    MIDQ->wStart = PCTODSP_BASED( MIDQ_DATA_BUFF );
    MIDQ->wSize  = PCTODSP_OFFSET( MIDQ_BUFF_SIZE ) - 1;
    MIDQ->wHead  = 0;
    MIDQ->wTail  = 0;

    MODQ->wStart = PCTODSP_BASED( MODQ_DATA_BUFF );
    MODQ->wSize  = PCTODSP_OFFSET( MODQ_BUFF_SIZE ) - 1;
    MODQ->wHead  = 0;
    MODQ->wTail  = 0;

    DAPQ->wStart = PCTODSP_BASED( DAPQ_DATA_BUFF );
    DAPQ->wSize  = PCTODSP_OFFSET( DAPQ_BUFF_SIZE ) - 1;
    DAPQ->wHead  = 0;
    DAPQ->wTail  = 0;

    DARQ->wStart = PCTODSP_BASED( DARQ_DATA_BUFF );
    DARQ->wSize  = PCTODSP_OFFSET( DARQ_BUFF_SIZE ) - 1;
    DARQ->wHead  = 0;
    DARQ->wTail  = 0;

    DSPQ->wStart = PCTODSP_BASED( DSPQ_DATA_BUFF );
    DSPQ->wSize  = PCTODSP_OFFSET( DSPQ_BUFF_SIZE ) - 1;
    DSPQ->wHead  = 0;
    DSPQ->wTail  = 0;

    SMA->wCurrPlayBytes = 0;
    SMA->wCurrRecordBytes = 0;
    SMA->wCurrPlayVolLeft = 0x7FFF;
    SMA->wCurrPlayVolRight =0x7FFF;
	SMA->wCurrInVolLeft = 0x7FFF;           // current wave output volume - left
	SMA->wCurrInVolRight = 0x7FFF;       // current wave output volume - right

	SMA->dwCurrPlayPitch =  0x00010000;     // current play pitch: MSW = int, LSW = frac
	SMA->dwCurrPlayRate =   0x0001;         // current play rate: MSW = int, LSW = frac

    SMA->wCurrDSPStatusFlags = 0;
    SMA->wCurrHostStatusFlags = 0;
    SMA->wCurrInputTagBits = 0x303;
    SMA->wCurrLeftPeak = 0;
    SMA->wCurrRightPeak = 0;


    SMA->bInPotPosRight = 0;
    SMA->bInPotPosLeft = 0;

    SMA->bAuxPotPosRight = 0;
    SMA->bAuxPotPosLeft = 0;


    SMA->wCurrPlayFormat = 1;
    SMA->wCurrPlaySampleSize = 16;
    SMA->wCurrPlayChannels = 2;
    SMA->wCurrPlaySampleRate = 44100;

    SMA->wCurrRecordFormat = 1;
    SMA->wCurrRecordSampleSize = 16;
    SMA->wCurrRecordChannels = 2;
    SMA->wCurrRecordSampleRate = 44100;

    SMA->wCurrMastVolLeft = 0;
    SMA->wCurrMastVolRight = 0;

    SMA->wCalFreqAtoD = 44100;


}




/************************************************************************
 *
 *    function : BYTE SetAuxVolume( BYTE bValue , BYTE bLRBN )
 *
 *    params   : bValue - The new volume value
 *               bLRNM  - Can be any of the following
 *                                  RIGHT - Set right volume to bValue
 *                                  LEFT  - Set left volume to bValue
 *                                  GANG  - Set left and right to bValue
 *                                  NONE  - Don't change value but send update
 *                                                  message to DSP
 *
 *    operation: This function tells the DSP to change the the auxiliary
 *                           volume based on the parameters sent to it.
 *
 *    returns  : 1  -  if succesful
 *                         0  -  if function fails
 *
 * **********************************************************************/

BYTE SetAuxVolume( BYTE bValue , BYTE bLRBN )
{
    if( bLRBN != NONE ) 
    {
        if( bLRBN & RIGHT ) 
            SMA->bAuxPotPosRight = bValue;
        if( bLRBN & LEFT ) 
            SMA->bAuxPotPosLeft = bValue;
    }

    if( SendHostWord( 0x00 , 0x00 , HDEXAR_AUX_SET_POTS ) &&
        SendDSPCommand( HDEX_AUX_REQ ) ) 
    {
        return( 1 );
    }
    return( 0 );
}
/************************************************************************
 *
 *    function : BYTE SetMicVolume( BYTE bValue , BYTE bLRBN )
 *
 *    params   : bValue - The new volume value
 *               bLRNM  - Can be any of the following
 *                                  RIGHT - Set right volume to bValue
 *                                  LEFT  - Set left volume to bValue
 *                                  GANG  - Set left and right to bValue
 *                                  NONE  - Don't change value but send update
 *                                                  message to DSP
 *
 *    operation: This function tells the DSP to change the the auxiliary
 *                           volume based on the parameters sent to it.
 *
 *    returns  : 1  -  if succesful
 *                         0  -  if function fails
 *
 * **********************************************************************/

BYTE SetMicVolume( BYTE bValue , BYTE bLRBN )
{
    if( bLRBN != NONE ) 
    {
        if( bLRBN & RIGHT ) 
            SMA->bMicPotPosRight = bValue;
        if( bLRBN & LEFT ) 
            SMA->bMicPotPosLeft = bValue;
    }

    if( SendHostWord( 0x00 , 0x00 , HDEXAR_MIC_SET_POTS) &&
        SendDSPCommand( HDEX_AUX_REQ ) ) 
    {
        return( 1 );
    }
    return( 0 );
}

/************************************************************************
 *
 *    function : BYTE SetInVolume( BYTE bValue , BYTE bLRBN )
 *
 *    params   : bValue - The new volume value
 *               bLRNM  => Can be any of the following
 *                                  RIGHT - Set right volume to bValue
 *                                  LEFT  - Set left volume to bValue
 *                                  GANG  - Set left and right to bValue
 *                                  NONE  - Don't change value but send update
 *                                                  message to DSP
 *
 *    operation: This function tells the DSP to change the the record
 *                           input volume based on the parameters sent to it.
 *
 *    returns  : 1  -  if successful
 *                           0  -  if not successful
 *
 * **********************************************************************/

BYTE SetInVolume( BYTE bValue , BYTE bLRBN )
{
    if( bLRBN != NONE ) 
    {
        if( bLRBN & RIGHT ) 
            SMA->bInPotPosRight = bValue;
        if( bLRBN & LEFT ) 
            SMA->bInPotPosLeft = bValue;
    }
    if( SendHostWord( 0x00 , 0x00 , HDEXAR_IN_SET_POTS ) &&
            SendDSPCommand( HDEX_AUX_REQ ) ) 
    {
        return( 1 );
    }
    return( 0 );
}

/************************************************************************
 *
 *    function : BYTE ClearPeaks( void )
 *
 *    params   : none
 *
 *    operation: This function clears the peak values stored in the
 *                           SMA
 *
 *    returns  : 1  -  if succesful
 *                         0  -  if function fails
 *
 * **********************************************************************/

BYTE ClearPeaks( void )
{
    if( SendHostWord( 0x00 , 0x00 , HDEXAR_CLEAR_PEAKS ) &&
                SendDSPCommand( HDEX_AUX_REQ ) ) return( 1 );
    return( 0 );
}

/************************************************************************
 *
 *    function : WORD GetExtDSPBits( void )
 *
 *    params   : none
 *
 *    operation: Updates and returns the values of the external DSP bits
 *
 *    returns  : The value of SMA->wExtDSPbits
 *
 * **********************************************************************/

WORD GetExtDSPBits(void )
{

	WORD ExtDSPBits = 0;

    // NOTE: wExtDSP bits are no longer part of the Pinnacle SMA.

//	SendHostWord(pHw,0x00,0x00,HDEXAR_RD_EXT_DSP_BITS);
//	SendDSPCommand(pHw,HDEX_AUX_REQ);

//	bBank = SetSMABank(pHw,HPBLKSEL_0);
//	ExtDSPBits = READ_REGISTER_USHORT(&pHw->SMA->wExtDSPbits);
//	bBank = SetSMABank(pHw,bBank);
	
	return ExtDSPBits;
}

/************************************************************************
 *
 *    function : BYTE CalibrateAD( WORD wSampleRate , BYTE bGndType )
 *
 *    params   : wSampleRate - The sample rate to calibrate to
 *               bGndType  => Can be one of the following
 *                                   SIGNAL - calibrate to signal ground
 *                   AGND   - calibrate to board AGND
 *
 *    operation: This function calibrates MultiSounds A/D converters
 *                           based on the sample rate and calibration type passed
 *
 *    returns  : 1 - success
 *                           0 - failure
 *
 * **********************************************************************/

BYTE CalibrateAD( WORD wSampleRate , BYTE bGndType )
{

    // NOTE: this routine appears in the NT driver, but not the pinnacle
    // Win '95 or Win 3.1 driver.
    BYTE bRetVal;


    if( bGndType == AGND )
        SMA->wCurrHostStatusFlags |= 0x0001;
    else
        SMA->wCurrHostStatusFlags &= ~(0x0001);


    SMA->wCalFreqAtoD = wSampleRate;

    if( SendHostWord( 0x00 , 0x00 , HDEXAR_CAL_A_TO_D ) &&
                    SendDSPCommand( HDEX_AUX_REQ ) ) 
    {
        bRetVal = 1;
    }
    else
        bRetVal = 0;

    // The actual time for the calibration is equal to
    // 4096 / SAMPLE_RATE
    // We will poll DCAL to wait untill the calibration is complete

    // NOTE: For the pinnacle, there may be another way for the calibration to be 
    // complete.. Have to check here.
    while( GetExtDSPBits() & ~( EXT_DSP_BIT_DCAL ) );
    return( bRetVal );
}

/************************************************************************
 *
 *    function : WORD MIDPortRead()
 *
 *    params   : none
 *
 *    operation: This function reads the MIDI data from the MIDI in queue
 *                           and returns the data
 *
 *    returns  : The MIDI data read from the queue
 *
 * **********************************************************************/

WORD MIDPortRead( void )
{
    WORD wRetVal , wHead;

    wHead = MIDQ->wHead;
    if( ++wHead > MIDQ->wSize ) wHead = 0;

    wRetVal = *( pwMIDQData + MIDQ->wHead );

    MIDQ->wHead = wHead;

    return( wRetVal );
}


/************************************************************************
 *
 *    function : void MODPortWrite( WORD wMIDIData )
 *
 *    params   : wMIDIData =>
 *                                  0xFFxx = Patch Change
 *               0x01xx - 0xFExx = Host defined message
 *                                  0x00xx = MIDI out data
 *
 *    operation: This function sends the MIDI data contained in
 *                           wMIDIData out to the MIDI queue
 *
 *    returns  : none
 *
 * **********************************************************************/

void MODPortWrite( WORD wMIDIData )
{
    WORD wNewTail , wOldTail;

    wNewTail = MODQ->wTail;
    wOldTail = wNewTail;
    if( ++wNewTail > MODQ->wSize ) 
        wNewTail = 0;

    while( wNewTail == MODQ->wHead );     // Wait if Buffer is full

    *( pwMODQData + MODQ->wTail ) = wMIDIData;

    MODQ->wTail = wNewTail;
    if( MODQ->wHead == wOldTail ) 
        SendDSPCommand( HDEX_MIDI_OUT_START );
}


/************************************************************************
 *
 *    function : void SetMidiInPort( BYTE bNewMIP )
 *
 *    params   : bNewMIP  =  The new MIP mask to change to
 *
 *    operation: This function changes the midi input port to
 *               the value defined by bNewMIP.
 *
 *    returns  : none
 *
 * **********************************************************************/

void SetMidiInPort( BYTE bNewMIP )
{
    BYTE bNewPatch;
    WORD wMipData;
    WORD MIPDataTable[] = {             // midi input port selection data table
        0x0003,                         // 0 = ext in
        0x0403                          // 1 = WAVEHDR
    };

    if (bNewMIP == bCurrMIP)    // no change, nothing to do.
        return;
    if (bNewMIP & 0x80)         // no port selected, should not happen.
        return;
    if (bNewMIP > MAX_MOP)      // range check this.
        return;
    
    wMipData = MIPDataTable[bNewMIP]; // get mask and set data.
    bNewPatch = (bCurrentMidiPatch & LOBYTE(wMipData)) | HIBYTE(wMipData);

    if (bNewPatch == bCurrentMidiPatch) // any real changes???
        return;                         // no, get out.
    bCurrentMidiPatch = bNewPatch;      // remember this
    
    MODPortWrite( 0xFE00 );
    MODPortWrite( MAKEWORD( bCurrentMidiPatch, 0xFF ) );

}


/************************************************************************
 *
 *    function : void SetMidiOutPort( BYTE bNewMOP )
 *
 *    params   : bNewMOP  =  The new MOP mask to change to
 *
 *    operation: This function changes the midi output port to
 *               the value defined by bNewMOP.
 *
 *    returns  : none
 *
 * **********************************************************************/

void SetMidiOutPort( BYTE bNewMOP )
{
    BYTE bNewPatch;
    WORD wMopData;
    WORD MOPDataTable[] = {             // midi output port selection data table
        0x0104,                         // 0 = WAVEHDR
        0x0204                          // 1 = ext out
                                        // and - turns off all outputs leaves input active
                                        // or - turns on correct output
    };

    if (bNewMOP == bCurrMOP)    // no change, nothing to do.
        return;
    if (bNewMOP & 0x80)         // no port selected, should not happen.
        return;
    if (bNewMOP > MAX_MOP)      // range check this.
        return;
    
    wMopData = MOPDataTable[bNewMOP]; // get mask and set data.
    bNewPatch = (bCurrentMidiPatch & LOBYTE(wMopData)) | HIBYTE(wMopData);

    if (bNewPatch == bCurrentMidiPatch) // any real changes???
        return;                         // no, get out.
    bCurrentMidiPatch = bNewPatch;      // remember this
    
    MODPortWrite( 0xFE00 );
    MODPortWrite( MAKEWORD( bCurrentMidiPatch, 0xFF ) );

    
}



/************************************************************************
 *
 *    function : word wsnaptimer( void )
 *
 *    params   : none
 *
 *    operation: This function get the current timer value.
 *
 *    returns  : The value from the timer.
 *
 * **********************************************************************/
WORD wsnaptimer(void)
{
  BYTE b0,b1;

    outp(0x43,0);           // freeze the timer
    b0 = inp(0x40);                 // read it
    b1 = inp(0x40);
    return(MAKEWORD(b0,b1));

}

 /************************************************************************
 *
 *    function : void DelayOneMsec( void )
 *
 *    params   : none
 *
 *    operation: This function returns after one millisecond.
 *
 *    returns  : none
 *
 * **********************************************************************/

void DelayOneMsec(void)
{
    WORD w0,w1;

    w0 = wsnaptimer();
    while ((w0-wsnaptimer()) < 2381) {};
}

/************************************************************************
 *
 *    function : void GetDelayTime( WORD wDelay )
 *
 *    params   : wDelay  -  The thousandths of seconds to wait
 *
 *    operation: This function delays for wDelay milliseconds
 *               using the DelayOneMsec function
 *
 *    returns  : none
 *
 * **********************************************************************/

void GetDelayTime( WORD wDelay )
{
    while ( wDelay--)  DelayOneMsec();
}


/************************************************************************
 *
 *    function : void _interrupt _far DSPInterrupt( void )
 *
 *    params   : none
 *
 *    operation: This function handles the interrupts from MultiSound
 *               It reads the message into the host queue, and then
 *                           terminates the interrupt with EOI
 *
 *    returns  : none
 *
 * **********************************************************************/

void _interrupt _far DSPInterrupt( void )
{
#define EOI_AT_END 0
    if( bDspIsrRunning != 1 ) 
    {
        bDspIsrRunning = 1;
        if( bCurrBank ) 
            outp( wBASEIO + HP_BLKS , HPBLKSEL_0 );
        inp( wBASEIO + HP_RXL );

#if (EOI_AT_END == 0)
        if( nIRQValue > 7 ) 
            outp( PIC2 , EOI );
        outp( PIC1 , EOI );
#endif


        while( DSPQ->wTail != DSPQ->wHead )  
        {
            if( ++pwHostTail >= ( pwHostQueue + HOSTQ_SIZE ) ) 
                pwHostTail = pwHostQueue;
        
            *pwHostTail = *( pwDSPQData + DSPQ->wHead );
        
            if( ++DSPQ->wHead > DSPQ->wSize ) 
                DSPQ->wHead = 0x00;
        }
        if( bCurrBank ) 
            outp( wBASEIO + HP_BLKS , HPBLKSEL_1 );

        bDspIsrRunning = 0;
    }
#if EOI_AT_END
        if( nIRQValue > 7 ) 
            outp( PIC2 , EOI );
        outp( PIC1 , EOI );
#endif
}

/************************************************************************
 *
 *    function : BYTE UploadDSPCode( void )
 *
 *    params   : none
 *
 *    operation: uploads the DSP code contained in the code segment
 *                           (dspcode.obj) to the DSP.  The initial code is sent up
 *                           through the host port, while the permanent code is
 *                           copied into the SMA.
 *
 *    returns  : 1 if upload successful
 *               0 if upload not successful
 *
 * **********************************************************************/

BYTE UploadDSPCode( void )
{
    int nInker;
    int nTimeOutCount = 20000;
    int nWait;


    pbDSPperm = &bDspPermStart;
    pbDSPinit = &bDspInitStart;



    // reset and select bank 0
    outp( wBASEIO + HP_DSPR , HPDSPRESET_ON );

    //outp( ( wBASEIO+ HP_BLKS ), HPBLKSEL_0);


    _fmemcpy( pMEM.p , pbDSPperm , wDspPermSize );


    // release the DSP, it should be waiting to upload the code.
    outp( wBASEIO + HP_DSPR , HPDSPRESET_OFF );

    DelayOneMsec();
    // wait for the DSP to settle down.
    while(nTimeOutCount-- >= 0) {
      if( inp(wBASEIO + HP_CVR) == 0x12 ) 
      {
         break;
      }
      DelayOneMsec();
    }
    if (nTimeOutCount < 0)
        return(0);


    for( nInker = 0; nInker < 3*wDspInitSize ; nInker += 3) {
      if( ( SendHostWord( *(pbDSPinit+nInker), *(pbDSPinit+nInker+1), *(pbDSPinit+nInker+2)))==0) {
      return(0);
      }
    }

    inp(wBASEIO + HP_RXL);      // clear rx data port(s)

    inp(wBASEIO + HP_CVR);      // Fetch the cmd vect reg

    // now wait until we get the green light.
    nWait = 2000;   // wait 2000 milliseconds.

    // wait until we get the green light.
    while( *( pMEM.p ) || *(pMEM.p + 1) )
    {
        DelayOneMsec();
        if (--nWait < 0)
            return(0);
    }


    return(1);
}

/************************************************************************
 *
 *    function : int uploadbin( char *szBinName )
 *
 *    params   : filename - then name of the bin file to upload
 *
 *    operation: uploads the bin file to the DSP
 *
 *    returns  : 1 if upload successful
 *               0 if upload not successful
 *
 * **********************************************************************/

BYTE uploadbin(char *szBinName )
{
   FILE *fBinFile;
   int lRSize, nInker = 0;
   static BYTE bBinBuff[512*3];

   while( nInker < sizeof( bBinBuff ) ) {
      bBinBuff[ nInker++ ] = 0;
      }

   if((fBinFile = fopen( szBinName, "rb" )) == NULL ) {
      printf("Error opening file : %s\n", szBinName );
      return(0);
      }

   // lRSize = fread( bBinBuff, sizeof( BYTE ), sizeof(bBinBuff), fBinFile);
   fread( bBinBuff, sizeof( BYTE ), sizeof(bBinBuff), fBinFile);

   for( nInker = 0; nInker < sizeof( bBinBuff ); nInker += 3) {
      if( ( SendHostWord( bBinBuff[nInker], bBinBuff[nInker+1], bBinBuff[nInker+2] ) ) == 0 ) {
         printf("Error uploading word %d\n", nInker );
         fclose( fBinFile );
         return(0);
      }
   }

   fclose(fBinFile);
   return(1);
}


/************************************************************************
 *
 *    function : int uploadreb( char *szrebstrname )
 *
 *    params   : filename - then name of the reb file to upload
 *
 *    operation: uploads the reb file to the DSP
 *
 *    returns  : 1 if upload successful
 *               0 if upload not successful
 *
 * **********************************************************************/

BYTE uploadreb(char *rebstrname)
  {
  FILE *rebfile;
  int i;

  if ((rebfile = fopen(rebstrname,"rb")) == NULL )
      {
      printf("Trouble opening %s\n",rebstrname);
      return(0);
      }
  outp( ( wBASEIO+ HP_BLKS ), HPBLKSEL_0);
  outp( ( wBASEIO+ HP_MEMM ), bMem );

  fread( pMEM.p , sizeof(BYTE), wDspPermSize, rebfile);
//  fread( pMEM.p , sizeof(BYTE), BUFFSIZE, rebfile);
  fclose( rebfile );
  return(1);
  }


/************************************************************************
 *
 *    function : BYTE SendHostWord( BYTE bHi , BYTE bMid , BYTE bLow )
 *
 *    params   : The hi, mid, and low byte respectively of the data
 *               to be sent to the DSP receive data address
 *
 *    operation: calls WaitTXDEmpty to wait until the host transmit
 *               data register has been cleared by the DSP.  The
 *               loading of the low order byte tells the DSP that
 *               the data is present.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
BYTE SendHostWord(BYTE bHi, BYTE bMid , BYTE bLow )
{
   if( WaitTXDEmpty() ) {
      outp( wBASEIO + HP_TXH , bHi );
      outp( wBASEIO + HP_TXM , bMid );
      outp( wBASEIO + HP_TXL , bLow );
      return(1);
      }
   return(0);
}

/************************************************************************
 *
 *    function : BYTE WaitTXDEmpty( void )
 *
 *    params   : none
 *
 *    operation: Waits for the DSP to read any data in the transmit
 *               registers.
 *
 *    returns  : 1 if transmit registers are clear
 *               0 if function times out waiting for TXDE
 *
 * **********************************************************************/
BYTE WaitTXDEmpty( void )
{
   int nTimeOutCount = 20000;

   while(nTimeOutCount-- >= 0) {
      if( inp(wBASEIO + HP_ISR) & HPISR_TXDE ) {
         return(1);
      }
   }
   return(0);
}

/************************************************************************
 *
 *    function : BYTE WaitHostClear( void )
 *
 *    params   : none
 *
 *    operation: Waits for HP_CVR to match HPCVR_HC.  Used in sending
 *               commands to the DSP
 *
 *    returns  : 1 if successful
 *               0 if operation times out
 *
 * **********************************************************************/

BYTE WaitHostClear( void )
{
   int nTimeOutCount = 20000;

   while(nTimeOutCount-- >= 0) {
      if( !(inp(wBASEIO + HP_CVR) & HPCVR_HC )) {
         return(1);
      }
   }
   return(0);
}

/************************************************************************
 *
 *    function : BYTE ResetDSP( void )
 *
 *    params   : none
 *
 *    operation: resets the DSP to initialize operation
 *
 *    returns  : 1 if successful
 *               0 if DSP times out
 *
 * **********************************************************************/

BYTE ResetDSP( void )
{

    int nTimeOutCount = 20000;


    outp( wBASEIO + HP_DSPR , HPDSPRESET_ON );
    DelayOneMsec();
    outp( wBASEIO + HP_DSPR , HPDSPRESET_OFF );


    DelayOneMsec();
    // wait for the DSP to settle down.
    while(nTimeOutCount-- >= 0) {
      if( inp(wBASEIO + HP_CVR) == 0x12 ) 
      {
         break;
      }
      DelayOneMsec();
    }
    if (nTimeOutCount < 0)
        return(0);

    return(1);

}


/************************************************************************
 *
 *    function : BYTE SendDSPCommand( BYTE bCommand )
 *
 *    params   : bCommand - the command to send to the DSP
 *
 *    operation: Sends the command bCommand to the DSP
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/

BYTE SendDSPCommand( BYTE bCommand )
{
    if( WaitHostClear() ) {
        outp( wBASEIO + HP_CVR , bCommand );
        return(1);
    }
    return(0);
}


/************************************************************************
 *
 *    function : BYTE SetIntMap( bIrq )
 *
 *    params   : bIrq - The interrupt the MultiSound is on
 *
 *    operation: Sets up the Hardware Interrupt on the MultiSound.  The order
 *               of the calls below must be followed to successfully setup
 *               the interrupt.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/


BYTE SetIntMap( BYTE bIrq )
{
  if( WaitTXDEmpty() ) {
    outp( wBASEIO + HP_ICR , inp( wBASEIO + HP_ICR ) | (HPICR_TREQ) );
    // this is done by the PNP programming now.  
    // Should we reconfigure the hardware???
    //outp( wBASEIO + HP_IRQM , bIrq );
    if (bIrq != HPIRQ_NONE)
        outp( wBASEIO + HP_ICR , inp( wBASEIO + HP_ICR ) & ~(HPICR_TREQ));
    return(1);
    }
return(0);
}


/************************************************************************
 *
 *    function : BYTE SetupMsndIRQ()
 *
 *    params   : none
 *
 *    operation: Sets up the PC to be able to receive interrupts from
 *               the MultiSound, and saves the previous DOS interrupt
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/


BYTE SetupMsndIRQ()
{

    WORD wGP;
    WORD wPic;
    BYTE cIrqMask;
    BYTE cIntBit;


    if( nIRQValue > 7 ) 
    {
        wGP = nIRQValue + 0x68;
        cIntBit = ( 1 << (nIRQValue - 8));
        wPic = PIC2+1;
    }
    else
    {
        wGP = nIRQValue + 8;
        cIntBit = (1 <<  nIRQValue );
        wPic = PIC1+1;
    }



    if( SetIntMap( bIrq ) == 0 ) {
        return(0);
    }


    nDSPSaveVect = _dos_getvect( wGP );
    _dos_setvect( wGP , DSPInterrupt );


    CLI;    // lock out interrupts.

    cIrqMask = inp( wPic);
        // only enable it if it is not already enabled.
    if (cIrqMask & cIntBit)
        outp( wPic, ((~cIntBit) & cIrqMask) );

    outp( wBASEIO + HP_ICR , inp( wBASEIO + HP_ICR ) | HPICR_RREQ );

    STI;

    return(1);
}


/************************************************************************
 *
 *    function : BYTE ResetMsndIRQ()
 *
 *    params   : none
 *
 *    operation: Resets the PC link to the MultiSound interrupt from
 *               the MultiSound, and restores the DOS interrupt
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/

BYTE ResetMsndIRQ()
{
    WORD wGP;
    WORD wPic;
    BYTE cIrqMask;
    BYTE cIntBit;


    if( nIRQValue > 7 ) 
    {
        wGP = nIRQValue + 0x68;
        cIntBit = ( 1 << (nIRQValue - 8));
        wPic = PIC2+1;
    }
    else
    {
        wGP = nIRQValue + 8;
        cIntBit = (1 <<  nIRQValue );
        wPic = PIC1+1;
    }

    // turn off both interrupt sources
    outp( wBASEIO + HP_ICR , inp( wBASEIO + HP_ICR ) & ~(HPICR_RREQ | HPICR_TREQ));

    if( SetIntMap( HPIRQ_NONE ) == 0 ) {
        return(0);
    }

    CLI;

    cIrqMask = inp( wPic);

    // already disabled....
    if ((cIrqMask & cIntBit) == 0)
        outp( wPic,  (cIntBit | cIrqMask) );

    STI;

    _dos_setvect( wGP , nDSPSaveVect );
  
    return(1);
}



/************************************************************************
 *
 *    function : WriteCFGRegister(WORD wRegister, BYTE bValue, WORD wPortAddress)
 *
 *    params   : wRegister -- configuration register to write to.
 *               bValue -- value to write
 *               wPortAddress -- PNP configuration port address.
 *
 *    operation: Write the PNP configuration register.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET WriteCFGRegister(WORD wRegister, BYTE bValue, WORD wPortAddress)
{
  BYTE bValIn;

  outp(wPortAddress, wRegister);
  outp(wPortAddress+1, bValue);

  bValIn = inp(wPortAddress+1);
  if (bValIn != bValue)
    {
      return CR_FAILURE;
    }

  return CR_SUCCESS;
}


/************************************************************************
 *
 *    function : CONFIGRET ActivateLogicalDevice( BYTE bLogDev, WORD wPortAddress)
 *
 *    params   : bLogDev -- logical device to configure the IO for.
 *               wPortAddress -- port address to use.
 *
 *    operation: Activate the device.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET ActivateLogicalDevice( BYTE bLogDev, WORD wPortAddress)
{
  if (CR_FAILURE == WriteCFGRegister(IREG_LOGDEVICE, bLogDev, wPortAddress))
    return CR_FAILURE;

  return (WriteCFGRegister(IREG_ACTIVATE, LD_ACTIVATE, wPortAddress));
}

/************************************************************************
 *
 *    function : CONFIGRET ConfigureIO0(BYTE bLogDev, WORD wIOAddress, WORD wPortAddress)
 *
 *    params   : bLogDev -- logical device to configure the IO for.
 *               wIOAddress -- IO address to program.
 *               wPortAddress -- port address to use.
 *
 *    operation: program first IO address.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET ConfigureIO0(BYTE bLogDev, WORD wIOAddress, WORD wPortAddress)
{
  if (CR_FAILURE == WriteCFGRegister(IREG_LOGDEVICE, bLogDev, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == WriteCFGRegister(IREG_IO0_BASEHI, HIBYTE(wIOAddress), wPortAddress))
    return CR_FAILURE;
  return (WriteCFGRegister(IREG_IO0_BASELO, LOBYTE(wIOAddress), wPortAddress));
}

/************************************************************************
 *
 *    function : ConfigureIO1(BYTE bLogDev, WORD wIOAddress, WORD wPortAddress)
 *
 *    params   : bLogDev -- logical device to configure the IO for.
 *               wIOAddress -- IO address to program.
 *               wPortAddress -- port address to use.
 *
 *    operation: program second IO address.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET ConfigureIO1(BYTE bLogDev, WORD wIOAddress, WORD wPortAddress)
{
  if (CR_FAILURE == WriteCFGRegister(IREG_LOGDEVICE, bLogDev, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == WriteCFGRegister(IREG_IO1_BASEHI, HIBYTE(wIOAddress), wPortAddress))
    return CR_FAILURE;
  return (WriteCFGRegister(IREG_IO1_BASELO, LOBYTE(wIOAddress), wPortAddress));
}

/************************************************************************
 *
 *    function : ConfigureIRQ(BYTE bLogDev, WORD wIRQ, WORD wPortAddress)
 *
 *    params   : bLogDev -- logical device to configure the IRQ for.
 *               wIRQ -- IRQ to program.
 *               wPortAddress -- port address to use.
 *
 *    operation: Configure the PNP registgers for the IRQ specified.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET ConfigureIRQ(BYTE bLogDev, WORD wIRQ, WORD wPortAddress)
{
  if (CR_FAILURE == WriteCFGRegister(IREG_LOGDEVICE, bLogDev, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == WriteCFGRegister(IREG_IRQ_NUMBER, LOBYTE(wIRQ), wPortAddress))
    return CR_FAILURE;
  return (WriteCFGRegister(IREG_IRQ_TYPE, IRQTYPE_EDGE, wPortAddress));
}

/************************************************************************
 *
 *    function : ConfigureRAM(BYTE bLogDev, WORD wRAMAddress, WORD wPortAddress)
 *
 *    params   : bLogDev -- logical device to configure the RAM for.
 *               wRamAddress -- RAM adress to program.
 *               wPortAddress -- port address to write.               
 *
 *    operation: configure the RAM address for the logical device specified.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET ConfigureRAM(BYTE bLogDev, WORD wRAMAddress, WORD wPortAddress)
{
  if (CR_FAILURE == WriteCFGRegister(IREG_LOGDEVICE, bLogDev, wPortAddress))
    return CR_FAILURE;
  wRAMAddress = (wRAMAddress >> 4) & 0x0FFF;
  if (CR_FAILURE == WriteCFGRegister(IREG_MEMBASEHI, HIBYTE(wRAMAddress), wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == WriteCFGRegister(IREG_MEMBASELO, LOBYTE(wRAMAddress), wPortAddress))
    return CR_FAILURE;
  if (wRAMAddress)
    {
    return (WriteCFGRegister(IREG_MEMCONTROL, (MEMTYPE_HIADDR | MEMTYPE_16BIT), wPortAddress));
    }
  else
    {
    return (CR_SUCCESS);
    }
}

/************************************************************************
 *
 *    function : WriteLogicalDevice( LDDATA *ld, WORD wPortAddress)
 *
 *    params   : ld -- logical data containing the information to write.
 *
 *    operation: write PNP data to the PNP configuratin registers.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET WriteLogicalDevice( LDDATA *ld, WORD wPortAddress)
{
  if (CR_FAILURE == WriteCFGRegister(IREG_LOGDEVICE, ld->bLDNumber, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == ConfigureIO0( ld->bLDNumber , ld->wIOAddr0, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == ConfigureIO1( ld->bLDNumber , ld->wIOAddr1, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == ConfigureIRQ( ld->bLDNumber , ld->wIRQ, wPortAddress))
    return CR_FAILURE;
  if (CR_FAILURE == ConfigureRAM( ld->bLDNumber , ld->wRAMAddr, wPortAddress))
    return CR_FAILURE;
  return (ActivateLogicalDevice( ld->bLDNumber, wPortAddress));
}

/************************************************************************
 *
 *    function : Config_hardware();
 *
 *    params   : none
 *
 *    operation: configure the Memory, IO and IRQ by programming
 *               the PNP registers.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/

CONFIGRET Config_Hardware(void)
{
    LDDATA ld;
    int n;

    // range check the parameters.
    if ( (wCFGIO!=  0x250)
        &&(wCFGIO !=  0x260)
        &&(wCFGIO !=  0x270) )
    {
        return CR_FAILURE;
    }

#if 0       // This does not seem to work the second time around.  
    if (VerifyHWExists(wCFGIO) != CR_SUCCESS)
        return(CR_FAILURE);
#endif

    ld.bLDNumber    = 0;
    ld.wIOAddr0     = wBASEIO;        // dsp address
    ld.wIOAddr1     = 0x00;
    ld.wIRQ         = nIRQValue;
    ld.wRAMAddr     = pMEM.dw.h;

    if (CR_FAILURE == WriteLogicalDevice(&ld, wCFGIO))
      return CR_FAILURE;

    if (wMPUIO != 0)     // kurtzweil??
    {
        ld.bLDNumber    = 1;
        ld.wIOAddr0     = wMPUIO;
        ld.wIOAddr1     = 0x00;
        ld.wIRQ         = nMPUIRQ;
        ld.wRAMAddr     = 0x00;

        if (CR_FAILURE == WriteLogicalDevice(&ld, wCFGIO))
          return CR_FAILURE;
    }

    if (wCDIO0 != 0) 
    {
        ld.bLDNumber    = 2;
        ld.wIOAddr0     = wCDIO0;
        ld.wIOAddr1     = wCDIO1;
        ld.wIRQ         = nCDIRQ;
        ld.wRAMAddr     = 0x00;

        if (CR_FAILURE == WriteLogicalDevice(&ld, wCFGIO))
          return CR_FAILURE;
    }

    if (wJOYIO == 0)
    {
        ld.bLDNumber    = 3;
        ld.wIOAddr0     = wJOYIO;
        ld.wIOAddr1     = 0x00;
        ld.wIRQ         = 0x00;
        ld.wRAMAddr     = 0x00;

        if (CR_FAILURE == WriteLogicalDevice(&ld, wCFGIO))
          return CR_FAILURE;
    }

    return(CR_SUCCESS);
    

}



/************************************************************************
 *    function : CheckDSPHardware();
 *
 *    operation: check to see if the DSP hardware is there.  First check the
 *               memory then check to see if the DSP responds with its magic number
 *               to the DSP reset.
 *
 *               
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 ************************************************************************/
CONFIGRET CheckDSPHardware()
{
    BYTE nVersion;
    BYTE nReg;   // 
    int n;

    outp( wBASEIO + HP_DSPR , HPDSPRESET_ON );
                                                //  it won't bother us.
   
    nVersion = inp(wBASEIO+HP_NU);
    nVersion &= 0xf0;   // mask off the board version number

    // Board information register layout
    //
    //    7 6 5 4 3 2 1 9
    //    x x x x x x x x
    //    | | | | | | | |
    //    | | | | | | \----- Board Version (BVER 1:0)
    //    | | | | | \------- Board Type (BTYPE)
    //    | | | | \--------- Reserverd always 1.
    //    \----------------- Xylinx Version (XVER:3:0)
    //
    //   Possible combinations of board type and Board Version (1:0) are
    //
    //   BTYPE   BVER
    //     1      1:1  Figi Rev A-B or Pinnacle Rev A-E early boards without 
    //                 hardware information.
    //     0      0:1  Pinnacle Rev F (or modified early pinnacles).
    //     0      1:0  Pinnacle Rev G.
    //     0      1:1  Pinnacle Rev H.
    //     0      0:0  Pinnacle Rev I.
    //     1      0:1  Figi Rev C
    //     1      1:0  Figi Rev D.
    //     1      0:0  Figi Rev E.
    //
    //   Xylinx Version number (XVER 3:0)
    // 
    //   1:1:1:1   -- Xilinx version 1.5 or earlier.
    //   0:0:0:0   -- not defined.
    //   0:0:0:1   -- Xilinx version 1.18 (as report as 1.2).
    //   0:0:1:0   -- Xilinx version 1.3x
    //   0:0:1:1   -- Xilinx Version 1.4x.
    //   0:1:0:0   -- Possible future versions.
    //   1:1:1:0   -- ------
    //
    if ((nVersion != 0xf0) &&  // only new versions get a memory check.
        (nVersion > 0x10))      // Xylinx 1.3 or newer.
    {
        CONFIGRET nMemOk;
        // disable the memory.
        outp(wBASEIO+HP_BLKS, HPMEMDISABLED);   

        nMemOk = CheckDSPMem();

        // if the memory is not ok, bail right here.
        if (nMemOk != CR_SUCCESS)       // if we failed, bug out.
            FAIL;
    }

    // now either the memory is OK or we could not check it.
    outp( wBASEIO + HP_DSPR , HPDSPRESET_OFF ); // unleash the DSP.

    // actually 10 microseconds would do, but we don't have an accurate
    // enough timer here.
    GetDelayTime(2);        // wait 2 milliseconds....

    // wait for the DSP to reset.
    for (n = 0; n < 1000; n++)
    {
        BYTE nIsrReg;   // interrupt status register.
        nIsrReg = inp(wBASEIO+HP_ISR);
        if (nIsrReg == HP_ISR_DEF)    // must be the default contents.
            break;
        DelayOneMsec();

        // wait, go around and try again.
    }
    if (n >= 1000)
    {
        printf("DSP failed to reset\n");
        FAIL;
    }
    nReg = inp(wBASEIO+HP_ICR);// interrupt ctrl register.
    if (nReg != HP_ICR_DEF)
        FAIL;
    nReg = inp(wBASEIO+HP_CVR);// Command vector register.
    if (nReg != HP_CVR_DEF)
        FAIL;
    nReg = inp(wBASEIO+HP_ISR);// Interrupt status register.
    if (nReg != HP_ISR_DEF)
        FAIL;
    nReg = inp(wBASEIO+HP_IVR);// Interrupt vector register.
    if (nReg != HP_IVR_DEF)
        FAIL;



    // put the DSP back into reset...
    outp( wBASEIO + HP_DSPR , HPDSPRESET_ON );
    return(CR_SUCCESS);
Fail:
    // put the DSP back into reset...
    outp( wBASEIO + HP_DSPR , HPDSPRESET_ON );
    return(CR_FAILURE);
    

}

/************************************************************************
 *
 *    function : ReadCFGRegister(WORD wRegister, WORD wPortAddress)
 *
 *    params   : wRegister -- configuration register to write to.
 *               wPortAddress -- PNP configuration port address.
 *
 *    operation: Read the PNP configuration register.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/
BYTE ReadCFGRegister(WORD wRegister, WORD wPortAddress)
{

  outp(wPortAddress, wRegister);
  return(inp(wPortAddress+1));
}

/************************************************************************
 *
 *    function : VerifyHWExists(WORD wPortAddress)
 *
 *    params   : wPortAddress -- PNP configuration port address.
 *
 *    operation: Verify the PNP hardware exists at the port 
 *               found here.
 *               
 *
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 * **********************************************************************/
CONFIGRET VerifyHWExists(WORD wPortAddress)
{
	BYTE bSaveReg;

	WriteCFGRegister(IREG_LOGDEVICE, 0x00, wPortAddress);
	bSaveReg = ReadCFGRegister(IREG_IO0_BASELO, wPortAddress); // save the config value
	WriteCFGRegister(IREG_IO0_BASELO,0xAA, wPortAddress);
	if(ReadCFGRegister(IREG_IO0_BASELO, wPortAddress) != 0xAA)
		return(CR_FAILURE);

	WriteCFGRegister(IREG_IO0_BASELO, 0x55, wPortAddress );
	if ( ReadCFGRegister(IREG_IO0_BASELO, wPortAddress) != 0x55 )
		return(CR_FAILURE);
	WriteCFGRegister(IREG_IO0_BASELO,bSaveReg, wPortAddress);   // restore the config value

	WriteCFGRegister(IREG_LOGDEVICE, 0x01 , wPortAddress);
	bSaveReg = ReadCFGRegister(IREG_IO0_BASELO, wPortAddress); // save the config value
	WriteCFGRegister(IREG_IO0_BASELO, 0xAA , wPortAddress);
	if ( ReadCFGRegister(IREG_IO0_BASELO, wPortAddress) != 0xAA )
		return(CR_FAILURE);

	WriteCFGRegister(IREG_IO0_BASELO, 0x55 , wPortAddress);
	if ( ReadCFGRegister(IREG_IO0_BASELO, wPortAddress) != 0x55 )
		return(CR_FAILURE);
	WriteCFGRegister(IREG_IO0_BASELO,bSaveReg, wPortAddress);   // restore the config value

	return(CR_SUCCESS);
}

/************************************************************************
 *    function : CheckDSPMem();
 *
 *    operation: Check to see if the dsp memory area is clear of
 *               any other application using it.
 *
 *               
 *    returns  : CR_SUCCESS = success.
 *               CR_FAILURE = failure
 *
 ************************************************************************/
CONFIGRET CheckDSPMem()
{
    WORD wInker;

    // before initializing the SMA area. first check to see if this
    // are is free.  (if there is read/write memory any where in here)
    // don't use this.

    for( wInker = 0; wInker < 0x8000 ; wInker+= 2) 
    {   
        WORD wOrigData;
        WORD wData;
        WORD *pWord;
        pWord = (WORD *)( pMEM.p + wInker );
        wOrigData= *pWord;  // save the original dat
        *pWord  = 0x55AA;    // write to it.
        wData = *pWord ;     // read it.
        *pWord = wOrigData; // restore it.

        if (wData == 0x55AA)
        {
            printf("Memory at %x is not Free\n",pMEM.dw.h);
            return(CR_FAILURE);
        }

    }
    return(CR_SUCCESS);
}

/************************************************************************
 *
 *    function : int EnableMPU( void )
 *
 *    params   : none
 *
 *    operation: This function initializes the Kurtzweil MPU port
 *
 *
 *    returns  : returns 1 if successful
 *               returns 0 if failed.
 *
 * **********************************************************************/
int EnableMPU()
{
    if (wMPUIO == 0)
    {
        printf("NO MPU address specified\n");
        return(0);
    }

    SetupMPUIRQ();

    bMpuIsrRunning = 1;     // keep the ISR from actually doing anything.

    if (!SetSmartMPU())
    {
        printf("SetSmartMPU\n");
        ResetMPUIRQ();
        return(0);
    }


    if (!VerifyMPUIrq())
    {
        printf("VerifyMPUIrq()\n");
        ResetMPUIRQ();
        return(0);
    }


    if (!SetDumbMPU())
    {
        printf("SetDumbIrq\n");
        ResetMPUIRQ();
        return(0);
    }

    bMpuIsrRunning = 0; // ISR is free to run.

    // set up the MIDI circular buffer.
    if (pMpuDataBuffer == NULL)
    {
        pMpuDataBuffer = calloc(sizeof(BYTE), MPU_BUFFER_SIZE);
        pMpuDataHead = pMpuDataBuffer;
        pMpuDataTail = pMpuDataBuffer;
        pMpuDataEnd = &pMpuDataBuffer[MPU_BUFFER_SIZE];
    }

    return(1);  // made it this far, return success

}
/************************************************************************
 *
 *    function : void DisableMPU( void )
 *
 *    params   : none
 *
 *    operation: This function disables the Kurtzweil MPU port
 *
 *
 *    returns  : returns 1 if successful
 *               returns 0 if failed.
 *
 * **********************************************************************/
 void DisableMPU( void )
 {
    if (wMPUIO == 0)
    {
        printf("NO MPU address specified\n");
        return;
    }


    ResetMPUIRQ();

    if (pMpuDataBuffer == NULL)
    {
        free(pMpuDataBuffer);
        pMpuDataBuffer = NULL;
    }


 }

/************************************************************************
 *
 *    function : int SetSmartMPU()
 *
 *    params   : none
 *
 *    operation: Sets up the MPU port in Smart mode.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int SetSmartMPU()
{
    int n;
    CLI;        // don't let em interrupt us on the reset.

    WriteMPUCmd(MPU_RESET);
    for (n = 0; n < MAX_FIFO_READ; n++)
    {
        BYTE bStatus;
        BYTE bData;
        bStatus = INP(wMPUIO + HP_MPU_CMD);

        // drain the FIFO...
        if (!ReadMPUQuick(&bData))
            break;
        
    }
    STI;    // re-enable interrupts.
    GetDelayTime(2); // delay 2 MS.

    return(1);
}

/************************************************************************
 *
 *    function : int SetDumbMPU()
 *
 *    params   : none
 *
 *    operation: Sets up the MPU port in Dump Uart mode.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int SetDumbMPU()
{
    BYTE bData;

    if (!WriteMPUCmd(MPU_RESET))
        return(0);


    if (!ReadMPU(&bData))
        return(0);

    if (bData != MPU_ACK)
        return(0);


    if (!WriteMPUCmd(MPU_UART_MODE))
        return(0);

    if (!ReadMPU(&bData))
        return(0);

    if (bData != MPU_ACK)
        return(0);

    
    return(1);
}

/************************************************************************
 *
 *    function : int VerifyMPUIrq()
 *
 *    params   : none
 *
 *    operation: Verifies that the IRQ is hooked up and working.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int VerifyMPUIrq()
{
    int n;
    BYTE bData;
    WORD wTimeOut;
    if (!WriteMPUCmd(MPU_RESET))
    {   
        printf("WriteMPUCmd(MPU_RESET)\n");
        return(0);
    }

    wTimeOut = wsnaptimer(); 
    do
    {
        printf(".");
        if (dwMPUInterruptCounter)
            break;
    } while (wTimeOut-wsnaptimer() < TIMER_MS*10);

    // we never did field an interrupt, so we are screwed.
    if (dwMPUInterruptCounter == 0)
    {
        printf("dwMPUInterruptCounter == 0\n");
        return(0);
    }

    // clear the ACK.
    ReadMPU(&bData);
            
    return(1);
}

/************************************************************************
 *
 *    function : int WriteMPUCmd(BYTE bCmd)
 *
 *    params   : bCmd -- command to write
 *
 *    operation: Writes an MPU command to the MPU command port.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int WriteMPUCmd(BYTE bCmd)
{
    BYTE bStatus;
    WORD wTimeOut;

    if (wMPUIO == 0)    // don't try to write if we don't have an MPU.
        return(0);

    wTimeOut = wsnaptimer(); 

    do
    {
        bStatus = INP(wMPUIO + HP_MPU_CMD);
        if ((bStatus & MPUSR_TX_BIT) == 0)
            break;    
    } while (wTimeOut-wsnaptimer() < TIMER_MS*10);

    // transmitter not read, bug out.
    if (bStatus & MPUSR_TX_BIT)
        return(0);

    // now we are clear to send it.
    OUTP(wMPUIO + HP_MPU_CMD, bCmd);


}


/************************************************************************
 *
 *    function : int ReadMPU()
 *
 *    params   : pbData -- pointer to place to stuff the data being read.
 *
 *    operation: Reads data from the MPU port.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int ReadMPU(BYTE *pbData)
{
    BYTE bStatus;
    WORD wTimeOut;

    if (wMPUIO == 0)    // don't try to write if we don't have an MPU.
        return(0);

    wTimeOut = wsnaptimer(); 
    do
    {
        bStatus = INP(wMPUIO + HP_MPU_CMD);
        if ((bStatus & MPUSR_RX_BIT) == 0)
            break;    
    } while (wTimeOut-wsnaptimer() < TIMER_MS*10);

    // transmitter not read, bug out.
    if (bStatus & MPUSR_RX_BIT)
        return(0);
    

    *pbData = INP(wMPUIO + HP_MPU_DATA);
}

/************************************************************************
 *
 *    function : int ReadMPUQuick()
 *
 *    params   : pbData -- pointer to place to stuff the data being read.
 *
 *    operation: Reads data from the MPU port, but with a shorter timerout
 *               than read MPU.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int ReadMPUQuick(BYTE *pbData)
{
    BYTE bStatus;
    

    WORD wTimeOut;

    wTimeOut = wsnaptimer(); 
    do
    {
        bStatus = INP(wMPUIO + HP_MPU_CMD);
        if ((bStatus & MPUSR_RX_BIT) == 0)
            break;    
    } while (wTimeOut-wsnaptimer() < TIMER_MS);

    // transmitter not read, bug out.
    if (bStatus & MPUSR_RX_BIT)
        return(0);
    
    *pbData = INP(wMPUIO + HP_MPU_DATA);

}

/************************************************************************
 *
 *    function : int WriteMPU()
 *
 *    params   : bData -- Data to write
 *
 *    operation: Writes data to the MPU data port.
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
int WriteMPU(BYTE bData)
{
    BYTE bStatus;
    WORD wTimeOut;

    if (wMPUIO == 0)    // don't try to write if we don't have an MPU.
        return(0);
    


    wTimeOut = wsnaptimer(); 
    do
    {
        bStatus = INP(wMPUIO + HP_MPU_CMD);
        if ((bStatus & MPUSR_TX_BIT) == 0)
            break;    
    } while (wTimeOut-wsnaptimer() < TIMER_MS*10);

    // transmitter not read, bug out.
    if (bStatus & MPUSR_TX_BIT)
        return(0);

    OUTP(wMPUIO + HP_MPU_DATA, bData);

}




/************************************************************************
 *
 *    function : BYTE SetupMPUIRQ()
 *
 *    params   : none
 *
 *    operation: Sets up the PC to be able to receive interrupts from
 *               the MultiSound, and saves the previous DOS interrupt
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/
BYTE SetupMPUIRQ()
{

    WORD wGP;
    WORD wPic;
    BYTE cIrqMask;
    BYTE cIntBit;


    if( nMPUIRQ > 7 ) 
    {
        wGP = nMPUIRQ + 0x68;
        cIntBit = ( 1 << (nMPUIRQ - 8));
        wPic = PIC2+1;
    }
    else
    {
        wGP = nMPUIRQ + 8;
        cIntBit = (1 <<  nMPUIRQ );
        wPic = PIC1+1;
    }



    nMPUSaveVect = _dos_getvect( wGP );
    _dos_setvect( wGP , MPUInterrupt );


    CLI;    // lock out interrupts.

    cIrqMask = inp( wPic);
        // only enable it if it is not already enabled.
    if (cIrqMask & cIntBit)
        outp( wPic, ((~cIntBit) & cIrqMask) );

    STI;

    return(1);
}


/************************************************************************
 *
 *    function : BYTE ResetMPUIRQ()
 *
 *    params   : none
 *
 *    operation: Resets the PC link to the MultiSound interrupt from
 *               the MultiSound, and restores the DOS interrupt
 *
 *    returns  : 1 if successful
 *               0 if not successful
 *
 * **********************************************************************/

BYTE ResetMPUIRQ()
{
    WORD wGP;
    WORD wPic;
    BYTE cIrqMask;
    BYTE cIntBit;


    if( nMPUIRQ > 7 ) 
    {
        wGP = nMPUIRQ + 0x68;
        cIntBit = ( 1 << (nMPUIRQ - 8));
        wPic = PIC2+1;
    }
    else
    {
        wGP = nMPUIRQ + 8;
        cIntBit = (1 <<  nMPUIRQ );
        wPic = PIC1+1;
    }


    CLI;

    cIrqMask = inp( wPic);

    // already disabled....
    if ((cIrqMask & cIntBit) == 0)
        outp( wPic,  (cIntBit | cIrqMask) );

    STI;

    _dos_setvect( wGP , nMPUSaveVect );
  
    return(1);
}


/************************************************************************
 *
 *    function : void _interrupt _far MPUInterrupt( void )
 *
 *    params   : none
 *
 *    operation: This reads the data the MPU wants to send us...
 *
 *    returns  : none
 *
 * **********************************************************************/

void _interrupt _far MPUInterrupt( void )
{
#define EOI_AT_END 0

#if (EOI_AT_END == 0)
    if( nMPUIRQ > 7 ) 
        outp( PIC2 , EOI );
    outp( PIC1 , EOI );

    STI;        // enable interrupts here.
#endif

    dwMPUInterruptCounter++;        // count this..

    if( bMpuIsrRunning != 1 ) 
    {
        bMpuIsrRunning = 1;



        // loop untin there is nothing left to read.
        for (;;)
        {
            BYTE bStatus;
            BYTE bData;
            bStatus = INP(wMPUIO + HP_MPU_CMD);
            if (bStatus & MPU_RX_DATA_RDY)  // RDY high, break out.
                break;
                
            bData = INP(wMPUIO + HP_MPU_DATA);
            // are we initialized and receiving data???
            if (bMPUInStarted)
            {
                *pMpuDataHead++ = bData;

                // did the data wrap.
                if (pMpuDataHead >= pMpuDataEnd)
                    pMpuDataHead == pMpuDataBuffer;
                
                // it would probably be a good idea to check for an overflow.
            }

        }

        

        bMpuIsrRunning = 0;
    }
#if EOI_AT_END
        if( nMPUIRQ > 7 ) 
            outp( PIC2 , EOI );
        outp( PIC1 , EOI );
#endif
}
/************************************************************************
 *
 *    function : int GetMPUInData(BYTE *pData)
 *
 *    params   : pData -- pointer to where to stuff the byte.
 *
 *    operation: This retrieves MPU data from the circular record buffer.
 *
 *    returns  : 1 -- data retrieved.
 *               0 -- no more data.
 *
 *
 * **********************************************************************/
int GetMPUInData(BYTE *pData)
{
    CLI;                                // begin the critical section.
    // nothing here.
    if (pMpuDataHead == pMpuDataTail)
        return(0);

    *pData = *pMpuDataTail++;

    // did it wrap.
    if (pMpuDataTail >= pMpuDataEnd)
        pMpuDataTail == pMpuDataBuffer;
    STI;                                // end of critical section.

}

/************************************************************************
 *
 *    function : void SetMPUIn(int bEnable)
 *
 *    params   : bEnable -- turn on/iff MPU in
 *
 *    operation: This enables/disables the MPU input data.
 *
 *    returns  : none
 *
 * **********************************************************************/
 void SetMPUIn(int bEnable)
 {
    bMPUInStarted = bEnable;
    
 }
