/* File name: mwindib.c
 *
 * Synopsis:  This program displays a welcoming message to the user
 *            under Windows and does some GDI drawing in the displayed
 *            MIL buffer. It uses MIL, the MIL VGA system and its Windows
 *            internal DIB to display the message and the drawings in a 
 *            user created client window.
 *
 *            Note that this program only applies to the MIL VGA system 
 *            and is given to ADVANCED Windows programmers to show how
 *            they can use the internal DIB structures created by MIL to 
 *            control the display of those DIB in their own client window.
 *
 *            We strongly suggest that you try the regular MdispSelect()
 *            (see MDISPLAY.C) and MvgaDispSelectClientArea()
 *            (see MWINDISP.C) before you start using the mecanism
 *            described in this program. The functions mentionned above
 *            are designed to be simple and meet the majority of the
 *            regular Windows imaging application display requirements.
 *
 *            Note also that the data in any type of MIL buffer can be
 *            read and written using the regular MbufGet() and MbufPut()
 *            function. This can be used to create your own data buffer
 *            to interface with Windows in a way that meet your
 *            requirements.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <windows.h>
#include <mil.h>
#include <mwinmil.h>
#include <wingdi.h>

#define BUFFERSIZEX 512
#define BUFFERSIZEY 480
#define MAX_PATH_NAME_LEN  256

/* Prototypes */
void MilApplication(HWND UserWindowHandle);
void MyModifyDIBfunction(M_DIB_INFO MPTYPE *DibInfoPtr);
void MilApplicationPaint(HWND UserWindowHandle);
void MilApplicationEraseBkGnd(HWND UserWindowHandle, HDC UserWindowDC);
long MilApplicationPalette(HWND UserWindowHandle, UINT message, WPARAM wParam);
void MFTYPE MyUpdateScreen(M_DIB_INFO MPTYPE *ModifiedVgaBufferInfoPtr,
                           void MPTYPE *ExtraUserInfoPtr);
void DrawScreen(long         ClearBackground,
                MIL_ID       BufferId,
                int          ModifOffsetX,
                int          ModifOffsetY,
                int          ModifSizeX,
                int          ModifSizeY,
                HWND         hWnd,
                LPBITMAPINFO DIBHeaderPtr,
                LPSTR        DIBDataPtr,
                HPALETTE     DIBPaletteHdl,
                HBITMAP      DIBBitmapHdl,
                UINT         DIBColorUse
                );

// This prototype is added to remove a warning in Borland C 4.5
#ifdef __BORLANDC__  
   WINGDIAPI BOOL  WINAPI GdiFlush(void);
#endif

/***************************************************************************/
/*/
* Name:         MilApplication()
*
* synopsis:     This function is the core of the MIL application that
*               will be executed when the "Start" menu item of this
*               Windows program will be selected. See WinMain() below
*               for the program entry point.
*
*               It will display a welcoming message in the user window
*               using MIL.
*
*/

void MilApplication(HWND UserWindowHandle)
{
  /* MIL variables */
  MIL_ID MilApplication,     /* MIL Application identifier.  */
         MilVgaSystem,       /* VGA System identifier.       */
         MilVgaDisplay,      /* VGA Display identifier       */
         MilVgaImage;        /* VGA Image buffer identifier. */

  /* Variables for DIB info data pointer */
  M_DIB_INFO MPTYPE *DibInfoPtr;

  /* Allocate a MIL application. */
  MappAlloc(M_DEFAULT, &MilApplication);

  /* Allocate a VGA system. */
  MsysAlloc(M_SYSTEM_VGA, M_DEV0, M_DEFAULT, &MilVgaSystem);

  /* Allocate a VGA display. */
  MdispAlloc(MilVgaSystem, M_DEV0, "M_DEFAULT", M_DEFAULT, &MilVgaDisplay);

  /* Allocate a VGA buffer. */
  MbufAlloc2d(MilVgaSystem,BUFFERSIZEX,BUFFERSIZEY,
              8+M_UNSIGNED,M_IMAGE+M_DISP,&MilVgaImage);

  /* Clear the buffer */
  MbufClear(MilVgaImage,0);

  /* Hook a custom screen update function to the VGA display to be
     called when system's buffer are modified.
  */
  MvgaHookModified(MilVgaDisplay,M_HOOK_MODIFIED_DIB,MyUpdateScreen,&UserWindowHandle);

  /* Get the pointer to a MIL DIB information structure */
  DibInfoPtr = MvgaDispAllocDIBInfo(MilVgaDisplay, MilVgaImage);

  /* Store DIB ptr in the reserved extra window memory for WM_PAINT update */
  SetWindowLong(UserWindowHandle,0,(DWORD)DibInfoPtr);

  /* On a VGA system, if ever the DIB content of a MIL buffer is modified
     without going through MIL regular interface, MnatModified() must
     be called to signal it to MIL (see the MIL native mode manual).

     Note: On system other than VGA, DIB modifications might not be supported
     and should be avoided if possible for portability and compatibility with
     future releases.
  */

  /* For example, the following lines write something in the
     MIL DIB with help of GDI:
  */
  MyModifyDIBfunction(DibInfoPtr); // User's code to modify the DIB.
  MnatModified(MilVgaImage);       // Advise MIL for the modification.

  /* Print a string in the VGA image buffer using MIL.
     Note: After each MIL command writing in a MIL VGA buffer,
     the MyUpdateScreen() hooked function will be automaticly called
     by MIL to signal a change in a certain region of the VGA
     buffer's DIB. In this example it will be called 3 times, one for
     each MgraText() call.
  */
  MgraText(M_DEFAULT, MilVgaImage, 200L, 250L, " ------------------ ");
  MgraText(M_DEFAULT, MilVgaImage, 200L, 275L, " Welcome to MIL !!! ");
  MgraText(M_DEFAULT, MilVgaImage, 200L, 300L, " ------------------ ");

  /* Windows code to open a message box for the user to wait a key. */
  MessageBox(0,"""Welcome to MIL !!!"" was printed",
               "MIL application example",
               MB_APPLMODAL | MB_ICONEXCLAMATION );

  /* Clear DIB ptr in the reserved extra window memory for WM_PAINT update */
  SetWindowLong(UserWindowHandle,0,(DWORD)NULL);

  /* Free the MIL DIB information structure */
  MvgaDispFreeDIBInfo(MilVgaDisplay, MilVgaImage);

  /* Unhook the custom screen update function from VGA system's
     buffer modifications.
  */
  MvgaHookModified(MilVgaDisplay, M_HOOK_MODIFIED_DIB, M_NULL, M_NULL);

  /* Free allocated objects */
  MbufFree(MilVgaImage);
  MdispFree(MilVgaDisplay);
  MsysFree(MilVgaSystem);
  MappFree(MilApplication);
}


/***************************************************************************/
/*/
* Name:       MyModifyDIBfunction()
*
* synopsis:  This function draws in the DIB with help of GDI functions
*
*/

void MyModifyDIBfunction(M_DIB_INFO MPTYPE *DibInfoPtr)
{
   HDC      hDC = (HDC)NULL;
   HPEN     hPen, hOldPen;
   HBRUSH   hBrush, hOldBrush, hHatchBrush;
   HBITMAP  hOldBitmap = (HBITMAP)NULL;
   LOGBRUSH MyBrush;

   #if M_MIL_USE_NT_WINDOWS
      /* Verify if DIB is ok to use GDI under Windows NT */
      /*  Mil buffer must be a parent                    */

      if (!MbufInquire(DibInfoPtr->MilVgaBufferId,M_PARENT_ID,M_NULL))
         {
         /* Open the DIB driver and give it the target packed DIB */
         hDC = CreateCompatibleDC(NULL);
         hOldBitmap = SelectObject(hDC,DibInfoPtr->DIBDisplayBitmapHdl);
         }
   #else
      /* Verify if DIB is ok to use GDI under Windows and DIB.DRV   */
      /*  Color table of DIB must be in DIB_RGB_COLORS              */
      /*  DIB must be in 8 bits with no compression                 */
      /*  Mil buffer must be a parent                               */

      if ( (DibInfoPtr->DIBDisplayColorUse == DIB_RGB_COLORS) &&
           (DibInfoPtr->DIBDisplayHeaderPtr->bmiHeader.biBitCount <= 8) &&
           (!MbufInquire(DibInfoPtr->MilVgaBufferId,M_PARENT_ID,M_NULL)))
         {
         /* Open the DIB driver and give it the target packed DIB */
         hDC = CreateDC("DIB",NULL,NULL,DibInfoPtr->DIBDisplayHeaderPtr);
         }
   #endif

   /* Chekc if one DC is available to do GDI draw */
   if (hDC)
      {
      /* Build objects to draw with */
      hPen = CreatePen(PS_DASH,10,RGB(200,200,200));
      MyBrush.lbStyle = BS_SOLID;
      MyBrush.lbColor = RGB(100,100,100);
      hBrush = CreateBrushIndirect(&MyBrush);
      MyBrush.lbStyle = BS_HATCHED;
      MyBrush.lbHatch = HS_BDIAGONAL;
      hHatchBrush = CreateBrushIndirect(&MyBrush);

      /* Select objects to draw with and do draw */
      hOldPen = SelectObject(hDC,hPen);
      hOldBrush = SelectObject(hDC,hHatchBrush);
      Ellipse(hDC,20,20,490,450);
      SelectObject(hDC,hBrush);
      Ellipse(hDC,140,130,490,450);
      Ellipse(hDC,260,240,490,450);
      Ellipse(hDC,340,310,490,450);
      TextOut(hDC,385,365,"GDI draw",8);

      #if M_MIL_USE_NT_WINDOWS
         GdiFlush();
      #endif

      SelectObject(hDC,hOldPen);
      SelectObject(hDC,hOldBrush);

      /* Destroy objects used to draw */
      DeleteObject(hPen);
      DeleteObject(hBrush);
      DeleteObject(hHatchBrush);

      /* Delete created DC */
      SelectObject(hDC,hOldBitmap);
      DeleteDC(hDC);
      }
}

/***************************************************************************/
/*/
* Name:       MilApplicationPaint()
*
* synopsis:  This function update the screen with the DIB data buffer.
*
*            This is called everytime the WindowProc receives a
*            WM_PAINT message
*/

void MilApplicationPaint(HWND UserWindowHandle)
{
  PAINTSTRUCT ps;
  M_DIB_INFO MPTYPE  *DIBPtr;

  BeginPaint(UserWindowHandle, &ps);

  /* Get DIB Ptr */
  DIBPtr = (M_DIB_INFO MPTYPE *)GetWindowLong(UserWindowHandle,0);

  /* Check if DIB pointer is available */
  if (DIBPtr)
     /* Use the DrawScreen function to update the DIB modified area */
     DrawScreen(FALSE,
                DIBPtr->MilVgaBufferId,
                ps.rcPaint.left,
                ps.rcPaint.top,
                ps.rcPaint.right-ps.rcPaint.left,
                ps.rcPaint.bottom-ps.rcPaint.top,
                UserWindowHandle,
                DIBPtr->DIBDisplayHeaderPtr,
                DIBPtr->DIBDisplayDataPtr,
                DIBPtr->DIBDisplayPaletteHdl,
                DIBPtr->DIBDisplayBitmapHdl,
                DIBPtr->DIBDisplayColorUse
                );

  EndPaint(UserWindowHandle, &ps);
}


/***************************************************************************/
/*/
* Name:       MilApplicationEraseBkGnd()
*
* synopsis:  This function handles the erase background of the window
*
*            This is called everytime the WindowProc receives a
*            WM_ERASEBKGND message
*/

void MilApplicationEraseBkGnd(HWND UserWindowHandle, HDC UserWindowDC)
{
  M_DIB_INFO MPTYPE  *DIBPtr;
  RECT  ClipRect;

  /* Get DIB Ptr */
  DIBPtr = (M_DIB_INFO MPTYPE *)GetWindowLong(UserWindowHandle,0);

  /* Check if DIB pointer is available */
  if (DIBPtr)
     {
     ClipRect.left   = 0;
     ClipRect.top    = 0;
     ClipRect.right  = (int)MbufInquire(DIBPtr->MilVgaBufferId, M_SIZE_X, NULL);
     ClipRect.bottom = (int)MbufInquire(DIBPtr->MilVgaBufferId, M_SIZE_Y, NULL);

     ExcludeClipRect(UserWindowDC,
                     ClipRect.left,ClipRect.top,
                     ClipRect.right,ClipRect.bottom);
     }
}


/***************************************************************************/
/*/
* Name:       MilApplicationPalette()
*
* synopsis:  This function handles the palette messages of the window
*
*            This is called everytime the WindowProc receives a
*            WM_QUERYNEWPALETTE or WM_PALETTECHAGNED message
*/

long MilApplicationPalette(HWND UserWindowHandle, UINT message, WPARAM wParam)
{
  M_DIB_INFO MPTYPE  *DIBPtr;
  HDC                 hWndDC;
  long                RetVal = 0;

  /* Get DIB Ptr */
  DIBPtr = (M_DIB_INFO MPTYPE *)GetWindowLong(UserWindowHandle,0);

  /* Check if DIB pointer is available */
  if (DIBPtr)
     {
     hWndDC = GetDC(UserWindowHandle);

     /* Message of palette activation when we get the input focus */
     if (message == WM_QUERYNEWPALETTE)
        {
        SelectPalette(hWndDC,DIBPtr->DIBDisplayPaletteHdl,FALSE);
        if (RealizePalette(hWndDC))
           {
           InvalidateRect(UserWindowHandle,(LPRECT)NULL,TRUE);
           RetVal = 1;
           }
        }
     /* Message of palette activation when other application has changed it */
     else
        {
        if ((HWND)wParam != UserWindowHandle)
           {
           SelectPalette(hWndDC,DIBPtr->DIBDisplayPaletteHdl,FALSE);
           if (RealizePalette(hWndDC))
              {
              InvalidateRect(UserWindowHandle,(LPRECT)NULL,TRUE);
              RetVal = 1;
              }
           }
        }

     ReleaseDC(UserWindowHandle,hWndDC);
     }

  return (RetVal);
}

/***************************************************************************/
/*/
* Name:         MyUpdateScreen()
*
* synopsis:     This function is the hook installed to respond to any
*               modification of the DIB.
*
* comments:     This function is using a Window handle that correspond
*               to data pointed by ExtraUserInfoPtr.
*/

void MFTYPE MyUpdateScreen(M_DIB_INFO MPTYPE *ModifiedVgaBufferInfoPtr,
                           void MPTYPE *ExtraUserInfoPtr)
{

  long ClearBackground;

  /* Windows code to update the a region of the specified VGA buffer
     if the DIB is currenly used on the Windows screen.

    Here,

    ModifiedVgaBufferInfoPtr->MilVgaBufferId;
    ModifiedVgaBufferInfoPtr->MilVgaDisplayId;
    ModifiedVgaBufferInfoPtr->ModificationType;
    ModifiedVgaBufferInfoPtr->ModificationOffsetX;
    ModifiedVgaBufferInfoPtr->ModificationOffsetY;
    ModifiedVgaBufferInfoPtr->ModificationSizeX;
    ModifiedVgaBufferInfoPtr->ModificationSizeY;
    ModifiedVgaBufferInfoPtr->DIBDisplayHeaderPtr;
    ModifiedVgaBufferInfoPtr->DIBDisplayColorUse;
    ModifiedVgaBufferInfoPtr->DIBDisplayDataPtr;
    ModifiedVgaBufferInfoPtr->DIBDisplayPaletteHdl;
    ModifiedVgaBufferInfoPtr->DIBDisplayBitmapHdl;

    fields can be used to verify that the modified VGA Buffer Id is
    effectively a DIB displayed and to update the screen region with
    the corresponding DIB and palette.

  */

  /* Check if DIB info is valid and if we need to clear the background */
  if (  (!ModifiedVgaBufferInfoPtr) ||
        (ModifiedVgaBufferInfoPtr &&
         ModifiedVgaBufferInfoPtr->ModificationType
         == M_MODIFIED_DIB_DESTRUCTION
        )
     )
     ClearBackground = TRUE;
  else
     ClearBackground = FALSE;

  /* Use the DrawScreen() function to update the DIB modified area
     The window to update is taken from the ExtraUserInfoPtr.
  */
  DrawScreen(ClearBackground,
             ModifiedVgaBufferInfoPtr->MilVgaBufferId,
             (int)ModifiedVgaBufferInfoPtr->ModificationOffsetX,
             (int)ModifiedVgaBufferInfoPtr->ModificationOffsetY,
             (int)ModifiedVgaBufferInfoPtr->ModificationSizeX,
             (int)ModifiedVgaBufferInfoPtr->ModificationSizeY,
             *((HWND *)ExtraUserInfoPtr),
             ModifiedVgaBufferInfoPtr->DIBDisplayHeaderPtr,
             ModifiedVgaBufferInfoPtr->DIBDisplayDataPtr,
             ModifiedVgaBufferInfoPtr->DIBDisplayPaletteHdl,
             ModifiedVgaBufferInfoPtr->DIBDisplayBitmapHdl,
             ModifiedVgaBufferInfoPtr->DIBDisplayColorUse
             );
}


/***************************************************************************/
/*/
* Name:         DrawScreen()
*
* synopsis:     This function update the screen with the DIB data buffer
*               using the coordinates of the modified region in that buffer.
*/

void DrawScreen(long         ClearBackground,
                MIL_ID       BufferId,
                int          ModifOffsetX,
                int          ModifOffsetY,
                int          ModifSizeX,
                int          ModifSizeY,
                HWND         hWnd,
                LPBITMAPINFO DIBHeaderPtr,
                LPSTR        DIBDataPtr,
                HPALETTE     DIBPaletteHdl,
                HBITMAP      DIBBitmapHdl,
                UINT         DIBColorUse
                )
{
  HPALETTE  hOldPalette;
  HDC       hDC;
  RECT      DIBRect;
  HBRUSH    hBkBrush;

  /* Check if DIB info is valid and if we need to clear the background. */
  if (ClearBackground)
     {
     /* Get the DC associated to user's windows */
     hDC = GetDC(hWnd);

     /* Get the background brush to clear the DIB area */
     #if M_MIL_USE_NT_WINDOWS
      hBkBrush  = (HBRUSH)GetClassLong(hWnd,GCL_HBRBACKGROUND);
     #else
      hBkBrush  = (HBRUSH)GetClassWord(hWnd,GCW_HBRBACKGROUND);
     #endif

     /* Fill area to clear it */
     DIBRect.left   = ModifOffsetX;
     DIBRect.top    = ModifOffsetY;
     DIBRect.right  = DIBRect.left + ModifSizeX;
     DIBRect.bottom = DIBRect.top  + ModifSizeY;

     FillRect(hDC,&DIBRect,hBkBrush);

     /* Release the DC associated to user's windows */
     ReleaseDC(hWnd,hDC);
     }
  else
     {
     /* Get the DC associated to user's windows */
     hDC = GetDC(hWnd);

     /* Create and realize palette on the Display DC. */
     hOldPalette = SelectPalette(hDC,DIBPaletteHdl,0);
     RealizePalette(hDC);

     /* When a handle to a bitmap is given, this bitmap can be     
        selected into a display context and StretchBlt() can be used
        to blit it to the display. Note that the handle to the bitmap
        is a handle to a DIB bitmap and it is only available under
        Window NT.
     */
     if ((DIBBitmapHdl) &&
         !MbufInquire(BufferId,M_PARENT_ID,M_NULL))
        {
        HDC       hSrcMemDC;
        HBITMAP   hOldBitmap;

        hSrcMemDC  = CreateCompatibleDC(hDC);
        if (hSrcMemDC)
           {
           /* StretchBlt Bits display DC */
           hOldBitmap = SelectObject(hSrcMemDC,DIBBitmapHdl);
           StretchBlt(hDC,             /* Dest DC     */
                      ModifOffsetX,    /* Dest  X     */
                      ModifOffsetY,    /* Dest  Y     */
                      ModifSizeX,      /* Dest sX     */
                      ModifSizeY,      /* Dest sY     */
                      hSrcMemDC,       /* Src DC      */
                      ModifOffsetX,    /* Src   X     */
                      ModifOffsetY,    /* Src   Y     */
                      ModifSizeX,      /* Src  sX     */
                      ModifSizeY,      /* Src  sY     */
                      SRCCOPY          /* Raster Op   */
                      );
           SelectObject(hSrcMemDC,hOldBitmap);
           DeleteDC(hSrcMemDC);
           }
        }
     else
        {
        /* Find the position for proper blit on the destination
        *
        *      DIB
        *   ---------
        *  | A.....  | <- Offset  <-        <-
        *  |  .....  |    Y         | Size    |
        *  |  .....  |              | Y       | DIB
        *  |  .....  |            <-          | Total
        *  | B       |                        | Size Y
        *  |         |                        |
        *   ---------                       <-
        *
        *  |---------|  <-------------- DIB Total Size X
        *
        *     |---|     <-------------- Size X
        *
        *     |         <-------------- Offset Y
        *
        *  When DIB origin is at upper left corner
        *   A is at Offset X, Offset Y
        *   B is at Offset X, Offset Y + Y Size
        *
        *  When DIB origin is at bottom left corner
        *   A is at Offset X, (DIB Total Size Y - Offset Y)
        *   B is at Offset X, (DIB Total Size Y - Offset Y - Size Y)
        *
        *  SourceYPosition is defined by the Y coordinate of point B
        *  when the origin of the DIB is the bottom left corner.
        */

        int SourceYPosition;

        SourceYPosition =
           (int)DIBHeaderPtr->bmiHeader.biHeight-ModifOffsetY-ModifSizeY;

        /* Stretch DI Bits directly to display DC */
        StretchDIBits(hDC,             /* Dest DC     */
                      ModifOffsetX,    /* Dest  X     */
                      ModifOffsetY,    /* Dest  Y     */
                      ModifSizeX,      /* Dest sX     */
                      ModifSizeY,      /* Dest sY     */
                      ModifOffsetX,    /* Src   X     */
                      SourceYPosition, /* Src   Y     */
                      ModifSizeX,      /* Src  sX     */
                      ModifSizeY,      /* Src  sY     */
                      DIBDataPtr,      /* DIB data    */
                      DIBHeaderPtr,    /* DIB header  */
                      DIBColorUse,     /* Color usage */
                      SRCCOPY          /* Raster Op   */
                      );
        }

     /* Restore old palette of Display DC only if old one exist */
     SelectPalette(hDC,hOldPalette,0);

     /* Release the DC associated to user's windows */
     ReleaseDC(hWnd,hDC);
     }
}


/***************************************************************************/
/*
 * Synopsis:  The next functions let the user create a Windows program
 *            easily and very quickly. They will call the MilApplication()
 *            function each time the Start menu item is selected. The
 *            function MilApplicationPaint() is called each time the
 *            window receives a paint message to update the display.
 */

/* Prototypes for windows application */
int WINAPI WinMain(HINSTANCE, HINSTANCE, LPSTR, int);
long WINAPI MainWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL InitApplication(HINSTANCE, int);

/***************************************************************************/
/*
 *   Name:     WinMain()
 *
 *   Synopsis: Calls initialization function, processes message loop.
 */

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
   MSG msg;

   /* Avoid parameter not used warning */
   lpCmdLine = lpCmdLine;

   /* Only allow one instance of program. */
   if (hPrevInstance)
      {
      MessageBox(0, "Sample MIL Host Windows Application already active.\n"
                    "This version sample application does not allow multiple instances.",
                    "Sample MIL Host Application - Error",
                     MB_OK | MB_ICONSTOP);
      return (FALSE);
      }

   /* Exits if unable to initialize */
   if (!InitApplication(hInstance,nCmdShow))
      return (FALSE);

   /* Get and dispatch messages until a WM_QUIT message is received. */
   while (GetMessage(&msg,(HWND)NULL,(UINT)NULL,(UINT)NULL))
      {
      TranslateMessage(&msg);   /* Translates virtual key codes      */
      DispatchMessage(&msg);    /* Dispatches message to window      */
      }

   return (msg.wParam);    /* Returns the value from PostQuitMessage */
}


/***************************************************************************/
/*
 *   Name:      MainWndProc()
 *
 *   Synopsis:  Processes messages:
 *              WM_COMMAND     - application menu
 *              WM_ENDSESSION  - destroy window
 *              WM_CLOSE       - destroy window
 *              WM_DESTROY     - destroy window
 */

long WINAPI MainWndProc(HWND   hWnd, UINT message,
                        WPARAM wParam, LPARAM lParam)
{
   HMENU hAppMenu;

   switch (message)
      {
      case WM_COMMAND:     /* message: command from application menu */
         switch(wParam)
            {
            case IDM_STARTMAIN:
               hAppMenu = GetMenu(hWnd);
               EnableMenuItem(hAppMenu,MF_BYCOMMAND | IDM_STARTMAIN,MF_GRAYED);
               DrawMenuBar(hWnd);
               MilApplication(hWnd);
               EnableMenuItem(hAppMenu,MF_BYCOMMAND | IDM_STARTMAIN,MF_ENABLED);
               DrawMenuBar(hWnd);
               break;
            default:
               return (DefWindowProc(hWnd, message, wParam, lParam));
            }
         break;
      case WM_ERASEBKGND:
         MilApplicationEraseBkGnd(hWnd,(HDC)wParam);
         return (DefWindowProc(hWnd, message, wParam, lParam));
      case WM_PAINT:
         MilApplicationPaint(hWnd);
         return (DefWindowProc(hWnd, message, wParam, lParam));
      case WM_QUERYNEWPALETTE:
      case WM_PALETTECHANGED:
         return (MilApplicationPalette(hWnd, message, wParam));
      case WM_ENDSESSION:
      case WM_CLOSE:
         PostMessage(hWnd, WM_DESTROY, (WORD)0, (LONG)0);
         break;
      case WM_DESTROY:
         /* Free the reserved extra window LPSTR to szAppName */
         free((LPSTR)GetWindowLong(hWnd,sizeof(LPVOID)));
         PostQuitMessage(0);

      default:
         return (DefWindowProc(hWnd, message, wParam, lParam));
      }
   return(0L);
}


/***************************************************************************/
/*
 *   Name:     InitApplication()
 *
 *   Synopsis: Initializes window data, registers window class and
 *             creates main window.
 */

BOOL InitApplication(HINSTANCE hInstance, int nCmdShow)
{
   WNDCLASS        wc;
   HWND            hwnd;
   char            TmpName[MAX_PATH_NAME_LEN];
   LPSTR           szAppName;

   /* Allocation and storage of application name */
   szAppName = (LPSTR)malloc((40+MAX_PATH_NAME_LEN) * sizeof(char));
   strcpy(szAppName,MILAPPLNAME);

   wc.style          = CS_VREDRAW | CS_HREDRAW;
   wc.lpfnWndProc    = (WNDPROC)MainWndProc;
   wc.cbClsExtra     = 0;
   wc.cbWndExtra     = sizeof(LPVOID)+sizeof(LPSTR);
   wc.hInstance      = hInstance;
   wc.hIcon          = LoadIcon(0,IDI_APPLICATION);
   wc.hCursor        = LoadCursor(0, IDC_ARROW);
   wc.hbrBackground  = CreateSolidBrush(RGB(BACKCOLORRED,
                                            BACKCOLORGREEN,
                                            BACKCOLORBLUE));
   wc.lpszMenuName   = "MILAPPLMENU";
   wc.lpszClassName  = szAppName;

   /* Register the class */
   if (!RegisterClass(&wc))
      return (FALSE);

   /* Create the window */
   hwnd = CreateWindow (szAppName,
                        szAppName,
                        WS_OVERLAPPEDWINDOW,
                        CW_USEDEFAULT, CW_USEDEFAULT,
                        CW_USEDEFAULT, CW_USEDEFAULT,
                        0, 0, hInstance, 0) ;
   if (!hwnd)
      return (FALSE);

   lstrcat(szAppName," - ");
   GetModuleFileName(hInstance,TmpName,MAX_PATH_NAME_LEN);
   lstrcat(szAppName,TmpName);
   SetWindowText(hwnd,szAppName);

   /* Set the reserved extra window LPVOID to NULL */
   SetWindowLong(hwnd,0,0L);

   /* Set the reserved extra window LPSTR to szAppName */
   SetWindowLong(hwnd,sizeof(LPVOID),(DWORD)szAppName);

   /* Display the window */
   ShowWindow (hwnd, nCmdShow) ;
   UpdateWindow (hwnd) ;

   return (TRUE);
}

