/***************************************************************************/
/*                     IMF Device Specific Functions                       */
/*                                                                         */
/*    DevInit()            Initialize Device                               */
/*    DevOpen()            Open device, hook ints, etc.                    */
/*    DevClose()           Close device, free ints, etc.                   */
/*    DevWrite(data)       Write data                                      */
/*    DevNoteOn()                                                          */
/*    DevNoteOff()                                                         */
/*    DevAllNotesOff()     Make sure all notes are off                     */
/*    DevAfterTouch()                                                      */
/*    DevControlChange                                                     */
/*    DevProgramChange                                                     */
/*    DevChannelPressure()                                                 */
/*    DevPitchBend                                                         */
/*    DevMTC()                                                             */
/*    DevSysEx()                                                           */
/*    DevIOCTLinit(dptr)   Process IOCTL init functions                    */
/*    DevIOCTLstatus(dptr) Return status                                   */
/*    DevIOCTLload()       Load DSP module                                 */
/*    DevIOCTLread8()      diagnostic read byte from ACPA port             */
/*    DevIOCTLwrite8()     diagnostic write byte to ACPA port              */
/*    DevIOCTLread16()     diagnostic read word from ACPA port             */
/*    DevIOCTLwrite16()    diagnostic write word to ACPA port              */
/*    DevIOCTLwait()       wait routine                                    */
/*    SetClock()           Called when Tempo or PPQN changes               */
/*    DevInt()             (optional)                                      */
/*    TimerInt()           (optional)                                      */
/*                                                                         */
/* Some not-so-obvious things that I've learned about the IMF:             */
/*    1. IBE must be set a good while before setting TMSK to force the     */
/*       IRQ low before any pending interrupt.                             */
/*    2. EXT8 must be held for a while after writing PIU1.                 */
/***************************************************************************/
#if IS_DEBUG
/*************************** Debug stuff ***********************************/
/* Note:  In DOSDD.ASM, change @DEBUG equ 0 to @DEBUG equ 1 */
  char testbuf[80];                    /* buffer for test message */
#endif

#if IS_OS2
#define  INCL_DOS
#define  INCL_DOSINFOSEG
#include <os2.h>
#endif

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>

#include <audiodd.h>
#include <audiodd2.h>
#if IS_OS2
#include <mme.h>
#endif
#include <auddef.h>
#include <audproto.h>
#include <audmsg.h>

#define TRK 0

/***************************/
/* Device Specific Equates */
/***************************/

/* IMF 1 I/O Map: (2nd IMF = 2A3x) */
#define  IMFBASE  0x2a20      /* Base Address of 1st IMF          */
#define  IMFBASE2 0x2a30      /* Base Address of 2nd IMF          */
#define  PIU0     0x2a20      /* Input Port                       */
#define  PIU1     0x2a21      /* Output Port                      */
#define  PIU2     0x2a22      /* In/Out bit 8, and Int status     */
#define  PCR      0x2a23      /* PIU Control Register             */
#define  CNTR0    0x2a24      /* Counter 0                        */
#define  CNTR1    0x2a25      /* Counter 1                        */
#define  CNTR2    0x2a26      /* Counter 2                        */
#define  TCWR     0x2a27      /* Timer Control Word Register      */
#define  TCR      0x2a28      /* Total Control Register           */
#define  TSR      0x2a2c      /* Total Status Register            */

#define  INT_PEND 0x80        /* Set when an interrupt is pending */
#define  TIMER_B_STATUS 2     /* Timer B int pending              */
#define  TIMER_A_STATUS 1     /* Timer A int pending              */
#define  IBE            0x80  /* IRQ Bus Enable (1 = ints enbld)  */
#define  TMSK           0x40  /* Total IRQ mask (1 = ints enbld)  */
#define  EXT8           0x10  /* Transmit data bit 8              */
#define  TBE            8     /* Timer B enable (1 = enabled)     */
#define  TAE            4     /* Timer A enable (1 = enabled)     */
#define  TBC            2     /* !Timer B clear (0 = clear)       */
#define  TAC            1     /* !Timer A clear (0 = clear)       */
#define  EXR8           0x80  /* Receive data bit 8               */
#define  RIE            0x10  /* Read interrupt enable            */
#define  RxRDY          8     /* Receiver ready                   */
#define  WIE            4     /* Write interrupt enable           */
#define  TxRDY          1     /* Transmitter ready                */
#define  CLEAR_WIE      4     /* Clears Write Interrupt Enable    */
#define  SET_WIE        5     /* Sets Write Interrupt Enable      */
#define  CLEAR_RIE      8     /* Clears Read Interrupt Enable     */
#define  SET_RIE        9     /* Sets Read Interrupt Enable       */

#define  TIMER             0x40

#if IS_OS2
#define STDOUT          1
#define MAXMSGLEN       256
#endif

/*************************/
/* Function declarations */
/*************************/
int   SetPaths(void);
int   GetAck(int data);
int   init_piu(void);
int   imf_error(int errorid, int trk);
void  GetParms(char far *parms);


/****************************/
/* External data references */
/****************************/
extern unsigned int midiflags,timingflags;
extern int initflags;
extern int ppqncntr,ppqndiv;
extern struct vscb recio,xmitio;
extern unsigned char *prvtimemsg,*prvtimecounthi,*prvtimecountlo;
extern long operation;
extern unsigned int hostbase = IMFBASE;
extern unsigned char hostrnge = 16;
extern long delay;

unsigned long pprocessed = 1;
unsigned long rprocessed = 1;
int      host_buffer_count = 0;        /* # of buffers written by host     */
int      prv_buffer_count = 0;         /* Previous # TMS buffers played 7191*/

/************************/
/* Device Specific Data */
/************************/
int   irqnum = 3;
int   GotAnInt = 0;           /* Indicates that interrupts are working     */
int   AckData = 0;

unsigned char curtcr = 0;

unsigned char qcaps_sysex[10] = { 0xf0,0,0,0x3a,5,1,0,0x10,0,0xf7 };
unsigned char qid_sysex[10] = { 0xf0,0,0,0x3a,5,4,'I','M','F',0xf7 };

/* Path Enables */
unsigned char sys2sp = 0x1f;  /* System to sound generator  0x01 = 8x-9x   */
unsigned char sys2out = 0x1f; /* System to MIDI-Out         0x02 = Ax,Dx,Ex*/
unsigned char in2sp = 0;      /* MIDI-In to sound generator 0x04 = Bx,Cx   */
unsigned char in2out = 0;     /* MIDI-In to MIDI-Out        0x08 = F0-F7   */
unsigned char in2sys = 0x1f;  /* MIDI-In to System          0x10 = F8-FF   */

char curbank[8] = { -1,-1,-1,-1,-1,-1,-1,-1 };  /* Current voice bank      */


#define  NUM_MODES         2
int   num_modes = NUM_MODES;
#define  CLAIM_HDWR_MODE   1
#define  UNKNOWN_MODE      NUM_MODES
struct mode_data mode_table[NUM_MODES] = {
   MIDI,0,0,8,8,0,0,0,0,1,DATAFLAGS+BPS+SRATE+CHANNELS+BSIZE,      /* MIDI_MODE   */
   CLAIM_HDWR,0,0,0,0,0,0,1,1,0,DATAFLAGS+BPS+SRATE+CHANNELS+BSIZE /* Get hardware*/
};

#if IS_OS2
//*******************************************************
// Global stream data.
//******************************************************* */

UCHAR   nszMsgBuff [MAXMSGLEN];
UCHAR   nszMsgFile [] = "AUDIODD.MSG\0";
UCHAR   nszMsgNum[] = "0\0";
UCHAR   MsgId[] = "AUD";
ULONG   IvTable;
USHORT  MsgLen = 0;

#endif

/****************************/
/* Device Interrupt Handler */
/****************************/
int DevInt()
{
   static int   data,count,intstat;
   unsigned int trk = 0;

   if(!((intstat=inp(TSR)) & INT_PEND)){  /* Is this our int?              */
      eoi(irqnum);                        /* Reset Interrupt Controller    */
      return(-1);                         /* If not, return -1             */
   }
   curtcr &= ~TMSK;
   outp(TCR,curtcr);                        /* Clear TMSK                    */

   do{
      if(intstat & TIMER_B_STATUS){         /* Is Timer B interrupting us?   */
         outp(TCR,curtcr & ~TBC);           /* Reset Timer B (leave disabled)*/
      }
      if(intstat & TIMER_A_STATUS){         /* Timer A interrupting us?      */
         outp(TCR,curtcr & ~TAC);           /* Clear timer A interrupt       */
         if(!is_running(TRK)){              /* If we're not running yet,     */
            GotAnInt=1;                     /*  then simply record an int    */

         }else if(--PPQNCNTR <= 0){    /* Prescale ppqn counter         */
            PPQNCNTR = PPQNDIV;
            if(!(TIMINGFLAGS & 0x4000)){  /* Is external clocking set?     */
               ((IOB)XMITIO.Ptr)->position++;        /* increment position            */
               ((IOB)RECIO.Ptr)->position++;         /*  in both buffers              */
               if( (((IOB)XMITIO.Ptr)->count)        /* Has xmit buffer been updated? */
               && !(MIDIFLAGS & OUTSYNC)){/*  without us knowing?          */
                  MIDIFLAGS |= OUTSYNC;   /* If so, then make sure that    */
                  DELAY = 1;              /*  we process it                */
               }
               if(MIDIFLAGS & OUTSYNC){   /* Is output synch pending?      */
                  if(--DELAY <= 0){       /* Time to do updates?         */
                     DoOutput(0);
#if IS_OS2
                     /* If thread is blocked on room in xmit buffer,       */
                     if(MIDIFLAGS & OUTPUT_WAITING){  /* then unblock it   */
                        DevHlp_Run((long)((char far *)&MIDIFLAGS));
                    }
                     /* If thread is blocked on AUDIO_WAIT                 */
                     if(INITFLAGS & WAITING){ /* then unblock it           */
                        DevHlp_Run((long)((char far *)&XMITIO));
                    }
#endif
                  }
               }
               if(TIMINGFLAGS & 0x1000){  /* Are input timing clocks enabled? */
                  if(TIMINGFLAGS & 0x400){/* Is timing compression enabled?*/
                     QTimeSX(TRK);           /*  if so, go do it              */
                  }else{                  /*  if not, queue a F8           */
                     *((IOB)RECIO.Ptr)->head = 0xf8;     /* Queue a MIDI timing clock     */
                     if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){    /* Time to wrap?*/
                        ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;
                     }
                     if(++((IOB)RECIO.Ptr)->count > ((IOB)RECIO.Ptr)->buf[0].length){    /* Max count reached */
                        ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;       /*  if so, limit it  */
                     }
                  }
               }
               if(TIMINGFLAGS & 0x2000){/* Are output timing clocks enabled?*/
                  DevWrite(0xf8,TRK);        /*  if so, write a F8            */
                  /* this sets TMSK! */
               }
            }
         }
      }

      if(inp(PIU2) & TxRDY){                 /* Spurious xmit interrupt?   */
         outp(PCR,CLEAR_WIE);                /*  clear WIE                 */
      }

      while((data = inp(PIU2)) & RxRDY){     /* Loop until all data read   */
         data = (data << 1) & 0x0100;        /* Shift EXR8 into bit 8 position*/
         data += inp(PIU0);                  /*  Read data from IMF        */
         if(!rec_running(TRK)){              /* If we're not running,      */
            AckData = data;                  /* Save it for GetAck()       */
         }else if(data >= 0x1f0 && data <= 0x1f5){ /* Error message?       */
            /* What do we want to do with error messages?   */
         }else if(data==0xf8){               /* Is this a MIDI timing clock?  */
            if(TIMINGFLAGS & 0x4000){        /* Is external clocking enabled? */
               if(TIMINGFLAGS & 0x400){      /* Is timing compression enabled?*/
                  QTimeSX(TRK);
               }else{
                  *((IOB)RECIO.Ptr)->head = 0xf8;           /* Queue up a MIDI clock      */
                  if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){
                     ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;
                  }
                  if(++((IOB)RECIO.Ptr)->count >= ((IOB)RECIO.Ptr)->buf[0].length){ /* Limit counter to 512   */
                     ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;
                  }
               }
            }
         }else if(data!=0xfe){               /* Not an active sense?       */
            if(MIDIFLAGS & 1){               /* Is time sysex last in queue?*/
               if(*PRVTIMEMSG==0xf0){        /* Verify it's still valid    */
                  count = *PRVTIMECOUNTLO + (*PRVTIMECOUNTHI << 7);
                  if(count < 3){             /* Is count < 3?              */
                     ((IOB)RECIO.Ptr)->count -= 8;          /* Remove sysex from queue    */
                     ((IOB)RECIO.Ptr)->head = PRVTIMEMSG;   /* Move head pointer back     */
                     while(count--){         /* Convert sysex to F8's      */
                        *((IOB)RECIO.Ptr)->head = 0xf8;     /* Convert to timing clocks   */
                        if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){
                           ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;   /*  wrap pointer if needed    */
                        }
                        if(++((IOB)RECIO.Ptr)->count > ((IOB)RECIO.Ptr)->buf[0].length){ /* Increment rec counter*/
                           ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;   /* Limit it to 512    */
                        }
                     }
                  }
               }
               MIDIFLAGS &= 0xfffe;          /* Clear timing sysex queued flag */
            }
            *((IOB)RECIO.Ptr)->head = (unsigned char)data;
            if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){
               ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;
            }
            if(++((IOB)RECIO.Ptr)->count > ((IOB)RECIO.Ptr)->buf[0].length){
               ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;
            }
         }
      }

   }while((intstat=inp(TSR)) & INT_PEND);

   _asm cli;                              /* No int's until we return      */
   eoi(irqnum);                           /* Reset Interrupt Controller    */

   curtcr |= TMSK;                        /* Set TMSK back on              */
   outp(TCR,curtcr);                      /* Set TMSK back on              */

   return(0);                             /* and return 0                  */
}

/**************************/
/* Dummy TimerInt routine */
/**************************/
int TimerInt()
{
   /* Nothing to do */
   return(0);
}

/***********/
/* Note On */
/***********/
void DevNoteOn(chan,key,vel)
char  chan,key,vel;
{
   DevWrite(0x90+chan,TRK);
   DevWrite(key,TRK);
   DevWrite(vel,TRK);
}

/************/
/* Note Off */
/************/
void DevNoteOff(chan,key,velocity)
char  chan,key,velocity;
{
   DevWrite(0x80+chan,TRK);
   DevWrite(key,TRK);
   DevWrite(0,TRK);
}

/*****************/
/* All Notes Off */
/*****************/
void DevAllNotesOff()
{
   static int chan;

   for(chan=0; chan<16; chan++){
      DevWrite(0xb0+chan,TRK);       /* Issue All Notes Off */
      DevWrite(123,TRK);
      DevWrite(0,TRK);
   }
}

/*******************/
/* ChannelPressure */
/*******************/
void  DevChannelPressure(chan,press)
char  chan,press;
{
   DevWrite(0xa0+chan,TRK);
   DevWrite(press,TRK);
}

/******************/
/* Control Change */
/******************/
void  DevControlChange(chan,cnum,cval)
char  chan,cnum,cval;
{
   DevWrite(0xb0+chan,TRK);
   DevWrite(cnum,TRK);
   DevWrite(cval,TRK);
}

/******************/
/* Program Change */
/******************/
void  DevProgramChange(chan,prog)
char  chan,prog;
{
   DevWrite(0xc0+chan,TRK);
   DevWrite(prog,TRK);
}

/**************/
/* AfterTouch */
/**************/
void  DevAfterTouch(chan,key,press)
char  chan,key,press;
{
   DevWrite(0xd0+chan,TRK);
   DevWrite(key,TRK);
   DevWrite(press,TRK);
}

/*************/
/* PitchBend */
/*************/
void  DevPitchBend(chan,lsb,msb)
char  chan,lsb,msb;
{
   DevWrite(0xe0+chan,TRK);
   DevWrite(lsb,TRK);
   DevWrite(msb,TRK);
}

/*************/
/* MTC       */
/*************/
void  DevMTC(byte1,byte2)
char  byte1,byte2;
{
   DevWrite(0xf2,TRK);
   DevWrite(byte1,TRK);
   DevWrite(byte2,TRK);
}

/**************************/
/* Device Dependent SysEx */
/**************************/
void DevDepSX(d1,d2,d3)
char d1,d2,d3;
{
   return;
}

/************************/
/* Generic Voice Select */
/************************/
void DevVoiceSel(inst,type,sel)
char inst,type,sel;
{
   return;
}

/**************************/
/* Query Timbre Parameter */
/**************************/
void DevQTimbreParm(parmid)
long parmid;
{
   return;
}

/************************/
/* Set Timbre Parameter */
/************************/
void DevSetTimbreParm(parmid,parmval)
long parmid;
int parmval;
{
   return;
}

/************************/
/* Request Timbre Block */
/************************/
void DevReqTimbreBlk(blknum)
int blknum;
{
   return;
}

/**********************/
/* Write Timbre Block */
/**********************/
void DevWriteTimbreBlk(blknum,len,blk)
int blknum,len;
unsigned char *blk;
{
   return;
}

/****************/
/* Track Volume */
/****************/
void DevTrkVolume(level,duration,trk)
int level,duration,trk;
{
   return;
}

/*****************/
/* Track Balance */
/*****************/
void DevTrkBalance(balance,duration,trk)
int balance,duration,trk;
{
   return;
}

/*****************/
/* Master Volume */
/*****************/
void DevMstVolume(level,duration,trk)
int duration,trk;
unsigned int level;
{
   return;
}

/********************/
/* Set Clock values */
/********************/
void SetClock(trk)
int trk;
{
#if NUM_TRACKS > 1
   extern int tempo[],ppqn[];
#else
   extern int tempo,ppqn;
#endif
   int ttempo,tppqn;

   ttempo = TEMPO;
   tppqn = PPQN;

   _asm{                ; 500000 / ((tempo * ppqn) / 600);
      mov   dx,ttempo
      mov   ax,tppqn
      mul   dx          ; dx:ax = tempo*ppqn
      mov   bx,600
      div   bx          ; ax = (tempo*ppqn)/600
      mov   bx,ax       ;  save it in bx
      mov   dx,7        ; dx:ax = 500000
      mov   ax,41248
      div   bx          ; ax = Timer Divisor
      mov   bx,ax       ;  save it in bx

      mov   dx,TCWR     ; Initialize Timer Control Word Register
      mov   al,34H      ; Set timer 0 mode 3
      out   dx,al
      jmp   $+2
      mov   dx,CNTR0    ; Write lsb
      mov   al,bl
      out   dx,al       ;  Set divisor LSB
      jmp   $+2
      mov   al,bh
      out   dx,al       ;  and MSB
   }
}


/***************/
/* Device Open */
/***************/
int DevOpen(trk)
int trk;
{
   static int   x,y;

   if(init_piu()){                           /* This leaves curtcr = IBE   */
      return(imf_error(4,trk));
   }
   GetDevInt(irqnum);                        /* Go Hook Device Interrupt   */
   curtcr = IBE+TMSK+TAE+TAC+TBC;            /* Set tcr = 0xc7             */
   outp(TCR,curtcr);                         /* Now enable timer A         */
   _asm sti;                                 /* Enable ints                */
   for(x=0,GotAnInt=0; x<32000 && GotAnInt==0; x++){  /* Wait for timer int*/
      for(y=0; y<1000 && GotAnInt==0; y++);  /*   time delay               */
   }
   if(GotAnInt==0){                          /* Did we time out?           */
      FreeDevInt();                          /*  release int vector        */
      return(imf_error(3,trk));              /*  return error              */
   }
   outp(PCR,SET_RIE);                        /* Enable RIE                 */

   DevWrite(0x1e0,trk);                         /* Set Music Mode             */
   DevWrite(0x100,trk);
   GetAck(0x1e0);

   DevWrite(0x1e1,trk);                         /* Set error reporting        */
   DevWrite(0x101,trk);                         /*  enable bit on             */
   GetAck(0x1e1);

   sys2sp = 0x1f;                            /* Enable all system to SP    */
   in2sys = 0x1f;                            /* Enable all MIDI-In to sys  */
   in2sp = 0;                                /* Disable MIDI-In to SP      */
   in2out = 0;                               /* Disable MIDI-In to MIDI-Out*/
   sys2out = 0;                              /* Disable system to MIDI-Out */
   SetPaths();                               /*  (during setup stuff)      */

   DevWrite(0x1e3,trk);                         /* Set node data              */
   DevWrite(0x100,trk);                         /*  Node 0                    */
   DevWrite(0x100,trk);                         /*  Non-protect               */
   DevWrite(0x111,trk);                         /*  config 17                 */
   DevWrite(0x100,trk);                         /*  master tune = 0           */
   DevWrite(0x17f,trk);                         /*  full volume               */
   DevWrite(0x100,trk);                         /*  chain disabled            */
   DevWrite(0x100,trk);                         /*  (reserved)                */
   DevWrite(0x100,trk);                         /*  (reserved)                */
   GetAck(0x1e3);

   for(x=0; x<8; x++){                       /* Reset current bank values  */
      DevWrite(0xf0,trk);                       /*  send IMF bank select SysEx*/
      DevWrite(0x43,trk);                       /*  Yamaha manuf id           */
      DevWrite(0x75,trk);                       /*  subid                     */
      DevWrite(0,trk);                          /*  res                       */
      DevWrite(0x18+x,trk);                     /*  18+i                      */
      DevWrite(4,trk);                          /*  sel bank msg id           */
      DevWrite(2,trk);                          /*  Bank #2                   */
      curbank[x] = 2;                           /*   and copy it to curbank   */
      DevWrite(0xf7,trk);                       /*  EOX                       */
      DevWrite(0xc0+x,trk);                     /*  Prog change               */
      DevWrite(5,trk);                          /*  select piano voice        */
      DevWrite(0xb0+x,trk);                     /*  Set mono mode             */
      DevWrite(0x7e,trk);                       /*                            */
      DevWrite(1,trk);                          /*                            */
   }

   sys2out = 0x1f;                           /* enable system to MIDI-Out  */
   SetPaths();

   return(0);
}

/****************/
/* Device Close */
/****************/
int DevClose(trk)
int trk;
{
   DevWrite(0x1e5,trk);                         /* Issue REBOOT               */
   GetAck(0x1e5);                             /*  Wait for an ack           */
   outp(PCR,CLEAR_WIE);                      /* Disable WIE                */
   outp(PCR,CLEAR_RIE);                      /* Disable RIE                */
   outp(TCR,IBE);                            /* Force IRQ low              */
   curtcr = IBE;
   outp(TCR,0);                              /* Remove IRQ from bus        */
   curtcr = 0;
   FreeDevInt();
   return(0);
}

/************/
/* DevStart */
/************/
void DevStart(trk)
int trk;
{
   /* nothing to do */
}

/************/
/* DevPause */
/************/
void DevPause(trk)
int trk;
{
   /* nothing to do */
}

/*************/
/* DevResume */
/*************/
void DevResume(trk)
int trk;
{
   /* nothing to do */
}

/***********/
/* DevStop */
/***********/
void DevStop(trk)
int trk;
{
   /* nothing to do */
}

/*****************************************************************************/
/* DevNotify                                                                 */
/*****************************************************************************/
char DevNotify(pos,trk)
unsigned long pos;
int trk;
{
   /* Nothing to do */
   return(0);
}

/*************/
/* DevDeInit */
/*************/
void DevDeInit(trk)
int trk;
{
   /* nothing to do */
}

/*************/
/* DevChange */
/*************/
int DevChange(changes, dev_info, any_dev_info, mode_info, any_mode_info, trk)
struct audio_change far *changes;
int *dev_info;
char any_dev_info;
int *mode_info;
char any_mode_info;
int trk;
{
   /* nothing to do */
   return(0);
}

/*********************/
/* Device init IOCTL */
/*********************/
int DevIOCTLinit(dptr,index,trk)             /* Process ioctl init requests   */
AIP   dptr;
int   index,trk;
{

   switch(index){
   default:
      index = MIDI_MODE;

      dptr->flags &= ~(mode_table[index].flags_mask);
      dptr->flags |= mode_table[index].flags | BESTFIT_PROVIDED;
      dptr->mode = mode_table[index].mode;
      dptr->srate = mode_table[index].srate_low;
      dptr->bits_per_sample = mode_table[index].bps_low;
      dptr->bsize = mode_table[index].bsize;
      dptr->channels = mode_table[index].channels_low;

   case MIDI_MODE:
      dptr->flags |= INPUT | OUTPUT | VOLUME | BALANCE;
      break;
   case CLAIM_HDWR:
      break;
   }/*switch*/

   dptr->rc = 0;                          /*  set rc = 0                   */
   dptr->slot_number = 0;
   dptr->device_id = IMF;                 /* Set device ID number          */

   return(index);
}


/****************/
/* Status IOCTL */
/****************/
void DevIOCTLstatus(dptr,trk)
ASP dptr;
int trk;
{
#if NUM_TRACKS > 1
   extern int tempo[];
#else
   extern int tempo;
#endif
   struct audio_change far *cptr;
   struct track_info far *tptr;

   dptr->mode = MIDI;                        /*    MIDI mode               */
   dptr->srate = TEMPO;                      /*    rate = tempo            */
   dptr->bits_per_sample = 8;                /*    byte data               */
   dptr->bsize = 1;                          /*    block size = 1          */
   dptr->channels = 1;                       /*    1 channel               */
   dptr->flags = 0;                          /*    flags                   */
   dptr->operation = OPERATION_G;            /*    operation               */
   cptr=&(dptr->change);                     /* ->audio_change block       */
   cptr->input = 0;                          /*    input                   */
   cptr->output = OUTPUT_1;                  /* output                     */
   cptr->monitor = 0;                        /*    monitor                 */
   cptr->volume = 0;                         /*    volume                  */
   cptr->volume_delay = 0;                   /*    volume delay            */
   cptr->balance = 0;                        /*    balance                 */
   cptr->balance_delay = 0;                  /*    balance delay           */
   cptr->treble = 0;                         /*    treble                  */
   cptr->bass = 0;                           /*    bass                    */
   cptr->pitch = 0;                          /*    pitch                   */
   if(tptr = cptr->dev_info){                /*    dev_info                */
      /* not defined for IMF */
   }
}


/*********************/
/* Device load IOCTL */
/*********************/
void DevIOCTLload(buf,size,flags,trk)
unsigned char far *buf;
unsigned long size,flags;
int trk;
{
   return;                                /* Nothing to do for IMF         */
}

/*********************/
/* Device wait IOCTL */
/*********************/
void DevIOCTLwait(void)
{
   return;                                /* Nothing to do for IMF         */
}

/*********************/
/* Diagnostic IOCTLs */
/*********************/
void DevIOCTLread8(dptr)
DR8 dptr;
{
   dptr->data=(unsigned char)inp(IMFBASE+dptr->offset);  /* Read data from reg*/
}

void DevIOCTLwrite8(dptr)
DR8 dptr;
{
   outp(IMFBASE+dptr->offset,dptr->data); /* Write data to port            */
}

void DevIOCTLread16(dptr)
DR16 dptr;
{
   dptr->data=inpw(IMFBASE+dptr->offset); /* Read data from reg            */
}

void DevIOCTLwrite16(dptr)
DR16 dptr;
{
   outpw(IMFBASE+dptr->offset,dptr->data);   /* Write data to port         */
}


/*************************/
/* Device Initialization */
/*************************/
int DevInit(parms)
char far *parms;
{
   static int   x,y;

   GetParms(parms);

   if(init_piu()){                     /* Also sets curtcr = 0x80          */
      return(imf_error(1,TRK));
   }

   GetDevInt(irqnum);                  /* Go get interrupt vector          */
   SetClock(TRK);                      /* Set timer parameters             */

   /* TCR must be set to 0x80 LONG before enabling TMSK!  (thus init_piu)  */
   curtcr = IBE+TMSK+TAE+TBC+TAC;
   outp(TCR,curtcr);                   /* Enable TCR & Timer A (clear both)*/
   /* Note that if 0xc6 is written, then no timer A interrupt will occur!  */

   _asm sti;                           /* Ints on                          */

/* DevWrite(0x1e5,trk);  */               /* Issue REBOOT                     */
/* GetAck(0x1e5);        */               /*  wait for ack                    */

   /* An Interrupt should occur right about here */

   /* Verify that timer int occurs                                         */
   for(x=0,GotAnInt=0; x<16 && GotAnInt==0; x++){
      for(y=0; y<32000 && GotAnInt==0; y++); /* Time delay                 */
   }
   outp(TCR,IBE);                      /* Disable TMSK and Timer A (leave IBE) */
   curtcr = IBE;

   _asm cli;                           /* Ints back off                    */

   FreeDevInt();                       /* Restore interrupt vector         */

   /* Make sure we found a card */
   if(GotAnInt==0){

#if IS_OS2
      nszMsgNum[0] = IMF_ERROR_MSG;
#endif
      imf_error(2,TRK);

#if IS_OS2
   } else {
      nszMsgNum[0] = IMF_OK_INSTALL_MSG;
#endif

   }

#if IS_OS2
   //*************************
   // Get message from AUDIODD.MSG
   //*************************
   DosGetMessage ((PCHAR FAR *)IvTable, 1,
                  nszMsgBuff, MAXMSGLEN,
                  *nszMsgNum, nszMsgFile, &MsgLen);

   nszMsgBuff[IRQ_NUM_POS] = (char)('0'+irqnum);

   DosPutMessage (STDOUT, MsgLen, nszMsgBuff);
#endif

   return(0);
}

int imf_error(errcode,trk)
int errcode,trk;
{

   MIDIFLAGS |= 0x40;                     /* Set IMF not responding flag   */
   /* display error message */
#if IS_OS2
   nszMsgBuff[ERROR_NUM_POS] += errcode;
   //*************************
   // Get message from AUDIODD.MSG
   //*************************
   DosGetMessage ((PCHAR FAR *)IvTable, 1,
                  nszMsgBuff, MAXMSGLEN,
                  *nszMsgNum, nszMsgFile, &MsgLen);

   DosPutMessage (STDOUT, MsgLen, nszMsgBuff);
#else
   errormsg[ERROR_NUM_POS] += errcode;
   _asm{
      MOV   AH,9
      MOV   DX,OFFSET errormsg
      INT   21H
   }
#endif
   return(-1);
}

/*********************************************************/
/* IMF Write - Wait for Xmit Rdy, then write word in AX  */
/* Leaves with WIE disabled when done, AX=0              */
/*********************************************************/
int DevWrite(ch,trk)                         /* DEVWRITE */
int ch,trk;                                  /* bit 8 set = IMF command       */
{
   static int   x;

   if(curtcr & TMSK) outp(TCR,curtcr & ~TMSK);  /* Clear TMSK (if needed)  */
   outp(PCR,SET_WIE);                     /* Enable WIE                    */
   for(x=0; x<32000 && !(inp(PIU2) & TxRDY); x++);  /* Wait for Xmit ready */
   if(ch & 0x100){
      curtcr |= EXT8;                     /* Set EXT8                      */
   }else{
      curtcr &= ~EXT8;                    /* Clear EXT8                    */
   }
   outp(TCR,curtcr & ~TMSK);              /* Write new EXT8 (TMSK off)     */
   outp(PIU1,ch & 0xff);                  /* Write lower 8 bits            */
   outp(PCR,CLEAR_WIE);                   /* Disable TxRDY interrupt       */
   if(curtcr & TMSK) outp(TCR,curtcr);    /* Set TMSK back on (if it was)  */

   return(0);
}

/************/
/* GET_ACK  */
/************/
int GetAck(val)           /* Wait until 'val' received */
int val;
{
   extern int AckData;
   static int   x;

   for(x=0; x < 32000 && AckData != val && midiread(TRK) != (val & 0xff); x++);
   if(x==32000){
      return(-1);
   }
   return(0);
}

/*************/
/* Set paths */
/*************/

int SetPaths()
{
   DevWrite(0x1e2,TRK);                      /* Set Paths                     */
   DevWrite(0x100 + in2sys,TRK);             /* MIDI in to System             */
   DevWrite(0x100 + sys2out,TRK);            /* System to MIDI Out            */
   DevWrite(0x100 + in2sp,TRK);              /* MIDI In to SP                 */
   DevWrite(0x100 + sys2sp,TRK);             /* System to SP                  */
   DevWrite(0x100 + in2out,TRK);             /* MIDI-In to MIDI-Out           */

   return(GetAck(0x1e2));                 /* Wait for ack then return      */
}




/***************************************************************************/
/* Initialize PIU, verify operation, then leave WIE and RIE off, TCR = IBE */
/***************************************************************************/
int init_piu()
{
   static int x;

   outp(TCR,IBE);                /* Ints disabled, but IBE gated to bus    */
   curtcr = IBE;
   outp(PCR,0xbc);               /* Initialize PIU                         */
   outp(PCR,CLEAR_WIE);          /* Turn off Write Interrupt Enable (WIE)  */
   outp(PCR,CLEAR_RIE);          /* Turn off Read Interrupt Enable  (RIE)  */

   /* Wait for RIE and WIE to clear                                        */
   for(x=0; x<32000 && (inp(PIU2) & (RIE+WIE)); x++);
   if(x==32000) return(-1);      /* If they don't, then theres a problem   */

   outp(PCR,SET_WIE);            /* Set Write Interrupt Enable (WIE)       */
   outp(PCR,SET_RIE);            /* Set Read Interrupt Enable (RIE)        */

   /* Wait for both bits to come back on                                   */
   for(x=0; x<32000 && (inp(PIU2) & (RIE+WIE))!=(RIE+WIE); x++);
   if(x==32000) return(-1);      /*   If they don't, then there's a problem */

   outp(PCR,CLEAR_WIE);          /* Turn WIE back off                      */
   outp(PCR,CLEAR_RIE);          /* Turn RIE back off                      */

   return(0);
}

#if IS_OS2
/*****************************************************************************/
/* PositionNotCountedYet                                                     */
/*****************************************************************************/
unsigned long PositionNotCountedYet(trk)
int trk;
{
   /* Do Nothing */
   return(0);
} /* end PositionNotCountedYet */
#endif

/************************************************************************/
/* This routine will parse the parameters from the CONFIG.SYS.          */
/*                                                                      */
/* Expected parameters are:                                             */
/*                                                                      */
/*              DEVICE=d:\path\IMFDD.SYS midi# int# pbufsiz scrsize               */
/*                                                                      */
/*      Written by Ron Lisle, Multi Media Products, Austin, TX.              */
/*      Portions copied from D. Dievendorfs VDISK listing                    */
/************************************************************************/
void GetParms(parms)
char far *parms;
{
   extern char midinum;
   static char ch;

   while((ch = *parms++)!=10 && ch!=13){
      /* find DOS delimiter */
      if(ch==0 || ch==' ' || ch==',' || ch==';' || ch=='+' || ch=='=' || ch=='\t'){
         break;
      }
   }
   if(ch==10 || ch==13) return;

   while((ch = *parms++)!=10 && ch!=13){
      /* Look for MIDI# */
      if(ch>='1' && ch<='9'){
         midinum=ch;
#if NOT_OS2
         startmsg2[AUDIO_NUM_POS] = ch;
#else
         nszMsgBuff[AUDIO_NUM_POS] = ch;
#endif

         /* Got it.  Now look for IRQ # */
         while((ch = *parms++)!=10 && ch!=13){
            if(ch>='1' && ch<='9'){
#if NOT_OS2
               startmsg2[IRQ_NUM_POS] = ch;
#else
               nszMsgBuff[IRQ_NUM_POS] = ch;
#endif
               irqnum = ch-'0';
            }
         }
      }
   }
}


/*****************************************************/
/* Generic Voice Translation Table (Bank #, Voice #) */
/*****************************************************/
char soundtbl[128][2] = {
   3,47,    /* 0 Accordian    Squeeze     */
   2,0,     /* 1 Bagpipe       ?          */
   2,0,     /* 2 Balalaika     ?          */
   6,22,    /* 3 Banjo         Banjo      */
   2,10,    /* 4 Bass          WodBass    */
   2,10,    /* 5 Bass, Double  WodBass    */
   2,9,     /* 6 Bass, Elec    EBass      */
   5,21,    /* 7 Slap Bass     FlapBas    */
   4,40,    /* 8 Bassoon       Bassoon    */
   2,26,    /* 9 Bell          Bells      */
   5,42,    /* 10 Bongo        HandDr     */
   2,00,    /* 11 Brass sect   Brass      */
   4,5,     /* 12 Bugle        Trumpt2    */
   2,0,     /* 13 Calliope     ?          */
   6,44,    /* 14 Cannon       Smash      */
   5,32,    /* 15 Castanets    Xyloph2    */
   3,46,    /* 16 Celeste      Celeste    */
   4,24,    /* 17 Cello        Cello1     */
   5,38,    /* 18 Chimes       Bells 2    */
   2,39,    /* 19 Choir, Fem   Voices     */
   6,33,    /* 20 Male Choir   M.Voice    */
   2,39,    /* 21 Mixed Choir  Voices     */
   5,32,    /* 22 Claps        Xyloph2    */
   2,18,    /* 23 Clarinet     Clarine    */
   5,32,    /* 24 Claves       Xyloph2    */
   2,24,    /* 25 Clavichord   Clav       */
   3,35,    /* 26 Clav, muted  MuteClv    */
   4,5,     /* 27 Cornet       Trumpt2    */
   5,45,    /* 28 Cow Bell     Heifer     */
   2,43,    /* 29 Crash Cymbal RD Cymb    */
   2,43,    /* 30 Cymbal       RD Cymb    */
   2,41,    /* 31 Drum         Elec Dr    */
   2,41,    /* 32 Drum, Bass   Elec Dr    */
   5,32,    /* 33 Drum, Rimsht Xyloph2    */
   2,42,    /* 34 Drum, Snare  SnareDr    */
   5,40,    /* 35 Drum, Steel  SteelDr    */
   4,03,    /* 36 Flugel Horn  Flugelh    */
   2,15,    /* 37 Flute        Flute      */
   4,0,     /* 38 French Horn  Horn2      */
   2,19,    /* 39 Glockenspiel Glocken    */
   5,39,    /* 40 Gong         TempleG    */
   6,15,    /* 41 Guitar       Guitar     */
   6,18,    /* 42 Guitar, Elec BriteGt    */
   6,19,    /* 43 Guitar, Fuzz FuzzGt     */
   2,8,     /* 44 Guitar, Jazz Jazz Gt    */
   6,18,    /* 45 Guitar, Stl  BriteGt    */
   2,35,    /* 46 Guitar, Waa  ZingPlp    */
   4,42,    /* 47 Harmonica    Harmon2    */
   2,27,    /* 48 Harp         Harp       */
   3,41,    /* 49 Harpsichord  Harsi2     */
   2,43,    /* 50 Hi Hat (cls) RD Cymb    */
   2,43,    /* 51 Hi Hat (opn) RD Cymb    */
   2,35,    /* 52 Jew's Harp   ZingPlp    */
   2,22,    /* 53 Koto         Koto       */
   6,21,    /* 54 Lute         Lute       */
   6,25,    /* 55 Mandolin     Harp3      */
   5,46,    /* 56 Maraca       SnareD2    */
   2,40,    /* 57 Marimba      Marimba    */
   2,17,    /* 58 Oboe         Oboe       */
   4,29,    /* 59 Orchestra    Orchest    */
   2,11,    /* 60 Organ, elec  EOrgan1    */
   2,11,    /* 61 Organ, Hmnd  EOrgan1    */
   6,12,    /* 62 Organ, Pipe  BigPipe    */
   4,36,    /* 63 Pan pipe     Pan Flt    */
   2,05,    /* 64 Piano        Piano      */
   2,06,    /* 65 Piano, Elec  NewEP      */
   3,07,    /* 66 Piano, Grand Grand      */
   3,12,    /* 67 Piano, Hnky  Honkey1    */
   2,16,    /* 68 Piccolo      Picolo     */
   6,42,    /* 69 Rattle       SpTalk     */
   2,43,    /* 70 Ride Cymbal  RC Cymb    */
   4,46,    /* 71 Saxaphone    Sax1       */
   6,28,    /* 72 Sitar        Sitar1     */
   5,32,    /* 73 Snap, finger Xyloph2    */
   2,04,    /* 74 String Sect  Strings    */
   4,31,    /* 75 Strings, Piz Pizzic1    */
   2,00,    /* 76 Synth, brass Brass      */
   2,00,    /* 77 Synth, growl            */
   2,15,    /* 78 Synth, PWN   Flute      */
   2,04,    /* 79 Synth, pwm   Strings    */
   2,04,    /* 80 Synth, str   Strings    */
   5,42,    /* 81 Tamborine sh HandDr     */
   5,42,    /* 82 Tamborine st HandDr     */
   2,31,    /* 83 Timpani      Timpani    */
   2,44,    /* 84 Tom Tom      TomTom     */
   2,43,    /* 85 Triangle     RD Cymb    */
   4,04,    /* 86 Trombone     Trombone   */
   2,02,    /* 87 Trumpet      Trumpet    */
   2,01,    /* 88 Tuba         Horn       */
   5,36,    /* 89 Tubular Bell TubeBe1    */
   6,17,    /* 90 Ukulele      PluckGt    */
   2,20,    /* 91 Vibes        Vibes      */
   2,03,    /* 92 Viola        LoStrig    */
   4,19,    /* 93 Violin       SoloVio    */
   4,32,    /* 94 Violin, pizz Pizzic2    */
   2,39,    /* 95 Voice, femal Voices     */
   6,33,    /* 96 Voice, male  M.Voice    */
   2,34,    /* 97 Whistle      Whistle    */
   2,47,    /* 98 Wind Chime   WindBel    */
   5,32,    /* 99 Wood Block   Xyloph2    */
   2,21,    /* 100 Xylophone   Xylophn    */
   2,23,    /* 101 Zither      Zither     */
   2,00,    /* 102                        */
   2,00,    /* 103                        */
   2,00,    /* 104                        */
   2,00,    /* 105                        */
   2,00,    /* 106                        */
   2,00,    /* 107                        */
   2,00,    /* 108                        */
   2,00,    /* 109                        */
   2,00,    /* 110                        */
   2,00,    /* 111                        */
   2,00,    /* 112                        */
   2,00,    /* 113                        */
   2,00,    /* 114                        */
   2,00,    /* 115                        */
   2,00,    /* 116                        */
   2,00,    /* 117                        */
   2,00,    /* 118                        */
   2,00,    /* 119                        */
   2,00,    /* 120                        */
   2,00,    /* 121                        */
   2,00,    /* 122                        */
   2,00,    /* 123                        */
   2,00,    /* 124                        */
   2,00,    /* 125                        */
   2,00,    /* 126                        */
   2,00     /* 127                        */
};

/* Generic Voice Translation Table (Bank #, Voice #) */
char effects_tbl[128][2] = {
   0,1,     /* 0 Air Horn         Horn       */
   6,46,    /* 1 Airplane         Helecopter */
   5,44,    /* 2 Alarm Clock      Clock      */
   6,35,    /* 3 Auto Engine      Racing     */
   6,47,    /* 4 Bee              SineWav    */
   6,36,    /* 5 Bird Chirp       Water      */
   6,44,    /* 6 Cannon           Smash      */
   6,42,    /* 7 Cat Meow         SpTalk     */
   5,45,    /* 8 Cow Bell         Heifer     */
   6,44,    /* 9 Crash            Smash      */
   6,47,    /* 10 Dial Tone       SineWav    */
   2,00,    /* 11 Dog Bark        ?          */
   2,00,    /* 12 Door Slam       ?          */
   2,00,    /* 13 Door Knock      ?          */
   6,40,    /* 14 Elephant        Space 1    */
   6,44,    /* 15 Explosion       Smash      */
   2,00,    /* 16 Fog Horn        Brass      */
   2,44,    /* 17 Foot steps      TomTom     */
   6,44,    /* 18 Gun Shot        Smash      */
   6,46,    /* 19 Helicopter      Helicop    */
   6,42,    /* 20 Jungle Sounds   SpTalk     */
   6,36,    /* 21 Ocean Waves     Water?     */
   6,47,    /* 22 Phone busy sgnl SineWav    */
   6,45,    /* 23 Phone ring      Alarm      */
   6,47,    /* 24 Phone ring sgnl SineWav    */
   6,45,    /* 25 Pneumatic Chisl Alarm      */
   6,36,    /* 26 Rain            Water      */
   6,37,    /* 27 Ray gun         WildWar    */
   6,37,    /* 28 Rooster Crow    WildWar    */
   6,37,    /* 29 Siren           WildWar    */
   2,35,    /* 30 Spring          ZingPlp    */
   2,46,    /* 31 Thunder         Storm      */
   6,35,    /* 32 Train           Racing     */
   6,10,    /* 33 Train whistle   SmlPipe    */
   6,36,    /* 34 Water Drop      Water      */
   6,44,    /* 35 Whip            Smash      */
   6,43,    /* 36 Wind            Winds      */
   2,00,    /* 37                            */
   2,00,    /* 38                            */
   2,00,    /* 39                            */
   2,00,    /* 40                            */
   2,00,    /* 41                            */
   2,00,    /* 42                            */
   2,00,    /* 43                            */
   2,00,    /* 44                            */
   2,00,    /* 45                            */
   2,00,    /* 46                            */
   2,00,    /* 47                            */
   2,00,    /* 48                            */
   2,00,    /* 49                            */
   2,00,    /* 50                            */
   2,00,    /* 51                            */
   2,00,    /* 52                            */
   2,00,    /* 53                            */
   2,00,    /* 54                            */
   2,00,    /* 55                            */
   2,00,    /* 56                            */
   2,00,    /* 57                            */
   2,00,    /* 58                            */
   2,00,    /* 59                            */
   2,00,    /* 60                            */
   2,00,    /* 61                            */
   2,00,    /* 62                            */
   2,00,    /* 63                            */
   2,00,    /* 64                            */
   2,00,    /* 65                            */
   2,00,    /* 66                            */
   2,00,    /* 67                            */
   2,00,    /* 68                            */
   2,00,    /* 69                            */
   2,00,    /* 70                            */
   2,00,    /* 71                            */
   2,00,    /* 72                            */
   2,00,    /* 73                            */
   2,00,    /* 74                            */
   2,00,    /* 75                            */
   2,00,    /* 76                            */
   2,00,    /* 77                            */
   2,00,    /* 78                            */
   2,00,    /* 79                            */
   2,00,    /* 80                            */
   2,00,    /* 81                            */
   2,00,    /* 82                            */
   2,00,    /* 83                            */
   2,00,    /* 84                            */
   2,00,    /* 85                            */
   2,00,    /* 86                            */
   2,00,    /* 87                            */
   2,00,    /* 88                            */
   2,00,    /* 89                            */
   2,00,    /* 90                            */
   2,00,    /* 91                            */
   2,00,    /* 92                            */
   2,00,    /* 93                            */
   2,00,    /* 94                            */
   2,00,    /* 95                            */
   2,00,    /* 96                            */
   2,00,    /* 97                            */
   2,00,    /* 98                            */
   2,00,    /* 99                            */
   2,00,    /* 100                           */
   2,00,    /* 101                           */
   2,00,    /* 102                           */
   2,00,    /* 103                           */
   2,00,    /* 104                           */
   2,00,    /* 105                           */
   2,00,    /* 106                           */
   2,00,    /* 107                           */
   2,00,    /* 108                           */
   2,00,    /* 109                           */
   2,00,    /* 110                           */
   2,00,    /* 111                           */
   2,00,    /* 112                           */
   2,00,    /* 113                           */
   2,00,    /* 114                           */
   2,00,    /* 115                           */
   2,00,    /* 116                           */
   2,00,    /* 117                           */
   2,00,    /* 118                           */
   2,00,    /* 119                           */
   2,00,    /* 120                           */
   2,00,    /* 121                           */
   2,00,    /* 122                           */
   2,00,    /* 123                           */
   2,00,    /* 124                           */
   2,00,    /* 125                           */
   2,00,    /* 126                           */
   2,00     /* 127                           */
};


