
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Prog. VOC2WAVE : Ein Konvertierungsprogramm, das VOC-Dateien in WAVE-Da-  }
{                  teien konvertiert.                                       }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : VOC2WAVE.PAS                                             }
{ entwickelt am  : 12.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.03                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

PROGRAM VOC2WAVE;

{$D-}                                        { keine Debugger-Informationen }
{$F-}                                        { FAR-Aufrufe sind nicht ntig }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

{$M 16384,65536,655360}         { 16 KB Stack, min. 64 KB Heap, max. 640 KB }

USES CRT, DOS;                                { CRT- und DOS-Unit einbinden }

TYPE

  IDString = ARRAY[0..$13] OF Char;                { Identifikations-String }
  String4  = ARRAY[0..3] OF Char;                    { String aus 4 Zeichen }

CONST

  {-------------------------------------------------------------------------}
  { String-Konstanten                                                       }
  {-------------------------------------------------------------------------}

  RIFFStr : String4 = ('R', 'I', 'F', 'F');
  WAVEStr : String4 = ('W', 'A', 'V', 'E');
  fmt_Str : String4 = ('f', 'm', 't', #$20);

TYPE

  {=========================================================================}
  { TVOCRec: Struktur des VOC-Header                                        }
  {=========================================================================}

  TVOCRec = RECORD
    IDStr                : IDString;                        { Datei-Kennung }
    DataOffs, Ver, Check : Word;   { Datenblock-Offset, Version, Check-Code }
  END;

  {=========================================================================}
  { TWAVERec: Struktur des WAVE-Header                                      }
  {=========================================================================}

  TWAVERec = RECORD
    RIFF      : String4;                                           { "RIFF" }
    FileLen   : LongInt;                         { Lnge der WAVE-Datei - 8 }
    WAVE      : String4;                                           { "WAVE" }
    fmt_      : String4;                                           { "fmt_" }
    BlocksLen : LongInt; { Lngenangabe bestehend aus Bytes 10h 00h 00h 00h }

    { Headerblock: wave-format }

    FormatType  : Word;                         { Art des benutzten Formats }
    Channels    : Word;   { Anzahl der benutzten Kanle (1: Mono, 2: Stereo }
    SampleRate  : LongInt;                                  { Sampling-Rate }
    BytesProSec : LongInt;   { durschnittl. Anz. Bytes pro Sek. bei Ausgabe }
    SampleType  : Word;                           { Anzahl Bytes pro Sample }

    { Headerblock: format-specific }

    SampleBits : Word;     { Anzahl Bits pro Sample (8: 8 Bits, 16: 16 Bits }

    { Danach folgt abhngig von der Lnge der beiden Headerblcke }
    { der String "data" und anschlieend die Lnge der Daten (ein }
    { Doppelwort).                                                }

  END;

VAR
  VOCName  : String;                                   { Name der VOC-Datei }
  VOCFile  : File;                                      { VOC-Dateivariable }
  WAVEName : String;                                  { Name der WAVE-Datei }
  WAVEFile : File;                                     { WAVE-Dateivariable }

{===========================================================================}
{ Prozedur ShowHelp: Hilfe zum Programm auf dem Bildschirm anzeigen.        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ShowHelp;

BEGIN
  WriteLn('Konvertiert eine VOC-Datei in eine WAVE-Datei.');
  WriteLn;
  WriteLn('Syntax: VOC2WAVE [Ausgangsdatei].VOC [Zieldatei].WAV');
END;

{===========================================================================}
{ Funktion UpperString: Wandelt einen String beliebiger Lnge in Gro-      }
{                       schreibung um und liefert ihn an den Aufrufer der   }
{                       Funktion zurck.                                    }
{===========================================================================}
{ Eingabe: S = String, der in Groschreibung umgewandelt werden soll        }
{ Ausgabe: String in Groschreibung                                         }
{---------------------------------------------------------------------------}

FUNCTION UpperString(S : String) : String;

VAR
  Count : Word;                                                { ein Zhler }
  Upper : String;                  { in Groschreibung umgewandelter String }

BEGIN
  UpperString := '';
  IF S <> '' THEN                               { Ist S kein leerer String? }
    BEGIN                                          { nein, S ist nicht leer }
      Upper := '';
      FOR Count := 1 TO Length(S) DO                     { String umwandeln }
        Upper := Upper+UpCase(S[Count]);
      UpperString := Upper;                      { neuen String zurckgeben }
    END;
END;

{===========================================================================}
{ Prozedur Error: Zeigt einen Fehler an und beendet das Programm mit einem  }
{                 Halt-Befehl.                                              }
{===========================================================================}
{ Eingabe: Text = Fehlermeldung                                             }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Error(Text : String);

BEGIN
  WriteLn;
  WriteLn(Text);                                   { Fehlermeldung ausgeben }
  Halt;                                                  { Programm beenden }
END;

{===========================================================================}
{ Prozedur Convert: Kmmert sich um die Konvertierung VOC -> WAVE.          }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Convert;

VAR
  VOCHeader     : TVOCRec;                                     { VOC-Header }
  WAVEHeader    : TWAVERec;                                   { WAVE-Header }
  VOC_IDStr     : String[20];                  { VOC-Identifikations-String }
  Count         : Word;                                        { ein Zhler }
  Result        : Integer;                          { Fehlerstatus-Variable }
  VOCBlockType  : Byte;                                      { VOC-Blocktyp }
  VOCBlockSize  : LongInt;                    { Gre eines VOC-Datenblocks }
  OldVOCPos     : Word;  { speichert die Position innerhalb einer VOC-Datei }

  {=========================================================================}
  { Prozedur ReadError: Wird beim Auftreten eines Lesefehlers aufgerufen.   }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadError(FileName : String);

  BEGIN
    Close(VOCFile);                                   { VOC-Datei schlieen }
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Erase(WAVEFile);                                   { WAVE-Datei lschen }
    Error('Fehler: Lesefehler bei '+FileName);     { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur WriteError: Wird beim Auftreten eines Schreibfehlers aufgeru-  }
  {                      fen.                                               }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE WriteError(FileName : String);

  BEGIN
    Close(VOCFile);                                   { VOC-Datei schlieen }
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Erase(WAVEFile);                                   { WAVE-Datei lschen }
    Error('Fehler: Schreibfehler bei '+FileName);  { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur SeekError: Wird beim Auftreten eines Positionierungsfehlers    }
  {                     aufgerufen.                                         }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE SeekError(FileName : String);

  BEGIN
    Close(VOCFile);                                   { VOC-Datei schlieen }
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Erase(WAVEFile);                                   { WAVE-Datei lschen }
    Error('Fehler: Positionierungsfehler '+        { Fehlermeldung ausgeben }
      ' bei '+FileName);
  END;

  {=========================================================================}
  { Prozedur ReadVOCBlockData: Liest die Daten eines VOC-Blocks ein. Die    }
  {                            Variablen VOCBlockType und VOCBlockSize wer- }
  {                            den entsprechend gesetzt.                    }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadVOCBlockData;

  VAR
    W : Word;
    B : Byte;

  BEGIN
    BlockRead(VOCFile, VOCBlockType, SizeOf(Byte));    { VOC-Blocktyp lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError(VOCName);

    { VOC-Blockgre lesen }

    BlockRead(VOCFile, W, SizeOf(Word));
    BlockRead(VOCFile, B, SizeOf(Byte));

    Result := IOResult;
    IF Result <> 0 THEN
      ReadError(VOCName);

    { Blockgre aus B und W ermitteln (24 Bit lang) }

    VOCBlockSize := B;
    VOCBlockSize := VOCBlockSize*65536+W;
  END;

  {=========================================================================}
  { Funktion ReadSampleRate1: Liest die Sample-Rate aus einem VOC-Daten-    }
  {                           block und liefert diese in Hz umgerechnet zu- }
  {                           rck. Die Funktion wird nur bei dem Blocktyp  }
  {                           mit dem ID-Code 1 aufgerufen.                 }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Sample-Rate in Hz                                              }
  {-------------------------------------------------------------------------}

  FUNCTION ReadSampleRate1 : Word;

  VAR
    TC : Byte;        { Sample-Rate in komprimierter Form - "time constant" }

  BEGIN
    ReadSampleRate1 := 0;
    BlockRead(VOCFile, TC, SizeOf(Byte));            { Sample-Rate auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError(VOCName);

    { Berechnung der eigentlichen Sample-Rate }

    IF TC <> 0 THEN
      ReadSampleRate1 := 1000000 DIV (256-TC);
  END;

  {=========================================================================}
  { Funktion ReadSampleRate2: Liest die Sample-Rate aus einem VOC-Daten-    }
  {                           block und liefert diese zurck. Die Funktion  }
  {                           wird nur bei dem Blocktyp 9 aufgerufen.       }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Sample-Rate in Hz                                              }
  {-------------------------------------------------------------------------}

  FUNCTION ReadSampleRate2 : Word;

  VAR
    TC : Word;        { Sample-Rate in komprimierter Form - "time constant" }

  BEGIN
    BlockRead(VOCFile, TC, SizeOf(Word));            { Sample-Rate auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError(VOCName);
    ReadSampleRate2 := TC;
  END;

  {=========================================================================}
  { Funktion ReadMode: Liest den Modus, in dem ein Sample aufgenommen wur-  }
  {                    de, aus einem VOC-Datenblock und liefert diesen zu-  }
  {                    rck (gemeint ist Mono bzw. Stereo).                 }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Modus                                                          }
  {          0: Mono (bei 16-Bits-Samples der Wert 1)                       }
  {          1: Stereo (bei 16-Bits-Samples der Wert 2)                     }
  {-------------------------------------------------------------------------}

  FUNCTION ReadMode : Byte;

  VAR
    Mode : Byte;                                      { speichert den Modus }

  BEGIN
    BlockRead(VOCFile, Mode, SizeOf(Byte));                   { Modus lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError(VOCName);
    ReadMode := Mode;
  END;

  {=========================================================================}
  { Funktion ReadBits: Liest die Anzahl der Bits pro Sample und liefert     }
  {                    diesen Wert zurck.                                  }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Anzahl der Bits                                                }
  {-------------------------------------------------------------------------}

  FUNCTION ReadBits : Byte;

  VAR
    Bits : Byte;                  { speichert die Zahl der Bits (8 oder 16) }

  BEGIN
    BlockRead(VOCFile, Bits, SizeOf(Byte));         { Anzahl der Bits lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError(VOCName);
    ReadBits := Bits;
  END;

  {=========================================================================}
  { Prozedur TransferData: bertrgt die Sample-Daten von einer VOC-Datei   }
  {                        in eine WAVE-Datei.                              }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE TransferData;

  CONST
    BufSize = 60*1024;                                  { Puffergre 60 KB }

  VAR
    DataBuf      : Pointer;                                   { Datenpuffer }
    TransferSize : LongInt;  { Gre der noch zu bertragenden Sample-Daten }
    B            : Byte;                                    { ein Datenbyte }

  BEGIN
    TransferSize := VOCBlockSize;
    GetMem(DataBuf, BufSize);              { Puffer auf dem Heap einrichten }

    { solange wiederholen, bis alle Sample-Daten bertragen sind }

    REPEAT
      IF TransferSize >= BufSize THEN     { Datenblockgre >= Puffergre? }
        BEGIN
          BlockRead(VOCFile, DataBuf^, BufSize); { Daten in Puffer einlesen }
          Result := IOResult;                        { Fehlerstatus sichern }
          IF Result <> 0 THEN                       { Liegt ein Fehler vor? }
            BEGIN                                                      { ja }
              FreeMem(DataBuf, BufSize);            { Puffer deinstallieren }
              ReadError(VOCName);       { Fehlerbehandlungsroutine aufrufen }
            END;
          BlockWrite(WAVEFile, DataBuf^, BufSize); { Pufferinhalt speichern }
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              WriteError(WAVEName);
            END;
          Dec(TransferSize, BufSize);
        END
      ELSE
        BEGIN
          BlockRead(VOCFile, DataBuf^, TransferSize);
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              ReadError(VOCName);
            END;
          BlockWrite(WAVEFile, DataBuf^, TransferSize);
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              WriteError(WAVEName);
            END;
          TransferSize := 0;
        END;
    UNTIL TransferSize = 0;

    FreeMem(DataBuf, BufSize);                  { Puffer vom Heap entfernen }
    IF VOCBlockSize MOD 2 <> 0 THEN      { Gre der Sample-Daten ungerade? }
      BEGIN                                                            { ja }
        B := 0;                 { eine Null am Ende der WAVE-Datei anhngen }
        BlockWrite(WAVEFile, B, SizeOf(Byte));
        Result := IOResult;
        IF Result <> 0 THEN
          WriteError(WAVEName);
      END;
  END;

{ Convert }

VAR
  BlockFound : Boolean;  { TRUE, wenn VOC-Datenblock 0, 1, 9 gefunden wurde }
  Rate       : Word;                                          { Sample-Rate }
  PackMethod : Byte;                                { Komprimierungsmethode }
  Mode       : Byte;                             { Modus (Mono oder Stereo) }
  Bits       : Byte;                               { Anzahl Bits pro Sample }
  dataStr    : String4;                                            { "data" }
  DataLen    : LongInt;                            { Lnge der Sample-Daten }

BEGIN
  WriteLn('Konvertierung ', VOCName, ' -> ', WAVEName);
  BlockRead(VOCFile, VOCHeader, SizeOf(TVOCRec));        { VOC-Header lesen }
  Result := IOResult;                              { Fehlerstatus speichern }
  IF Result <> 0 THEN                                 { Fehler aufgetreten? }
    ReadError(VOCName);
  VOC_IDStr := '';
  FOR Count := 0 TO $13 DO
    VOC_IDStr := VOC_IDstr+UpCase(VOCHeader.IDStr[Count]);
  IF (VOC_IDStr <> 'CREATIVE VOICE FILE'+#26)      { VOC-Datei-Kennung oder }
  OR ($1233-VOCHeader.Ver <> VOCHeader.Check) THEN     { Check-Code falsch? }
    BEGIN                                                              { ja }
      Close(VOCFile);                                 { VOC-Datei schlieen }
      Close(WAVEFile);             { bereits angelegte WAVE-Datei schlieen }
      Erase(WAVEFile);                           { und anschlieend lschen }
      Error('Fehler: Datei '+VOCName+' nicht im VOC-Format');
    END;
  Seek(VOCFile, VOCHeader.DataOffs);     { Sprung zum ersten VOC-Datenblock }
  Result := IOResult;
  IF Result <> 0 THEN
    SeekError(VOCName);
  Mode := 0;

  { solange wiederholen, bis einer der Datenblcke 0, 1, 9 }
  { oder Ende der VOC-Datei erreicht ist.                  }

  BlockFound := FALSE;
  REPEAT
    ReadVOCBlockData;
    IF VOCBlockType IN [0, 1, 9] THEN
      BlockFound := TRUE;
    OldVOCPos := FilePos(VOCFile);
    IF (VOCBlockType IN [2..8])
    AND (FilePos(VOCFile) < FileSize(VOCFile)-1) THEN
      BEGIN
        IF VOCBlockType = 8 THEN
          BEGIN
            Seek(VOCFile, FilePos(VOCFile)+3);     { drei Positionen weiter }
            Result := IOResult;
            IF Result <> 0 THEN
              SeekError(VOCName);
            Mode := ReadMode;              { Modus (Mono oder Stereo) lesen }
          END;
        Seek(VOCFile, OldVOCPos+VOCBlockSize);  { Sprung zum nchsten Block }
        Result := IOResult;
        IF Result <> 0 THEN
          SeekError(VOCName);
      END;
  UNTIL BlockFound OR (FilePos(VOCFile) >= FileSize(VOCFile)-1);

  IF (VOCBlockType IN [1, 9]) { VOC-Blocktyp fr die Konvertierung gefunden }
  OR (FilePos(VOCFile) < FileSize(VOCFile)-1) THEN    { und noch kein Ende? }
    BEGIN                                 { ja, Konvertierung kann beginnen }
      WriteLn;
      WriteLn('Konvertierung luft ...');
      CASE VOCBlockType OF       { Daten des gefundenen VOC-Blocks einlesen }
        1 :
          BEGIN
            Rate := ReadSampleRate1;                  { Sampling-Rate lesen }
            Bits := 8;                                      { 8-Bits-Sample }
            Seek(VOCFile, FilePos(VOCFile)+1);       { eine Position weiter }
            Result := IOResult;
            IF Result <> 0 THEN
              SeekError(VOCName);
            Dec(VOCBlockSize, 2);
          END;
        9 :
          BEGIN
            Rate := ReadSampleRate2;                  { Sampling-Rate lesen }
            Seek(VOCFile, FilePos(VOCFile)+2);     { zwei Positionen weiter }
            Result := IOResult;
            IF Result <> 0 THEN
              ReadError(VOCName);
            Bits := ReadBits;                       { Anzahl der Bits lesen }
            Mode := ReadMode;              { Modus (Mono oder Stereo) lesen }
            Dec(Mode);
            Dec(VOCBlockSize, 12);
            Seek(VOCFile, FilePos(VOCFile)+6);    { sechs Positionen weiter }
            Result := IOResult;
            IF Result <> 0 THEN
              SeekError(VOCName);
          END;
      END;

      { WAVE-Header mit Daten fllen }

      WITH WAVEHeader DO
        BEGIN
          RIFF := RIFFStr;
          FileLen := SizeOf(TWAVERec)+8+VOCBlockSize-8;    { Dateigre - 8 }
          IF VOCBlockSize MOD 2 <> 0 THEN       { Gre der Daten ungerade? }
            Inc(FileLen);         { ja, Gre der WAVE-Datei gerade stellen }
          WAVE := WAVEStr;
          fmt_ := fmt_Str;
          BlocksLen := $10;    { Lnge der beiden Headerblcke ist 16 Bytes }
          FormatType := 1;                           { WAVE-Format-ID ist 1 }
          Channels := Mode+1;          { Anzahl der benutzten Kanle setzen }
          SampleRate := Rate;                       { und die Sampling-Rate }

          { durchschnittliche Anzahl von Bytes setzen, }
          { die pro Sekunde ausgegeben werden          }

          BytesProSec := Rate;
          CASE VOCBlockType OF
            1 :
              IF Mode = 1 THEN
                BytesProSec := BytesProSec*2;                { 8 Bit Stereo }
            9 :
              IF Bits = 8 THEN
                IF Mode = 1 THEN
                  BytesProSec := BytesProSec*2               { 8 Bit Stereo }
                ELSE
              ELSE
                IF Mode = 0 THEN
                  BytesProSec := BytesProSec*2                { 16 Bit Mono }
                ELSE
                  BytesProSec := BytesProSec*4;             { 16 Bit Stereo }
          END;

          { Typ des Sample und die Tiefe setzen }

          CASE VOCBlockType OF
            1 :
              BEGIN
                IF Mode = 0 THEN
                  SampleType := 1                              { 8 Bit Mono }
                ELSE
                  SampleType := 2;                           { 8 Bit Stereo }
                SampleBits := Bits;        { Tiefe (Anzahl der Bits) setzen }
              END;
            9 :
              BEGIN
                IF Mode = 0 THEN
                  SampleType := 3                             { 16 Bit Mono }
                ELSE
                  SampleType := 4;                          { 16 Bit Stereo }
                SampleBits := Bits;                          { Tiefe setzen }
              END;
          END;
        END;

      BlockWrite(WAVEFile, WAVEHeader,              { WAVE-Header speichern }
        SizeOf(TWAVERec));
      Result := IOResult;                              { Fehlerstatus lesen }
      IF Result <> 0 THEN                             { Fehler aufgetreten? }
        WriteError(WAVEName);                           { ja, Schreibfehler }
      dataStr[0] := 'd';
      dataStr[1] := 'a';
      dataStr[2] := 't';
      dataStr[3] := 'a';
      BlockWrite(WAVEFile, dataStr, SizeOf(String4)); { "data"-ID speichern }
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(WAVEName);
      DataLen := VOCBlockSize;
      IF VOCBlockSize MOD 2 <> 0 THEN           { Gre der Daten ungerade? }
        Inc(DataLen);                            { ja, Gre gerade stellen }
      BlockWrite(WAVEFile, DataLen,                  { Datenlnge speichern }
        SizeOf(LongInt));
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(WAVEName);
      TransferData;                               { Sample-Daten bertragen }
    END
  ELSE
    BEGIN
      Close(VOCFile);                                 { VOC-Datei schlieen }
      Close(WAVEFile);                               { WAVE-Datei schlieen }
      Erase(WAVEFile);                                 { WAVE-Datei lschen }
      Error('Fehler: Datenblock fr die Konvertierung nicht gefunden.');
    END;

  { Dateien schlieen und Konvertierung beenden }

  Close(VOCFile);                                     { VOC-Datei schlieen }
  Close(WAVEFile);                                   { WAVE-Datei schlieen }
  WriteLn;
  WriteLn('Konvertierung beendet.');
END;

{---------------------------------------------------------------------------}
{ Hauptprogramm                                                             }
{---------------------------------------------------------------------------}

VAR
  Result : Integer;                                 { Fehlerstatus-Variable }
  Dir    : DirStr;                              { Verzeichnis der VOC-Datei }
  Name   : NameStr;                   { Name der VOC-Datei ohne Erweiterung }
  Ext    : ExtStr;                                       { Dateierweiterung }

BEGIN
  TextColor(LightGray);                                { Textfarbe hellgrau }
  WriteLn;
  WriteLn('VOC2WAVE  *  Version 1.03  *  (c) 1993 by Arthur Burda');
  WriteLn(''+
    '');
  IF ParamCount = 0 THEN               { Kommandozeilenparameter angegeben? }
    Error('Fehler: Keine Parameter angegeben.')         { nein, also Fehler }
  ELSE
    IF ParamCount = 1 THEN                     { nur 1 Parameter angegeben? }
      IF ParamStr(1) = '/?' THEN  { Ja. Ist der Parameter ein Fragezeichen? }
        ShowHelp                                       { ja, Hilfe anzeigen }
      ELSE                                                           { nein }
        Error('Fehler: Zu wenig Parameter')                 { Fehlermeldung }
    ELSE
      BEGIN
        VOCName := UpperString(ParamStr(1));      { VOC-Dateinamen auslesen }
        FSplit(VOCName, Dir, Name, Ext);          { VOC-Dateinamen zerlegen }
        IF Ext = '' THEN
          VOCName := VOCName+'.VOC';
        WAVEName := UpperString(ParamStr(2));    { WAVE-Dateinamen auslesen }
        FSplit(WAVEName, Dir, Name, Ext);        { WAVE-Dateinamen zerlegen }
        IF Ext = '' THEN
          WAVEName := WAVEName+'.WAV';
        Assign(VOCFile, VOCName);          { VOC-Datei mit Namen verknpfen }
        Reset(VOCFile, 1);                               { VOC-Datei ffnen }
        Result := IOResult;                         { Fehlerstatus abfragen }
        IF Result <> 0 THEN                           { Fehler aufgetreten? }
          Error('Fehler: Datei '+VOCName+      { ja, Fehlermeldung ausgeben }
            ' konnte nicht geffnet werden.');
        Assign(WAVEFile, WAVEName);       { WAVE-Datei mit Namen verknpfen }
        Rewrite(WAVEFile, 1);             { WAVE-Datei zum Schreiben ffnen }
        Result := IOResult;                         { Fehlerstatus abfragen }
        IF Result <> 0 THEN                           { Fehler aufgetreten? }
          BEGIN                                                        { ja }
            Close(VOCFile);         { bereits geffnete VOC-Datei schlieen }
            Error('Fehler: Datei '+WAVEName+       { Fehlermeldung ausgeben }
              ' konnte nicht geffnet werden.');
          END;
        Convert;                                    { Konvertierung starten }
      END;
END.
