
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit SBFMDrv   : Enthlt elementare Routinen zum Zugriff auf den Treiber  }
{                  SBFMDRV.COM. Mit Hilfe dieses Treibers knnen Musikstk- }
{                  ke (z.B. im CMF-Format) leicht abgespielt werden.        }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : SBFMDRV.PAS                                              }
{ entwickelt am  : 09.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.04                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT SBFMDrv;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

INTERFACE

USES DOS;                                              { Unit DOS einbinden }

CONST

  {-------------------------------------------------------------------------}
  { Treiberfunktionen                                                       }
  {-------------------------------------------------------------------------}

  {=========================================================================}
  { sbfmd_DriverVer: Nummer der Treiberversion (SB, SB Pro und hher)       }
  {=========================================================================}
  { Eingabe: BX = sbfmd_DriverVer                                           }
  { Ausgabe: AH = Hauptversion                                              }
  {          AL = Unterversion                                              }
  {-------------------------------------------------------------------------}

  sbfmd_DriverVer = $00;

  {=========================================================================}
  { sbfmd_SetStatusByteAddr: Adresse des Musik-Statusbyte einstellen (SB,   }
  {                          SB Pro und hher)                              }
  {=========================================================================}
  { Eingabe: BX    = sbfmd_SetStatusByteAddr                                }
  {          DX:AX = Adress des Statusbyte (Segment, Offset)                }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbfmd_SetStatusByteAddr = $01;

  {=========================================================================}
  { sbfmd_SetInstrTable: Instrumenten-Tabelle setzen (SB, SB Pro und hher) }
  {=========================================================================}
  { Eingabe: BX    = sbfmd_SetInstrTable                                    }
  {          CX    = Anzahl der Instrumente                                 }
  {          DX:AX = Adresse der Instrumenten-Tabelle (Segment, Offset)     }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbfmd_SetInstrTable = $02;

  {=========================================================================}
  { sbfmd_SetClockRate1: Einstellung der Frequenz fr den Timer-Interrupt,  }
  {                      wenn der Treiber keine Musik abspielt (SB, SB Pro  }
  {                      und hher)                                         }
  {=========================================================================}
  { Eingabe: AX = Faktor (1193180/Frequenz)                                 }
  {          BX = sbfmd_SetClockRate1                                       }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbfmd_SetClockRate1 = $03;

  {=========================================================================}
  { sbfmd_SetClockRate2: Einstellung der Frequenz fr den Timer-Interrupt,  }
  {                      wenn der Treiber Musik abspielt (zu unterscheiden  }
  {                      von sbfmd_SetClockRate1; SB, SB Pro und hher)     }
  {=========================================================================}
  { Eingabe: AX = Faktor (1193180/Frequenz)                                 }
  {          BX = sbfmd_SetClockRate2                                       }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbfmd_SetClockRate2 = $04;

  {=========================================================================}
  { sbfmd_TransposeMusic: Musik transponieren, d.h. um eine bestimmte An-   }
  {                       zahl halber Noten nach unten oder oben verschie-  }
  {                       ben (SB, SB Pro und hher)                        }
  {=========================================================================}
  { Eingabe: AX = Anzahl der halben Noten                                   }
  {          BX = sbfmd_TransposeMusic                                      }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbfmd_TransposeMusic = $05;

  {=========================================================================}
  { sbfmd_PlayMusic: Abspielen der Musikdaten (SB, SB Pro und hher)        }
  {=========================================================================}
  { Eingabe: BX    = sbfmd_PlayMusic                                        }
  {          DX:AX = Adresse des Musik-Datenblocks (Segment, Offset)        }
  { Ausgabe: AX    = Ergebnis (00h bedeutet fehlerfrei, 01h andere Musik    }
  {                  ist zur Zeit aktiv)                                    }
  {-------------------------------------------------------------------------}
  { ANMERKUNG: Bei fehlerfreier Ausfhrung der Funktion enthlt das Status- }
  {            byte den Wert FFh.                                           }
  {-------------------------------------------------------------------------}

  sbfmd_PlayMusic = $06;

  {=========================================================================}
  { sbfmd_StopMusic: Musikausgabe beenden (SB, SB Pro und hher)            }
  {=========================================================================}
  { Eingabe: BX = sbfmd_StopMusic                                           }
  { Ausgabe: AX = Ergebnis (00h bedeutet fehlerfrei, 01h zur Zeit wird kei- }
  {               ne Musik abgespielt)                                      }
  {-------------------------------------------------------------------------}

  sbfmd_StopMusic = $07;

  {=========================================================================}
  { sbfmd_DriverReset: Reset des FM-Treibers (SB, SB Pro und hher)         }
  {=========================================================================}
  { Eingabe: BX = sbfmd_DriverReset                                         }
  { Ausgabe: AX = Ergebnis (00h bedeutet fehlerfrei, 01h Musik ist derzeit  }
  {               aktiv)                                                    }
  {-------------------------------------------------------------------------}

  sbfmd_DriverReset = $08;

  {=========================================================================}
  { sbfmd_PauseMusic: Musikausgabe anhalten (SB, SB Pro und hher)          }
  {=========================================================================}
  { Eingabe: BX = sbfmd_PauseMusic                                          }
  { Ausgabe: AX = Ergebnis (00h bedeutet fehlerfrei, 01h zur Zeit wird kei- }
  {               ne Musik abgespielt)                                      }
  {-------------------------------------------------------------------------}

  sbfmd_PauseMusic = $09;

  {=========================================================================}
  { sbfmd_ContinueMusic: Musikausgabe fortsetzen (SB, SB Pro und hher)     }
  {=========================================================================}
  { Eingabe: BX = sbfmd_ContinueMusic                                       }
  { Ausgabe: AX = Ergebnis (00h fehlerfrei, 01h es wurde keine Musik ange-  }
  {               halten)                                                   }
  {-------------------------------------------------------------------------}

  sbfmd_ContinueMusic = $0A;

  {=========================================================================}
  { sbfmd_SetUserRoutine: Einstellen einer benutzerdefinierten Routine (SB, }
  {                       SB Pro und hher)                                 }
  {=========================================================================}
  { Eingabe: BX    = sbfmd_SetUserRoutine                                   }
  {          DX:AX = Adresse der sog. Trap-Routine (wenn DX und AX jeweils  }
  {                  den Wert 0000h haben, wird die Routine ausgeschaltet)  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}
  { ANMERKUNGEN: Die eingestellte Routine wird dann eingeschaltet, wenn der }
  {              FM-Treiber einem System-Exclusive-Event begegnet. Danach   }
  {              zeigt das Registerpaar ES:BX auf das Byte hinter diesem    }
  {              Event. Bei der Programmierung der Routine mu beachtet     }
  {              werden, da alle benutzten Register gespeichert werden und }
  {              die Routine selbst mit einem RETF-Befehl beendet wird.     }
  {-------------------------------------------------------------------------}

  sbfmd_SetUserRoutine = $0B;

  {=========================================================================}
  { sbfmd_GetMIDIAddr: Bestimmung der Adresse des MIDI-Kanals (erst ab Ver- }
  {                    sion 1.10 verfgbar; SB, SB Pro und hher)           }
  {=========================================================================}
  { Eingabe: BX    = sbfmd_GetMIDIAddr                                      }
  { Ausgabe: DX:AX = Adresse, wo die Einstellungen des MIDI-Kanals zu fin-  }
  {                  den sind (Segment, Offset)                             }
  {-------------------------------------------------------------------------}

  sbfmd_GetMIDIAddr = $0C;

  {=========================================================================}
  { sbfmd_GetMusicBlockAddr: Bestimmung der aktuellen Adresse innerhalb des }
  {                          abzuspielenden Musikblocks (erst ab Version    }
  {                          1.10 verfgbar; SB, SB Pro und hher)          }
  {=========================================================================}
  { Eingabe: BX    = sbfmd_GetMusicBlockAddr                                }
  { Ausgabe: DX:AX = aktuelle Adresse des Musikblocks (Segment, Offset)     }
  {-------------------------------------------------------------------------}

  sbfmd_GetMusicBlockAddr = $0D;

  {=========================================================================}
  { sbfmd_SetFadeParam: Einstellung der Parameter fr einen Fade-Effekt am  }
  {                     Ende des Musikstckes (SB, SB Pro und hher)        }
  {=========================================================================}
  { Eingabe: AX = Start-Lautstrke (Angabe in Prozent)                      }
  {          BX = sbfmd_SetFadeParam                                        }
  {          CX = Vernderungsgre                                         }
  {          DX = End-Lautstrke (Angabe in Prozent)                        }
  {          DI = Schrittgre                                              }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbfmd_SetFadeParam = $0E;

  {=========================================================================}
  { sbfmd_SetRepeats: Anzahl der Wiederholungen einstellen (SB, SB Pro und  }
  {                   hher)                                                }
  {=========================================================================}
  { Eingabe: AX = Anzahl der Wiederholungen des Musikstckes                }
  {          BX = sbfmd_SetRepeats                                          }
  { Ausgabe: AX = letzte Einstellung                                        }
  {-------------------------------------------------------------------------}

  sbfmd_SetRepeats = $0F;

VAR
  FMDrvIntr : Byte;    { Nummer des von dem FM-Treiber benutzten Interrupts }

{===========================================================================}
{ Prozedur InitSBFMDrv: Initialisiert den FM-Treiber. Wenn der Treiber ge-  }
{                       funden wird, enthlt die Variable FMDrvIntr die     }
{                       Nummer des verwendeten Interrupts. Daran kann ein   }
{                       Programm feststellen, ob der Treiber geladen ist.   }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE InitSBFMDrv;

{===========================================================================}
{ Prozedur CallFMDrv: Ruft den FM-Treiber mit den bergebenen Parametern    }
{                     auf.                                                  }
{===========================================================================}
{ Eingabe: AX   = Inhalt des AX-Registers                                   }
{          Func = Nummer der Treiberfunktion (BX-Register)                  }
{          CX   = Inhalt des CX-Registers                                   }
{          DX   = Inhalt des DX-Registers                                   }
{          DI   = Inhalt des DI-Registers                                   }
{ Ausgabe: AX, CX, DX, DI                                                   }
{---------------------------------------------------------------------------}

PROCEDURE CallFMDrv(VAR AX : Word; Func : Word; VAR CX, DX, DI : Word);

{===========================================================================}
{ Prozedur SBFMDrvGetVer: Liefert die Nummer der Haupt- und Unterversion    }
{                         des FM-Treibers zurck.                           }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Major = Hauptversion                                             }
{          Minor = Unterversion                                             }
{---------------------------------------------------------------------------}

PROCEDURE SBFMDrvGetVer(VAR Major, Minor : Byte);

{===========================================================================}
{ Prozedur SBFMDrvSetStatusByte: Setzt die Adresse des Musik-Statusbyte.    }
{===========================================================================}
{ Eingabe: Addr = Adresse des Statusbyte                                    }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBFMDrvSetStatusByte(Addr : Pointer);

{===========================================================================}
{ Prozedur SBFMDrvSetInstrTable: Stellt die Instrumenten-Parameter und ein. }
{===========================================================================}
{ Eingabe: Nmb = Anzahl der Instrumente                                     }
{          P   = Zeiger auf die Tabelle mit den Instrumenten-Parametern     }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBFMDrvSetInstrTable(Nmb : Word; P : Pointer);

{===========================================================================}
{ Prozedur SBFMDrvSetClockRate1: Stellt die Frequenz fr den Timer-Inter-   }
{                                rupt fr den Fall, da der Treiber keine   }
{                                Musik abspielt.                            }
{===========================================================================}
{ Eingabe: Freq = Timer-Frequenz in Hz (19..50000)                          }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBFMDrvSetClockRate1(Freq : Word);

{===========================================================================}
{ Prozedur SBFMDrvSetClockRate2: Stellt die Frequenz fr den Timer-Inter-   }
{                                rupt fr den Fall, da der Treiber Musik   }
{                                abspielt.                                  }
{===========================================================================}
{ Eingabe: Freq = Timer-Frequenz in Hz (19..50000)                          }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBFMDrvSetClockRate2(Freq : Word);

{===========================================================================}
{ Funktion SBFMDrvPlayMusic: Spielt einen Musikblock ab. Die Funktion lie-  }
{                            fert TRUE zurck, wenn der Aufruf der Treiber- }
{                            funktion ohne Fehler beendet wurde.            }
{===========================================================================}
{ Eingabe: P = Zeiger auf die Musikdaten                                    }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SBFMDrvPlayMusic(P : Pointer) : Boolean;

{===========================================================================}
{ Funktion SBFMDrvStopMusic: Beendet die Musikausgabe. Die Funktion liefert }
{                            TRUE zurck, wenn der Aufruf der entsprechen-  }
{                            den Treiberfunktion ohne Fehler beendet wurde. }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SBFMDrvStopMusic : Boolean;

{===========================================================================}
{ Funktion SBFMDrvPauseMusic: Unterbricht die Musikausgabe, beendet sie     }
{                             aber nicht endgltig. Die Funktion liefert    }
{                             TRUE zurck, wenn der Aufruf der entsprechen- }
{                             den Treiberfunktion ohne Fehler beendet wur-  }
{                             de, sonst wird FALSE als Ergebnis geliefert.  }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SBFMDrvPauseMusic : Boolean;

{===========================================================================}
{ Funktion SBFMDrvContinueMusic: Nimmt die zuvor unterbrochene Musikausga-  }
{                                be wieder auf. Bezglich des Ausgabepara-  }
{                                meters gilt hier das gleiche wie bei der   }
{                                Funktion SBFMDrvPauseMusic.                }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SBFMDrvContinueMusic : Boolean;

{===========================================================================}
{ Funktion FMDriverReset: Fhrt einen Reset des FM-Treibers aus. Es wird    }
{                         TRUE zurckgeliefert, wenn der Aufruf der ent-    }
{                         sprechenden Treiberfunktion erfolgreich war.      }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION FMDriverReset : Boolean;

IMPLEMENTATION

PROCEDURE InitSBFMDrv;

TYPE
  PDrvBeginRec = ^TDrvBeginRec;    { Zeiger auf die Treiber-Anfangsstruktur }
  TDrvBeginRec = RECORD                           { Treiber-Anfangsstruktur }
    Extend : ARRAY[0..2] OF Char;                               { Ergnzung }
    IDStr  : ARRAY[0..4] OF Char;                  { Identifikations-String }
  END;

VAR
  DrvIntr     : Byte;                                       { Hilfsvariable }
  DrvBeginRec : PDrvBeginRec;                      { Zeiger auf den Treiber }
  Found       : Boolean;            { TRUE, wenn der Treiber gefunden wurde }
  P           : Pointer;                                       { ein Zeiger }

BEGIN
  FMDrvIntr := $00;
  DrvIntr := $80;
  Found := FALSE;

  { solange suchen, bis der Treiber gefunden wird oder kein freier }
  { Interrupt-Vektor mehr brig bleibt                             }

  REPEAT
    GetIntVec(DrvIntr, P);         { Inhalt des Interrupt-Vektors ermitteln }
    DrvBeginRec :=  { Zeiger auf den Treiber-Start bestimmen (COM-Programm) }
      Ptr(Seg(P^), $100);
    IF DrvBeginRec^.IDStr = 'FMDRV' THEN        { Treiver-Kennung gefunden? }
      Found := TRUE                                                    { ja }
    ELSE
      Inc(DrvIntr);                              { Interrupt-Nummer erhhen }
  UNTIL (DrvIntr = $C0) OR Found;

  IF Found THEN
    FMDrvIntr := DrvIntr;
END;

PROCEDURE CallFMDrv(VAR AX : Word; Func : Word; VAR CX, DX, DI : Word);

VAR
  Regs : Registers;                                    { Prozessor-Register }

BEGIN
  IF FMDrvIntr <> 0 THEN                                 { Treiber geladen? }
    BEGIN                                                              { ja }
      Regs.AX := AX;
      Regs.BX := Func;
      Regs.CX := CX;
      Regs.DX := DX;
      Regs.DI := DI;
      Intr(FMDrvIntr, Regs);
      AX := Regs.AX;
      CX := Regs.CX;
      DX := Regs.DX;
      DI := Regs.DI;
    END;
END;

PROCEDURE SBFMDrvGetVer(VAR Major, Minor : Byte);

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  Major := 0;
  Minor := 0;
  AX := $0000;
  Func := sbfmd_DriverVer;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
  Major := Hi(AX);
  Minor := Lo(AX);
END;

PROCEDURE SBFMDrvSetStatusByte(Addr : Pointer);

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  AX := Ofs(Addr^);
  Func := sbfmd_SetStatusByteAddr;
  CX := $0000;
  DX := Seg(Addr^);
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
END;

PROCEDURE SBFMDrvSetInstrTable(Nmb : Word; P : Pointer);

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  AX := Ofs(P^);
  Func := sbfmd_SetInstrTable;
  CX := Nmb;
  DX := Seg(P^);
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
END;

PROCEDURE SBFMDrvSetClockRate1(Freq : Word);

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN

  { Prfen, ob die angegebene Frequenz im Wertebereich zwischen }
  { 19 und 50000 Hz liegt. Falls nicht, korrigieren.            }

  IF Freq < 19 THEN
    Freq := 19;
  IF Freq > 50000 THEN
    Freq := 50000;

  AX := 1193180 DIV Freq;
  Func := sbfmd_SetClockRate1;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
END;

PROCEDURE SBFMDrvSetClockRate2(Freq : Word);

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN

  { Prfen, ob die angegebene Frequenz im Wertebereich zwischen }
  { 19 und 50000 Hz liegt. Falls nicht, korrigieren.            }

  IF Freq < 19 THEN
    Freq := 19;
  IF Freq > 50000 THEN
    Freq := 50000;

  AX := 1193180 DIV Freq;
  Func := sbfmd_SetClockRate2;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
END;

FUNCTION SBFMDrvPlayMusic(P : Pointer) : Boolean;

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  SBFMDrvPlayMusic := FALSE;
  AX := Ofs(P^);
  Func := sbfmd_PlayMusic;
  CX := $0000;
  DX := Seg(P^);
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
  IF AX = $0000 THEN                                          { alles o.k.? }
    SBFMDrvPlayMusic := TRUE;             { ja, den Wert TRUE zurckliefern }
END;

FUNCTION SBFMDrvStopMusic : Boolean;

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  SBFMDrvStopMusic := FALSE;
  AX := $0000;
  Func := sbfmd_StopMusic;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
  IF AX = $0000 THEN                          { War der Aufruf ohne Fehler? }
    SBFMDrvStopMusic := TRUE;             { ja, den Wert TRUE zurckliefern }
END;

FUNCTION SBFMDrvPauseMusic : Boolean;

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  SBFMDrvPauseMusic := FALSE;
  AX := $0000;
  Func := sbfmd_PauseMusic;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
  IF AX = $0000 THEN                           { War der Aufruf fehlerfrei? }
    SBFMDrvPauseMusic := TRUE;                     { ja, TRUE zurckliefern }
END;

FUNCTION SBFMDrvContinueMusic : Boolean;

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  SBFMDrvContinueMusic := FALSE;
  AX := $0000;
  Func := sbfmd_ContinueMusic;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
  IF AX = $0000 THEN                                   { Aufruf fehlerfrei? }
    SBFMDrvContinueMusic := TRUE;                  { ja, TRUE zurckliefern }
END;

FUNCTION FMDriverReset : Boolean;

VAR
  AX, Func, CX, DX, DI : Word;                         { Prozessor-Register }

BEGIN
  FMDriverReset := FALSE;
  AX := $0000;
  Func := sbfmd_DriverReset;
  CX := $0000;
  DX := $0000;
  DI := $0000;
  CallFMDrv(AX, Func, CX, DX, DI);                       { Treiber aufrufen }
  IF AX = $0000 THEN                                          { alles o.k.? }
    FMDriverReset := TRUE;                         { ja, TRUE zurckliefern }
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

BEGIN
  InitSBFMDrv;
END.
