
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit SBDrv     : Stellt Routinen zur Ansteuerung der Sound-Blaster-Karte  }
{                  ber die Treiber CT-VOICE.DRV und CTVDSK.DRV zur Verf-  }
{                  gung. Die wichtigste Routine in dem Zusammenhang ist das }
{                  Senden einer Steuersequenz an den Treiber, um eine Trei- }
{                  berfunktion auszufhren. Darber hinaus stellt die Unit  }
{                  einige Routinen zur Verfgung (wie zum Beispiel An- und  }
{                  Ausschalten des Lautsprechers) quasi als Beispiele, die  }
{                  alle auf der Funktion zum Senden einer Steuersequenz     }
{                  an den Treiber basieren. Intern werden auch die Umge-    }
{                  bungsvariablen aus der AUTOEXEC.BAT-Datei ausgelesen und }
{                  nach Treiberverzeichnis, I/O-Adresse, Interrupt- und     }
{                  DMA-Kanal-Nummer ausgewertet.                            }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : SBDRV.PAS                                                }
{ entwickelt am  : 29.04.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.13                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT SBDrv;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

{$M 16384,65536,655360}         { 16 KB Stack, min. 64 KB Heap, max. 640 KB }

INTERFACE

USES DOS;                                              { Unit DOS einbinden }

TYPE

  {-------------------------------------------------------------------------}
  { String-Typen                                                            }
  {-------------------------------------------------------------------------}

  String4  = String[4];                          { ein String aus 4 Zeichen }
  String12 = String[12];                        { ein String aus 12 Zeichen }
  String79 = String[79];                        { ein String aus 79 Zeichen }

VAR

  DrvPtr  : Pointer;                         { Zeiger auf den Treiberpuffer }
  DrvSize : Word;                                            { Treibergre }
  CTVOICE : Boolean;          { TRUE, wenn der Treiber CT-VOICE geladen ist }
  CTVDSK  : Boolean;            { TRUE, wenn der Treiber CTVDSK geladen ist }

  {-------------------------------------------------------------------------}
  { Basis-I/O-Adresse, Interrupt-Nummer und Nummer des DMA-Kanals aus der   }
  { BLASTER-Umgebungsvariablen in der AUTOEXEC.BAT-Datei                    }
  {-------------------------------------------------------------------------}

  SB_BaseAddress : Word;
  SB_Interrupt   : Word;
  SB_DMAChannel  : Word;

  {-------------------------------------------------------------------------}
  { Treiber-Verzeichnisname, der aus der SOUND-Umgebungsvariablen ermittelt }
  { wird                                                                    }
  {-------------------------------------------------------------------------}

  SB_DriverPath : String79;

CONST

  {-------------------------------------------------------------------------}
  { Sound Blaster Kartentypen                                               }
  {-------------------------------------------------------------------------}

  type_SB_SBMCV = 1;             { Sound Blaster 1.5 oder Sound Blaster MCV }
  type_SBPro    = 2; { Sound Blaster Pro, Audio Blaster 2.5 oder AB Pro 4.0 }
  type_SB20     = 3;                                    { Sound Blaster 2.0 }
  type_SB16     = 6;   { Sound Blaster 16 ASP oder eine vergleichbare Karte }

  {-------------------------------------------------------------------------}
  { Treiberfunktionen                                                       }
  {-------------------------------------------------------------------------}

  {=========================================================================}
  { sbd_DriverVer: Nummer der Treiberversion (SB, SB Pro und hher)         }
  {=========================================================================}
  { Eingabe: BX = sbd_DriverVer                                             }
  { Ausgabe: AH = Hauptversion                                              }
  {          AL = Unterversion                                              }
  {-------------------------------------------------------------------------}

  sbd_DriverVer = $00;

  {=========================================================================}
  { sbd_SetBaseAddress: Basis-I/O-Adresse festlegen (SB, SB Pro und hher)  }
  {=========================================================================}
  { Eingabe: AX = Adresse                                                   }
  {          BX = sbd_SetBaseAddress                                        }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_SetBaseAddress = $01;

  {=========================================================================}
  { sbd_SetInterrupt: IRQ-Interrupt festlegen (SB, SB Pro und hher)        }
  {=========================================================================}
  { Eingabe: AX = Interrupt-Nummer                                          }
  {          BX = sbd_SetInterrupt                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_SetInterrupt = $02;

  {=========================================================================}
  { sbd_DriverInit: Initialisierung des Treibers (SB, SB Pro und hher)     }
  {=========================================================================}
  { Eingabe: AX = Puffergre in 4-KByte-Einheiten (nur bei CTVDSK)         }
  {          BX = sbd_DriverInit                                            }
  { Ausgabe: AX = Ergebnis (00h bedeutet fehlerfrei)                        }
  {-------------------------------------------------------------------------}
  { ANMERKUNG: Bei CTVDSK mu vor der Initialisierung unbedingt die Funk-   }
  {            tion sbd_SetBuffer aufgerufen werden.                        }
  {-------------------------------------------------------------------------}

  sbd_DriverInit = $03;

  {=========================================================================}
  { sbd_SetSpeakerOnOff: Lautsprecher ein bzw. aus (SB, SB Pro und hher)   }
  {=========================================================================}
  { Eingabe: AL = 00h (aus), AL = 01h (ein)                                 }
  {          BX = sbd_SetSpeakerOnOff                                       }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_SetSpeakerOnOff = $04;

  {=========================================================================}
  { sbd_SetStatusword: Definition der Statuswort-Adresse (SB, SB Pro und    }
  {                    hher)                                               }
  {=========================================================================}
  { Eingabe: BX = sbd_SetStatusword                                         }
  {          CTVDSK   : DX:AX = Adresse des Statuswortes                    }
  {          CT-VOICE : ES:DI = Adresse des Statuswortes                    }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_SetStatusword = $05;

  {=========================================================================}
  { sbd_VoiceOutput: Ausgabe von Geruschen (SB, SB Pro und hher)          }
  {=========================================================================}
  { Eingabe: BX = sbd_VoiceOutput                                           }
  {          CTVDSK   : AX    = Dateihandle                                 }
  {          CT-VOICE : ES:DI = Adresse des Puffers ohne Header             }
  { Ausgabe: AX = Ergebnis (00h bedeutet fehlerfrei)                        }
  {-------------------------------------------------------------------------}

  sbd_VoiceOutput = $06;

  {=========================================================================}
  { sbd_VoiceRecord: Aufnahme von Geruschen (SB, SB Pro und hher)         }
  {=========================================================================}
  { Eingabe: BX = sbd_VoiceRecord                                           }
  {          CTVDSK   : AX    = Dateihandle                                 }
  {                     DX    = Sampling-Rate                               }
  {          CT-VOICE : AX    = Sampling-Rate                               }
  {                     DX:CX = Lnge des Puffers                           }
  {                     ES:DI = Adresse des Puffers                         }
  { Ausgabe: AX = Ergebnis (00h bedeutet fehlerfrei)                        }
  {-------------------------------------------------------------------------}

  sbd_VoiceRecord = $07;

  {=========================================================================}
  { sbd_StopInOutput: Ein-/Ausgabe stoppen (SB, SB Pro und hher)           }
  {=========================================================================}
  { Eingabe: BX = sbd_StopInOutput                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_StopInOutput = $08;

  {=========================================================================}
  { sbd_DriverEnd: Treiber ausschalten (SB, SB Pro und hher)               }
  {=========================================================================}
  { Eingabe: BX = sbd_DriverEnd                                             }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_DriverEnd = $09;

  {=========================================================================}
  { sbd_OutputPause: Ausgabe anhalten (SB, SB Pro und hher)                }
  {=========================================================================}
  { Eingabe: BX = sbd_OutputPause                                           }
  { Ausgabe: AX = Ergebnis (nur CT-VOICE: 00h bedeutet Unterbrechung,       }
  {               01h kein Sample wird abgespielt)                          }
  {-------------------------------------------------------------------------}

  sbd_OutputPause = $0A;

  {=========================================================================}
  { sbd_ContinueOutput: Ausgabe fortsetzen (SB, SB Pro und hher)           }
  {=========================================================================}
  { Eingabe: BX = sbd_ContinueOutput                                        }
  { Ausgabe: AX = Ergebnis (nur CT-VOICE: 00h bedeutet Fortsetzung,         }
  {               01h Ausgabe wurde nicht unterbrochen)                     }
  {-------------------------------------------------------------------------}

  sbd_ContinueOutput = $0B;

  {=========================================================================}
  { sbd_QuitLoop: Verlassen (Herausspringen aus) einer Schleife (SB, SB Pro }
  {               und hher)                                                }
  {=========================================================================}
  { Eingabe: AX = 00h (Schleife erst am Ende verlassen)                     }
  {          AX = 01h (Schleife sofort verlassen)                           }
  {          BX = sbd_QuitLoop                                              }
  { Ausgabe: AX = Ergebnis (nur CT-VOICE: 00h bedeutet Verlassen)           }
  {-------------------------------------------------------------------------}

  sbd_QuitLoop = $0C;

  {=========================================================================}
  { sbd_DefUserRoutine: Definition einer Benutzerroutine (nur CT-VOICE; SB, }
  {                     SB Pro und hher)                                   }
  {=========================================================================}
  { Eingabe: BX    = sbd_DefUserRoutine                                     }
  {          DX:AX = Adresse der Benutzerroutine                            }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_DefUserRoutine = $0D;

  {=========================================================================}
  { sbd_VoiceOutputEMM: Abspielen eines Sample-Blocks, der in einem Exten-  }
  {                     ded Memory Block gespeichert ist (nur CT-VOICE; nur }
  {                     SB Pro und hher)                                   }
  {=========================================================================}
  { Eingabe: BX    = sbd_VoiceOutputEMM                                     }
  {          DX    = EMM-Handle                                             }
  {          DI:SI = Start in Extended Memory (Offset)                      }
  { Ausgabe: AX    = Ergebnis (00h bedeutet fehlerfrei)                     }
  {-------------------------------------------------------------------------}

  sbd_VoiceOutputEMM = $0E;

  {=========================================================================}
  { sbd_GetProcessError: Fehlerstatus der VOICE-Eingabe bzw. -Ausgabe (nur  }
  {                      CTVDSK; SB, SB Pro und hher)                      }
  {=========================================================================}
  { Eingabe: BX = sbd_GetProcessError                                       }
  { Ausgabe: DX = System-Fehlercode                                         }
  {          AX = Laufwerk-Fehlercode                                       }
  {-------------------------------------------------------------------------}

  sbd_GetProcessError = $0E;

  {=========================================================================}
  { sbd_VoiceRecordEMB: Aufnahme eines Sample mit Hilfe eines EMB (nur CT-  }
  {                     VOICE; nur SB Pro und hher)                        }
  {=========================================================================}
  { Eingabe: AX    = Sampling-Rate                                          }
  {          BX    = sbd_VoiceRecordEMB                                     }
  {          CX    = Pufferlnge in KByte                                   }
  {          DX    = EMM-Handle                                             }
  {          DI:SI = Start in Extended Memory (Offset)                      }
  { Ausgabe: AX    = Ergebnis (00h bedeutet fehlerfrei)                     }
  {-------------------------------------------------------------------------}

  sbd_VoiceRecordEMB = $0F;

  {=========================================================================}
  { sbd_SetBuffer: Puffer festlegen (nur CTVDSK; SB, SB Pro und hher)      }
  {=========================================================================}
  { Eingabe: BX    = sbd_SetBuffer                                          }
  {          DX:AX = Adresse des Puffers                                    }
  {          CX    = Puffergre in 4-KByte-Einheiten                       }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_SetBuffer = $0F;

  {=========================================================================}
  { sbd_SetRecMode: Aufnahmemodus setzen (nur SB Pro bzw. hher)            }
  {=========================================================================}
  { Eingabe: AX = 00h (Mono), 01h (Stereo)                                  }
  {          BX = sbd_SetRecMode                                            }
  { Ausgabe: AX = vorheriger Modus                                          }
  {-------------------------------------------------------------------------}

  sbd_SetRecMode = $10;

  {=========================================================================}
  { sbd_SelectRecSource: Aufnahmequelle whlen (nur SB Pro bzw. hher)      }
  {=========================================================================}
  { Eingabe: AX = 00h (Mikrofon)                                            }
  {          AX = 01h (CD-ROM)                                              }
  {          AX = 02h (Mikrofon)                                            }
  {          AX = 03h (LINE-Eingang)                                        }
  {          BX = sbd_SelectRecSource                                       }
  { Ausgabe: AX = alte Quelle                                               }
  {-------------------------------------------------------------------------}

  sbd_SelectRecSource = $11;

  {=========================================================================}
  { sbd_SelectRecFilter: Aufnahmefilter whlen (nur SB Pro bzw. hher)      }
  {=========================================================================}
  { Eingabe: AX = 00h (tiefer Filter)                                       }
  {          AX = 01h (hoher Filter)                                        }
  {          BX = sbd_SelectRecFilter                                       }
  { Ausgabe: AX = alter Filter                                              }
  {-------------------------------------------------------------------------}

  sbd_SelectRecFilter = $12;

  {=========================================================================}
  { sbd_SetDMAChannel: DMA-Kanal setzen (nur SB Pro bzw. hher)             }
  {=========================================================================}
  { Eingabe: AX = DMA-Kanal                                                 }
  {          BX = sbd_SetDMAChannel                                         }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_SetDMAChannel = $13;

  {=========================================================================}
  { sbd_BoardType: Kartentyp feststellen (SB, SB Pro und hher)             }
  {=========================================================================}
  { Eingabe: BX = sbd_BoardType                                             }
  { Ausgabe: AX = 01h (SB 1.5 oder SB MCV)                                  }
  {          AX = 02h (SB Pro 3, Audio Blaster 2.5 oder AB Pro 4)           }
  {          AX = 03h (SB 2.0)                                              }
  {          AX = 06h (SB 16 ASP oder vergleichbar)                         }
  {-------------------------------------------------------------------------}
  { ANMERKUNG: Bei der heute sehr schnell fortschreitenden Technik kann es  }
  {            sein, da andere Werte andere neue bzw. weniger bekannte     }
  {            Kartentypen kennzeichnen.                                    }
  {-------------------------------------------------------------------------}

  sbd_BoardType = $14;

  {=========================================================================}
  { sbd_SetVolume: Lautstrke einstellen (nur CT-VOICE; nur SB Pro und      }
  {                hher)                                                   }
  {=========================================================================}
  { Eingabe: AX = 00h (Ausgabesignal; Stereo)                               }
  {          AX = 01h (VOICE-Teil; Stereo)                                  }
  {          AX = 02h (Mikrofon-Eingang)                                    }
  {          AX = 03h (CD-ROM; Stereo)                                      }
  {          AX = 04h (Stereo-LINE-Eingang; Stereo)                         }
  {          BX = sbd_SetVolume                                             }
  {          CX = Lautstrke-Pegel (1..15)                                  }
  {          DX = 01h (linker Kanal), DX = 02h (rechter Kanal)              }
  { Ausgabe: AX = FFFFh (Fehler aufgetreten), AX <> FFFFh (vorheriger Wert) }
  {-------------------------------------------------------------------------}

  sbd_SetVolume = $15;

  {=========================================================================}
  { sbd_SetFilterOnOff: Filtereinstellungen (nur SB Pro bzw. hher)         }
  {=========================================================================}
  { Eingabe: AX = 00h (Eingabefilter), AX = 01h (Ausgabefilter)             }
  {          BX = sbd_SetFilterOnOff                                        }
  {          CX = 00h (ein), CX = 01h (aus)                                 }
  { Ausgabe: AX = FFFFh (Fehler aufgetreten), AX <> FFFFh (vorheriger Wert) }
  {-------------------------------------------------------------------------}

  sbd_SetFilterOnOff = $16;

  {=========================================================================}
  { sbd_MixerReset: Reset des Mixer-Chips (nur CT-VOICE; nur SB Pro und     }
  {                 hher)                                                  }
  {=========================================================================}
  { Eingabe: BX = sbd_MixerReset                                            }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_MixerReset = $17;

  {=========================================================================}
  { sbd_VolumeSettings: Bestimmung der Lautstrke-Einstellungen (nur CT-    }
  {                     VOICE; nur SB Pro und hher)                        }
  {=========================================================================}
  { Eingabe: BX = sbd_VolumeSettings                                        }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  sbd_VolumeSettings = $18;

  {=========================================================================}
  { sbd_GetVolume: Abfrage der Lautstrke (nur CT-VOICE; nur SB Pro und     }
  {                hher)                                                   }
  {=========================================================================}
  { Eingabe: AX = 00h (Ausgabesignal; Stereo)                               }
  {          AX = 01h (VOICE-Teil; Stereo)                                  }
  {          AX = 02h (Mikrofon-Eingang)                                    }
  {          AX = 03h (CD-ROM; Stereo)                                      }
  {          AX = 04h (Stereo-LINE-Eingang)                                 }
  {          BX = sbd_GetVolume                                             }
  {          DX = 01h (linker Kanal), DX = 02h (rechter Kanal)              }
  { Ausgabe: AX = FFFFh (Fehler aufgetreten), AX <> FFFFh (gesuchter Wert)  }
  {-------------------------------------------------------------------------}

  sbd_GetVolume = $19;

  {=========================================================================}
  { sbd_SamplingRate: Sampling-Rate ermitteln (SB, SB Pro und hher)        }
  {                                                                         }
  {                   ACHTUNG: Bei CTVDSK hat diese Funktion die Nummer 15h }
  {=========================================================================}
  { Eingabe: AX = 00h (Aufnahme), AX = 01h (Wiedergabe)                     }
  {          BX = sbd_SamplingRate                                          }
  {          DX = 00h (Mono), DX = 01h (Stereo; nur SB Pro)                 }
  { Ausgabe: AX = kleinste Sampling-Rate                                    }
  {          DX = hchste Sampling-Rate                                     }
  {-------------------------------------------------------------------------}

  sbd_SamplingRate = $20;

  {=========================================================================}
  { sbd_GetFilterOnOff: Abfrage der Einstellung eines Filters (nur SB Pro   }
  {                     bzw. hher)                                         }
  {                                                                         }
  {                     ACHTUNG: Bei CTVDSK hat diese Funktion die Nummer   }
  {                              17h.                                       }
  {=========================================================================}
  { Eingabe: AX = 00h (Eingabefilter), AX = 01h (Ausgabefilter)             }
  {          BX = sbd_GetFilterOnOff                                        }
  { Ausgabe: AX = FFFFh (Fehler aufgetreten)                                }
  {          AX = 00h (an), AX = 01h (aus)                                  }
  {-------------------------------------------------------------------------}

  sbd_GetFilterOnOff = $21;

{===========================================================================}
{ Prozedur InitSBDrv: Initialisiert die Unit, indem die Treibervariablen    }
{                     DrvPtr und DrvSize initialisiert werden, die Umge-    }
{                     bungsvariablen aus der AUTOEXEC.BAT-Datei ausgelesen  }
{                     und ausgewertet werden.                               }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE InitSBDrv;

{===========================================================================}
{ Funktion ReadDSP: Fragt den Wert des DSP-Ports SB_BaseAddress+0Ah ab.     }
{                   Die Funktion wird von CheckBaseAddress aufgerufen, um   }
{                   zu prfen, ob die Basis-I/O-Adresse richtig gesetzt     }
{                   ist.                                                    }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: ein Bytewert                                                     }
{---------------------------------------------------------------------------}

FUNCTION ReadDSP : Byte;

{===========================================================================}
{ Prozedur CheckBaseAddress: Prft, ob die Basis-Adresse richtig gesetzt    }
{                            ist. Dazu wird ein Reset des DSP (Digital      }
{                            Sound Processor) ausgelst und es wird gewar-  }
{                            tet, bis an der Adresse SB_BaseAddress+0Ah     }
{                            der Wert AAh zu lesen ist. Kann dieser Zustand }
{                            innerhalb von ungefhr 1000 Mikrosekunden      }
{                            nicht erreicht werden, so ist die I/O-Adresse  }
{                            nicht korrekt. Es wird dann eine Fehlermeldung }
{                            ausgegeben und die Initialisierung der Sound-  }
{                            karte abgebrochen.                             }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE CheckBaseAddress;

{===========================================================================}
{ Funktion Load_CTVOICE: Ldt den CT-VOICE-Treiber in den Heap-Speicher.    }
{                        Die Funktion liefert einen Fehlercode zurck. Die  }
{                        Variablen DrvPtr und DrvSize werden bei fehler-    }
{                        freier Ausfhrung entsprechend gesetzt.            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: 0, wenn der Ladevorgang fehlerfrei abgelaufen ist, sonst Fehler- }
{          code                                                             }
{---------------------------------------------------------------------------}

FUNCTION Load_CTVOICE : Integer;

{===========================================================================}
{ Funktion Load_CTVDSK: Ldt den CTVDSK-Treiber in den Heap-Speicher. Wie   }
{                       Load_CTVOICE liefert auch diese Funktion einen Feh- }
{                       lercode zurck. Die Variablen DrvPtr und DrvSize    }
{                       werden bei fehlerfreier Ausfhrung entsprechend ge- }
{                       setzt.                                              }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: 0, wenn der Ladevorgang fehlerfrei abgelaufen ist, sonst Fehler- }
{          code                                                             }
{---------------------------------------------------------------------------}

FUNCTION Load_CTVDSK : Integer;

{===========================================================================}
{ Funktion SendToDriver: Sendet eine Steuersequenz an den sich im Speicher  }
{                        befindlichen Treiber und veranlat somit die Aus-  }
{                        fhrung einer Funktion, die innerhalb des Trei-    }
{                        bers implementiert ist, zum Beispiel Ausgabe von   }
{                        Geruschen. Als Ergebnis gibt die Funktion in den  }
{                        meisten, jedoch treiberbedingt nicht allen Fllen  }
{                        den Inhalt des AX-Registers zurck.                }
{===========================================================================}
{ Eingabe: AX_  = Inhalt des AX-Registers                                   }
{          Func = Nummer der auszufhrenden Treiberfunktion (BX-Reg.)       }
{          CX_  = Inhalt des CX-Registers                                   }
{          DX_  = Inhalt des DX-Registers                                   }
{          ES_  = Inhalt des ES-Registers                                   }
{          DI_  = Inhalt des DI-Registers                                   }
{ Ausgabe: Inhalt des AX-Registers                                          }
{---------------------------------------------------------------------------}

FUNCTION SendToDriver(AX_, Func, CX_, DX_, ES_, DI_ : Word) : Word;

{===========================================================================}
{ Prozedur GetDriverVer: Liefert die Versionsnummer des sich im Speicher    }
{                        befindlichen Treibers zurck. Der Treiber mu ini- }
{                        tialisiert sein.                                   }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Version    = Hauptversion                                        }
{          SubVersion = Unterversion                                        }
{---------------------------------------------------------------------------}

PROCEDURE GetDriverVer(VAR Version, SubVersion : Byte);

{===========================================================================}
{ Prozedur SetBaseAddress: Setzt die Basis-I/O-Adresse der Sound-Blaster-   }
{                          Karte.                                           }
{===========================================================================}
{ Eingabe: Addr = I/O-Adresse                                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetBaseAddress(Addr : Word);

{===========================================================================}
{ Prozedur SetInterrupt: Setzt die IRQ-Nummer der Sound-Blaster-Karte.      }
{===========================================================================}
{ Eingabe: IRQ = Nummer des Interrupts                                       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetInterrupt(IRQ : Word);

{===========================================================================}
{ Prozedur SetDMAChannel: Setzt die Nummer des DMA-Kanals der Sound         }
{                         Blaster-Karte.                                    }
{                                                                           }
{                         WICHTIG: Diese Funktion ist nur bei Sound Blaster }
{                                  Pro und hher verfgbar.                 }
{===========================================================================}
{ Eingabe: DMA = DMA-Kanal                                                  }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetDMAChannel(DMA : Word);

{===========================================================================}
{ Funktion DriverInit: Initialisiert den sich im Speicher befindlichen      }
{                      Treiber. Die Funktion liefert im Normalfall den Wert }
{                      TRUE zurck, wenn die Initialisierung fehlerfrei ab- }
{                      gelaufen ist.                                        }
{===========================================================================}
{ Eingabe: BufSize4KB = Puffergre in 4-KByte-Einheiten                    }
{ Ausgabe: TRUE, wenn Initialisierung korrekt, sonst FALSE                  }
{---------------------------------------------------------------------------}

FUNCTION DriverInit(BufSize4KB : Word) : Boolean;

{===========================================================================}
{ Prozedur SpeakerOn: Schalter den Lautsprecher ein.                        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SpeakerOn;

{===========================================================================}
{ Prozedur SpeakerOff: Schaltet den Lautsprecher aus.                       }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SpeakerOff;

{===========================================================================}
{ Funktion HaltOutput: Unterbricht das Abspielen eines Sample. Die Funktion }
{                      liefert den Wert TRUE zurck, wenn die Ausgabe un-   }
{                      terbrochen wurde und den Wert FALSE, wenn kein Sam-  }
{                      ple zur Zeit abgespielt wird.                        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE = Unterbrechung, FALSE = keine VOICE-Ausgabe zur Zeit       }
{---------------------------------------------------------------------------}

FUNCTION HaltOutput : Boolean;

{===========================================================================}
{ Funktion ContinueOutput: Setzt eine unterbrochene VOICE-Ausgabe fort. Die }
{                          Funktion liefert TRUE zurck, wenn das Abspielen }
{                          eines Sample wieder fortgesetzt wird, sonst FAL- }
{                          SE.                                              }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE = Fortsetzung der Ausgabe, FALSE = Ausgabe wurde nicht un-  }
{          terbrochen                                                       }
{---------------------------------------------------------------------------}

FUNCTION ContinueOutput : Boolean;

{===========================================================================}
{ Prozedur DriverEnd: Beendet den aktuellen Treiber. Der Treiber wird auch  }
{                     aus dem Speicher entfernt.                            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE DriverEnd;

{===========================================================================}
{ Funktion GetBoardType: Liefert den Typ der Sound-Blaster-Karte. Der Trei- }
{                        ber, von dem die Informationen geholt werden, mu  }
{                        vorher initialisiert werden (Treiberfunktion       }
{                        sbd_DriverInit).                                   }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Kartentyp (Beschreibung s. Funktion sbd_BoardType in der Defini- }
{          tion der Treiber-Funktionskonstanten) bzw. 0, wenn DrvPtr = NIL  }
{---------------------------------------------------------------------------}

FUNCTION GetBoardType : Word;

IMPLEMENTATION

USES CRT;                                       { intern CRT-Unit einbinden }

{===========================================================================}
{ Funktion CharPos: Liefert die Position eines Zeichens in einem String     }
{                   bzw. den Wert 0, wenn das Zeichen in dem angegebenen    }
{                   String nicht vorkommt oder der String leer ist.         }
{                                                                           }
{                   Diese Funktion wird unter anderem von den Routinen      }
{                   GetBaseAddress, GetInterrupt und  GetDMAChannel aufge- }
{                   rufen, um die Werte aus den Umgebungsvariablen in der   }
{                   AUTOEXEC.BAT-Datei auszulesen.                          }
{                                                                           }
{                   ANMERKUNG: Die Unterscheidung zwischen Klein- und Gro- }
{                              schreibung findet nicht statt.               }
{===========================================================================}
{ Eingabe: Ch = Zeichen, nach dem im String S gesucht wird                  }
{          S  = String                                                      }
{ Ausgabe: Position des Zeichens Ch im String S bzw. 0, wenn das Zeichen    }
{          nicht gefunden wird                                              }
{---------------------------------------------------------------------------}

FUNCTION CharPos(Ch : Char; S : String) : Byte;

VAR
  Count : Word;                                                { ein Zhler }
  Found : Boolean;  { wird auf TRUE gesetzt, wenn das Zeichen gefunden wird }
  Len   : Byte;                                            { String-Lnge S }

BEGIN
  CharPos := 0;
  IF S <> '' THEN                                      { String nicht leer? }
    BEGIN                           { nein, dann das Zeichen Ch in S suchen }
      Count := 1;
      Len := Length(S);                            { String-Lnge ermitteln }
      REPEAT
        Found := UpCase(S[Count]) = Ch;                 { Zeichen gefunden? }
        Inc(Count);                                        { Zhlen erhhen }
      UNTIL Found OR (Count > Len);
      IF Found THEN   { wenn Zeichen gefunden, dann dessen Pos. zurckgeben }
        CharPos := Count-1;
    END;
END;

{===========================================================================}
{ Funktion HexToDec: Wandelt eine hexadezimale Zahl, die als String vor-    }
{                    liegt, in einen Dezimalwert um.                        }
{===========================================================================}
{ Eingabe: HexStr = hexadezimale Zahl in String-Form                        }
{ Ausgabe: eine dezimale Zahl bzw. der Wert 0, wenn der Hex-String nicht    }
{          korrekt oder leer ist                                            }
{---------------------------------------------------------------------------}

FUNCTION HexToDec(HexStr : String4) : Word;

VAR
  Len     : Byte;                                     { String-Lnge HexStr }
  Count   : Word;                                              { ein Zhler }
  Fail    : Boolean;  { TRUE, wenn ein Zeichen in dem Hex-String falsch ist }
  Mult    : LongInt;                                        { Multiplikator }
  Ch      : Char;                    { Zeichen, das gerade umgewandelt wird }
  Decimal : Word;                                             { Dezimalwert }

BEGIN
  HexToDec := 0;
  IF HexStr <> '' THEN                             { Hex-String nicht leer? }
    BEGIN                                        { nein, Umwandlung starten }
      Len := Length(HexStr);                   { Lnge von HexStr ermitteln }
      Count := Len;                                         { Zhler setzen }
      Fail := FALSE;                                      { noch alles o.k. }
      Mult := 1;                               { Multiplikator auf 1 setzen }
      Decimal := 0;                           { Dezimalwert ist am Anfang 0 }
      REPEAT                                              { Schleifenanfang }
        Ch := UpCase(HexStr[Count]);                    { Zeichen bestimmen }
        CASE Ch OF                     { die eigentliche Umwandlung beginnt }
          '0'..'9' :
            Inc(Decimal, Mult*(Byte(Ch)-48));
          'A'..'F' :
            Inc(Decimal, Mult*(Byte(Ch)-55));
        ELSE                           { ein falsches Zeichen im Hex-String }
          Fail := TRUE;                              { Umwandlung abbrechen }
        END;
        Dec(Count);                                    { Zhler erniedrigen }
        Mult := 16*Mult;                             { Multiplikator mal 16 }
      UNTIL (Count = 0) OR Fail;                            { Schleifenende }
      IF NOT Fail THEN      { wenn alles o.k., dann Dezimalwert zurckgeben }
        HexToDec := Decimal
      ELSE                                     { ein Fehler ist aufgetreten }
        HexToDec := 0;
    END;
END;

{===========================================================================}
{ Funktion GetBaseAddress: Wertet die BLASTER-Umgebungsvariable aus, indem  }
{                          daraus die Basis-I/O-Adresse der Sound-Blaster-  }
{                          Karte ermittelt und zurckgeliefert wird.        }
{                                                                           }
{                          ANMERKUNG: Die zurckgelieferte Basis-Adresse    }
{                                     ist nicht hexadezimal. So bedeutet    }
{                                     zum Beispiel der Wert 544 die Stan-   }
{                                     dard-Adresse 220 hex.                 }
{===========================================================================}
{ Eingabe: BlasterEnv = BLASTER-Umgebungsvariable                           }
{ Ausgabe: Basis-I/O-Adresse bzw. der Wert 0, wenn ein entsprechender Ein-  }
{          trag in der Umgebungsvariablen nicht gefunden wird               }
{---------------------------------------------------------------------------}

FUNCTION GetBaseAddress(BlasterEnv : String) : Word;

VAR
  AddrPos  : Byte;  { Position der Basis-Adresse in der Umgebungsvariablen }
  AddrStr  : String;                        { Basis-Adresse als Hex-String }

BEGIN
  GetBaseAddress := 0;

  { Die Basis-I/O-Adresse wird in der BLASTER-Umgebungsvariablen gefolgt }
  { von dem Zeichen "A" eingetragen.                                     }

  AddrPos := CharPos('A', BlasterEnv)+1;

  { Ermittlung der Basis-Adresse }

  IF AddrPos > 1 THEN
    BEGIN
      AddrStr := Copy(BlasterEnv, AddrPos, 3);
      GetBaseAddress := HexToDec(AddrStr);
    END;
END;

{===========================================================================}
{ Funktion GetInterrupt: Sucht in der BLASTER-Umgebungsvariablen nach dem   }
{                        Interrupt-Eintrag und liefert diesen ggf. zurck.  }
{                        Wird der Interrupt-Eintrag nicht gefunden, so gibt }
{                        die Funktion den Wert 0 zurck.                    }
{===========================================================================}
{ Eingabe: BlasterEnv = Umgebungsvariable - ein String                      }
{ Ausgabe: Interrupt-Nummer der Soundkarte bzw. 0, wenn der Interrupt-Ein-  }
{          trag nicht gefunden wurde                                        }
{---------------------------------------------------------------------------}

FUNCTION GetInterrupt(BlasterEnv : String) : Word;

VAR
  IntrPos : Byte;                     { Position des Eintrags in BlasterEnv }
  IntrStr : String;                           { Interrupt-Nummer als String }
  IntrNmb : Word;       { Interrupt-Nummer (nach String -> Word-Umwandlung) }
  Code    : Word;                    { zeigt, ob die Umwandlung korrekt war }

BEGIN
  GetInterrupt := 0;

  { Der Interrupt-Eintrag wird in der BLASTER-Umgebungsvariablen mit einem }
  { "I" gekennzeichnet.                                                    }

  IntrPos := CharPos('I', BlasterEnv)+1;

  { Ermittlung der Interrupt-Nummer }

  IF IntrPos <> 0 THEN
    BEGIN
      IntrStr := Copy(BlasterEnv, IntrPos, 2);
      IF NOT (IntrStr[2] IN ['0'..'9']) THEN
        IntrStr := IntrStr[1];
      Val(IntrStr, IntrNmb, Code);              { String -> Word-Umwandlung }
      IF Code = 0 THEN                           { War die Umwandlung o.k.? }
        GetInterrupt := IntrNmb;         { ja, Interrupt-Nummer zurckgeben }
    END;
END;

{===========================================================================}
{ Funktion GetDMAChannel: Ermittelt aus der BLASTER-Umgebungsvariablen die  }
{                         Nummer des DMA-Kanals und liefert diese an den    }
{                         Aufrufer dieser Funktion zurck. Der Wert 65535   }
{                         (DMA-Kanal kann durchaus auch 0 sein) wird in dem }
{                         Falle zurckgegeben, wenn kein DMA-Eintrag fest-  }
{                         gestellt werden konnte oder dieser falsch ist.    }
{===========================================================================}
{ Eingabe: BlasterEnv = Umgebungsvariable                                   }
{ Ausgabe: DMA-Nummer bzw. 65535                                            }
{---------------------------------------------------------------------------}

FUNCTION GetDMAChannel(BlasterEnv : String) : Word;

VAR
  DMAPos : Byte;                  { Position des DMA-Eintrags in BlasterEnv }
  DMAStr : String;                                             { DMA-String }

BEGIN
  GetDMAChannel := 65535;

  { Der Eintrag des DMA-Kanals ist an dem Buchstaben "D" zu erkennen. }

  DMAPos := CharPos('D', BlasterEnv)+1;

  { Ermittlung des DMA-Kanals }

  IF DMAPos <> 0 THEN
    BEGIN
      DMAStr := Copy(BlasterEnv, DMAPos, 1);
      IF DMAStr[1] IN ['0'..'9'] THEN
        GetDMAChannel := Byte(DMAStr[1])-48;
    END;
END;

{===========================================================================}
{ Funktion LoadDriver: Ldt einen Treiber (CT-VOICE.DRV oder CTVDSK.DRV)    }
{                      in den Heap-Speicher und bietet somit eine Mglich-  }
{                      keit Steuerbefehle an den Treiber direkt zu senden   }
{                      (s. Funktion SendToDriver). Tritt whrend des Lade-  }
{                      vorgangs ein Fehler auf, so wird der Fehlercode als  }
{                      Ergebnis dieser Funktion zurckgeliefert. Bei feh-   }
{                      lerfreier Ausfhrung des Ladevorgangs wird der Wert  }
{                      0 zurckgegeben.                                     }
{===========================================================================}
{ Eingabe: Name = Name der Treiberdatei                                     }
{ Ausgabe: 0 bei fehlerfreier Ausfhrung, sonst Fehlercode                  }
{---------------------------------------------------------------------------}

FUNCTION LoadDriver(Name : String12) : Integer;

VAR
  DrvFile    : File;                                        { Dateivariable }
  Result     : Integer;    { Status der letzten I/O-Operation auf der Datei }
  HelpPtr    : Pointer;                                   { ein Hilfszeiger }
  Segm, Offs : Word;                          { Segment- und Offset-Adresse }
  Address    : LongInt;                         { Adresse in gepackter Form }

BEGIN
  LoadDriver := 0;
  IF DrvPtr = NIL THEN                         { noch kein Treiber geladen? }
    IF Name <> '' THEN                   { Ja. Ist Name kein leerer String? }
      BEGIN                                { nein, dann Ladevorgang starten }
        Assign(DrvFile,                        { Datei mit Namen verknpfen }
          SB_DriverPath+'\'+Name);
        Reset(DrvFile, 1);                                   { Datei ffnen }
        Result := IOResult;           { Status der Datei-Operation abfragen }
        IF Result = 0 THEN                                { Ist alles o.k.? }
          BEGIN
            DrvSize := FileSize(DrvFile)+15;   { Dateigre ermitteln (+15) }
            GetMem(HelpPtr, DrvSize);           { Speicherplatz reservieren }

            { Treiber-Offset auf 0 setzen }

            Segm := Seg(HelpPtr^);
            Offs := Ofs(HelpPtr^);
            Address := (LongInt(Segm) SHL 4)+LongInt(Offs);
            Segm := (Address SHR 4)+1;
            Offs := 0;
            DrvPtr := Ptr(Segm, Offs);

            BlockRead(DrvFile, DrvPtr^, DrvSize-15);   { Treiberdaten laden }
            Result := IOResult;                       { Fehler aufgetreten? }
            IF Result = 0 THEN                          { nein, kein Fehler }
              Close(DrvFile)                       { Treiberdatei schlieen }
            ELSE                                        { leider ein Fehler }
              LoadDriver := Result;                { Fehlercode zurckgeben }
          END
        ELSE                                             { nein, ein Fehler }
          LoadDriver := Result;                    { Fehlercode zurckgeben }
      END
    ELSE                                       { Name ist ein leerer String }
      LoadDriver := 32766    { Fehlercode 32766: Name ist ein leerer String }
  ELSE                      { ein Treiber befindet sich bereits im Speicher }
    LoadDriver := 32767;  { Fehlercode 32767: ein Treiber ist schon geladen }
END;

{---------------------------------------------------------------------------}
{ Implementation der modulexternen Routinen                                 }
{---------------------------------------------------------------------------}

PROCEDURE InitSBDrv;

VAR
  BlasterEnv : String;             { BLASTER-Umgebungsvariable - ein String }
  SoundEnv   : String;     { SOUND-Umgebungsvariable - ebenfalls ein String }

BEGIN

  { Treibervariablen initialisieren }

  DrvPtr := NIL;
  DrvSize := 0;
  CTVDSK := FALSE;
  CTVDSK := FALSE;

  { BLASTER- und SOUND-Umgebungsvariablen liefern }

  BlasterEnv := GetEnv('BLASTER');
  SoundEnv := GetEnv('SOUND');

  { Prfen, ob BlasterEnv und SoundEnv keine leeren Strings sind. }
  { Wenn ja, eine Fehlermeldung ausgeben und die Initialisierung  }
  { der Unit per Halt-Befehl beenden.                             }

  IF (BlasterEnv = '')
  OR (SoundEnv = '') THEN
    BEGIN
      WriteLn;
      WriteLn('Fehler: BLASTER- bzw. SOUND-Umgebungsvariable in der '+
        'AUTOEXEC.BAT-Datei');
      WriteLn('        nicht gefunden.');
      Halt;
    END;

  SB_DriverPath := SoundEnv+'\DRV';       { Treiber-Verzeichnisnamen setzen }
  SB_BaseAddress := GetBaseAddress(BlasterEnv);     { I/O-Adresse ermitteln }

  { Prfen, ob die Basis-Adresse gefunden werden konnte und richtig ist. }
  { Wenn nicht, Fehlermeldung auf dem Bildschirm anzeigen und die Ini-   }
  { tialisierung der Unit mittels Halt beenden.                          }

  IF (SB_BaseAddress <> $210)
  AND (SB_BaseAddress <> $220)
  AND (SB_BaseAddress <> $230)
  AND (SB_BaseAddress <> $240)
  AND (SB_BaseAddress <> $250)
  AND (SB_BaseAddress <> $260)
  AND (SB_BaseAddress <> $280) THEN
    BEGIN
      WriteLn;
      WriteLn('Fehler: Die Basis-I/O-Adresse konnte nicht gefunden werden '+
        'oder ist nicht');
      WriteLn('        richtig gesetzt. berprfen Sie Ihre AUTOEXEC.BAT-'+
        'Datei.');
      Halt;
    END;
  CheckBaseAddress;

  SB_Interrupt := GetInterrupt(BlasterEnv);        { Interrupt-Nummer lesen }

  { Prfen, ob der Interrupt-Eintrag gefunden wurde und richtig }
  { gesetzt ist. Wenn nicht, Ausgabe einer Fehlermeldung und    }
  { Abbruch der Initialisierung.                                }

  IF NOT (SB_Interrupt IN [2, 5, 7, 10]) THEN
    BEGIN
      WriteLn;
      WriteLn('Fehler: Die Interrupt-Nummer Ihrer Sound Blaster-Karte '+
        'konnte nicht ');
      WriteLn('        ermittelt werden oder ist nicht richtig gesetzt. '+
        'Am besten, Sie');
      WriteLn('        berprfen Ihre AUTOEXEC.BAT-Datei nach dem "SET '+
        'BLASTER"-Eintrag.');
      Halt;
    END;

  SB_DMAChannel := GetDMAChannel(BlasterEnv);         { DMA-Kanal ermitteln }

  { Prfen, ob der DMA-Eintrag korrekt ist. }

  IF NOT (SB_DMAChannel IN [0, 1, 3]) THEN
    BEGIN
      WriteLn;
      WriteLn('Fehler: Die DMA-Einstellung wurde nicht gefunden oder ist '+
        'nicht korrekt');
      WriteLn('        gesetzt. Schauen Sie in Ihrer AUTOEXEC.BAT-Datei '+
        'nach.');
      Halt;
    END;
END;

FUNCTION ReadDSP : Byte;

VAR
  Count   : Word;                                              { ein Zhler }
  Reading : Boolean;              { TRUE, wenn der DSP zum Lesen bereit ist }

BEGIN
  Count := 0;
  Reading := FALSE;
  WHILE NOT Reading DO          { wiederholen, bis DSP zum Lesen bereit ist }
    BEGIN
      Inc(Count);                                          { Zhler erhhen }
      Reading :=                        { Status des Lesevorgangs ermitteln }
        (Port[SB_BaseAddress+$0E] AND $80) = $80;
      IF Count = 10000 THEN      { Hat der Zhler den Stand 10000 erreicht? }
        BEGIN                                             { ja, also Fehler }
          WriteLn;
          WriteLn('Fehler: Der DSP der Sound Blaster-Karte kann nicht '+
            'gelesen werden.');
          WriteLn('        Vielleicht ist keine Soundkarte vorhanden.');
          Halt;
        END;
    END;
  ReadDSP := Port[SB_BaseAddress+$0A];   { DSP o.k., Bytewert zurckliefern }
END;

PROCEDURE CheckBaseAddress;

VAR
  Count   : Word;                                              { ein Zhler }
  Reading : Boolean;    { TRUE, wenn der Bytewert AAh gelesen werden konnte }

BEGIN
  Count := 0;
  Reading := FALSE;
  Port[SB_BaseAddress+$06] := $01;     { Port Basis+06h mit 01h beschreiben }
  Delay(1);                                      { eine Millisekunde warten }
  Port[SB_BaseAddress+$06] := $00;     { Port Basis+06h mit 00h beschreiben }
  WHILE NOT Reading DO          { solange wiederholen, bis Wert AAh gelesen }
    BEGIN
      Inc(Count);                                          { Zhler erhhen }
      Reading := (ReadDSP = $AA);                       { Wert AAh gelesen? }
      IF Count = 1000 THEN   { 1000 Mikrosekunden (1 Millisek.) abgelaufen? }
        BEGIN                                             { ja, also Fehler }
          WriteLn;
          WriteLn('Fehler: Falsche Basis-I/O-Adresse oder Reset-Fehler');
          Halt;
        END;
    END;
END;

FUNCTION Load_CTVOICE : Integer;

VAR
  Result : Integer;

BEGIN
  Result := LoadDriver('CT-VOICE.DRV');
  IF Result = 0 THEN                         { Treiber erfolgreich geladen? }
    CTVOICE := TRUE;                                                   { ja }
  Load_CTVOICE := Result;
END;

FUNCTION Load_CTVDSK : Integer;

VAR
  Result : Integer;

BEGIN
  Result := LoadDriver('CTVDSK.DRV');
  IF Result = 0 THEN
    CTVDSK := TRUE;
  Load_CTVDSK := Result;
END;

FUNCTION SendToDriver(AX_, Func, CX_, DX_, ES_, DI_ : Word) : Word;

BEGIN
  ASM
    MOV   AX,AX_
    MOV   BX,Func
    MOV   CX,CX_
    MOV   DX,DX_
    MOV   DI,ES_
    MOV   ES,DI
    MOV   DI,DI_
    CALL  [DrvPtr]
    MOV   AX_,AX
  END;
  SendToDriver := AX_;
END;

PROCEDURE GetDriverVer(VAR Version, SubVersion : Byte);

VAR
  VerPack : Word;          { gepackte Version-Info (Haupt- und Unterversion }

BEGIN
  Version := 0;
  SubVersion := 0;
  IF DrvPtr <> NIL THEN            { Ist ein Treiber im Speicher vorhanden? }
    BEGIN                                           { ja, Version ermitteln }
      VerPack := SendToDriver(0, sbd_DriverVer, 0, 0, 0, 0);
      Version := Hi(VerPack);                       { Hi-Byte: Hauptversion }
      SubVersion := Lo(VerPack);                    { Lo-Byte: Unterversion }
    END;
END;

PROCEDURE SetBaseAddress(Addr : Word);

BEGIN

  { Wenn ein Treiber geladen ist und die angegebene Basis-Adresse eine }
  { der mglichen I/O-Adressen der Soundkarte ist, dann wird diese Ad- }
  { resse mittels SendToDriver gesetzt.                                }

  IF (DrvPtr <> NIL)
  AND ((Addr = $210)
  OR (Addr = $220)
  OR (Addr = $230)
  OR (Addr = $240)
  OR (Addr = $250)
  OR (Addr = $260)
  OR (Addr = $280)) THEN
    SendToDriver(Addr, sbd_SetBaseAddress, 0, 0, 0, 0);
END;

PROCEDURE SetInterrupt(IRQ : Word);

BEGIN

  { Wenn ein Treiber geladen ist und die angegebene IRQ-Nummer eine }
  { der mglichen ist, dann wird diese Interrupt-Nummer gesetzt.    }

  IF (DrvPtr <> NIL)
  AND (IRQ IN [2, 5, 7, 10]) THEN
    SendToDriver(IRQ, sbd_SetInterrupt, 0, 0, 0, 0);
END;

PROCEDURE SetDMAChannel(DMA : Word);

BEGIN

  { Wenn ein Treiber (CT-VOICE oder CTVDSK) geladen ist und die Nummer }
  { des DMA-Kanals einer der mglichen ist, dann wird diese DMA-Nummer }
  { gesetzt.                                                           }

  IF (DrvPtr <> NIL)
  AND (DMA IN [0, 1, 3]) THEN
    SendToDriver(DMA, sbd_SetDMAChannel, 0, 0, 0, 0);
END;

FUNCTION DriverInit(BufSize4KB : Word) : Boolean;

VAR
  Result : Word;                             { Ergebnis der Initialisierung }

BEGIN
  DriverInit := FALSE;
  IF DrvPtr <> NIL THEN            { Befindet sich ein Treiber im Speicher? }
    BEGIN                                     { ja, Initialisierung starten }
      IF CTVOICE THEN
        Result := SendToDriver(0, sbd_DriverInit, 0, 0, 0, 0)
      ELSE
        IF CTVDSK THEN
          Result := SendToDriver(BufSize4KB, sbd_DriverInit, 0, 0, 0, 0);
      IF Result = 0 THEN                         { Initialisierung korrekt? }
        DriverInit := TRUE;                                            { ja }
    END;
END;

PROCEDURE SpeakerOn;

BEGIN
  IF DrvPtr <> NIL THEN                                  { Treiber geladen? }
    SendToDriver(1, sbd_SetSpeakerOnOff, 0, 0, 0, 0); { ja, Lautsprecher an }
END;

PROCEDURE SpeakerOff;

BEGIN
  IF DrvPtr <> NIL THEN                                  { Treiber geladen? }
    SendToDriver(0, sbd_SetSpeakerOnOff,             { ja, Lautsprecher aus }
      0, 0, 0, 0);
END;

FUNCTION HaltOutput : Boolean;

VAR
  Result : Word;                             { Ergebnis der Treiberfunktion }

BEGIN
  HaltOutput := FALSE;
  IF DrvPtr <> NIL THEN            { Befindet sich ein Treiber im Speicher? }
    BEGIN                                  { ja, VOICE-Ausgabe unterbrechen }
      Result := SendToDriver(0, sbd_OutputPause, 0, 0, 0, 0);
      IF Result = 0 THEN                   { Ausgabe wirklich unterbrochen? }
        HaltOutput := TRUE;                 { ja, den Wert TRUE zurckgeben }
    END;
END;

FUNCTION ContinueOutput : Boolean;

VAR
  Result : Word;                             { Ergebnis der Treiberfunktion }

BEGIN
  ContinueOutput := FALSE;
  IF DrvPtr <> NIL THEN                                  { Treiber geladen? }
    BEGIN                             { ja, VOICE-Ausgabe wieder fortsetzen }
      Result := SendToDriver(0, sbd_ContinueOutput, 0, 0, 0, 0);
      IF Result = 0 THEN       { Abspielen des Sample wirklich fortgesetzt? }
        ContinueOutput := TRUE;           { ja, den Wert TRUE zurckliefern }
    END;
END;

PROCEDURE DriverEnd;

BEGIN
  IF DrvPtr <> NIL THEN                          { Ist ein Treiber geladen? }
    BEGIN                                                              { ja }
      SendToDriver(0, sbd_DriverEnd, 0, 0, 0, 0);      { Treiber abschalten }
      FreeMem(DrvPtr, DrvSize);                   { Speicherplatz freigeben }
      DrvPtr := NIL;
      DrvSize := 0;
      IF CTVOICE THEN
        CTVOICE := FALSE;
      IF CTVDSK THEN
        CTVDSK := FALSE;
    END;
END;

FUNCTION GetBoardType : Word;

BEGIN
  IF DrvPtr <> NIL THEN                          { Ist ein Treiber geladen? }
    GetBoardType := SendToDriver(0, sbd_BoardType, 0, 0, 0, 0)         { ja }
  ELSE                                                               { nein }
    GetBoardType := 0;
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

END.
