
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Prog. PlayWAVE : Ein Programm zum Abspielen von WAVE-Dateien (8 Bits pro  }
{                  Sample, Mono) via direkten DSP-Zugriff mit Hilfe einer   }
{                  benutzerdefinierten Interrupt-Behandlungsroutine.        }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : PLAYWAVE.PAS                                             }
{ entwickelt am  : 07.08.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.03                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

PROGRAM PlayWAVE;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$D-}                                        { keine Debugger-Informationen }
{$F-}                                        { FAR-Aufrufe sind nicht ntig }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

{$M 16384,65536,655360}         { 16 KB Stack, min. 64 KB Heap, max. 640 KB }

USES CRT, DOS, SBDSP;                 { CRT-, DOS- und SBDSP-Unit einbinden }

TYPE

  String4  = ARRAY[0..3] OF Char;                    { String aus 4 Zeichen }

  {=========================================================================}
  { TWAVERec: Struktur des WAVE-Header                                      }
  {=========================================================================}

  TWAVERec = RECORD
    RIFF      : String4;                                           { "RIFF" }
    FileLen   : LongInt;                         { Lnge der WAVE-Datei - 8 }
    WAVE      : String4;                                           { "WAVE" }
    fmt_      : String4;                                           { "fmt_" }
    BlocksLen : LongInt; { Lngenangabe bestehend aus Bytes 10h 00h 00h 00h }

    { Headerblock: wave-format }

    FormatType  : Word;                         { Art des benutzten Formats }
    Channels    : Word;   { Anzahl der benutzten Kanle (1: Mono, 2: Stereo }
    SampleRate  : LongInt;                                  { Sampling-Rate }
    BytesProSec : LongInt;   { durschnittl. Anz. Bytes pro Sek. bei Ausgabe }
    SampleType  : Word;                           { Anzahl Bytes pro Sample }

    { Headerblock: format-specific }

    SampleBits : Word;     { Anzahl Bits pro Sample (8: 8 Bits, 16: 16 Bits }

    { Danach folgt abhngig von der Lnge der beiden Headerblcke }
    { der String "data" und anschlieend die Lnge der Daten (ein }
    { Doppelwort).                                                }

  END;

VAR
  WAVEName : String;                                  { Name der WAVE-Datei }
  WAVEFile : File;                                     { WAVE-Dateivariable }

{===========================================================================}
{ Prozedur ShowHelp: Hilfe zum Programm auf dem Bildschirm anzeigen.        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ShowHelp;

BEGIN
  WriteLn('Spielt eine WAVE-Datei (8 Bits pro Sample, Mono) via direkten '+
    'DSP-Zugriff ab.');
  WriteLn;
  WriteLn('Syntax: PLAYWAVE [Dateiname].WAV');
END;

{===========================================================================}
{ Funktion UpperString: Wandelt einen String beliebiger Lnge in Gro-      }
{                       schreibung um und liefert ihn an den Aufrufer der   }
{                       Funktion zurck.                                    }
{===========================================================================}
{ Eingabe: S = String, der in Groschreibung umgewandelt werden soll        }
{ Ausgabe: String in Groschreibung                                         }
{---------------------------------------------------------------------------}

FUNCTION UpperString(S : String) : String;

VAR
  Count : Word;                                                { ein Zhler }
  Upper : String;                  { in Groschreibung umgewandelter String }

BEGIN
  UpperString := '';
  IF S <> '' THEN                               { Ist S kein leerer String? }
    BEGIN                                          { nein, S ist nicht leer }
      Upper := '';
      FOR Count := 1 TO Length(S) DO                     { String umwandeln }
        Upper := Upper+UpCase(S[Count]);
      UpperString := Upper;                      { neuen String zurckgeben }
    END;
END;

{===========================================================================}
{ Prozedur Error: Zeigt einen Fehler an und beendet das Programm mit einem  }
{                 Halt-Befehl.                                              }
{===========================================================================}
{ Eingabe: Text = Fehlermeldung                                             }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Error(Text : String);

BEGIN
  WriteLn;
  WriteLn(Text);                                   { Fehlermeldung ausgeben }
  Halt;                                                  { Programm beenden }
END;

{===========================================================================}
{ Prozedur UserIntr: Eine benutzerdefinierte Interrupt-Routine, die das     }
{                    Abspielen der Sample-Daten via direkten DSP-Zugriff    }
{                    bernimmt.                                             }
{===========================================================================}
{ Eingabe: Flags, CS, IP, AX, BX ... = CPU-Register als Pseudo-Parameter    }
{ Ausgabe: wie Eingabe                                                      }
{---------------------------------------------------------------------------}

PROCEDURE UserIntr(Flags, CS, IP, AX, BX, CX, DX, SI, DI, DS, ES,
  BP : Word); INTERRUPT;

BEGIN

  { Funktionen des neuen Interrupt-Handler auswerten }

  CASE AX OF

    {=======================================================================}
    { Funktion 00h: Datenblock abspielen                                    }
    {=======================================================================}
    { Eingabe: AX    = 00h                                                  }
    {          BX    = Gre des Datenblocks in Bytes                       }
    {          CX    = Ausgabe-Frequenz in Hz                               }
    {          SI:DI = Startadresse des Datenblocks                         }
    { Ausgabe: keine                                                        }
    {-----------------------------------------------------------------------}

    $00 :
      PlayDirect(Ptr(SI, DI), BX, CX, FALSE);
  END;
END;

{===========================================================================}
{ Prozedur Play: Spielt die WAVE-Datei ab.                                  }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Play;

VAR
  WAVEHeader    : TWAVERec;                                   { WAVE-Header }
  WAVEBlockSize : LongInt;                    { Gre der WAVE-Sample-Daten }
  Result        : Integer;                          { Fehlerstatus-Variable }
  Count         : Word;                                        { ein Zhler }
  OldIntrVec    : Pointer;                         { alter Interrupt-Vektor }
  IntrNmb       : Byte;             { Nummer des "freien" Interrupt-Vektors }

  {=========================================================================}
  { Prozedur ReadError: Wird beim Auftreten eines Lesefehlers aufgerufen.   }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadError(FileName : String);

  BEGIN
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Error('Fehler: Lesefehler bei '+FileName);     { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur SeekError: Wird beim Auftreten eines Positionierungsfehlers    }
  {                     aufgerufen.                                         }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE SeekError(FileName : String);

  BEGIN
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Error('Fehler: Positionierungsfehler '+        { Fehlermeldung ausgeben }
      ' bei '+FileName);
  END;

  {=========================================================================}
  { Prozedur ReadDataAndPlay: Liest einen max. 60 KB groen Datenblock ein  }
  {                           und spielt ihn mittels einer benutzerdefi-    }
  {                           nierten Interrupt-Routine ab.                 }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadDataAndPlay;

  CONST
    BufSize = 60*1024;                                  { Puffergre 60 KB }

  VAR
    DataBuf      : Pointer;                                   { Datenpuffer }
    TransferSize : LongInt;  { Gre der noch zu bertragenden Sample-Daten }
    B            : Byte;                                    { ein Datenbyte }
    Regs         : Registers;                                { CPU-Register }

  BEGIN
    TransferSize := WAVEBlockSize;
    GetMem(DataBuf, BufSize);              { Puffer auf dem Heap einrichten }

    { solange wiederholen, bis alle Sample-Daten abgespielt sind }

    REPEAT
      IF TransferSize >= BufSize THEN     { Datenblockgre >= Puffergre? }
        BEGIN
          BlockRead(WAVEFile, DataBuf^, BufSize);          { Daten einlesen }
          Result := IOResult;                        { Fehlerstatus sichern }
          IF Result <> 0 THEN                       { Liegt ein Fehler vor? }
            BEGIN                                                      { ja }
              FreeMem(DataBuf, BufSize);            { Puffer deinstallieren }
              SetIntVec(IntrNmb, OldIntrVec);     { alten Int.-Vektor setzen}
              ReadError(WAVEName);      { Fehlerbehandlungsroutine aufrufen }
            END;

          { CPU-Register mit Werten belegen und benutzerdefinierte }
          { Interrupt-Routine aufrufen                             }

          WITH Regs DO
            BEGIN
              AX := $00;
              BX := BufSize;
              CX := WAVEHeader.SampleRate;
              SI := Seg(DataBuf^);
              DI := Ofs(DataBuf^);
              Intr(IntrNmb, Regs);
            END;

          Dec(TransferSize, BufSize);
        END
      ELSE
        BEGIN
          BlockRead(WAVEFile, DataBuf^, TransferSize);
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              SetIntVec(IntrNmb, OldIntrVec);
              ReadError(WAVEName);
            END;

          { CPU-Register mit Werten belegen und benutzerdefinierte }
          { Interrupt-Routine aufrufen                             }

          WITH Regs DO
            BEGIN
              AX := $00;
              BX := TransferSize;
              CX := WAVEHeader.SampleRate;
              SI := Seg(DataBuf^);
              DI := Ofs(DataBuf^);
              Intr(IntrNmb, Regs);
            END;

          TransferSize := 0;
        END;
    UNTIL TransferSize = 0;

    FreeMem(DataBuf, BufSize);                  { Puffer vom Heap entfernen }
  END;

{ Play }

VAR
  RIFFStr : String;                                        { "RIFF"-Kennung }
  WAVEStr : String;                                        { "WAVE"-Kennung }

BEGIN
  BlockRead(WAVEFile, WAVEHeader, SizeOf(TWAVERec));    { WAVE-Header lesen }
  Result := IOResult;                              { Fehlerstatus speichern }
  IF Result <> 0 THEN                                 { Fehler aufgetreten? }
    ReadError(WAVEName);
  RIFFStr := '';
  FOR Count := 0 TO 3 DO
    RIFFStr := RIFFStr+UpCase(WAVEHeader.RIFF[Count]);
  WAVEStr := '';
  FOR Count := 0 TO 3 DO
    WAVEStr := WAVEStr+UpCase(WAVEHeader.WAVE[Count]);
  IF (RIFFStr <> 'RIFF')                              { "RIFF"-Kennung oder }
  OR (WAVEStr <> 'WAVE') THEN              { "WAVE"-Kennung nicht gefunden? }
    BEGIN                                                            { nein }
      Close(WAVEFile);                               { WAVE-Datei schlieen }
      Error('Fehler: Datei '+WAVEName+' nicht im WAVE-Format');
    END;

  { berprfen, ob es sich um eine Mono-WAVE-Datei }
  { mit 8 Bits pro Sample handelt                  }

  IF (WAVEHeader.Channels = 1)
  AND (WAVEHeader.SampleType = 1)
  AND (WAVEHeader.SampleBits = 8) THEN
    BEGIN
      WriteLn('Datei '+WAVEName+' wird nun via direkten DSP-Zugriff '+
        'abgespielt.');
      Seek(WAVEFile, FilePos(WAVEFile)+4);          { 4 Positionen vorwrts }
      Result := IOResult;
      IF Result <> 0 THEN
        SeekError(WAVEName);
      BlockRead(WAVEFile, WAVEBlockSize,       { Gre der WAVE-Daten lesen }
        SizeOf(LongInt));
      Result := IOResult;
      IF Result <> 0 THEN
        ReadError(WAVEName);

      { "freien" Interrupt-Vektor suchen }

      IntrNmb := $77;
      REPEAT
        Inc(IntrNmb);                { Nummer des Interrupt-Vektors erhhen }
        GetIntVec(IntrNmb, OldIntrVec);            { Interrupt-Vektor holen }
      UNTIL (OldIntrVec = NIL) OR (IntrNmb = $80);

      IF (OldIntrVec = NIL)            { Interrupt-Vektor noch nicht belegt }
      AND (IntrNmb < $80) THEN                    { und kleiner als 80 hex? }
        BEGIN                                                          { ja }
          WriteLn;
          WriteLn('Hierzu wird der Interrupt ', IntrNmb, ' (dez.) '+
            'verwendet.');
          WriteLn;
          WriteLn('WAVE-Dateien, die lnger als 60 KB sind, werden ab und '+
            'zu kurz unterbrochen,');
          WriteLn('um neue Daten in den Puffer zu laden. Dies liegt daran, '+
            'da hier kein Doppel-,');
          WriteLn('sondern ein Einzelpuffer benutzt wird.');
          SetIntVec(IntrNmb, @UserIntr);    { neuen Interrupt-Vektor setzen }
          ReadDataAndPlay;                               { Sample abspielen }
          SetIntVec(IntrNmb, OldIntrVec);   { alten Interrupt-Vektor setzen }
        END
      ELSE             { ein "freier" Interrupt-Vektor wurde nicht gefunden }
        BEGIN
          Close(WAVEFile);                           { WAVE-Datei schlieen }
          Error('Fehler: Ein "freier" Interrupt-Vektor fr die DSP-Ausgabe '+
            'nicht gefunden.');
        END;
    END
  ELSE
    BEGIN
      Close(WAVEFile);
      Error('Fehler: Keine Mono-WAVE-Datei mit 8 Bits pro Sample.');
    END;

  Close(WAVEFile);                                   { WAVE-Datei schlieen }
END;

{---------------------------------------------------------------------------}
{ Hauptprogramm                                                             }
{---------------------------------------------------------------------------}

VAR
  Result  : Integer;                                { Fehlerstatus-Variable }
  Dir     : DirStr;                            { Verzeichnis der WAVE-Datei }
  Name    : NameStr;                 { Name der WAVE-Datei ohne Erweiterung }
  Ext     : ExtStr;                                      { Dateierweiterung }

BEGIN
  TextColor(LightGray);                                { Textfarbe hellgrau }
  WriteLn;
  WriteLn('PLAYWAVE  *  Version 1.03  *  (c) 1993 by Arthur Burda');
  WriteLn(''+
    '');
  IF ParamCount = 0 THEN                   { keine Kommandozeilenparameter? }
    BEGIN                 { nein, Fehlermeldung auf dem Bildschirm ausgeben }
      WriteLn;
      WriteLn('Fehler: Keine WAVE-Datei zum Abspielen angegeben.');
      Halt;
    END
  ELSE
    IF ParamStr(1) = '/?' THEN                         { Hilfe angefordert? }
      ShowHelp                    { ja, Hilfetext auf dem Bildschirm zeigen }
    ELSE                                { keine Hilfe, WAVE-Datei abspielen }
      BEGIN
        WAVEName := UpperString(ParamStr(1));    { WAVE-Dateinamen auslesen }
        FSplit(WAVEName, Dir, Name, Ext);        { WAVE-Dateinamen zerlegen }
        IF Ext = '' THEN
          WAVEName := WAVEName+'.WAV';
        Assign(WAVEFile, WAVEName);       { WAVE-Datei mit Namen verknpfen }
        Reset(WAVEFile, 1);                             { WAVE-Datei ffnen }
        Result := IOResult;                         { Fehlerstatus abfragen }
        IF Result <> 0 THEN                           { Fehler aufgetreten? }
          Error('Fehler: Datei '+WAVEName+     { ja, Fehlermeldung ausgeben }
            ' konnte nicht geffnet werden.');
        InitDSP;
        Play;                                            { Sample abspielen }
      END;
END.
