//
//    CONVERT.EXE - (c) 1992 Covox Inc., Eugene, Oregon USA
//
//    Command line program to convert between all Covox sound file
//    formats.
//
//    Created:       Jan 90
//    Last updated:  Feb 92 Ryan Hanlon
//
//    CONVERT.EXE can be compiled using either MicroSoft C 6.0 or Borland
//    C 2.0.  See CONVERT.MMK and CONVERT.BMK for more information.
//
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------


#if defined(__TURBOC__)  
#include <dir.h>
#endif

#include <bios.h>
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include "parse.h"
#include "cvxdigi.h"
#include "cvxutil.h"

//  Version string for CONVERT.C
//
PSTR    convertVersionString  = "2.20"; 

#define  _BUFFER_SIZE         8000L  // Size (in bytes) of source and destination buffers.
#define  _FORMAT_V8_INDEX     4      // Index for the default destination format.

// Error codes for CONVERT.C
//
enum
{
   _ERROR_RATE_EXCEEDED = 1,       // Rate specified on command line exceeded 229
   _ERROR_NO_COMMAND_LINE_OPTIONS, // No options were found on command line.
   _ERROR_FILE_FIND,               // No source file specified on command line.
   _ERROR_SOURCE_EXTENSION,
   _ERROR_PARSE,
   _ERROR_SAME_FORMAT,
   _ERROR_CONVERT,
   _ERROR_NO_FORMAT
};


typedef struct _formatTableStruct
{
   WORD code;                   // code in header 
   PSTR extension;              // file extension 
   PSTR commandLineStr;         // string used to compare on the command line 
   PSTR endOfDataStr;           // end of data strings used by the ADPCM formats 
} _FORMAT_TABLE;


_FORMAT_TABLE formatTable[] =
{
   {_FORMAT_V8S, ".V8S", "8s",  _NULL               },
   {_FORMAT_V4S, ".V4S", "4s",  "\xFF\xAA\xAA\xAA"  },
   {_FORMAT_V3S, ".V3S", "3s",  "\xFF\xAA\xAA"      },
   {_FORMAT_V2S, ".V2S", "2s",  "\xFF\xAA\xFF\xAA"  },
   {_FORMAT_V8,  ".V8",  "8",   _NULL               },
   {_FORMAT_V8,  ".VMD", _NULL, _NULL               },
   {_FORMAT_V8,  ".VMF", _NULL, _NULL               },
};



// global variables 
//
WORD  sourceFormatIndex      = 0;
WORD  destFormatIndex        = _FORMAT_V8_INDEX;
WORD  rate                   = 0;
BYTE  sourceFileName[80]     = "";
BYTE  destFileName[80]       = "";
BYTE  filePath[80];
BYTE  sourceBuffer  [_BUFFER_SIZE];
BYTE  destBuffer    [_BUFFER_SIZE];

// Structure used by Covox pack and unpack routines.
//
_CONVERT_DATA    cnvrt =
{
   ( LPSTR )sourceBuffer, 0L, 0L,   // source pointer and lengths 
   ( LPSTR )destBuffer,   0L, 0L,   // destination pointer amd lengths 
   _FORMAT_V8,                      // fileFormat 
   _CVX_RATE_DEFAULT,               // sampleRate 
   _SILENCE_3,                      // silenceRange 
   _FALSE,                          // trebleFlag 
   _FALSE                           // noiseFlag 
};

// Function prototypes.
//
WORD  parseCommandLine ( WORD argc, PSTR  argv[] );
WORD  parseFileName    ( PSTR );
WORD  parseSilence     ( PSTR );
WORD  parseRate        ( PSTR );
WORD  parseTreble      ( PSTR );
WORD  parseNoise       ( PSTR );
WORD  parseFormat      ( PSTR );

WORD  findSourceFormat ( PSTR );

// Function to display valid command line options to the screen.
//
VOID  displayCommandLineSyntax( VOID );

// Function to print out error message to screen
//
VOID error( WORD );


// ------------------------------------------------------------------------
// --------------------------   BEGIN MAIN   ------------------------------
VOID main( WORD argc, PSTR argv[] )
{
   WORD   i, j, firstPass;
   BYTE   fileOverwrite;
   WORD   fileRate;
   PSTR   stringPointer;
   HANDLE sourceFileHandle;
   HANDLE destFileHandle;
   LONG   bytesWritten;
   LONG   bytesRead;

   WORD   returnValue = _ERROR_NONE;

   struct find_t findTable;  // Stucture for Microsoft _dos_find functions


   // Get all options from the command line that have been entered
   // by user. If illegal options have been used then display  
   // the legal command line syntax and exit.
   //
   if( parseCommandLine( argc, argv ) )
   {
      displayCommandLineSyntax();

      exit(0);
   }

   // If no file name was given on the command line then display
   // the command line options and exit.
   //
   if( !strlen( sourceFileName ) )
   {
      displayCommandLineSyntax();

      exit(0);
   }

   // Print out say version and ID
   printf( "\nCONVERT Version %s - Covox Inc (c) 1992\n\n", convertVersionString );

   // Find first instance of file specified by user of CONVERT
   //
   if( _dos_findfirst( sourceFileName,_A_NORMAL,&findTable ) )
      error( _ERROR_FILE_FIND );

   // Get path from file name 
   //
   for(i = 0, j = 0 ; ; i++)    // j is latest back slash 
   {
      if(sourceFileName[i] == '\\' || sourceFileName[i] == ':')
         j = i+1;

      if(sourceFileName[i] == '\0')
      {
         filePath[j] = '\0';
         break;
      }

      filePath[i] = sourceFileName[i];
   }

   // Do until all files specifeied have been converted
   //
   do
   {
      // Append file path onto file name.
      //
      strcpy(sourceFileName,filePath);
      strcat(sourceFileName,findTable.name);


      //
      findSourceFormat( sourceFileName );

      // If source and destination have same format then exit.
      //
      if( sourceFormatIndex == destFormatIndex )
         error( _ERROR_SAME_FORMAT );

      // If one of the files is not 8 bit then exit.
      //
      if( formatTable[sourceFormatIndex].code != _FORMAT_V8 &&
          formatTable[destFormatIndex].code != _FORMAT_V8)
         error( _ERROR_NO_FORMAT );

      // Make destination file name 
      //
      strcpy(destFileName,sourceFileName);
      stringPointer = strrchr(destFileName,'.');
      *stringPointer = '\0';
      strcat(destFileName,formatTable[destFormatIndex].extension);

      // Test if file exist.
      //
      if( !cvxFileOpen( destFileName, _OPEN_R_ONLY, &destFileHandle ) )
      {
         cvxFileClose( destFileHandle );

         printf( "File \"%s\" exists, overwrite (Y/N)? ", destFileName );
         fileOverwrite = getch();

         if( toupper( fileOverwrite ) != 'Y' )
            exit(1);
             
         printf("\n");
      }

      printf( "Converting file %s to %s\n\r", sourceFileName, destFileName );

      // Open source file.
      //
      returnValue =  cvxFileOpen( sourceFileName, _OPEN_R_ONLY, &sourceFileHandle );


      if( returnValue )
         error( returnValue );
      else
      {
         // .VMF files need special handing before being treated as .V8 files
         // with no header.
         // 
         if( strstr(sourceFileName,".VMF") != _NULL ) // strip VMF header 
         {
            // Read data from sound file for cvxHeaderParse.
            //
            returnValue = cvxFileRead( sourceFileHandle, 
                                       ( LPSTR )sourceBuffer, 
                                       ( LONG )_HEADER_LENGTH_VMF, 
                                       &bytesRead );

            if( returnValue )
               error( returnValue );

            if(rate == 0)
               fileRate = sourceBuffer[4];
            else
               fileRate = rate;
         }
         else
            fileRate = rate;

         // Create destination file.
         //
         returnValue =  cvxFileCreate( destFileName, _CREATE_NORMAL, &destFileHandle );

         if( returnValue )
            error( returnValue );

         firstPass = 1;
      
         cnvrt.sampleRate        = fileRate;
         cnvrt.sourceLength      = 0L;
         cnvrt.destinationLength = _BUFFER_SIZE;
   
         for ( ; ; )
         {

            // Read data from sound file for cvxHeaderParse.
            //
            returnValue = cvxFileRead( sourceFileHandle, 
                                       (LPSTR)sourceBuffer + cnvrt.sourceLength, 
                                       _BUFFER_SIZE - cnvrt.sourceLength, 
                                       &bytesRead );
            if( returnValue )
               error( returnValue );

            cnvrt.sourceLength += bytesRead;

            if(firstPass)
            {
               if( formatTable[sourceFormatIndex].code == _FORMAT_V8 )
               {
                  cnvrt.bufferFormat = formatTable[destFormatIndex].code;
   
                  if( packFirst( (_CONVERT_DATA far *) &cnvrt ) )
                     error( _ERROR_CONVERT );
   
                  if( cnvrt.sourceUsed == 0L )
                     break;
               }
               else
               {
                  cnvrt.bufferFormat = formatTable[sourceFormatIndex].code;
   
                  if( unpackFirst( (_CONVERT_DATA far *) &cnvrt ) )
                     error( _ERROR_CONVERT );
               }
               firstPass = 0;
            }
            else
            {
               if( formatTable[sourceFormatIndex].code == _FORMAT_V8 )
               {
                  if( packNext( (_CONVERT_DATA far *) &cnvrt ) )
                     error(  _ERROR_CONVERT );
   
                  if( cnvrt.sourceUsed == 0L )
                     break;
               }
               else
               {
                  if( unpackNext( (_CONVERT_DATA far *) &cnvrt ) )
                     error(  _ERROR_CONVERT );
               }
            }
   
            if(cnvrt.destinationFilled == 0L)
               break;
   
            returnValue = cvxFileWrite( destFileHandle, 
                                        ( LPSTR )destBuffer, 
                                        cnvrt.destinationFilled, 
                                        &bytesWritten );
            if( returnValue )
               error( returnValue );

            cnvrt.sourceLength -= cnvrt.sourceUsed;
   
            memmove( sourceBuffer,&sourceBuffer[(( WORD )cnvrt.sourceUsed)],
                     ( ( WORD )cnvrt.sourceLength ) );
         }
   
         // Since pack does not put end of data marker on file, we put it
         // on now if destination is an ADPCM format
         //
         if(formatTable[destFormatIndex].endOfDataStr != _NULL)
         {
            returnValue = cvxFileWrite( destFileHandle, 
                                        ( LPSTR )formatTable[ destFormatIndex ].endOfDataStr, 
                                        ( LONG )strlen(formatTable[ destFormatIndex ].endOfDataStr), 
                                        &bytesWritten );
            if( returnValue )
               error( returnValue );
         }
      }

   }while( _dos_findnext(&findTable) == 0 );
   
}
// ---------------------------    END MAIN    --------------------------------
//----------------------------------------------------------------------------




//
WORD findSourceFormat(PSTR fileName)
{
   WORD i;

   for( i = 0; i < sizeof(formatTable)/sizeof(_FORMAT_TABLE) ; i++)
   {
      if(strstr(fileName,formatTable[i].extension) != 0)
      {
         sourceFormatIndex = i;
         return( _ERROR_NONE );
      }
   }

   return( _ERROR_SOURCE_EXTENSION );
}


// Parse the options entered by user of program.
//
WORD parseCommandLine(WORD argc, PSTR argv[])
{
   BYTE  commandLineToken;
   unsigned short commandLineCount;
   WORD returnValue = _ERROR_NONE;

   static PARSE_INFO convertParse[] =
   {
      { _TRUE,   0  ,parseFileName},
      { _TRUE,  'R' ,parseRate},
      { _TRUE,  'S' ,parseSilence},
      { _FALSE, 'T' ,parseTreble},
      { _FALSE, 'N' ,parseNoise},
      { _TRUE,  'F' ,parseFormat},
   };

   commandLineToken = '/';

   commandLineCount = sizeof( convertParse ) / sizeof(PARSE_INFO);

   SetParse(commandLineToken, commandLineCount, convertParse );

   if( argc < 2 )
      returnValue = _ERROR_NO_COMMAND_LINE_OPTIONS;

   SetParse('/',sizeof(convertParse)/sizeof(PARSE_INFO),convertParse);

   returnValue = Parse(argc, argv);

   return( returnValue );

}

WORD parseFileName(PSTR x)
{
   strcpy( sourceFileName, x );
   return( 0 );
}

WORD parseSilence(PSTR x)
{
   cnvrt.silenceRange = atoi(x);
   return( 0 );
}

WORD parseRate(PSTR x)
{
   rate = atoi( x );

   if( rate > ( BYTE )_CVX_RATE_OUT_MAXIMUM )
   {
      return( _ERROR_RATE_EXCEEDED );
   }

   return( _ERROR_NONE );
}

WORD parseTreble(PSTR x)
{
   cnvrt.trebleFlag = 1;
   return( 0 );
}


WORD parseNoise(PSTR x)
{
   cnvrt.noiseFlag = 1;
   return( 0 );
}

WORD parseFormat(PSTR x)
{

   WORD i;

   for( i = 0; i < sizeof(formatTable)/sizeof(_FORMAT_TABLE) ; i++)
   {
      if(stricmp(x,formatTable[i].commandLineStr) == 0)
      {
         destFormatIndex = i;
         return( _ERROR_NONE );
      }
   }

   return( _ERROR_PARSE );
}

// Display to the screen the legal command line options avaiable.
//
VOID displayCommandLineSyntax( VOID )
{
   WORD  i, j;
   WORD  messageLength;

   static PSTR commandLineSyntaxMessage[] =
   {
      "        CONVERT Version ",
      "",
      " - Covox Inc (c) 1992\n\n",
      "Convert Covox sound files to different formats. Can do packing and unpacking\n",
      "of compressed files. One file must always be 8 bit PCM (.V8,.VMF,.VMD).\n\n",
      "CONVERT <filename> [<switches>]\n",
      "<filename> - Source DOS file name, can have wildcards.\n",
      "             Valid file extension are: V8,V8S,V4S,V3S,V2S,VMD,VMF\n",
      "/Fxx  - New file format: (an S appended has silence encoding)\n",
      "          8 - Eight bit PCM (default)\n",
      "          8S - Eight PCM bit encoding    4S - Four bit ADPCM encoding\n",
      "          3S - Three bit ADPCM encoding  2S - Two bit ADPCM encoding\n",
      "/Rxxx - Where xxx is a different playback rate for the new file.\n",
      "/Sx   - Silence theshold (0-5), default 3 (packing only).\n",
      "/T    - Treble output (unpacking only).\n",
      "/N    - Generate low level noise rather then silence when unpacking\n",
      "        silence encoding (unpacking only).\n\n",
      "CONVERT *.v8 /f4s\n",
      "        Converts all .v8 files in directory to 4 bit ADPCM files with\n",
      "        the same name and new extension .v4s (.v8 files remain the same).\n"
   };


   // Put version into array.
   commandLineSyntaxMessage[ 1 ] = convertVersionString;

   messageLength = sizeof(commandLineSyntaxMessage)/sizeof(PSTR);

   for(i = 0 ; ( (i < messageLength) && (i != 22) ) ; i++)
   {
      printf(commandLineSyntaxMessage[i]);

   }
}

// Function to print out error message to screen and exit to DOS.
//
VOID error( WORD errorNumber )
{
   PSTR   string;

   printf("\nERROR : ");

   if( errorNumber >= 1000 )
   {
      string = cvxGetErrorString( errorNumber );
      printf("%s\n", string);
   }
   else
   {
      switch( errorNumber )
      {
         case _ERROR_NO_COMMAND_LINE_OPTIONS :
            printf( "No options given on command line" );
   
            break;
   
         case _ERROR_FILE_FIND :
            printf( "Can not find file" );
   
            break;
   
         case _ERROR_PARSE :
            printf( "Parse error" );
   
            break;
   
         case _ERROR_SOURCE_EXTENSION :
            printf( "Unknown extension on source" );
   
            break;
   
         case _ERROR_SAME_FORMAT :
            printf( "Source and Destionation have same extension" );
   
            break;
   
         case _ERROR_CONVERT :
            printf( "Converting file" );
   
            break;
   
         case _ERROR_NO_FORMAT :           
            printf( "One of the file formats must be 8 bit PCM" );
   
            break;
   
         case  _ERROR_RATE_EXCEEDED :
            printf( "Maximum rate exceeded" );
   
            break;
      }
   }
   
   printf("\n");

   exit(1);
}
