/* Translate and compile a NetRexx program             */
/*                                                     */
/* use as:  NETREXXC hello                             */
/*                                                     */
/*   which will use the NetRexx translator to          */
/*     translate hello.nrx to hello.java               */
/*   then will use JAVAC to compile hello.java         */
/*                                                     */
/* Each step only takes place if all previous steps    */
/* complete without an error.                          */
/*                                                     */
/* Any NetRexxT (translator) flags may be used, along  */
/* with the following extra flags:                     */
/*                                                     */
/*   -run    = run class after compilation             */
/*   -runpm  = run class as a PM application           */
/*   -keep   = keep the java file after successful     */
/*             compilation (as xxx.java.keep)          */
/*   -noisy  = turn on -diag, -verbosegc, and -verbose */

parse arg args
w=wordpos('-noisy', args)
if w>0 then do; noisy=1;  args=delword(args,w,1); end; else noisy=0
w=wordpos('-run', args)
if w>0 then do; run=1;   args=delword(args,w,1); end; else run=0
w=wordpos('-runpm', args)
if w>0 then do; runpm=1; args=delword(args,w,1); end; else runpm=0
w=wordpos('-keep', args)
if w>0 then do; keep=1;  args=delword(args,w,1); end; else keep=0

/* collect the file specification */
file=''; flags=''
do i=1 to words(args)
  parse var args arg args
  if left(arg, 1)='-' then flags=flags arg
   else file=file arg
  end
file=strip(file)

if words(file)<>1 then do
  parse source . . me .
  if words(file)=0 then say me 'must be given a NetRexx file specification'
                   else say me 'must be given only one NetRexx file specification'
  exit 1; end

filename=filespec('n', file); parse var file fn '.' fe
if fe='' then file=file'.nrx'

do queued(); parse pull .; end
'@dir' file '/n /b 2>nul | rxqueue'     /* use DIR to get exact case */
if queued()<>1 then do
  if queued()=0 then say 'Cannot find file:' file
  if queued()>1 then do; do queued(); parse pull .; end
    say 'File "'file'" is not a unique specification'
    end
  exit 1; end
parse pull file                         /* is now correct case */
parse var file fn '.' fe

fsource=file
fjava  =fn'.java'
fclass =fn'.class'

numeric digits 20 /* for date comparisons */

/* ----- Translate step ----- */
if (filedate(fjava)>filedate(fsource) | (filedate(fjava)=0 & \keep)),
 & filedate(fclass)>filedate(fsource) then rc=0  /* no translate needed */
 else do
  parse version . level .
  if level>4 then do
    say 'Sorry,' me 'must be run with the standard Rexx interpreter as default'
    exit 1; end
  /* say '  Translating' fsource '...' */
  '@netrexxt' fsource flags
  end

/* ----- Compile step ----- */
if rc=0 then do
  if filedate(fclass)>filedate(fjava) then rc=0   /* no compile needed */
   else do
    javacopts='-ms4m -verbosegc'
    if noisy then javacopts=javacopts '-diag -ms2m -verbose -verbosegc -noasyncgc'
    javacclass='sun.tools.javac.Main'
    say 'Compiling' fjava 'to make' fclass '...'
    '@java' javacopts javacclass fjava
    javacrc=rc
    if keep then '@copy'  fjava fjava'.keep'
    '@erase' fjava
    /* Fix the primary class file to show correct origin */
    if javacrc=0 then call fixclass fclass, fsource
    rc=javacrc
    end
  end

/* ----- Run step ----- */
if rc=0 & (run | runpm) then do
  opts=''
  if noisy then opts='-diag -ms1m -verbosegc -verbose -noasyncgc'
  if runpm then exe='javapm'; else exe='java'
  say 'Running' fn '...'
  '@'exe opts fn args
  end

if rc<>0 then say '+++ RC='rc '+++'
exit rc

/* Filedate as all-digits
 * Arg1 is the name of the file to date
 * Returns file date as a pseudo-number, or 0 if file not found
 */
filedate: procedure
  arg file
  dt=stream(file, 'c', 'query datetime')
  if dt='' then return 0
  parse var dt date time .
  parse var date mon'-'day'-'year
  parse var time hh':'mm':'sec
  value=year''mon''day''hh''mm''sec
  /* say 'FD of' file 'is' value */
  return value

/* Fix a class file
 * Arg1 is the name of the class file
 * Arg2 is the name of the source file
 * Returns 1 if fixed OK, 0 if no change was made
 */
fixclass: procedure
  parse arg file, source

  filelen=stream(file, 'c', 'query size')
  if filelen='' then return 0
  cdata=charin(file, 1, filelen); call charout file

  parse var cdata magic +4 minor +2 major +2 cpcount +2
  if c2x(magic)\='CAFEBABE' then return 0

  /* Scan the constants, record the source string constant index */
  cons=c2d(cpcount)-1      /* number of constants */
  sourceindex=0            /* filename -- not found yet */
  sfindex=0                /* attribute keyword -- not found yet */
  start=11  /* first constant */
  do i=1 to cons
    parse var cdata =(start) tag +1 ctwo +2
    tag=c2d(tag)
    select                                /* calc len of constant */
      when tag=7  then len=3
      when tag=9  then len=5
      when tag=10 then len=5
      when tag=11 then len=5
      when tag=8  then len=3
      when tag=3  then len=5
      when tag=4  then len=5
      when tag=5  then do; len=9; i=i+1; end
      when tag=6  then do; len=9; i=i+1; end
      when tag=12 then len=5
      when tag=2  then len=c2d(ctwo)*2+3
      when tag=1  then do
        len=c2d(ctwo)+3
        string=substr(cdata, start+3, len-3)
        if sourceindex=0 then if string==source then sourceindex=i
        if sfindex=0 then if string=='SourceFile' then sfindex=i
        end
      otherwise return 0   /* bad tag */
      end
    start=start+len
    end i
  /* say 'Constants skipped, now at:' start */
  if sfindex=0 then return 0     /* Attribute name not found */
  if sourceindex=0 then return 0 /* Source constant field not found */

  /* Now, instead of parsing the rest of the class file, we assume the
     sourcefile attribute is at the back of the class.  We modify the
     attribute pointer rather than the actual string, as the string may be
     a shared constant */

  finatt=right(cdata,8)
  parse var finatt name +2 len +4 value +2
  name=c2d(name)
  len=c2d(len)
  value=c2d(value)
  if len\=2 then return 0            /* badly wrong */
  if name\=sfindex then return 0     /* not 'SourceFile' attribute */

  cdata=left(cdata, filelen-2)||d2c(sourceindex, 2)
  call charout file, cdata, 1
  call charout file
  return 1

