#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include "stm32f103xb.h"
#include "mapleComms.h"
#include "mapleWire.h"
#include "printf.h"


//B15 = single soldered wire
//A3  = CN3.1 (RX)
//A2  = CN3.2 (TX)
//A5 = B13 = CN4.2
//GND  = CN4.3
//B14  = CN4.4
//B0   = CN4.5
//A10  = CN4.6
//B3   = JP1.1
//B12  = must be floating as our board has it wired to B14


static uint64_t mTicks = 0;

void SysTick_Handler(void)
{
	mTicks += 0x01000000;
}

static uint64_t getTime(void)
{
	uint64_t hi;
	uint32_t lo;
	
	do {
		hi = mTicks;
		lo = SysTick->VAL;
		asm volatile("":::"memory");
	} while (hi != mTicks);
	
	return hi + (0x01000000 - lo);
}

void prPutchar(char chr)
{
	volatile uint32_t *addr = (volatile uint32_t*)0x20004ffc;
	
	while(addr[0] & 0x80000000ul);
	addr[0] = 0x80000000ul | (uint8_t)chr;
}

static void initHw(void)
{
	//setup clock tree for 80MHz speeds everywhere except APB1 (40MHz), enable clocks to periphs we care about
	//to get this high speed, we need an external crystal on this chip...
	{
		uint32_t cfgrVal = 0x00218400ul;		//pll on at hse * 10 = 80mhz (most we can do and then some)
	
		//start HSE and wait for it
		RCC->CR |= 0x00010000;
		while (!(RCC->CR & 0x00010000));
	
		//before we select the pll as the clock, set proper wait states on flash
		FLASH->ACR = 0x11;						//all prefetch types on, 2 wait states
		
		//configure pll and let it start
		RCC->CFGR = cfgrVal;
		RCC->CR |= 0x01000000;					//pll on, hsi on
		while(!(RCC->CR & 0x02000000));			//wait for pll on
		
		RCC->CFGR = cfgrVal | 2;				//switch to pll clock source
	}
	
	//setup systick for timing analysis (in debug only)
	{
		uint32_t i;
		
		SysTick->CTRL = 0;
		SysTick->LOAD = 0x00ffffff;
		SysTick->VAL = 0x00ffffff;
		SysTick->CTRL = SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_TICKINT_Msk | SysTick_CTRL_ENABLE_Msk;
		NVIC_SetPriority(SysTick_IRQn, 1);	

		//set priorities for all other ints to be less important to make sure we can always get proper time in any irq handler
		for (i = 0; i <= USBWakeUp_IRQn; i++)
			NVIC_SetPriority(i, 2);
	}
}

uint32_t swap32(uint32_t t)
{
	return (t >> 24) | (t << 24) | ((t & 0xff00) << 8) | ((t >> 8) & 0xff00);
}

static void stuff(void)
{
	uint8_t buf[1029 + 1 + 5] = {0};	//max valid length plus one, plus 5, times two
	int32_t len, i, crc = 0;


	asm volatile ("cpsid i");

	mapleCommsStart(MAPLE_CMD_REQ_DEV_NFO, 0x40, 0x41, 0);
	mapleCommsFinish();
	pr("did requested...waiting for timout of reply\n");

	//test image for screen
	static const uint8_t img[] = {
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc1, 0xff, 0xff, 0xbf, 0xff, 0xff,
		0xde, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0x74, 0x63, 0xb0, 0xd2, 0xfb,
		0xdf, 0xb3, 0x9d, 0xbb, 0xcf, 0x77, 0xdf, 0xb7, 0xbd, 0xbb, 0xdf, 0x77,
		0xdf, 0xb7, 0xbd, 0xbb, 0xdf, 0x77, 0xdf, 0xb7, 0xbd, 0xbb, 0xdf, 0xaf,
		0xdf, 0x77, 0xbd, 0xbb, 0xdf, 0xaf, 0xde, 0x77, 0xbd, 0xbb, 0xdf, 0xaf,
		0xc1, 0xf7, 0xbd, 0xbc, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
	};

	i = mapleCommsWriteScreen(0x41, img);
	pr("scr wr = 0x%08x\n", i);

	{
		uint16_t yr;
		uint8_t mon, day, hr, min, sec;
		
		mapleCommsGetClock(0x41, &yr, &mon, &day, &hr, &min, &sec);
		pr("TIME: %02u:%02u:%02u on %02u/%02u/%04u\n", hr, min, sec, mon, day, yr);
		
		mapleCommsSetClock(0x41, 1988, 3, 28, 20, 3, 0);
		pr("time set now\n");
		
		mapleCommsGetClock(0x41, &yr, &mon, &day, &hr, &min, &sec);
		pr("TIME: %02u:%02u:%02u on %02u/%02u/%04u\n", hr, min, sec, mon, day, yr);
		
		
		//mapleCommsSetBuzzer(0x41, 255, 100);
		
	}


	//game data
	static const uint8_t gamedata[] = {
  0x21, 0x20, 0x3d, 0xd9, 0x5d, 0xe8, 0xfe, 0xb0, 0x00, 0x00, 0x00, 0xdd,
  0x5d, 0xe9, 0xfe, 0xb0, 0x00, 0x00, 0x00, 0x28, 0x5f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x28, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdb,
  0x10, 0xe9, 0xff, 0xb0, 0x00, 0x00, 0x00, 0x28, 0x50, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xd8, 0x30, 0xec, 0xfe, 0xb0, 0x00, 0x00, 0x00, 0xd8,
  0x34, 0xed, 0xfe, 0xb0, 0x00, 0x00, 0x00, 0xda, 0x61, 0xee, 0xfe, 0xb0,
  0x00, 0x00, 0x00, 0xd8, 0x4e, 0xef, 0xfe, 0xb0, 0x99, 0x18, 0x04, 0xd9,
  0x18, 0xea, 0xff, 0x9b, 0x18, 0x04, 0xdb, 0x18, 0xeb, 0xff, 0xb0, 0x99,
  0x10, 0x04, 0xd9, 0x10, 0xe8, 0xff, 0x99, 0x5e, 0x04, 0xd9, 0x5e, 0xea,
  0xfe, 0xb0, 0x61, 0x08, 0xdf, 0x08, 0x61, 0x00, 0x61, 0x7f, 0xb8, 0x0d,
  0x21, 0x01, 0x30, 0x71, 0x08, 0x9a, 0x5e, 0x04, 0xda, 0x5e, 0xeb, 0xfe,
  0x71, 0x00, 0x99, 0x00, 0x04, 0xd9, 0x7f, 0xec, 0xff, 0x9b, 0x00, 0x04,
  0xdb, 0x7f, 0xed, 0xff, 0x71, 0x00, 0xb0, 0xb8, 0x0d, 0x21, 0xe0, 0x24,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xb8, 0x0d, 0x21, 0x01, 0x00, 0xa0, 0x00, 0x00,
  0x20, 0x00, 0x97, 0xa0, 0x00, 0x00, 0x00, 0x00, 0xb8, 0x0d, 0x21, 0x01,
  0x10, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xb8, 0x0d, 0x21, 0x01, 0x20, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x28, 0x7b, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb8, 0x0d, 0x21, 0x01,
  0x40, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xb8, 0x0d, 0x21, 0x01, 0xf0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x75, 0x4d, 0x32, 0x33,
  0x20, 0x6c, 0x61, 0x75, 0x6e, 0x63, 0x68, 0x65, 0x72, 0x20, 0x20, 0x20,
  0x56, 0x4d, 0x55, 0x20, 0x43, 0x2d, 0x4d, 0x32, 0x33, 0x20, 0x65, 0x6d,
  0x75, 0x6c, 0x61, 0x74, 0x6f, 0x72, 0x20, 0x62, 0x79, 0x20, 0x44, 0x6d,
  0x69, 0x74, 0x72, 0x79, 0x2e, 0x47, 0x52, 0x20, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x9e, 0xf7,
  0x08, 0x42, 0xe7, 0x39, 0x1c, 0xe7, 0x9a, 0xd6, 0x18, 0xc6, 0x96, 0xb5,
  0x14, 0xa5, 0x10, 0x84, 0x8e, 0x73, 0x8a, 0x52, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
  0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x10, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x01, 0x00, 0x10, 0x01, 0x01,
  0x44, 0x8a, 0xb9, 0x54, 0x40, 0x10, 0x00, 0x10, 0x11, 0x00, 0x00, 0x01,
  0x11, 0x11, 0x01, 0x41, 0x6a, 0xde, 0xee, 0xb8, 0x11, 0x10, 0x00, 0x00,
  0x10, 0x01, 0x00, 0x01, 0x00, 0x00, 0x11, 0x18, 0xbe, 0xa7, 0x23, 0xec,
  0x86, 0x10, 0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x16, 0x9c,
  0xe2, 0x94, 0x82, 0x2d, 0xd9, 0x65, 0x00, 0x11, 0x10, 0x11, 0x00, 0x00,
  0x00, 0x15, 0x7a, 0xdd, 0x22, 0xb4, 0x5a, 0x22, 0xde, 0xb6, 0x44, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x04, 0x47, 0xcd, 0x29, 0x8c, 0xc6, 0xa1, 0xb2,
  0x3d, 0xdc, 0x84, 0x11, 0x10, 0x00, 0x00, 0x01, 0x15, 0x92, 0xd2, 0x27,
  0x15, 0x99, 0x9a, 0x82, 0x33, 0xde, 0xc8, 0x61, 0x01, 0x11, 0x00, 0x14,
  0x7b, 0xed, 0x22, 0x2b, 0x59, 0x75, 0x7c, 0x22, 0x23, 0xd3, 0xde, 0x96,
  0x51, 0x00, 0x04, 0x68, 0xbe, 0x22, 0x22, 0x98, 0xb0, 0xbb, 0x92, 0xc2,
  0x2d, 0xd3, 0xdd, 0xeb, 0x64, 0x40, 0x55, 0x8d, 0xe3, 0x32, 0x22, 0xc6,
  0x9a, 0x4b, 0x22, 0x22, 0x2d, 0xdd, 0x3d, 0xdd, 0xc8, 0x41, 0x68, 0xcd,
  0x22, 0x23, 0x98, 0xcb, 0x0a, 0xb2, 0x2c, 0xc9, 0x92, 0xdd, 0x3d, 0xdd,
  0xeb, 0x76, 0x57, 0xbd, 0x32, 0x33, 0xc7, 0x9c, 0x7a, 0x23, 0xc2, 0x89,
  0x6b, 0xdd, 0x3d, 0xdd, 0xd2, 0x97, 0x66, 0x8f, 0xe2, 0x22, 0x3b, 0x56,
  0xa2, 0xc2, 0x33, 0x53, 0x70, 0x7d, 0xdd, 0xdd, 0xfb, 0x76, 0x57, 0x99,
  0xbf, 0xd2, 0x2c, 0xcb, 0x2c, 0x22, 0xab, 0xad, 0x92, 0x6c, 0xdd, 0xe2,
  0xc8, 0x66, 0x78, 0x86, 0xab, 0x2d, 0x22, 0x22, 0x22, 0x27, 0x85, 0xbd,
  0xbc, 0x72, 0xdd, 0xfb, 0x69, 0x75, 0x67, 0x79, 0x96, 0xce, 0x33, 0x22,
  0x22, 0xb8, 0x37, 0xbc, 0x76, 0xbd, 0xf2, 0x89, 0x88, 0x87, 0x05, 0x78,
  0x79, 0x98, 0xfe, 0xd3, 0x22, 0xc8, 0xd7, 0xbb, 0xcd, 0xef, 0xaa, 0x88,
  0x96, 0x64, 0x00, 0x46, 0x89, 0x79, 0xab, 0xfd, 0x32, 0x3d, 0xd6, 0xa5,
  0xbf, 0xcc, 0x87, 0xa7, 0x76, 0x10, 0x00, 0x04, 0x67, 0x8a, 0x5a, 0xc2,
  0xed, 0xd3, 0x31, 0x5c, 0xdd, 0xa6, 0xa8, 0x87, 0x50, 0x00, 0x00, 0x00,
  0x06, 0x87, 0x99, 0x5c, 0xe3, 0xdd, 0xd6, 0xbf, 0xc7, 0xa8, 0x88, 0x61,
  0x10, 0x00, 0x00, 0x00, 0x00, 0x47, 0x87, 0x99, 0x9e, 0xfe, 0xef, 0xf9,
  0xa8, 0x89, 0x75, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x68, 0x96,
  0x9a, 0xbf, 0xfb, 0xb8, 0x7a, 0x76, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x01, 0x56, 0x78, 0x95, 0x93, 0xc8, 0x79, 0x77, 0x75, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x68, 0x77, 0x99, 0x88, 0x78,
  0x86, 0x00, 0x01, 0x11, 0x10, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
  0x77, 0x66, 0x67, 0x76, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x01, 0x11, 0x11, 0x11, 0x44, 0x01, 0x00, 0x01, 0x10,
  0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x01, 0x10, 0x00,
  0x00, 0x00, 0x11, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
  0xfc, 0x63, 0x23, 0x06, 0xc2, 0x23, 0x01, 0x00, 0xe1, 0x00, 0x12, 0x04,
  0x12, 0x05, 0x12, 0x06, 0x12, 0x07, 0x12, 0xfe, 0x12, 0xff, 0x12, 0xc1,
  0x12, 0xc2, 0x23, 0x03, 0xb4, 0x23, 0x02, 0x04, 0x23, 0x00, 0x04, 0x20,
  0x16, 0xb6, 0x22, 0x04, 0x04, 0x23, 0x03, 0xbc, 0x23, 0x02, 0x04, 0x23,
  0x00, 0x04, 0x20, 0x16, 0xb6, 0xc8, 0xbc, 0x02, 0xbf, 0x41, 0xff, 0x03,
  0x21, 0x14, 0x3f, 0x68, 0xc0, 0x1e, 0x02, 0xfe, 0xe2, 0xc1, 0x80, 0x09,
  0x23, 0x02, 0x00, 0x20, 0x14, 0xf6, 0x21, 0x13, 0xc4, 0x02, 0xff, 0xe2,
  0xc2, 0x80, 0x09, 0x23, 0x02, 0x08, 0x20, 0x14, 0xf6, 0x21, 0x13, 0xc4,
  0x23, 0x03, 0x04, 0x20, 0x06, 0x00, 0x02, 0x05, 0xe1, 0xfe, 0x61, 0x00,
  0x23, 0x00, 0x05, 0x61, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2e, 0x1b, 0x2e, 0x1b,
  0x2e, 0x1b, 0x2e, 0x1b, 0x2e, 0x41, 0x2e, 0x41, 0x2e, 0x41, 0x2e, 0x41,
  0x2e, 0x97, 0x2e, 0x97, 0x2e, 0x97, 0x2e, 0x97, 0x2e, 0xf7, 0x2f, 0x24,
  0x2f, 0x38, 0x2f, 0x4b, 0x2f, 0x5e, 0x2f, 0x5e, 0x2f, 0x5e, 0x2f, 0x5e,
  0x2f, 0x7c, 0x2f, 0x7c, 0x2f, 0x7c, 0x2f, 0x7c, 0x2f, 0x8c, 0x2f, 0x8c,
  0x2f, 0x8c, 0x2f, 0x8c, 0x2f, 0x9b, 0x2f, 0x9b, 0x2f, 0x9b, 0x2f, 0x9b,
  0x2f, 0xaa, 0x2f, 0xc0, 0x3a, 0x4f, 0x3a, 0x8e, 0x3e, 0x8a, 0x3e, 0x8a,
  0x3e, 0x8a, 0x3e, 0x8a, 0x3e, 0xa9, 0x3e, 0xa1, 0x3e, 0x99, 0x3f, 0x11,
  0x3f, 0x00, 0x3e, 0xde, 0x3e, 0xbe, 0x3f, 0x35, 0x3e, 0x6b, 0x3e, 0x6b,
  0x3e, 0x6b, 0x3e, 0x6b, 0x3e, 0x29, 0x3e, 0x29, 0x3e, 0x29, 0x3e, 0x29,
  0x3e, 0x38, 0x3e, 0x38, 0x3e, 0x38, 0x3e, 0x38, 0x3d, 0xe3, 0x3d, 0xe3,
  0x3d, 0xe3, 0x3d, 0xe3, 0x3e, 0x5f, 0x3e, 0x5f, 0x3e, 0x5f, 0x3e, 0x5f,
  0x3e, 0x1b, 0x3e, 0x1b, 0x3e, 0x1b, 0x3e, 0x1b, 0x3e, 0x78, 0x3e, 0x78,
  0x3e, 0x78, 0x3e, 0x78, 0x3e, 0x81, 0x3e, 0x81, 0x3e, 0x81, 0x3e, 0x81,
  0x3a, 0xeb, 0x3a, 0xeb, 0x3a, 0xeb, 0x3a, 0xeb, 0x3b, 0x06, 0x3b, 0x06,
  0x3b, 0x06, 0x3b, 0x06, 0x3b, 0x17, 0x3b, 0x43, 0x3f, 0xb3, 0x3f, 0x9a,
  0x3b, 0xa0, 0x3b, 0xad, 0x3f, 0xa7, 0x3f, 0x5b, 0x3f, 0xec, 0x3f, 0xec,
  0x3f, 0xec, 0x3f, 0xec, 0x3f, 0xf6, 0x3f, 0xf6, 0x3f, 0xf6, 0x3f, 0xf6,
  0x3d, 0x95, 0x3d, 0xae, 0x3d, 0xb6, 0x3d, 0xbe, 0x3d, 0xc6, 0x3d, 0xd4,
  0x3d, 0xd1, 0x3f, 0x75, 0x3d, 0x4c, 0x3d, 0x4c, 0x3d, 0x4c, 0x3d, 0x4c,
  0x2e, 0x12, 0x3f, 0x94, 0x3f, 0x94, 0x3f, 0x94, 0x2e, 0x12, 0x2e, 0x12,
  0x2e, 0x12, 0x2e, 0x12, 0x3f, 0x94, 0x3c, 0x2a, 0x3f, 0x94, 0x3f, 0x94,
  0x23, 0x00, 0xbc, 0x23, 0x02, 0x02, 0x20, 0x16, 0xb6, 0x23, 0x00, 0x02,
  0x22, 0x00, 0xbc, 0x21, 0x13, 0x0a, 0x23, 0x03, 0x06, 0x20, 0x06, 0x00,
  0x21, 0x13, 0x2b, 0x20, 0x10, 0x8f, 0x12, 0x00, 0x20, 0x10, 0x98, 0x12,
  0x01, 0x20, 0x11, 0x5c, 0x20, 0x11, 0x49, 0x80, 0x11, 0x13, 0x02, 0x02,
  0x00, 0xa1, 0x04, 0x12, 0x00, 0x20, 0x12, 0xab, 0x53, 0x02, 0xf4, 0x20,
  0x11, 0x91, 0x21, 0x10, 0xce, 0x20, 0x10, 0x8f, 0x12, 0x00, 0x20, 0x10,
  0x98, 0x12, 0x01, 0x20, 0x11, 0x49, 0x80, 0x27, 0x13, 0x02, 0x20, 0x11,
  0x5c, 0x23, 0x03, 0x04, 0xdf, 0x01, 0x72, 0x00, 0x04, 0xd0, 0x14, 0x53,
  0x03, 0xf8, 0x53, 0x02, 0x03, 0x21, 0x06, 0x71, 0x02, 0x00, 0x81, 0x04,
  0x12, 0x00, 0x21, 0x06, 0x55, 0x20, 0x11, 0x91, 0x21, 0x10, 0xc8, 0xe1,
  0x00, 0x23, 0x02, 0x04, 0x14, 0x62, 0x00, 0x53, 0x02, 0xfa, 0x02, 0x01,
  0x81, 0x03, 0x12, 0x01, 0x05, 0xcf, 0xc0, 0x9f, 0x00, 0x02, 0xef, 0xc0,
  0xce, 0xc0, 0xed, 0xc0, 0x21, 0x04, 0xb7, 0x20, 0x10, 0x8f, 0x12, 0x00,
  0x20, 0x10, 0x98, 0x12, 0x01, 0x20, 0x11, 0x49, 0x80, 0x2b, 0x13, 0x02,
  0x20, 0x11, 0x5c, 0x72, 0x00, 0x04, 0xf0, 0x23, 0x03, 0x04, 0x04, 0xd0,
  0x14, 0x72, 0x00, 0x53, 0x03, 0xf8, 0x53, 0x02, 0x03, 0x21, 0x06, 0xc9,
  0x02, 0x00, 0x81, 0x05, 0x12, 0x00, 0x21, 0x06, 0xab, 0x20, 0x11, 0x91,
  0x62, 0x00, 0x21, 0x10, 0xc8, 0x20, 0x11, 0x36, 0x05, 0x9f, 0x00, 0x0b,
  0xcd, 0xc0, 0xee, 0xc0, 0xef, 0xc0, 0xd1, 0xff, 0x21, 0x06, 0xeb, 0xed,
  0xc0, 0xce, 0xc0, 0xcf, 0xc0, 0xe1, 0x00, 0x23, 0x02, 0x04, 0x14, 0x62,
  0x00, 0x53, 0x02, 0xfa, 0x21, 0x04, 0xb7, 0x20, 0x12, 0xe6, 0xdf, 0x01,
  0x22, 0x02, 0x02, 0x22, 0x0d, 0x04, 0x04, 0x62, 0x00, 0x95, 0x62, 0x01,
  0x16, 0x62, 0x02, 0x52, 0x0d, 0xf4, 0x20, 0x11, 0x91, 0x20, 0x11, 0x99,
  0x02, 0x03, 0x12, 0x00, 0x72, 0x02, 0x72, 0x02, 0x72, 0x02, 0x20, 0x11,
  0x82, 0x21, 0x10, 0xce, 0x20, 0x12, 0xe6, 0xdf, 0x01, 0x22, 0x02, 0x02,
  0x20, 0x12, 0x99, 0x20, 0x11, 0x8f, 0x20, 0x11, 0xae, 0x21, 0x07, 0x14,
  0x20, 0x10, 0x98, 0x12, 0x00, 0x20, 0x10, 0x8f, 0x12, 0x03, 0x20, 0x11,
  0x3d, 0x20, 0x11, 0x74, 0x21, 0x06, 0xfa, 0x20, 0x10, 0x98, 0x12, 0x00,
  0x20, 0x10, 0x8f, 0x12, 0x03, 0x20, 0x11, 0x3d, 0x20, 0x11, 0x74, 0x21,
  0x07, 0x27, 0x20, 0x10, 0xae, 0x12, 0x00, 0xce, 0xc0, 0xcd, 0xc0, 0x02,
  0x04, 0x90, 0x02, 0xed, 0xc0, 0x14, 0xe1, 0x00, 0x23, 0x02, 0x03, 0x62,
  0x00, 0x14, 0x53, 0x02, 0xfa, 0x21, 0x04, 0xb7, 0x20, 0x10, 0xae, 0x12,
  0x00, 0x02, 0x04, 0x20, 0x11, 0x74, 0x22, 0x03, 0x04, 0x21, 0x07, 0x27,
  0x20, 0x10, 0xae, 0x12, 0x00, 0x12, 0x03, 0x02, 0x04, 0x20, 0x11, 0x74,
  0x21, 0x06, 0xfa, 0x20, 0x10, 0xae, 0x12, 0x00, 0x12, 0x03, 0x02, 0x04,
  0x20, 0x11, 0x74, 0x21, 0x07, 0x27, 0x20, 0x12, 0xcd, 0x20, 0x11, 0xc3,
  0x2f, 0xd6, 0x2f, 0xed, 0x39, 0x68, 0x39, 0x91, 0x39, 0xce, 0x3a, 0x39,
  0x3a, 0x43, 0x39, 0xea, 0x20, 0x12, 0xcd, 0x20, 0x11, 0xc3, 0x38, 0x4c,
  0x3a, 0x21, 0x3a, 0x2d, 0x3a, 0x33, 0x38, 0x04, 0x38, 0x67, 0x38, 0x1b,
  0x38, 0x34, 0x23, 0x02, 0x04, 0x23, 0x03, 0x00, 0x04, 0xe5, 0x14, 0xd3,
  0x03, 0x13, 0x03, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf2, 0x21, 0x10,
  0x7a, 0x23, 0x02, 0x04, 0x23, 0x03, 0x00, 0x04, 0xf5, 0x14, 0xd3, 0x03,
  0x13, 0x03, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf2, 0x21, 0x10, 0x7a,
  0x23, 0x02, 0x04, 0x23, 0x03, 0x00, 0x04, 0xd5, 0x14, 0xd3, 0x03, 0x13,
  0x03, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf2, 0x21, 0x10, 0x7a, 0x23,
  0x02, 0x04, 0x23, 0x03, 0x00, 0x05, 0xf1, 0xff, 0xe4, 0x14, 0xd3, 0x03,
  0x13, 0x03, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf0, 0x21, 0x10, 0x7a,
  0x23, 0x02, 0x04, 0x23, 0x03, 0x00, 0x05, 0xf1, 0xff, 0x14, 0xd3, 0x03,
  0x13, 0x03, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf1, 0x21, 0x10, 0x7a,
  0x23, 0x02, 0x04, 0x23, 0x03, 0x00, 0x04, 0xe5, 0x12, 0x0d, 0xd3, 0x03,
  0x13, 0x03, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf1, 0x22, 0x00, 0x0e,
  0x21, 0x10, 0x7a, 0x02, 0x00, 0x12, 0x02, 0x60, 0x00, 0x62, 0x00, 0x62,
  0x00, 0x04, 0x62, 0x00, 0xd4, 0x70, 0x00, 0x90, 0x75, 0x60, 0x01, 0x62,
  0x01, 0x62, 0x01, 0x05, 0x62, 0x01, 0xd5, 0x70, 0x01, 0x90, 0x67, 0x62,
  0x00, 0x04, 0x72, 0x00, 0x80, 0x40, 0x62, 0x01, 0x05, 0x72, 0x01, 0x80,
  0x3f, 0x05, 0x13, 0x02, 0x04, 0x13, 0x03, 0x62, 0x00, 0x04, 0x30, 0x12,
  0x05, 0x03, 0x03, 0x12, 0x04, 0x03, 0x02, 0x12, 0x06, 0x62, 0x01, 0x72,
  0x00, 0x05, 0x13, 0x02, 0x04, 0x13, 0x03, 0x62, 0x00, 0x04, 0x30, 0x61,
  0x00, 0x03, 0x03, 0x82, 0x05, 0x12, 0x05, 0x71, 0x00, 0x92, 0x06, 0x12,
  0x06, 0x03, 0x02, 0x91, 0x00, 0x12, 0x07, 0x21, 0x09, 0x50, 0x02, 0x00,
  0xc2, 0x01, 0x12, 0x00, 0x05, 0x13, 0x02, 0x04, 0x13, 0x03, 0x62, 0x00,
  0x04, 0x30, 0x12, 0x05, 0x03, 0x03, 0x12, 0x04, 0x03, 0x02, 0x12, 0x06,
  0x22, 0x07, 0x00, 0x21, 0x09, 0x50, 0xe1, 0x00, 0x12, 0x0b, 0x20, 0x11,
  0x69, 0x22, 0x0c, 0x00, 0x02, 0x0b, 0x82, 0x0c, 0x12, 0x03, 0xa1, 0x04,
  0x9f, 0x01, 0x3f, 0x02, 0x02, 0x82, 0x0c, 0x12, 0x00, 0x05, 0x80, 0x36,
  0x13, 0x02, 0x04, 0x13, 0x03, 0x62, 0x00, 0x04, 0x30, 0x12, 0x0d, 0x02,
  0x03, 0x81, 0x04, 0x12, 0x00, 0x04, 0x83, 0x03, 0x14, 0x62, 0x03, 0x6a,
  0x03, 0x1c, 0x62, 0x00, 0x04, 0x92, 0x0d, 0x14, 0x62, 0x03, 0x6a, 0x03,
  0x11, 0x62, 0x00, 0x04, 0x93, 0x02, 0x14, 0x62, 0x03, 0x6a, 0x03, 0x06,
  0x62, 0x00, 0x04, 0x91, 0x00, 0x14, 0x62, 0x0c, 0x62, 0x0c, 0x8a, 0x0c,
  0xaf, 0x62, 0x01, 0x62, 0x0b, 0x8a, 0x0b, 0xa5, 0x02, 0x02, 0x12, 0x00,
  0x22, 0x01, 0x04, 0x20, 0x11, 0x5c, 0x02, 0x02, 0x12, 0x00, 0x20, 0x11,
  0x0b, 0x13, 0x03, 0x62, 0x00, 0x21, 0x10, 0x7a, 0x05, 0x31, 0x20, 0x1d,
  0x7f, 0x01, 0x14, 0xcf, 0xc0, 0xe1, 0x00, 0x23, 0x02, 0x04, 0x14, 0x62,
  0x00, 0x53, 0x02, 0xfa, 0xed, 0xc0, 0xce, 0xc0, 0x21, 0x04, 0xb7, 0x20,
  0x11, 0x02, 0x21, 0x06, 0x2b, 0x04, 0xd0, 0x20, 0x11, 0x91, 0x21, 0x09,
  0x71, 0x20, 0x11, 0x04, 0x05, 0x31, 0x20, 0x08, 0x7f, 0x01, 0x18, 0xcf,
  0xc0, 0x21, 0x09, 0xa5, 0x04, 0xf0, 0x20, 0x11, 0x91, 0xe1, 0x00, 0x23,
  0x02, 0x04, 0x14, 0x72, 0x00, 0x53, 0x02, 0xfa, 0x21, 0x09, 0x7c, 0x62,
  0x00, 0x80, 0x07, 0x13, 0x02, 0x21, 0x06, 0x55, 0x62, 0x00, 0x02, 0x00,
  0xa1, 0x04, 0x12, 0x00, 0x20, 0x11, 0x0b, 0x13, 0x03, 0x62, 0x00, 0x21,
  0x10, 0x7a, 0x02, 0x00, 0x13, 0x03, 0x20, 0x11, 0x04, 0x05, 0x80, 0xe4,
  0x13, 0x02, 0xa1, 0x20, 0x9f, 0x01, 0x03, 0x21, 0x06, 0xad, 0x61, 0x03,
  0x04, 0x70, 0x00, 0x21, 0x06, 0xd5, 0x02, 0x00, 0x13, 0x03, 0x20, 0x11,
  0x04, 0x05, 0x80, 0xc8, 0xe1, 0x0f, 0x80, 0x21, 0x13, 0x02, 0x03, 0x03,
  0x12, 0x01, 0x02, 0x00, 0x12, 0x02, 0x05, 0xd0, 0x23, 0x03, 0x04, 0x04,
  0xd0, 0x14, 0x72, 0x00, 0x53, 0x03, 0xf8, 0x02, 0x02, 0x12, 0x00, 0x53,
  0x02, 0xec, 0x21, 0x0a, 0x1b, 0x04, 0xf0, 0x20, 0x11, 0x91, 0x21, 0x09,
  0xbc, 0x02, 0x00, 0x12, 0x03, 0xe1, 0x00, 0x20, 0x11, 0x7b, 0x21, 0x07,
  0x27, 0x22, 0x03, 0x04, 0x21, 0x07, 0x27, 0x22, 0x03, 0x04, 0x21, 0x06,
  0xfa, 0x02, 0x00, 0x12, 0x03, 0x02, 0xc0, 0xf0, 0x21, 0x06, 0xfc, 0x02,
  0x00, 0x12, 0x03, 0x02, 0xc0, 0xf0, 0xbf, 0x01, 0x21, 0x07, 0x29, 0x68,
  0x05, 0x2c, 0x22, 0x0d, 0x00, 0x20, 0x10, 0xea, 0x12, 0x00, 0x12, 0x02,
  0x41, 0xbc, 0x02, 0xe9, 0x0d, 0x20, 0x10, 0xfa, 0x12, 0x01, 0x41, 0xbc,
  0x02, 0xe9, 0x0d, 0x20, 0x11, 0x16, 0x89, 0x0d, 0x0a, 0x02, 0x02, 0x12,
  0x00, 0x23, 0x00, 0x02, 0x20, 0x13, 0x0a, 0x21, 0x04, 0xb7, 0x20, 0x10,
  0xea, 0x12, 0x00, 0x20, 0x10, 0xfa, 0x12, 0x01, 0x22, 0x03, 0x04, 0x21,
  0x07, 0x27, 0x68, 0x05, 0x29, 0x20, 0x10, 0xea, 0x12, 0x00, 0x20, 0x10,
  0xfa, 0x12, 0x01, 0x31, 0xbc, 0x06, 0x20, 0x11, 0x5c, 0x21, 0x04, 0xb7,
  0x23, 0x02, 0x03, 0xdf, 0x01, 0x05, 0x81, 0x02, 0x14, 0x62, 0x00, 0x62,
  0x01, 0x04, 0x95, 0x14, 0x53, 0x02, 0xf6, 0x21, 0x04, 0xb7, 0x02, 0x04,
  0xe0, 0x13, 0x65, 0x20, 0x10, 0xfa, 0x61, 0x00, 0x98, 0x65, 0x0b, 0x22,
  0x01, 0xbc, 0x22, 0x00, 0x04, 0x20, 0x11, 0x5c, 0xe8, 0x04, 0x22, 0x00,
  0xbc, 0x70, 0x01, 0x20, 0x11, 0x5c, 0xc8, 0xbc, 0x98, 0x65, 0x09, 0x22,
  0x01, 0x04, 0x22, 0x00, 0xb8, 0x20, 0x11, 0x5c, 0x21, 0x04, 0xb7, 0x20,
  0x10, 0xae, 0x12, 0x00, 0x60, 0x00, 0x20, 0x12, 0x69, 0xf9, 0x02, 0x22,
  0x01, 0xbc, 0x20, 0x12, 0x84, 0x70, 0x00, 0x04, 0xe1, 0xfc, 0x14, 0x21,
  0x04, 0xb7, 0x20, 0x10, 0xae, 0x12, 0x00, 0x22, 0x01, 0xb4, 0x20, 0x12,
  0x69, 0x20, 0x12, 0x84, 0x21, 0x04, 0xb7, 0x68, 0x05, 0x63, 0x02, 0x04,
  0x13, 0x02, 0xf0, 0xf0, 0xe1, 0xfc, 0x12, 0x04, 0xe1, 0x00, 0x12, 0x06,
  0x12, 0x07, 0x91, 0x00, 0x12, 0x05, 0x22, 0x01, 0x04, 0x22, 0x00, 0xb4,
  0x7f, 0x02, 0x06, 0x20, 0x11, 0x16, 0x21, 0x04, 0xb7, 0x20, 0x11, 0x26,
  0x21, 0x04, 0xb7, 0x68, 0x05, 0x37, 0x20, 0x10, 0x8f, 0x12, 0x00, 0x20,
  0x10, 0x98, 0x12, 0x01, 0x23, 0x03, 0x01, 0x23, 0x02, 0x03, 0x6e, 0x04,
  0x04, 0x63, 0x03, 0x73, 0x02, 0x05, 0x14, 0x62, 0x00, 0x62, 0x01, 0x53,
  0x03, 0xf7, 0xaf, 0x04, 0xe2, 0x04, 0x7f, 0x00, 0x05, 0xe1, 0x00, 0x21,
  0x0b, 0x74, 0xd1, 0xff, 0x14, 0x62, 0x00, 0x53, 0x02, 0xfa, 0x21, 0x04,
  0xb7, 0x20, 0x10, 0x8f, 0x12, 0x00, 0x20, 0x11, 0x0b, 0x80, 0x03, 0x21,
  0x04, 0xb7, 0x02, 0x04, 0xc0, 0xc0, 0xe1, 0x3e, 0x89, 0x05, 0x02, 0xd1,
  0x40, 0x81, 0x02, 0x22, 0x00, 0xbc, 0x20, 0x13, 0x0a, 0x21, 0x04, 0xb7,
  0x20, 0x10, 0x8f, 0x12, 0x00, 0x20, 0x11, 0x0b, 0x90, 0xe0, 0x21, 0x04,
  0xb7, 0x68, 0x05, 0xf0, 0x20, 0x10, 0x8f, 0x12, 0x00, 0x20, 0x10, 0x98,
  0x6f, 0x04, 0x4b, 0x6e, 0x04, 0x1e, 0x60, 0x00, 0x12, 0x01, 0x22, 0x00,
  0x04, 0x20, 0x11, 0x5c, 0x70, 0x00, 0x22, 0x01, 0x07, 0x23, 0x02, 0x04,
  0x05, 0x14, 0x62, 0x00, 0x72, 0x01, 0x53, 0x02, 0xf7, 0x21, 0x04, 0xb7,
  0x60, 0x00, 0x12, 0x01, 0x22, 0x00, 0x04, 0x20, 0x11, 0x5c, 0x70, 0x00,
  0x22, 0x01, 0x05, 0x23, 0x03, 0x02, 0x23, 0x02, 0x02, 0x05, 0x14, 0x62,
  0x00, 0x72, 0x01, 0x53, 0x02, 0xf7, 0x02, 0x01, 0x81, 0x04, 0x12, 0x01,
  0x53, 0x03, 0xeb, 0x21, 0x04, 0xb7, 0x60, 0x00, 0x12, 0x01, 0x22, 0x00,
  0x04, 0x20, 0x11, 0x5c, 0x70, 0x00, 0x22, 0x01, 0x05, 0x23, 0x02, 0x02,
  0x05, 0x14, 0x62, 0x00, 0x72, 0x01, 0x53, 0x02, 0xf7, 0x23, 0x02, 0x02,
  0x21, 0x0b, 0x6a, 0x21, 0x15, 0x36, 0x8c, 0x04, 0xfa, 0x60, 0x0e, 0x60,
  0x0f, 0x60, 0x04, 0x02, 0x04, 0x20, 0x12, 0xdf, 0x61, 0x00, 0x23, 0x03,
  0x08, 0x20, 0x06, 0x00, 0x02, 0x08, 0x20, 0x12, 0xdf, 0x12, 0x03, 0x02,
  0x09, 0x20, 0x12, 0xdf, 0x13, 0x02, 0x22, 0x00, 0x0c, 0x70, 0x01, 0x20,
  0x11, 0x5c, 0xd8, 0x65, 0xd8, 0x25, 0x71, 0x00, 0x61, 0x02, 0x7d, 0x00,
  0x15, 0x02, 0x03, 0x12, 0x00, 0x20, 0x11, 0xde, 0x13, 0x65, 0x13, 0x25,
  0x22, 0x00, 0x0c, 0x20, 0x11, 0xde, 0xf3, 0x65, 0x13, 0x65, 0x02, 0x0e,
  0xd2, 0x0f, 0x90, 0x39, 0x02, 0x03, 0x81, 0x03, 0x12, 0x00, 0x04, 0x72,
  0x00, 0xd4, 0x72, 0x00, 0xd4, 0x90, 0x2a, 0x72, 0x00, 0x04, 0x80, 0x18,
  0x13, 0x02, 0x02, 0x0c, 0x13, 0x03, 0x02, 0x0d, 0x40, 0x12, 0x09, 0x03,
  0x03, 0x12, 0x08, 0xe1, 0x00, 0x12, 0x0a, 0x12, 0x0b, 0x21, 0x0d, 0x24,
  0xe1, 0x00, 0x12, 0x08, 0x12, 0x09, 0x12, 0x0a, 0x12, 0x0b, 0x21, 0x0d,
  0x24, 0x22, 0x04, 0x01, 0xe1, 0x00, 0x12, 0x05, 0x12, 0x06, 0x12, 0x07,
  0x12, 0x08, 0x12, 0x09, 0x12, 0x0a, 0x12, 0x0b, 0x02, 0x03, 0x81, 0x03,
  0x12, 0x00, 0x04, 0x7f, 0x00, 0x13, 0x22, 0x00, 0x04, 0x20, 0x12, 0xab,
  0x7f, 0x01, 0x6b, 0x02, 0x03, 0x12, 0x00, 0x20, 0x12, 0xab, 0x21, 0x0c,
  0xc8, 0x22, 0x00, 0x0c, 0x02, 0x03, 0x12, 0x01, 0x22, 0x02, 0x02, 0x20,
  0x12, 0x99, 0x7f, 0x01, 0x1c, 0x22, 0x02, 0x02, 0x22, 0x00, 0x0c, 0x20,
  0x11, 0x82, 0x22, 0x00, 0x08, 0x22, 0x01, 0x04, 0x23, 0x03, 0x04, 0x04,
  0xd5, 0x14, 0x62, 0x00, 0x62, 0x01, 0x53, 0x03, 0xf6, 0x22, 0x00, 0x04,
  0x20, 0x12, 0xb9, 0x7f, 0x01, 0x0a, 0x02, 0x03, 0x12, 0x00, 0x20, 0x12,
  0xb9, 0x21, 0x0c, 0xe5, 0x98, 0x25, 0x07, 0x02, 0x03, 0x12, 0x00, 0x20,
  0x11, 0xef, 0x98, 0x65, 0x06, 0x22, 0x00, 0x08, 0x20, 0x11, 0xef, 0x22,
  0x01, 0x08, 0x70, 0x00, 0x20, 0x11, 0x5c, 0x70, 0x0f, 0x70, 0x0e, 0x21,
  0x04, 0xb7, 0x22, 0x01, 0x04, 0x21, 0x0d, 0x3a, 0x23, 0x02, 0x00, 0x02,
  0x05, 0xe1, 0x07, 0x9a, 0x00, 0x05, 0xd1, 0xf8, 0x23, 0x02, 0xff, 0x12,
  0x05, 0x62, 0x04, 0x02, 0x04, 0x90, 0x02, 0x62, 0x05, 0xdf, 0x01, 0x02,
  0x04, 0xf0, 0x12, 0x04, 0x02, 0x05, 0xf0, 0x12, 0x05, 0x22, 0x00, 0xbc,
  0x22, 0x01, 0x04, 0x23, 0x03, 0x02, 0xdf, 0x01, 0x04, 0x95, 0x14, 0x62,
  0x00, 0x62, 0x01, 0x53, 0x03, 0xf6, 0x23, 0x03, 0x02, 0x04, 0x93, 0x02,
  0x14, 0x62, 0x00, 0x53, 0x03, 0xf7, 0x21, 0x04, 0xb7, 0x8d, 0xc0, 0x02,
  0x62, 0x05, 0x68, 0x05, 0x03, 0x21, 0x04, 0xb7, 0xe1, 0x00, 0x8f, 0x04,
  0x02, 0xd1, 0xff, 0x13, 0x02, 0x12, 0x05, 0x21, 0x0d, 0x5d, 0x8f, 0xc0,
  0xe9, 0xa8, 0x05, 0x21, 0x0d, 0x9a, 0x8e, 0xc0, 0xe1, 0xa8, 0x05, 0x21,
  0x0d, 0x9a, 0x8c, 0xc0, 0xd9, 0xa8, 0x05, 0x21, 0x0d, 0x9a, 0x8f, 0xc0,
  0xd1, 0x6d, 0xc0, 0xce, 0xa8, 0x05, 0x21, 0x0d, 0x9a, 0x6d, 0xc0, 0xc6,
  0xa8, 0x05, 0x8e, 0xc0, 0x02, 0xa8, 0x05, 0x8c, 0xc0, 0x02, 0xa8, 0x05,
  0x21, 0x0d, 0x9a, 0x20, 0x11, 0x49, 0x12, 0x03, 0x23, 0x00, 0x01, 0x61,
  0x00, 0x20, 0x10, 0x98, 0x12, 0x00, 0x20, 0x10, 0x8f, 0x61, 0x00, 0x61,
  0x00, 0x02, 0x03, 0x22, 0x01, 0x04, 0x20, 0x10, 0xb7, 0x70, 0x01, 0x23,
  0x02, 0x04, 0xe1, 0x00, 0x15, 0x62, 0x01, 0x53, 0x02, 0xfa, 0x23, 0x00,
  0x04, 0x71, 0x03, 0x71, 0x02, 0x20, 0x16, 0xb6, 0x21, 0x04, 0xb7, 0x20,
  0x11, 0x49, 0xe0, 0x12, 0x03, 0x23, 0x00, 0x02, 0x61, 0x00, 0x21, 0x0d,
  0xed, 0x20, 0x11, 0x49, 0xe0, 0xe0, 0x12, 0x03, 0x23, 0x00, 0x04, 0x61,
  0x00, 0x21, 0x0d, 0xed, 0x23, 0x00, 0x01, 0x61, 0x00, 0x20, 0x11, 0x49,
  0x61, 0x00, 0x20, 0x10, 0x8f, 0x13, 0x03, 0x20, 0x10, 0x98, 0x12, 0x00,
  0x71, 0x00, 0x22, 0x01, 0x04, 0x20, 0x10, 0xb7, 0x71, 0x02, 0x23, 0x00,
  0x04, 0x20, 0x17, 0xaa, 0x21, 0x04, 0xb7, 0x23, 0x00, 0x02, 0x61, 0x00,
  0x20, 0x11, 0x49, 0xe0, 0x21, 0x0e, 0x40, 0x23, 0x00, 0x04, 0x61, 0x00,
  0x20, 0x11, 0x49, 0xe0, 0xe0, 0x21, 0x0e, 0x40, 0x20, 0x12, 0x00, 0x20,
  0x17, 0xaa, 0x21, 0x04, 0xb7, 0x20, 0x12, 0x00, 0x20, 0x16, 0xb6, 0x21,
  0x04, 0xb7, 0x22, 0x00, 0xbc, 0x23, 0x03, 0x02, 0x20, 0x12, 0x06, 0x20,
  0x16, 0xb6, 0x21, 0x04, 0xb7, 0x23, 0x00, 0x01, 0x61, 0x00, 0x21, 0x0e,
  0xae, 0x23, 0x00, 0x02, 0x61, 0x00, 0x21, 0x0e, 0xae, 0x23, 0x00, 0x04,
  0x61, 0x00, 0x20, 0x12, 0x49, 0x13, 0x03, 0x23, 0x00, 0x04, 0x71, 0x02,
  0x20, 0x17, 0xaa, 0x21, 0x04, 0xb7, 0x20, 0x12, 0x49, 0x61, 0x00, 0x13,
  0x03, 0x23, 0x00, 0x04, 0x23, 0x02, 0x01, 0x20, 0x16, 0xb6, 0x70, 0x00,
  0x23, 0x03, 0x03, 0xe1, 0x00, 0x62, 0x00, 0x14, 0x53, 0x03, 0xfa, 0x21,
  0x04, 0xb7, 0x20, 0x12, 0x49, 0x61, 0x00, 0x13, 0x03, 0x23, 0x00, 0x04,
  0x23, 0x02, 0x02, 0x20, 0x16, 0xb6, 0x70, 0x00, 0x62, 0x00, 0x23, 0x03,
  0x02, 0xe1, 0x00, 0x62, 0x00, 0x14, 0x53, 0x03, 0xfa, 0x21, 0x04, 0xb7,
  0x20, 0x12, 0x49, 0x13, 0x03, 0x23, 0x00, 0x04, 0x23, 0x02, 0x04, 0x20,
  0x16, 0xb6, 0x21, 0x04, 0xb7, 0x20, 0x12, 0x49, 0x61, 0x00, 0x13, 0x03,
  0x23, 0x00, 0x04, 0x23, 0x02, 0x01, 0x20, 0x16, 0xb6, 0x70, 0x00, 0x04,
  0x9f, 0x00, 0xa9, 0x23, 0x03, 0x03, 0xd1, 0xff, 0x62, 0x00, 0x14, 0x53,
  0x03, 0xfa, 0x21, 0x04, 0xb7, 0x20, 0x12, 0x49, 0x61, 0x00, 0x13, 0x03,
  0x23, 0x00, 0x04, 0x23, 0x02, 0x02, 0x20, 0x16, 0xb6, 0x70, 0x00, 0x62,
  0x00, 0x04, 0x9f, 0x00, 0xa5, 0x23, 0x03, 0x02, 0xd1, 0xff, 0x62, 0x00,
  0x14, 0x53, 0x03, 0xfa, 0x21, 0x04, 0xb7, 0x88, 0x05, 0x36, 0x02, 0x04,
  0x31, 0x30, 0x06, 0x31, 0x20, 0x09, 0x21, 0x04, 0xb7, 0x20, 0x16, 0x6d,
  0x21, 0x04, 0xb7, 0x20, 0x16, 0x6a, 0x21, 0x04, 0xb7, 0x68, 0x05, 0x16,
  0x02, 0x04, 0x90, 0x02, 0x02, 0xb0, 0x80, 0x08, 0x20, 0x20, 0x40, 0x90,
  0x03, 0x21, 0x04, 0xb7, 0x23, 0x00, 0x03, 0x21, 0x13, 0xc4, 0x23, 0x00,
  0x0b, 0x21, 0x13, 0xc4, 0x23, 0x00, 0x03, 0x21, 0x13, 0xc4, 0x68, 0x05,
  0xf7, 0x02, 0x04, 0xe1, 0xec, 0x41, 0x60, 0xf0, 0x21, 0x04, 0xb7, 0x02,
  0x05, 0xe1, 0x01, 0x12, 0x0d, 0x23, 0x00, 0x0d, 0x21, 0x0f, 0xfd, 0x88,
  0x05, 0x06, 0x23, 0x03, 0xb8, 0x20, 0x0f, 0xe0, 0x22, 0x0d, 0xa0, 0x02,
  0x0d, 0xa1, 0x04, 0x12, 0x0d, 0x8f, 0x04, 0x07, 0x02, 0x0d, 0x13, 0x03,
  0x20, 0x0f, 0xe0, 0x02, 0x04, 0xe0, 0xe1, 0xfe, 0x80, 0x07, 0x12, 0x04,
  0x02, 0x0d, 0x41, 0x80, 0xe2, 0x21, 0x04, 0xb7, 0x20, 0x12, 0xf6, 0x23,
  0x00, 0xb4, 0x23, 0x02, 0x04, 0x21, 0x17, 0xaa, 0x22, 0x0d, 0x04, 0x02,
  0x05, 0xe1, 0x07, 0x21, 0x0f, 0xfd, 0x22, 0x0d, 0x00, 0x02, 0x05, 0xe1,
  0x07, 0xe9, 0x0d, 0x12, 0x0c, 0x20, 0x12, 0xe1, 0x12, 0x01, 0x02, 0x04,
  0x12, 0x0b, 0x22, 0x00, 0x04, 0x20, 0x11, 0x5c, 0x22, 0x0a, 0x00, 0x88,
  0x0b, 0x33, 0x02, 0x0a, 0x20, 0x12, 0xe1, 0x13, 0x03, 0x23, 0x02, 0x04,
  0x23, 0x00, 0x04, 0x6a, 0x0d, 0x06, 0x20, 0x16, 0xb6, 0x21, 0x10, 0x2f,
  0x20, 0x17, 0xaa, 0x22, 0x00, 0x04, 0x04, 0x81, 0x04, 0x14, 0x23, 0x02,
  0x03, 0x62, 0x00, 0x04, 0x91, 0x00, 0x14, 0x53, 0x02, 0xf7, 0x02, 0x0a,
  0x42, 0x0c, 0x02, 0xc9, 0x0d, 0x02, 0x0b, 0xc0, 0xe1, 0x7f, 0x80, 0x07,
  0x12, 0x0b, 0x62, 0x0a, 0x8b, 0x0a, 0xbc, 0x88, 0x0d, 0x0b, 0xc8, 0x0d,
  0x22, 0x0a, 0x0f, 0x22, 0x0b, 0x01, 0x21, 0x10, 0x13, 0x89, 0x0d, 0x0d,
  0x02, 0x0c, 0x20, 0x12, 0xe1, 0x12, 0x00, 0x22, 0x01, 0x04, 0x20, 0x11,
  0x5c, 0xc8, 0xbc, 0x21, 0x04, 0xb7, 0x72, 0x00, 0x04, 0xce, 0xc0, 0x9f,
  0x00, 0x02, 0xee, 0xc0, 0x03, 0x03, 0xcd, 0xc0, 0x90, 0x02, 0xed, 0xc0,
  0x21, 0x04, 0xb7, 0x02, 0x04, 0xe0, 0xe0, 0xe1, 0x1c, 0x81, 0x80, 0xa0,
  0x02, 0x04, 0xc0, 0xe1, 0x1c, 0x81, 0x80, 0xa0, 0x02, 0x05, 0xd0, 0x02,
  0x04, 0xd0, 0xc0, 0xc0, 0xc0, 0xe1, 0x1c, 0x81, 0x80, 0xa0, 0x02, 0x05,
  0xe0, 0xe0, 0xe1, 0x1c, 0x81, 0x80, 0xa0, 0x84, 0x15, 0x23, 0x02, 0x03,
  0x62, 0x00, 0x62, 0x01, 0x04, 0x91, 0x00, 0x15, 0x53, 0x02, 0xf5, 0xa0,
  0x02, 0x00, 0x81, 0x04, 0x12, 0x00, 0x72, 0x00, 0x04, 0x23, 0x02, 0x03,
  0xce, 0xc0, 0x9f, 0x00, 0x02, 0xee, 0xc0, 0x72, 0x00, 0xd4, 0x53, 0x02,
  0xfa, 0xcd, 0xc0, 0x90, 0x02, 0xed, 0xc0, 0x21, 0x04, 0xb7, 0x02, 0x04,
  0x13, 0x03, 0xe0, 0xe0, 0xe1, 0x1c, 0x8f, 0x04, 0x02, 0xd1, 0x20, 0x81,
  0x80, 0xa0, 0x02, 0x04, 0xc0, 0xe1, 0x3c, 0x81, 0x80, 0xa0, 0x62, 0x00,
  0x62, 0x00, 0x62, 0x00, 0x62, 0x00, 0xa0, 0x23, 0x02, 0x03, 0x04, 0x62,
  0x00, 0xd4, 0x53, 0x02, 0xfa, 0xa0, 0x23, 0x02, 0x04, 0xdf, 0x01, 0x04,
  0x95, 0x14, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02, 0xf6, 0xa0, 0x23, 0x02,
  0x04, 0xdf, 0x01, 0x04, 0xb5, 0x14, 0x62, 0x00, 0x62, 0x01, 0x53, 0x02,
  0xf6, 0xa0, 0x62, 0x01, 0x62, 0x01, 0x62, 0x01, 0xa0, 0x02, 0x05, 0xd0,
  0x02, 0x04, 0xd0, 0xe0, 0xe0, 0xe0, 0xe1, 0x07, 0xa0, 0x02, 0x04, 0xf0,
  0x13, 0x03, 0x02, 0x05, 0xf0, 0x13, 0x02, 0x03, 0x03, 0xf0, 0x03, 0x02,
  0xf0, 0xe1, 0x1f, 0xa0, 0x23, 0x03, 0x04, 0x05, 0x62, 0x01, 0x14, 0x62,
  0x00, 0x53, 0x03, 0xf7, 0xa0, 0x12, 0x04, 0xe1, 0x00, 0x12, 0x05, 0x12,
  0x06, 0x12, 0x07, 0xa0, 0x20, 0x11, 0x69, 0x22, 0x01, 0x04, 0xa0, 0x20,
  0x11, 0x69, 0x22, 0x00, 0x04, 0xa0, 0x23, 0x64, 0x04, 0x06, 0x62, 0x02,
  0x14, 0x62, 0x00, 0x53, 0x64, 0xf7, 0xa0, 0xbf, 0x01, 0xcf, 0xc0, 0x9f,
  0x01, 0x02, 0xef, 0xc0, 0xa0, 0xcc, 0xc0, 0x72, 0x00, 0x72, 0x01, 0x72,
  0x02, 0x04, 0xf5, 0x7f, 0x00, 0x07, 0x04, 0xf6, 0x9f, 0x00, 0x02, 0xec,
  0xc0, 0xa0, 0xcc, 0xc0, 0x72, 0x00, 0x72, 0x01, 0x72, 0x02, 0x04, 0xf5,
  0x9f, 0x00, 0x07, 0x04, 0xf6, 0x9f, 0x00, 0x02, 0xec, 0xc0, 0xa0, 0x02,
  0x05, 0xd0, 0x02, 0x04, 0xd0, 0xc0, 0xc0, 0xc0, 0xc0, 0xe1, 0x0e, 0x71,
  0x03, 0x71, 0x02, 0x83, 0x02, 0x61, 0x00, 0xe1, 0x00, 0x93, 0x03, 0x61,
  0x03, 0xa0, 0x60, 0x00, 0x02, 0x00, 0x81, 0x03, 0x12, 0x00, 0x04, 0x70,
  0x00, 0x7f, 0x00, 0x03, 0xe1, 0x00, 0xa0, 0x23, 0x02, 0x04, 0xdf, 0x01,
  0xe1, 0x00, 0xb4, 0x14, 0x62, 0x00, 0x53, 0x02, 0xf7, 0xd1, 0x01, 0xa0,
  0x22, 0x00, 0xb4, 0x23, 0x03, 0x00, 0x23, 0x02, 0x02, 0x02, 0x04, 0xf0,
  0x12, 0x04, 0x02, 0x05, 0xf0, 0x12, 0x05, 0x53, 0x02, 0xf3, 0x60, 0x05,
  0x02, 0x04, 0xe1, 0xfc, 0xd3, 0x03, 0x84, 0xe1, 0xfc, 0x62, 0x00, 0x12,
  0x04, 0x02, 0x05, 0xe1, 0x03, 0x94, 0x62, 0x00, 0x12, 0x05, 0xe1, 0x00,
  0x94, 0x62, 0x00, 0x12, 0x06, 0xe1, 0x00, 0x94, 0x12, 0x07, 0x71, 0x00,
  0xe1, 0x1c, 0x81, 0x80, 0x13, 0x03, 0x23, 0x02, 0x04, 0x23, 0x00, 0x04,
  0xa0, 0x20, 0x10, 0x8f, 0x61, 0x00, 0x20, 0x10, 0x98, 0x61, 0x00, 0x20,
  0x10, 0xa0, 0x12, 0x01, 0x22, 0x00, 0x04, 0x20, 0x11, 0x5c, 0x70, 0x01,
  0x22, 0x00, 0x04, 0x20, 0x11, 0x16, 0x71, 0x00, 0xa0, 0x02, 0x04, 0xf0,
  0x13, 0x02, 0xe1, 0x00, 0x13, 0x05, 0x13, 0x04, 0xf0, 0x13, 0x03, 0x03,
  0x02, 0xf0, 0xe1, 0xfc, 0x13, 0x02, 0x03, 0x03, 0xf0, 0x13, 0x03, 0xa0,
  0x22, 0x02, 0x02, 0xdf, 0x01, 0x23, 0x64, 0x04, 0x05, 0x96, 0x14, 0x62,
  0x00, 0x62, 0x01, 0x62, 0x02, 0x53, 0x64, 0xf4, 0xa0, 0x23, 0x64, 0x04,
  0xdf, 0x01, 0x04, 0x62, 0x00, 0xb5, 0x62, 0x01, 0x16, 0x62, 0x02, 0x53,
  0x64, 0xf4, 0xa0, 0x23, 0x03, 0x04, 0xdf, 0x01, 0x04, 0xf0, 0x14, 0x62,
  0x00, 0x53, 0x03, 0xf8, 0xa0, 0x23, 0x03, 0x04, 0xdf, 0x01, 0x02, 0x00,
  0x81, 0x03, 0x12, 0x00, 0x04, 0xd0, 0x14, 0x72, 0x00, 0x53, 0x03, 0xf8,
  0xa0, 0x20, 0x10, 0x8f, 0x12, 0x00, 0x20, 0x10, 0x98, 0x12, 0x01, 0xa0,
  0xc0, 0xc0, 0xe1, 0x1c, 0x81, 0x80, 0xa0, 0xe1, 0x0f, 0xe0, 0xe0, 0x81,
  0x80, 0xa0, 0x20, 0x10, 0x98, 0x12, 0x00, 0x20, 0x10, 0xa0, 0x12, 0x01,
  0x20, 0x10, 0x8f, 0x12, 0x03, 0xa0, 0x22, 0x00, 0xb4, 0x04, 0xa1, 0x04,
  0x14, 0x23, 0x02, 0x03, 0x62, 0x00, 0x04, 0xb1, 0x00, 0x14, 0x53, 0x02,
  0xf7, 0xa0, 0x84, 0x14, 0x23, 0x02, 0x03, 0x62, 0x00, 0x04, 0x91, 0x00,
  0x14, 0x53, 0x02, 0xf7, 0xa0, 0x21, 0x0f, 0x94, 0x21, 0x0f, 0x94, 0x8c,
  0x04, 0xf7, 0x8e, 0x06, 0xf7, 0x21, 0x04, 0xb7, 0x21, 0x13, 0xc1, 0x8c,
  0x05, 0xfa, 0x8f, 0x07, 0x59, 0x8c, 0x07, 0xeb, 0x8e, 0x07, 0x0b, 0x22,
  0x01, 0xbc, 0x22, 0x00, 0xb8, 0x20, 0x11, 0x5c, 0xe8, 0xb8, 0x22, 0x09,
  0xff, 0x6a, 0x05, 0x07, 0x22, 0x09, 0x00, 0xad, 0x07, 0xab, 0x07, 0x6d,
  0x07, 0x02, 0xcc, 0x07, 0x02, 0x06, 0xf0, 0xe1, 0xfe, 0x12, 0x06, 0x02,
  0x07, 0xf0, 0xe1, 0x3f, 0x12, 0x07, 0x88, 0x04, 0x02, 0xee, 0x07, 0x89,
  0x04, 0x02, 0xef, 0x07, 0x02, 0x04, 0xc0, 0xc0, 0xe1, 0x3f, 0x12, 0x08,
  0x88, 0x05, 0x02, 0xee, 0x08, 0x89, 0x05, 0x02, 0xef, 0x08, 0x22, 0x00,
  0xbc, 0x22, 0x01, 0x06, 0x20, 0x11, 0x16, 0x21, 0x04, 0xb7, 0x02, 0x07,
  0x20, 0x12, 0xdf, 0x12, 0x00, 0x81, 0x02, 0x6f, 0x04, 0x0b, 0x12, 0x01,
  0xe1, 0x00, 0x15, 0x62, 0x01, 0x15, 0x21, 0x13, 0xa3, 0x12, 0x00, 0x02,
  0x06, 0x14, 0x62, 0x00, 0x8a, 0x05, 0x02, 0xef, 0x07, 0x02, 0x04, 0xe1,
  0x0f, 0x12, 0x04, 0x02, 0x07, 0xe1, 0xf0, 0xd2, 0x04, 0xc0, 0xc0, 0xc0,
  0xc0, 0x14, 0x21, 0x04, 0xb7, 0x21, 0x0f, 0x94, 0x61, 0x00, 0xe1, 0x00,
  0x12, 0x04, 0x12, 0x05, 0x12, 0x06, 0x22, 0x07, 0x01, 0x8f, 0xc0, 0x02,
  0xed, 0x07, 0x8e, 0xc0, 0x02, 0xef, 0x07, 0x8d, 0xc0, 0x02, 0xee, 0x07,
  0x8c, 0xc0, 0x02, 0xec, 0x07, 0x88, 0xc0, 0x02, 0xe8, 0x04, 0x23, 0x03,
  0x04, 0x20, 0x0f, 0xe0, 0x23, 0x03, 0xbc, 0x20, 0x0f, 0xe0, 0x23, 0x03,
  0xb8, 0x20, 0x0f, 0xe0, 0x23, 0x03, 0xb0, 0x20, 0x0f, 0xe0, 0x23, 0x03,
  0x8c, 0x20, 0x0f, 0xe0, 0x23, 0x03, 0x88, 0x20, 0x0f, 0xe0, 0x23, 0x03,
  0x84, 0x20, 0x0f, 0xe0, 0x23, 0x03, 0x80, 0x20, 0x0f, 0xe0, 0xd1, 0xff,
  0x12, 0xbb, 0x12, 0xba, 0x12, 0xb9, 0xe1, 0xf9, 0x12, 0xb8, 0x71, 0x00,
  0xe0, 0xe0, 0x12, 0x04, 0xe1, 0x03, 0x12, 0x05, 0xc8, 0x04, 0xc9, 0x04,
  0x22, 0x06, 0x00, 0x22, 0x07, 0x00, 0xe8, 0xc0, 0x21, 0x04, 0xa9, 0x02,
  0xfe, 0xe2, 0xc1, 0x90, 0x06, 0x02, 0xff, 0xe2, 0xc2, 0x80, 0x3f, 0x22,
  0x01, 0xb4, 0x22, 0x00, 0x04, 0x20, 0x11, 0x5c, 0x22, 0x00, 0x04, 0x23,
  0x00, 0x1f, 0x20, 0x13, 0x0a, 0x23, 0x03, 0x08, 0x23, 0x00, 0x04, 0x23,
  0x02, 0x01, 0x20, 0x16, 0xb6, 0x68, 0x08, 0x1e, 0x02, 0xfe, 0xe2, 0xc1,
  0x80, 0x09, 0x23, 0x02, 0x00, 0x20, 0x14, 0xf6, 0x21, 0x14, 0x28, 0x02,
  0xff, 0xe2, 0xc2, 0x80, 0x09, 0x23, 0x02, 0x08, 0x20, 0x14, 0xf6, 0x21,
  0x14, 0x28, 0x23, 0x03, 0x80, 0x20, 0x14, 0xe4, 0x23, 0x03, 0x84, 0x20,
  0x14, 0xe4, 0x23, 0x03, 0x88, 0x20, 0x14, 0xe4, 0x23, 0x03, 0x8c, 0x20,
  0x14, 0xe4, 0x23, 0x03, 0xb0, 0x20, 0x14, 0xe4, 0x23, 0x03, 0xb8, 0x20,
  0x14, 0xe4, 0x23, 0x03, 0xbc, 0x20, 0x14, 0xe4, 0x23, 0x03, 0x04, 0x20,
  0x14, 0xe4, 0xc8, 0xbc, 0x22, 0xc0, 0x00, 0x88, 0x04, 0x02, 0xe8, 0xc0,
  0x8d, 0x07, 0x02, 0xef, 0xc0, 0x8f, 0x07, 0x02, 0xee, 0xc0, 0x8e, 0x07,
  0x02, 0xed, 0xc0, 0x8c, 0x07, 0x02, 0xec, 0xc0, 0x88, 0x07, 0x03, 0x21,
  0x04, 0xb7, 0x23, 0x00, 0x0b, 0x21, 0x13, 0xc4, 0x23, 0x00, 0xb4, 0x23,
  0x02, 0x04, 0x20, 0x16, 0xb6, 0x22, 0x00, 0xb4, 0x23, 0x00, 0x04, 0x21,
  0x13, 0x0a, 0x78, 0x00, 0x35, 0x79, 0x00, 0x2d, 0x7a, 0x00, 0x25, 0x7b,
  0x00, 0x1d, 0x7c, 0x00, 0x15, 0x7d, 0x00, 0x0d, 0x7e, 0x00, 0x05, 0x23,
  0x00, 0x07, 0x2d, 0x31, 0x23, 0x00, 0x06, 0x2d, 0x31, 0x23, 0x00, 0x05,
  0x2d, 0x31, 0x23, 0x00, 0x04, 0x2d, 0x31, 0x23, 0x00, 0x03, 0x2d, 0x31,
  0x23, 0x00, 0x02, 0x2d, 0x31, 0x23, 0x00, 0x01, 0x2d, 0x31, 0x23, 0x00,
  0x00, 0x83, 0x02, 0x81, 0x10, 0xa0, 0x23, 0x03, 0x06, 0x20, 0x06, 0x00,
  0x02, 0x07, 0x20, 0x12, 0xdf, 0x61, 0x00, 0x02, 0x07, 0x20, 0x12, 0xd8,
  0x61, 0x00, 0x02, 0x04, 0x13, 0x03, 0x20, 0x12, 0xdf, 0x12, 0x03, 0x02,
  0x06, 0x20, 0x12, 0xdf, 0x12, 0x02, 0x12, 0x00, 0x22, 0x0d, 0x00, 0x7d,
  0x03, 0x27, 0xa2, 0x03, 0x80, 0x1b, 0x20, 0x11, 0xde, 0x98, 0x00, 0x04,
  0xe8, 0x0d, 0xaa, 0x0d, 0x02, 0x03, 0x12, 0x00, 0x20, 0x11, 0xde, 0x98,
  0x00, 0x04, 0xe9, 0x0d, 0xaa, 0x0d, 0x21, 0x15, 0x89, 0x20, 0x11, 0xde,
  0x98, 0x00, 0x02, 0xe8, 0x0d, 0x7e, 0x03, 0x11, 0x22, 0x01, 0x04, 0x23,
  0x03, 0x08, 0xe1, 0x00, 0x15, 0x62, 0x01, 0x53, 0x03, 0xfa, 0x21, 0x15,
  0xc9, 0x71, 0x00, 0x12, 0x0c, 0x12, 0x01, 0x22, 0x00, 0x04, 0x20, 0x11,
  0x5c, 0x70, 0x01, 0x60, 0x01, 0x60, 0x0c, 0x22, 0x00, 0x08, 0x20, 0x11,
  0x5c, 0x8a, 0x0d, 0x11, 0x22, 0x00, 0x04, 0x23, 0x03, 0x08, 0xdf, 0x01,
  0xe1, 0x00, 0xb4, 0x14, 0x62, 0x00, 0x53, 0x03, 0xf7, 0x60, 0x0d, 0x02,
  0x03, 0x12, 0x01, 0x22, 0x0d, 0x00, 0x22, 0x0c, 0x00, 0x02, 0x0d, 0x82,
  0x0c, 0x12, 0x03, 0x02, 0x02, 0x82, 0x0c, 0x12, 0x00, 0x05, 0x80, 0x41,
  0x13, 0x02, 0x04, 0x13, 0x03, 0x62, 0x00, 0x04, 0x30, 0x13, 0x64, 0x02,
  0x03, 0x81, 0x04, 0x12, 0x00, 0x04, 0x83, 0x03, 0x14, 0x62, 0x03, 0x6b,
  0x03, 0x27, 0x62, 0x00, 0x04, 0x93, 0x64, 0x14, 0x62, 0x03, 0x6b, 0x03,
  0x1c, 0x62, 0x00, 0x04, 0x93, 0x02, 0x14, 0x62, 0x03, 0x6b, 0x03, 0x11,
  0x9f, 0x01, 0x0e, 0x62, 0x00, 0x62, 0x03, 0x04, 0x91, 0x00, 0x14, 0x6b,
  0x03, 0x03, 0x21, 0x16, 0x14, 0x62, 0x0c, 0x62, 0x0c, 0x8a, 0x0c, 0xa9,
  0x62, 0x01, 0x62, 0x0d, 0x8a, 0x0d, 0x9f, 0x70, 0x0d, 0x88, 0x0d, 0x07,
  0x02, 0x02, 0x12, 0x00, 0x20, 0x11, 0xef, 0x89, 0x0d, 0x09, 0x02, 0x01,
  0xa1, 0x04, 0x12, 0x00, 0x20, 0x11, 0xef, 0x8a, 0x0d, 0x09, 0x22, 0x00,
  0x04, 0x23, 0x02, 0x08, 0x20, 0x11, 0xf2, 0x70, 0x00, 0x22, 0x01, 0x04,
  0x20, 0x11, 0x5c, 0x70, 0x00, 0x22, 0x01, 0x08, 0x20, 0x11, 0x5c, 0x21,
  0x04, 0xb7, 0x21, 0x01, 0x40, 0xf8, 0x07, 0xa0, 0x05, 0x13, 0x64, 0x62,
  0x01, 0x05, 0x13, 0x65, 0xa0, 0x05, 0x12, 0x02, 0xef, 0x02, 0xf0, 0x62,
  0x01, 0x05, 0xf0, 0x13, 0x25, 0xa0, 0x05, 0x12, 0x02, 0xcf, 0x02, 0x61,
  0x0e, 0x21, 0x17, 0x51, 0x03, 0x03, 0x12, 0x00, 0x03, 0x02, 0x41, 0x02,
  0x4b, 0x05, 0x31, 0x00, 0x06, 0x31, 0x02, 0x09, 0x21, 0x16, 0xe4, 0x22,
  0x00, 0xfe, 0x21, 0x16, 0xac, 0x22, 0x00, 0xc1, 0x04, 0x15, 0x62, 0x00,
  0x62, 0x01, 0x53, 0x02, 0xf7, 0xa0, 0x12, 0x01, 0x81, 0x03, 0x12, 0x00,
  0x03, 0x02, 0xa1, 0x01, 0xe5, 0x80, 0x06, 0x23, 0x00, 0x03, 0x21, 0x13,
  0xc4, 0x04, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0x0f, 0x80, 0x34, 0x31, 0x02,
  0x52, 0x7f, 0x01, 0x0f, 0x31, 0x04, 0xab, 0x7f, 0x01, 0x59, 0x31, 0x06,
  0xaf, 0x7f, 0x01, 0x65, 0x21, 0x17, 0xd8, 0x03, 0x03, 0x12, 0x00, 0x05,
  0x80, 0x73, 0xf9, 0x01, 0x60, 0x00, 0x12, 0x00, 0x04, 0x62, 0x00, 0xd9,
  0x01, 0x14, 0x62, 0x00, 0xf9, 0x01, 0x53, 0x02, 0xf3, 0x70, 0x00, 0xd9,
  0x01, 0xa0, 0x05, 0x62, 0x01, 0x13, 0x04, 0x05, 0x62, 0x01, 0x82, 0x0e,
  0x13, 0x05, 0x05, 0x92, 0x0f, 0xe1, 0x01, 0x13, 0x54, 0x03, 0x03, 0x12,
  0x00, 0x50, 0x63, 0x04, 0x14, 0x62, 0x00, 0x53, 0x02, 0xf7, 0xa0, 0x20,
  0x16, 0x70, 0x03, 0x03, 0x12, 0x00, 0x03, 0x66, 0x14, 0x62, 0x00, 0x53,
  0x02, 0xf8, 0xa0, 0x05, 0x62, 0x01, 0x13, 0x04, 0x05, 0x62, 0x01, 0x13,
  0x05, 0x05, 0xe1, 0x01, 0x13, 0x54, 0x21, 0x17, 0x19, 0x20, 0x16, 0x79,
  0x61, 0x0e, 0x23, 0x0e, 0x81, 0x03, 0x03, 0x12, 0x00, 0x06, 0x62, 0x02,
  0x14, 0x62, 0x00, 0x53, 0x02, 0xf7, 0x71, 0x0e, 0xa0, 0xf9, 0x01, 0x60,
  0x08, 0x12, 0x08, 0xfc, 0x01, 0x04, 0xdc, 0x01, 0x62, 0x08, 0xd9, 0x01,
  0x14, 0x62, 0x00, 0xf9, 0x01, 0x53, 0x02, 0xef, 0x70, 0x08, 0xd9, 0x01,
  0xa0, 0x03, 0x02, 0x41, 0x02, 0x56, 0x03, 0x03, 0x12, 0x00, 0x05, 0x31,
  0x00, 0x06, 0x31, 0x02, 0x14, 0x21, 0x17, 0xd8, 0x04, 0xf1, 0xff, 0xe2,
  0xfe, 0x12, 0xfe, 0x62, 0x00, 0x04, 0xf1, 0xff, 0xe2, 0xff, 0x12, 0xff,
  0xa0, 0x04, 0x12, 0xc1, 0x62, 0x00, 0x04, 0x12, 0xc2, 0xa0, 0x12, 0x01,
  0x81, 0x03, 0x12, 0x00, 0x03, 0x02, 0xa1, 0x01, 0xe5, 0x80, 0x06, 0x23,
  0x00, 0x03, 0x21, 0x13, 0xc4, 0x04, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0x0f,
  0x80, 0x12, 0x31, 0x02, 0x50, 0x7f, 0x01, 0x12, 0x31, 0x04, 0x72, 0x7f,
  0x01, 0x06, 0x31, 0x06, 0xa8, 0x7f, 0x01, 0x51, 0x23, 0x00, 0x03, 0x21,
  0x13, 0xc4, 0x03, 0x03, 0x12, 0x00, 0x05, 0x80, 0x18, 0xf9, 0x01, 0x60,
  0x00, 0x12, 0x00, 0xd9, 0x01, 0x04, 0x62, 0x00, 0xf9, 0x01, 0x14, 0x62,
  0x00, 0x53, 0x02, 0xf3, 0x70, 0x00, 0xd9, 0x01, 0xa0, 0xf9, 0x01, 0x60,
  0x08, 0x12, 0x08, 0xd9, 0x01, 0x04, 0x62, 0x00, 0xf9, 0x01, 0xfc, 0x01,
  0x14, 0xdc, 0x01, 0x62, 0x08, 0x53, 0x02, 0xef, 0x70, 0x08, 0xd9, 0x01,
  0xa0, 0x03, 0x03, 0x12, 0x00, 0x20, 0x16, 0x70, 0x04, 0x62, 0x00, 0x13,
  0x66, 0x53, 0x02, 0xf8, 0xa0, 0x20, 0x16, 0x79, 0x61, 0x0e, 0x23, 0x0e,
  0x81, 0x03, 0x03, 0x12, 0x00, 0x04, 0x62, 0x00, 0x16, 0x62, 0x02, 0x53,
  0x02, 0xf7, 0x71, 0x0e, 0xa0, 0x05, 0x12, 0x02, 0xcf, 0x02, 0x61, 0x0e,
  0x21, 0x18, 0x31, 0x4e, 0x6f, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x20,
  0x20, 0x20, 0x20, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0x2e, 0x20, 0x4c, 0x6f,
  0x61, 0x64, 0x20, 0x73, 0x6f, 0x6d, 0x65, 0x20, 0x20, 0x62, 0x65, 0x66,
  0x6f, 0x72, 0x65, 0x72, 0x75, 0x6e, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x75,
  0x4d, 0x32, 0x33, 0x28, 0x63, 0x29, 0x20, 0x64, 0x6d, 0x69, 0x74, 0x72,
  0x79, 0x67, 0x72, 0x43, 0x6c, 0x75, 0x73, 0x74, 0x65, 0x72, 0x3a, 0x00,
  0x53, 0x69, 0x7a, 0x65, 0x3a, 0x00, 0x41, 0x3d, 0x67, 0x6f, 0x20, 0x20,
  0x42, 0x3d, 0x62, 0x61, 0x63, 0x6b, 0x00, 0x54, 0x68, 0x69, 0x73, 0x20,
  0x66, 0x69, 0x6c, 0x65, 0x20, 0x69, 0x73, 0x66, 0x72, 0x61, 0x67, 0x6d,
  0x65, 0x6e, 0x74, 0x65, 0x64, 0x20, 0x26, 0x63, 0x61, 0x6e, 0x27, 0x74,
  0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6e, 0x44, 0x65, 0x66, 0x72, 0x61,
  0x67, 0x6d, 0x65, 0x6e, 0x74, 0x3f, 0x20, 0x41, 0x3d, 0x79, 0x65, 0x73,
  0x20, 0x20, 0x20, 0x42, 0x3d, 0x6e, 0x6f, 0x23, 0x0e, 0x92, 0x23, 0x06,
  0x7f, 0x23, 0x22, 0x00, 0xf9, 0x01, 0x22, 0x12, 0xff, 0xf8, 0x54, 0x20,
  0x1b, 0x8f, 0x20, 0x1b, 0xc3, 0x02, 0x12, 0x41, 0xff, 0x1c, 0x20, 0x1b,
  0xa5, 0x23, 0x04, 0x4b, 0x23, 0x05, 0x18, 0xe1, 0x00, 0x13, 0x65, 0x13,
  0x02, 0x13, 0x03, 0x20, 0x1b, 0x03, 0x23, 0x0e, 0x81, 0x20, 0x1b, 0x5c,
  0x39, 0x0e, 0x72, 0x12, 0xe8, 0x12, 0xd1, 0xff, 0xa2, 0x12, 0xc0, 0x12,
  0x13, 0xe1, 0x00, 0x12, 0x14, 0x12, 0x15, 0x62, 0x12, 0x20, 0x1b, 0xa5,
  0x23, 0x02, 0x00, 0x02, 0x14, 0x83, 0x02, 0xe0, 0xe1, 0xfe, 0x82, 0x12,
  0x80, 0x1f, 0x12, 0x00, 0x04, 0x13, 0x05, 0x62, 0x00, 0x04, 0x81, 0x04,
  0x13, 0x04, 0xd8, 0x65, 0x03, 0x02, 0x42, 0x15, 0x02, 0xf8, 0x65, 0x20,
  0x1b, 0x24, 0x63, 0x02, 0x03, 0x02, 0x41, 0x05, 0xd6, 0xd8, 0x65, 0x23,
  0x02, 0x00, 0x23, 0x03, 0x0b, 0x23, 0x00, 0x20, 0x20, 0x1c, 0x81, 0x63,
  0x02, 0x03, 0x02, 0x41, 0x05, 0xf3, 0x02, 0x14, 0x80, 0x0c, 0x23, 0x00,
  0x80, 0x23, 0x02, 0x00, 0x23, 0x03, 0x0b, 0x20, 0x1c, 0x81, 0x02, 0x13,
  0xe1, 0xfc, 0x80, 0x14, 0x02, 0x13, 0xa2, 0x14, 0xa1, 0x05, 0x80, 0x0c,
  0x23, 0x00, 0x81, 0x23, 0x02, 0x04, 0x23, 0x03, 0x0b, 0x20, 0x1c, 0x81,
  0x20, 0x1b, 0x5c, 0x7c, 0x02, 0x3c, 0x79, 0x02, 0x18, 0x78, 0x02, 0x03,
  0x21, 0x19, 0x98, 0x02, 0x15, 0x80, 0x05, 0x72, 0x15, 0x21, 0x19, 0x29,
  0x02, 0x14, 0x80, 0xe4, 0x72, 0x14, 0x21, 0x19, 0x29, 0x02, 0x15, 0x31,
  0x04, 0x0c, 0x81, 0x01, 0xa2, 0x13, 0x9f, 0x01, 0xd3, 0x62, 0x15, 0x21,
  0x19, 0x29, 0x02, 0x14, 0x82, 0x15, 0x81, 0x01, 0xa2, 0x13, 0x9f, 0x01,
  0xc3, 0x62, 0x14, 0x21, 0x19, 0x29, 0x20, 0x1b, 0xa5, 0x02, 0x14, 0x82,
  0x15, 0xe0, 0xe1, 0xfe, 0x82, 0x12, 0x12, 0x00, 0x04, 0x13, 0x05, 0x62,
  0x00, 0x04, 0x81, 0x04, 0x13, 0x04, 0x23, 0x02, 0x00, 0xf8, 0x65, 0x20,
  0x1b, 0x24, 0xd8, 0x65, 0x23, 0x04, 0x87, 0x23, 0x05, 0x18, 0x23, 0x02,
  0x02, 0x23, 0x03, 0x00, 0x20, 0x1b, 0x03, 0x04, 0x81, 0x02, 0x13, 0x04,
  0x72, 0x00, 0x04, 0x13, 0x05, 0x50, 0x13, 0x64, 0xe1, 0x00, 0x23, 0x02,
  0x02, 0x23, 0x03, 0x0b, 0x20, 0x1a, 0xbe, 0x23, 0x04, 0x90, 0x23, 0x05,
  0x18, 0x23, 0x02, 0x03, 0x23, 0x03, 0x00, 0x20, 0x1b, 0x03, 0x04, 0x13,
  0x05, 0x62, 0x00, 0x04, 0x81, 0x18, 0x13, 0x04, 0x50, 0xe0, 0x61, 0x00,
  0xe1, 0x80, 0x13, 0x64, 0x71, 0x00, 0xe1, 0x7f, 0x23, 0x02, 0x03, 0x23,
  0x03, 0x0b, 0x20, 0x1a, 0xbe, 0x23, 0x04, 0x96, 0x23, 0x05, 0x18, 0xd8,
  0x65, 0x23, 0x02, 0x04, 0x23, 0x03, 0x00, 0x20, 0x1b, 0x03, 0x20, 0x1b,
  0x5c, 0x7c, 0x02, 0x06, 0x9d, 0x02, 0xf7, 0x21, 0x19, 0x16, 0x20, 0x1b,
  0xa5, 0x04, 0x81, 0x02, 0x13, 0x04, 0x72, 0x00, 0x04, 0x13, 0x05, 0x50,
  0x61, 0x00, 0x20, 0x1c, 0x3f, 0x80, 0x14, 0x71, 0x00, 0xd9, 0x01, 0xe0,
  0x61, 0x00, 0xe1, 0xfe, 0x12, 0x0e, 0x71, 0x00, 0xe1, 0x01, 0x12, 0x0f,
  0x21, 0x04, 0x80, 0x23, 0x04, 0xa3, 0x23, 0x05, 0x18, 0xe1, 0x00, 0x13,
  0x65, 0x13, 0x02, 0x13, 0x03, 0x20, 0x1b, 0x03, 0x20, 0x1b, 0x5c, 0x7c,
  0x02, 0x08, 0x9d, 0x02, 0xf7, 0x73, 0x06, 0x21, 0x19, 0xda, 0x71, 0x00,
  0x20, 0x1e, 0x68, 0x21, 0x19, 0xda, 0x61, 0x05, 0x61, 0x04, 0x61, 0x02,
  0x61, 0x03, 0x61, 0x00, 0x03, 0x64, 0x13, 0x03, 0x71, 0x00, 0x21, 0x1a,
  0xd6, 0x90, 0x03, 0x33, 0x03, 0x24, 0x23, 0x02, 0x0a, 0x40, 0x13, 0x05,
  0x03, 0x03, 0x13, 0x04, 0x03, 0x02, 0x81, 0x30, 0x71, 0x03, 0x71, 0x02,
  0x20, 0x1c, 0x81, 0x73, 0x03, 0x61, 0x02, 0x61, 0x03, 0x03, 0x04, 0x13,
  0x03, 0x03, 0x05, 0x21, 0x1a, 0xd1, 0x71, 0x03, 0x71, 0x02, 0x71, 0x04,
  0x71, 0x05, 0xa0, 0x23, 0x64, 0x00, 0x03, 0x64, 0xc1, 0x80, 0x18, 0x20,
  0x1c, 0x81, 0x63, 0x64, 0x63, 0x03, 0x03, 0x03, 0xa1, 0x0c, 0x90, 0xee,
  0x23, 0x03, 0x00, 0x63, 0x02, 0x03, 0x02, 0xa1, 0x05, 0x90, 0xe3, 0xa0,
  0x23, 0x03, 0x00, 0x50, 0x90, 0x03, 0x23, 0x00, 0x20, 0x61, 0x03, 0x13,
  0x64, 0x03, 0x04, 0xe1, 0x1f, 0xa1, 0x04, 0x13, 0x03, 0x03, 0x64, 0x20,
  0x1c, 0x81, 0x03, 0x64, 0x41, 0x2e, 0x04, 0x71, 0x00, 0x61, 0x03, 0x71,
  0x03, 0x63, 0x04, 0x9c, 0x04, 0xd9, 0x23, 0x00, 0x20, 0x20, 0x1c, 0x81,
  0x63, 0x03, 0x03, 0x03, 0x41, 0x0c, 0xf3, 0xa0, 0x23, 0x4c, 0xff, 0x03,
  0x4c, 0x41, 0xff, 0xf8, 0x23, 0x4c, 0xff, 0x03, 0x4c, 0xf1, 0xff, 0x80,
  0xf7, 0x13, 0x02, 0xa1, 0x01, 0xe3, 0x02, 0x90, 0xe7, 0x7f, 0x02, 0x04,
  0x7e, 0x02, 0x0e, 0xa0, 0x61, 0x0e, 0x23, 0x0e, 0x81, 0x20, 0x01, 0x40,
  0x71, 0x0e, 0x21, 0x18, 0xdf, 0x23, 0x0e, 0x81, 0x21, 0x01, 0xf0, 0x23,
  0x05, 0xfe, 0x23, 0x04, 0x46, 0x50, 0xe0, 0xe1, 0xfe, 0x12, 0x10, 0x23,
  0x04, 0x4a, 0x50, 0xe0, 0xe1, 0xfe, 0x12, 0x11, 0xa0, 0x61, 0x0e, 0x23,
  0x0e, 0x81, 0x60, 0x02, 0xe1, 0x00, 0x13, 0x25, 0x22, 0x02, 0x80, 0x16,
  0x62, 0x02, 0x6f, 0x02, 0xfa, 0x63, 0x25, 0x99, 0x25, 0xf2, 0x70, 0x02,
  0x71, 0x0e, 0xa0, 0x02, 0x12, 0x12, 0x00, 0x02, 0x11, 0x13, 0x05, 0x23,
  0x04, 0x00, 0x61, 0x04, 0x50, 0x41, 0x33, 0x42, 0x03, 0x04, 0x81, 0x0f,
  0x13, 0x04, 0x23, 0x02, 0x0c, 0x50, 0x73, 0x04, 0x31, 0x33, 0x09, 0xe1,
  0xdf, 0x90, 0x2f, 0x53, 0x02, 0xf3, 0x3c, 0x16, 0x03, 0x04, 0xe1, 0x1f,
  0xa1, 0x05, 0x7f, 0x01, 0x21, 0x50, 0x41, 0x32, 0x1d, 0x73, 0x04, 0x50,
  0x41, 0x4d, 0x17, 0x73, 0x04, 0x50, 0x41, 0x2e, 0x11, 0x71, 0x00, 0x61,
  0x00, 0x14, 0x72, 0x00, 0x03, 0x05, 0x14, 0x72, 0x00, 0x02, 0x00, 0x31,
  0x37, 0x24, 0x71, 0x00, 0x81, 0x20, 0x13, 0x04, 0x9f, 0x01, 0xaf, 0x78,
  0x05, 0x04, 0x63, 0x05, 0x3b, 0xce, 0x03, 0x05, 0xc0, 0x20, 0x1c, 0x5d,
  0x31, 0xff, 0x0b, 0xdf, 0x01, 0xf0, 0x13, 0x05, 0x23, 0x04, 0x00, 0x7f,
  0x01, 0x8f, 0x02, 0x00, 0x12, 0x12, 0xa0, 0x61, 0x02, 0x13, 0x02, 0x20,
  0x1c, 0x5d, 0x31, 0xff, 0x0a, 0xa3, 0x02, 0x41, 0x01, 0x0a, 0x83, 0x02,
  0x21, 0x1c, 0x41, 0xd1, 0xff, 0x71, 0x02, 0xa0, 0xe1, 0x00, 0x71, 0x02,
  0xa0, 0x61, 0x04, 0x61, 0x05, 0x61, 0x54, 0xf8, 0x54, 0xff, 0x01, 0xf0,
  0x13, 0x04, 0xe1, 0x00, 0x92, 0x10, 0x13, 0x05, 0x50, 0x80, 0x04, 0xd1,
  0xff, 0x3c, 0x7a, 0x73, 0x04, 0x50, 0x71, 0x54, 0x71, 0x05, 0x71, 0x04,
  0xa0, 0x61, 0x02, 0x61, 0x03, 0x60, 0x02, 0x61, 0x05, 0x61, 0x04, 0x60,
  0x0d, 0x60, 0x0c, 0x60, 0x0b, 0x1c, 0xa4, 0x70, 0x0b, 0x70, 0x0c, 0x70,
  0x0d, 0x71, 0x04, 0x71, 0x05, 0x70, 0x02, 0x71, 0x03, 0x71, 0x02, 0xa0,
  0xa1, 0x20, 0x61, 0x00, 0x81, 0x42, 0x13, 0x04, 0xe1, 0x00, 0x91, 0x1d,
  0x13, 0x05, 0x71, 0x00, 0xdf, 0x01, 0xf0, 0x83, 0x04, 0x13, 0x04, 0xe1,
  0x00, 0x93, 0x05, 0x13, 0x05, 0x03, 0x02, 0xe0, 0x83, 0x02, 0x22, 0x02,
  0x80, 0x23, 0x25, 0x00, 0x20, 0x1d, 0x29, 0x20, 0x1d, 0x2e, 0x03, 0x03,
  0xc0, 0xe1, 0x7f, 0x82, 0x02, 0x12, 0x02, 0x23, 0x02, 0x03, 0x23, 0x00,
  0x03, 0xa3, 0x02, 0xc1, 0x98, 0x65, 0x02, 0xf1, 0xff, 0x12, 0x0d, 0xe1,
  0xf0, 0x22, 0x0b, 0x02, 0x12, 0x0c, 0x61, 0x0e, 0x23, 0x0e, 0x81, 0x06,
  0x98, 0x03, 0x03, 0x20, 0x1d, 0x29, 0xe1, 0x0f, 0xd2, 0x0c, 0x98, 0x03,
  0x03, 0x20, 0x1d, 0x29, 0x16, 0x71, 0x0e, 0x23, 0x00, 0x06, 0x20, 0x1d,
  0x2e, 0x02, 0x0d, 0x20, 0x1d, 0x29, 0xe1, 0xf0, 0x12, 0x0c, 0x72, 0x0b,
  0x68, 0x0b, 0xd3, 0x23, 0x00, 0x04, 0x20, 0x1d, 0x2e, 0x53, 0x02, 0xb6,
  0xa0, 0xc0, 0xc0, 0xc0, 0xc0, 0xa0, 0x78, 0x25, 0x02, 0xcf, 0x02, 0x82,
  0x02, 0x23, 0x25, 0x00, 0x9f, 0x00, 0x02, 0x63, 0x25, 0xd1, 0x80, 0x12,
  0x02, 0xa0, 0x00, 0x00, 0x00, 0x44, 0x40, 0x40, 0xaa, 0x00, 0x00, 0xae,
  0xae, 0xa0, 0x6c, 0x6c, 0x40, 0x82, 0x48, 0x20, 0xcc, 0xea, 0x60, 0x44,
  0x00, 0x00, 0x24, 0x44, 0x20, 0x42, 0x22, 0x40, 0xa4, 0xa0, 0x00, 0x04,
  0xe4, 0x00, 0x00, 0x04, 0x80, 0x00, 0xe0, 0x00, 0x00, 0x00, 0x40, 0x22,
  0x48, 0x80, 0x6a, 0xaa, 0xc0, 0x4c, 0x44, 0x40, 0xc2, 0x48, 0xe0, 0xc2,
  0x42, 0xc0, 0xaa, 0xe2, 0x20, 0xe8, 0xc2, 0xc0, 0x68, 0xea, 0xe0, 0xe2,
  0x48, 0x80, 0xea, 0xea, 0xe0, 0xea, 0xe2, 0xc0, 0x04, 0x04, 0x00, 0x04,
  0x04, 0x80, 0x24, 0x84, 0x20, 0x0e, 0x0e, 0x00, 0x84, 0x24, 0x80, 0xe2,
  0x40, 0x40, 0x4a, 0xe8, 0x60, 0x4a, 0xea, 0xa0, 0xca, 0xca, 0xc0, 0x68,
  0x88, 0x60, 0xca, 0xaa, 0xc0, 0xe8, 0xe8, 0xe0, 0xe8, 0xe8, 0x80, 0x68,
  0xea, 0x60, 0xaa, 0xea, 0xa0, 0xe4, 0x44, 0xe0, 0x22, 0x2a, 0x40, 0xaa,
  0xca, 0xa0, 0x88, 0x88, 0xe0, 0xae, 0xea, 0xa0, 0xae, 0xee, 0xa0, 0x4a,
  0xaa, 0x40, 0xca, 0xc8, 0x80, 0x4a, 0xae, 0x60, 0xca, 0xec, 0xa0, 0x68,
  0x42, 0xc0, 0xe4, 0x44, 0x40, 0xaa, 0xaa, 0x60, 0xaa, 0xa4, 0x40, 0xaa,
  0xee, 0xa0, 0xaa, 0x4a, 0xa0, 0xaa, 0x44, 0x40, 0xe2, 0x48, 0xe0, 0xe8,
  0x88, 0xe0, 0x08, 0x42, 0x00, 0xe2, 0x22, 0xe0, 0x4a, 0x00, 0x00, 0x00,
  0x00, 0xe0, 0x84, 0x00, 0x00, 0x0c, 0x6a, 0xe0, 0x8c, 0xaa, 0xc0, 0x06,
  0x88, 0x60, 0x26, 0xaa, 0x60, 0x06, 0xac, 0x60, 0x24, 0xe4, 0x40, 0x06,
  0xae, 0x24, 0x8c, 0xaa, 0xa0, 0x40, 0x44, 0x40, 0x20, 0x22, 0xa4, 0x8a,
  0xcc, 0xa0, 0xc4, 0x44, 0xe0, 0x0e, 0xee, 0xa0, 0x0c, 0xaa, 0xa0, 0x04,
  0xaa, 0x40, 0x0c, 0xaa, 0xc8, 0x06, 0xaa, 0x62, 0x06, 0x88, 0x80, 0x06,
  0xc6, 0xc0, 0x4e, 0x44, 0x60, 0x0a, 0xaa, 0x60, 0x0a, 0xae, 0x40, 0x0a,
  0xee, 0xe0, 0x0a, 0x44, 0xa0, 0x0a, 0xa6, 0x24, 0x0e, 0x6c, 0xe0, 0x64,
  0x84, 0x60, 0x44, 0x04, 0x40, 0xc4, 0x24, 0xc0, 0x6c, 0x00, 0x00, 0xee,
  0xee, 0xe0, 0x4e, 0x44, 0x44, 0x44, 0x44, 0xe4, 0x62, 0x00, 0x04, 0x81,
  0x18, 0x13, 0x04, 0x72, 0x00, 0x04, 0x91, 0x00, 0x13, 0x05, 0xf8, 0x54,
  0x50, 0x12, 0x16, 0x22, 0x18, 0xff, 0x22, 0x19, 0xff, 0x23, 0x03, 0x00,
  0x20, 0x20, 0x02, 0x02, 0x03, 0x41, 0xff, 0x29, 0x02, 0x02, 0x41, 0xfc,
  0x24, 0x02, 0x18, 0x41, 0xff, 0x0a, 0x03, 0x03, 0x12, 0x18, 0x22, 0x17,
  0x01, 0x21, 0x1e, 0xa2, 0x62, 0x17, 0x02, 0x17, 0xa2, 0x16, 0x7f, 0x01,
  0x0f, 0x80, 0x04, 0x62, 0x18, 0x72, 0x17, 0x02, 0x18, 0x12, 0x19, 0x3e,
  0xb8, 0x22, 0x18, 0xff, 0x63, 0x03, 0x03, 0x03, 0x41, 0xc8, 0xc5, 0x02,
  0x19, 0x41, 0xff, 0x01, 0xa0, 0x13, 0x03, 0x03, 0x04, 0xe1, 0xe0, 0x81,
  0x02, 0x13, 0x04, 0x50, 0x13, 0x02, 0x12, 0x18, 0x20, 0x1f, 0xcb, 0x63,
  0x03, 0x61, 0x03, 0x03, 0x02, 0x13, 0x03, 0x20, 0x20, 0x02, 0x02, 0x03,
  0x90, 0x09, 0x02, 0x02, 0x13, 0x02, 0x71, 0x03, 0x21, 0x1e, 0xd4, 0x02,
  0x19, 0x13, 0x03, 0x02, 0x16, 0x13, 0x02, 0x03, 0x02, 0x41, 0x01, 0x09,
  0x22, 0x03, 0xff, 0x22, 0x02, 0xfa, 0x21, 0x1f, 0x0e, 0x03, 0x03, 0x81,
  0x01, 0x12, 0x02, 0x22, 0x03, 0x00, 0x20, 0x20, 0x26, 0x63, 0x03, 0x53,
  0x02, 0xe1, 0x03, 0x05, 0xd0, 0xd1, 0x80, 0x13, 0x03, 0x03, 0x04, 0xe1,
  0xe0, 0xd1, 0x02, 0xd0, 0x13, 0x02, 0x22, 0x03, 0x00, 0x02, 0x19, 0x12,
  0x02, 0x20, 0x1f, 0xa7, 0x02, 0x18, 0x13, 0x03, 0x20, 0x20, 0x02, 0x60,
  0x02, 0x60, 0x03, 0x22, 0x03, 0xff, 0x22, 0x02, 0xfc, 0x20, 0x20, 0x26,
  0x71, 0x00, 0x90, 0x04, 0x71, 0x00, 0x3f, 0x32, 0xa0, 0x03, 0x02, 0xf0,
  0xe1, 0xfe, 0x13, 0x04, 0x03, 0x03, 0xf0, 0x13, 0x05, 0xe1, 0x00, 0xf0,
  0x13, 0x54, 0xa0, 0x60, 0x00, 0x61, 0x04, 0x03, 0x04, 0xe1, 0x80, 0x13,
  0x04, 0x22, 0x00, 0x80, 0x50, 0x14, 0x62, 0x00, 0x63, 0x04, 0x6f, 0x00,
  0xf7, 0x71, 0x04, 0x70, 0x00, 0xa0, 0x61, 0x02, 0x61, 0x03, 0x60, 0x00,
  0x61, 0x04, 0x61, 0x05, 0x61, 0x54, 0x22, 0x7c, 0x00, 0x03, 0x54, 0x12,
  0x7d, 0x03, 0x05, 0x12, 0x7e, 0x03, 0x04, 0xe1, 0x80, 0x12, 0x7f, 0x20,
  0x01, 0x08, 0x71, 0x54, 0x71, 0x05, 0x71, 0x04, 0x70, 0x00, 0x71, 0x03,
  0x71, 0x02, 0xa0, 0x61, 0x05, 0x61, 0x04, 0x60, 0x00, 0x20, 0x1f, 0x4d,
  0x20, 0x1f, 0x5f, 0x03, 0x04, 0xd1, 0x80, 0x12, 0x00, 0x02, 0x02, 0x14,
  0x62, 0x00, 0x02, 0x03, 0x14, 0x20, 0x1f, 0x7a, 0x70, 0x00, 0x71, 0x04,
  0x71, 0x05, 0xa0, 0x61, 0x05, 0x61, 0x04, 0x60, 0x02, 0x22, 0x02, 0x00,
  0x61, 0x02, 0x61, 0x03, 0x03, 0x02, 0x13, 0x03, 0x02, 0x02, 0xc0, 0xc0,
  0x13, 0x02, 0x61, 0x02, 0x20, 0x1f, 0x4d, 0x20, 0x1f, 0x5f, 0x71, 0x02,
  0x71, 0x03, 0x20, 0x1f, 0x4d, 0x20, 0x1f, 0x7a, 0x71, 0x02, 0x62, 0x02,
  0x8a, 0x02, 0xd9, 0x70, 0x02, 0x71, 0x04, 0x71, 0x05, 0xa0, 0x61, 0x54,
  0x61, 0x05, 0x61, 0x04, 0x03, 0x03, 0xf0, 0xe1, 0xfe, 0x13, 0x04, 0x02,
  0x10, 0x91, 0x00, 0x13, 0x05, 0xf8, 0x54, 0x50, 0x12, 0x02, 0x63, 0x04,
  0x50, 0x12, 0x03, 0x71, 0x04, 0x71, 0x05, 0x71, 0x54, 0xa0, 0x61, 0x03,
  0x61, 0x02, 0x03, 0x03, 0x13, 0x02, 0x02, 0x10, 0xc0, 0xd1, 0x80, 0x13,
  0x03, 0x20, 0x1f, 0xa7, 0x71, 0x02, 0x71, 0x03, 0xa0, 0x21, 0x18, 0xdf,
  0x13, 0x03, 0xa1, 0x09, 0x03, 0x03, 0x7f, 0x01, 0x03, 0xd1, 0xff, 0xa0,
  0xe0, 0x81, 0x59, 0x61, 0x00, 0x23, 0x00, 0x20, 0x91, 0x00, 0x61, 0x00,
  0xa0, 0x28, 0x49, 0x28, 0x6e, 0x28, 0x73, 0x28, 0x76, 0x28, 0x87, 0x28,
  0x99, 0x28, 0xe2, 0x28, 0xf6, 0x28, 0x6b, 0x21, 0x18, 0xdf, 0x51, 0x04,
  0x21, 0x20, 0x96, 0x21, 0x20, 0x96, 0x02, 0x0e, 0x12, 0x81, 0x02, 0x0f,
  0x12, 0x82, 0x22, 0x80, 0x00, 0x22, 0x83, 0x03, 0x21, 0x20, 0x96, 0x22,
  0x80, 0x42, 0x22, 0x81, 0x1d, 0x22, 0x82, 0x00, 0x22, 0x83, 0x03, 0x21,
  0x20, 0x96, 0xe1, 0x00, 0xa0, 0x02, 0x83, 0x41, 0x30, 0xab, 0x02, 0x82,
  0xc0, 0xd2, 0x80, 0xe1, 0x7f, 0x90, 0xa2, 0x02, 0x80, 0x13, 0x02, 0x02,
  0x81, 0x13, 0x03, 0x02, 0x82, 0x13, 0x65, 0xf9, 0x01, 0x60, 0x7c, 0x60,
  0x7d, 0x60, 0x7e, 0x60, 0x7f, 0x22, 0x7c, 0x00, 0x03, 0x65, 0x12, 0x7d,
  0x03, 0x03, 0x12, 0x7e, 0x03, 0x02, 0x12, 0x7f, 0x61, 0x0e, 0x23, 0x0e,
  0x81, 0x20, 0x01, 0x08, 0x71, 0x0e, 0x70, 0x7f, 0x70, 0x7e, 0x70, 0x7d,
  0x70, 0x7c, 0xd9, 0x01, 0x28, 0x96, 0x02, 0x8c, 0x13, 0x65, 0x02, 0x88,
  0x13, 0x03, 0x02, 0x84, 0x13, 0x02, 0x02, 0x80, 0x20, 0x1c, 0xa4, 0x21,
  0x20, 0x96, 0x02, 0x83, 0xa1, 0x03, 0xd2, 0x80, 0x90, 0x98, 0x02, 0x82,
  0xe1, 0xfe, 0x90, 0x92, 0x02, 0x82, 0x12, 0x0f, 0x02, 0x81, 0x12, 0x0e,
  0x21, 0x04, 0x80


	};
	
	//one datafile to send
	static const uint8_t datafile[] = {
 0x00, 0x02, 0x00, 0x20, 0x47, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00,
  0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00,
  0x43, 0x00, 0x00, 0x00, 0x08, 0xde, 0x30, 0xbf, 0xfd, 0xe7, 0x00, 0x23,
  0x10, 0xb5, 0x0c, 0x49, 0x0c, 0x4a, 0x0d, 0x48, 0x52, 0x1a, 0x93, 0x42,
  0x0a, 0xd1, 0x00, 0x23, 0x18, 0x00, 0x0b, 0x49, 0x0b, 0x4a, 0x52, 0x1a,
  0x93, 0x42, 0x07, 0xd1, 0x00, 0xf0, 0x14, 0xf8, 0x30, 0xbf, 0xfd, 0xe7,
  0xc4, 0x58, 0xcc, 0x50, 0x04, 0x33, 0xee, 0xe7, 0xcc, 0x18, 0x20, 0x60,
  0x04, 0x33, 0xf1, 0xe7, 0x00, 0xff, 0xff, 0x1f, 0x00, 0xff, 0xff, 0x1f,
  0x84, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0x1f, 0x00, 0xff, 0xff, 0x1f,
  0xff, 0x23, 0x1b, 0x06, 0x9c, 0x46, 0xa0, 0x20, 0x00, 0x23, 0x80, 0x21,
  0xf0, 0xb5, 0x85, 0xb0, 0x02, 0x93, 0xc0, 0x05, 0xfe, 0x23, 0x1b, 0x06,
  0x01, 0x93, 0x3c, 0xe0, 0x83, 0xfb, 0x03, 0x45, 0x82, 0xfb, 0x02, 0x67,
  0x83, 0xfb, 0x02, 0x23, 0x1b, 0x02, 0x12, 0x0e, 0x1a, 0x43, 0x24, 0x0e,
  0x2b, 0x02, 0x36, 0x0e, 0x3d, 0x02, 0x1c, 0x43, 0x2e, 0x43, 0xa3, 0x1b,
  0x01, 0x9c, 0x52, 0x00, 0x1b, 0x19, 0x64, 0x46, 0x12, 0x19, 0x83, 0xfb,
  0x03, 0x67, 0x82, 0xfb, 0x02, 0x45, 0x2d, 0x02, 0x24, 0x0e, 0x2c, 0x43,
  0x36, 0x0e, 0x3d, 0x02, 0x2e, 0x43, 0x80, 0x25, 0xa4, 0x19, 0xed, 0x04,
  0xac, 0x42, 0x3e, 0xdc, 0x03, 0x9c, 0x01, 0x3c, 0x03, 0x94, 0x00, 0x2c,
  0xd6, 0xd1, 0x49, 0x08, 0x39, 0xb9, 0x02, 0x9b, 0x01, 0x33, 0xdb, 0xb2,
  0x02, 0x93, 0x0c, 0x2b, 0x10, 0xd0, 0x01, 0x30, 0x80, 0x31, 0x80, 0x23,
  0x01, 0x9a, 0x5b, 0x03, 0xd3, 0x18, 0x01, 0x93, 0x80, 0x23, 0x01, 0x9a,
  0x5b, 0x04, 0x9a, 0x42, 0x08, 0xd0, 0x11, 0x24, 0x62, 0x46, 0x01, 0x9b,
  0x03, 0x94, 0xe1, 0xe7, 0x02, 0x91, 0x05, 0x30, 0x80, 0x21, 0xec, 0xe7,
  0x80, 0x22, 0x52, 0x03, 0x63, 0x46, 0x94, 0x46, 0x01, 0x9a, 0x63, 0x44,
  0x9c, 0x46, 0x93, 0x42, 0xac, 0xd1, 0x40, 0xf2, 0x80, 0x13, 0x0b, 0x4a,
  0xc0, 0xf2, 0x00, 0x03, 0x11, 0x00, 0x18, 0x78, 0x10, 0x70, 0x13, 0x78,
  0xdb, 0xb2, 0xff, 0x2b, 0xfa, 0xd1, 0x0b, 0x70, 0x0a, 0x78, 0xff, 0x2a,
  0xfb, 0xd0, 0xff, 0xf7, 0x69, 0xff, 0x05, 0xb0, 0xf0, 0xbd, 0x03, 0x78,
  0x0b, 0x43, 0x03, 0x70, 0xc1, 0xe7, 0x00, 0xbf, 0x4c, 0x00, 0x00, 0x40,
  0xff, 0x00, 0x00, 0x00




	};
		
	uint32_t block, phase, byte, j;
	const void *rxedData;
	
	for (block = 0; block < (sizeof(gamedata) + 511) / 512; block++) {
		
		pr("writing block %u (game)\n", block);
		i = mapleCommsWriteBlock(0x41, 0, block, gamedata + block * 512);
		pr("wr = 0x%08x\n", i);
	}

	for (i = 0; i < (sizeof(datafile) + 511) / 512; i++, block++) {
		
		uint32_t j;
		
		pr("writing block %u (data)\n", block);
		j = mapleCommsWriteBlock(0x41, 0, block, datafile + i * 512);
		pr("wr = 0x%08x\n", j);
	}


	//read fat
	block = 254;
	pr("reading fat\n");
	i = mapleCommsReadBlock(0x41, 0, block, &rxedData);
	pr("fat rd = 0x%08x\n", i);
	memcpy(buf, rxedData, 512);
	
	//fat is little endian
	//link in blocks
	for (i = 0; i < (sizeof(gamedata) + 511) / 512 - 1; i++)
		((uint16_t*)buf)[i] = i + 1;
	((uint16_t*)buf)[i++] = 0xfffa;
	
	for (j = 0; j < (sizeof(datafile) + 511) / 512 - 1; j++, i++)
		((uint16_t*)buf)[i] = i + 1;
	((uint16_t*)buf)[i++] = 0xfffa;
	
	for (;i < 240; i++)
		((uint16_t*)buf)[i] = 0xfffc;	//free all other blocks
	
	//write fat
	i = mapleCommsWriteBlock(0x41, 0, block, buf);
	pr("fat wr = 0x%08x\n", i);
	
	
	block = 253;
	pr("reading dir\n");
	i = mapleCommsReadBlock(0x41, 0, block, &rxedData);
	pr("dir rd = 0x%08x\n", i);
	memcpy(buf, rxedData, 512);
	
	//write directory entry
	buf[0x00] = 0xcc;	//game
	buf[0x01] = 0;		//not copy protected
	buf[0x02] = 0;		//block_num.lo
	buf[0x03] = 0;		//block_num.hi
	buf[0x04] = 'u';	//name[12]
	buf[0x05] = 'M';
	buf[0x06] = '2';
	buf[0x07] = '3';
	buf[0x08] = '\0';
	buf[0x09] = '\0';
	buf[0x0a] = '\0';
	buf[0x0b] = '\0';
	buf[0x0c] = '\0';
	buf[0x0d] = '\0';
	buf[0x0e] = '\0';
	buf[0x0f] = '\0';
	buf[0x10] = 0x20;	//top 2 digits of year
	buf[0x11] = 0x17;	//bottom 2 digits of year
	buf[0x12] = 0x05;	//month
	buf[0x13] = 0x31;	//day
	buf[0x14] = 0x23;	//hour
	buf[0x15] = 0x05;	//minute
	buf[0x16] = 0x33;	//second
	buf[0x17] = 0x02;	//2 = wednesday
	buf[0x18] = ((sizeof(gamedata) + 511) / 512) >> 0;		//file size in block LO
	buf[0x19] = ((sizeof(gamedata) + 511) / 512) >> 8;		//file size in block HI
	buf[0x1a] = (0x200 / 512) >> 0;					//offset of header in file in blocks LO
	buf[0x1b] = (0x200 / 512) >> 8;					//offset of header in file in blocks HI
	buf[0x1c] = 0;		//SBZ
	buf[0x1d] = 0;
	buf[0x1e] = 0;
	buf[0x1f] = 0;

	//write directory entry
	buf[0x20] = 0x33;	//data
	buf[0x21] = 0;		//not copy protected
	buf[0x22] = buf[0x18];		//block_num.lo
	buf[0x23] = buf[0x19];		//block_num.hi
	buf[0x24] = 'M';	//name[12]
	buf[0x25] = 'N';
	buf[0x26] = 'D';
	buf[0x27] = 'L';
	buf[0x28] = 'B';
	buf[0x29] = 'R';
	buf[0x2a] = 'T';
	buf[0x2b] = '.';
	buf[0x2c] = 'M';
	buf[0x2d] = '2';
	buf[0x2e] = '3';
	buf[0x2f] = '\0';
	buf[0x30] = 0x20;	//top 2 digits of year
	buf[0x31] = 0x17;	//bottom 2 digits of year
	buf[0x32] = 0x05;	//month
	buf[0x33] = 0x31;	//day
	buf[0x34] = 0x23;	//hour
	buf[0x35] = 0x05;	//minute
	buf[0x36] = 0x33;	//second
	buf[0x37] = 0x02;	//2 = wednesday
	buf[0x38] = ((sizeof(datafile) + 511) / 512) >> 0;		//file size in block LO
	buf[0x39] = ((sizeof(datafile) + 511) / 512) >> 8;		//file size in block HI
	buf[0x3a] = 0;					//offset of header in file in blocks LO
	buf[0x3b] = 0;					//offset of header in file in blocks HI
	buf[0x3c] = 0;		//SBZ
	buf[0x3d] = 0;
	buf[0x3e] = 0;
	buf[0x3f] = 0;



	//write dir
	i = mapleCommsWriteBlock(0x41, 0, block, buf);
	pr("dir wr = 0x%08x\n", i);
	memcpy(buf, rxedData, 512);

out:


	pr(" DONE 1 ");
	while(1);
	
	
	
	for (j = 32; j < 128+32; j++) {
		//read the dumped data	
		i = mapleCommsReadBlock(0x41, 0, j, &rxedData);
		pr("read[%3u]=%u\n", j, i);
		memcpy(buf, rxedData, 512);
		for (i = 0; i < 512; i += 4) {
			
			uint8_t t;
			
			t = buf[i + 0];
			buf[i + 0] = buf[i + 3];
			buf[i + 3] = t;
			
			t = buf[i + 1];
			buf[i + 1] = buf[i + 2];
			buf[i + 2] = t;
		}
		
		for (i = 0; i < 512; i++)
		{
			if (!(i & 0x0f))
				pr("\n%07x", i + (j - 32) * 512);
			pr(" %02X", buf[i]);
		}
	}
	
	pr("done\n");
}

void main(void)
{
	volatile uint32_t t = 0;
	
	
	for (t = 0; t < 0x8000; t += 2)
		t--;
	
	initHw();
	mapleWireInit();
	stuff();
	
	while(1);
}




