#include <dal.h>
#include <kal.h>
#include <boot.h>
#include <StringMgr.h>
#include <FatalAlert.h>
#include "palmcard.h"
#include "printf.h"
#include "common.h"
#include "port.h"


#define PORT_NAME_UART				'ser0'
#define PORT_NAME_IR				'sir0'

#define RX_BUF_SZ					64			//we need to be able to RX this many bytes nonstop at max baudrate

struct OpenPort {
	
	union UartCfg cfg, prevCfg;
	uint32_t txTimeout;			//cts timeout
	uint32_t errors;
	const struct VdrvRecvQueue *recvQ;
	
	enum UartPort which;
};

struct Globals {					//limited to 8 bytes by 'amdi' resource. change that to get more
	struct OpenPort* ser;	//NULL if not open
	struct OpenPort* sir;	//NULL if not open
};

static struct Globals* __attribute__((const)) serGlobalsGet(void)
{
	register void* r9 asm("r9");
	void**** ret = r9;
	
	return (struct Globals*)&ret[0][MY_LIB_ID / 4][0x18/4];
}

static void portPrvRxfunc(void *userData, uint16_t *rawBuf, uint32_t nItems)
{
	struct OpenPort *prt = (struct OpenPort*)userData;
	uint32_t i, spaceAvail, lineErrors = 0, lineErrorsThisBlock = 0, tmpIdx = 0;
	uint8_t tmp[16];
	
	spaceAvail = prt->recvQ->RecvQueueGetQueueSpaceF(prt->recvQ->userData);
	
	if (spaceAvail < nItems) {
		lineErrors |= VDRV_LINE_ERR_SW_RX_OVERFLOW;
		nItems = spaceAvail;
	}
	
	for (i = 0; i < nItems; i++) {
		
		if (rawBuf[i] & UART_BIT_MASK_PAR_ERR)
			lineErrorsThisBlock |= VDRV_LINE_ERR_PARITY;
		
		if (rawBuf[i] & UART_BIT_MASK_BRK_RXED)
			lineErrorsThisBlock |= VDRV_LINE_ERR_BREAK;
		
		if (rawBuf[i] & UART_BIT_MASK_FRM_ERR)
			lineErrorsThisBlock |= VDRV_LINE_ERR_FRAMING;
		
		if (rawBuf[i] & UART_BIT_MASK_RX_OVERRUN)
			lineErrorsThisBlock |= VDRV_LINE_ERR_HW_RX_OVERFLOW;
		
		tmp[tmpIdx++] = rawBuf[i];
		
		if (tmpIdx == sizeof(tmp) || (i == nItems - 1 && tmpIdx)) {
			
			prt->recvQ->RecvQueueBlockRxedF(prt->recvQ->userData, tmp, tmpIdx, lineErrorsThisBlock);
			lineErrors |= lineErrorsThisBlock;
			
			lineErrorsThisBlock = 0;
			tmpIdx = 0;
		}
	}
	
	if (!nItems && lineErrors)
		prt->recvQ->RecvQueueBlockRxedF(prt->recvQ->userData, NULL, 0, lineErrors);
	
	prt->errors |= lineErrors;
}

static bool serPrvUpdateConfig(struct OpenPort *prt)
{
	if (prt->prevCfg.raw32 == prt->cfg.raw32)
		return true;
	
	if (!repalmUartConfig(prt->which, &prt->cfg, portPrvRxfunc, prt))
		return false;
	
	prt->prevCfg.raw32 = prt->cfg.raw32;
	return true;
}

static bool serDeinit(struct OpenPort *prt) 
{
	prt->cfg.rxEn = 0;
	prt->cfg.txEn = 0;
	
	return serPrvUpdateConfig(prt);
}

static Err portCfgSettings(struct OpenPort *prt, uint32_t reqCfg)
{
	prt->cfg.stopBits = (reqCfg & VDRV_UART_SETTING_2_STOP_BITS) ? 3 : 1;
	prt->cfg.parEna = !!(reqCfg & VDRV_UART_SETTING_PARITY_ON);
	prt->cfg.parEven = !!(reqCfg & VDRV_UART_SETTING_PARITY_EVEN);
	switch (reqCfg & VDRV_UART_SETTING_CHAR_SZ_MASK) {
		case VDRV_UART_SETTING_5_BIT_CHARS:
			prt->cfg.charBits = 0;
			break;
		case VDRV_UART_SETTING_6_BIT_CHARS:
			prt->cfg.charBits = 1;
			break;
		case VDRV_UART_SETTING_7_BIT_CHARS:
			prt->cfg.charBits = 2;
			break;
		case VDRV_UART_SETTING_8_BIT_CHARS:
			prt->cfg.charBits = 3;
			break;
		default:
			return serErrNotSupported;
	}
	
	return serPrvUpdateConfig(prt) ? errNone : serErrNotSupported;
}

Err portOpen(void **drvrDataP /* OUT */, const struct VdrvConfig *cfg, const struct VdrvRecvQueue* recvQ)
{
	struct Globals *g = serGlobalsGet();
	struct OpenPort *prt, **prtP;
	enum UartPort which;
	Err e;
	
	
	if (!drvrDataP || !cfg || !recvQ || !cfg->baudrate)
		return serErrBadParam;
	
	logt("Opening port with baudrate %u\n", cfg->baudrate);
	
	switch (cfg->portName) {
		//for ir port set cr3.USART_CR3_IREN and disable cr1.USART_CR1_RE
		
		case PORT_NAME_IR:
			which = UartPortIrDA;
			logi("opening IR\n");
			prtP = &g->sir;
			break;
			
		case PORT_NAME_UART:
			which = UartPortCradleSerial;
			logi("opening UART\n");
			prtP = &g->ser;
			break;
		
		default:
			return serErrBadParam;
	}
			
	prt = MemChunkNew(0, sizeof(struct OpenPort), 0x200);
	if (!prt)
		return memErrNotEnoughSpace;
	
	prt->which = which;
	prt->cfg.baudrate = cfg->baudrate;
	e = portCfgSettings(prt, VDRV_UART_SETTING_8_BIT_CHARS);
	if (e) {
		MemChunkFree(prt);
		return e;
	}
	*prtP = prt;
		
	prt->txTimeout = 5000;
	prt->recvQ = recvQ;
	prt->cfg.rxEn = 1;
	prt->cfg.txEn = (prt != g->sir);
	serPrvUpdateConfig(prt);

	*drvrDataP = prt;
	return errNone;
}

struct OpenPort* portVerify(void *drvrData)
{
	if (!drvrData || (serGlobalsGet()->ser != drvrData && serGlobalsGet()->sir != drvrData))
		return NULL;
	
	return (struct OpenPort*)drvrData;
}

Err portClose(void *drvrData)
{
	struct OpenPort *prt = portVerify(drvrData);
	struct Globals *g = serGlobalsGet();
	
	loge("Closing port\n");
	
	if (!prt)
		return serErrBadPort;
	
	prt->cfg.rxEn = 0;
	prt->cfg.txEn = 0;
	serPrvUpdateConfig(prt);
	
	if (g->ser == prt)
		g->ser = NULL;
	else if (g->sir == prt)
		g->sir = NULL;
	MemChunkFree(drvrData);
	
	return errNone;
}

Err portControl(void *drvrData, uint32_t controlOp /* VDRV_CTL_OP_* */, void* data, uint16_t *controlDataLenP /* never used */)
{
	struct OpenPort *prt = portVerify(drvrData);
	uint32_t t, *data32 = (uint32_t*)data;
	struct Globals *g = serGlobalsGet();
	uint16_t *data16 = (uint16_t*)data;
	uint8_t *data8 = (uint8_t*)data;
	
	if (!prt)
		return serErrBadPort;
	
	if (!data || !controlDataLenP)
		return serErrBadParam;
	
	switch (controlOp) {
		
		case VDRV_CTL_OP_SET_BAUDRATE:
			if (!*data32)
				return serErrBadParam;
			prt->cfg.baudrate = *data32;
			serPrvUpdateConfig(prt);
			return errNone;
		
		case VDRV_CTL_OP_SET_SETTINGS:
			return portCfgSettings(prt, *data32);
		
		case VDRV_CTL_OP_SET_CTS_TIMEOUT:
			prt->txTimeout = *data32;
			return errNone;
		
		case VDRV_CTL_OP_CLEAR_ERR:
			prt->errors = 0;
			return errNone;
		
		case VDRV_CTL_OP_SLEEP:
			serDeinit(prt);
			return errNone;
		
		case VDRV_CTL_OP_WAKE:
			prt->cfg.rxEn = 1;
			prt->cfg.txEn = (prt != g->sir);
			serPrvUpdateConfig(prt);
			return errNone;
		
		case VDRV_CTL_OP_GET_TX_FIFO_FREE:
			t = repalmUartGetSta(prt->which);
			if (t & UART_STA_BIT_TX_FIFO_EMPTY)
				*data32 = 4;		//why not?
			else if (t & UART_STA_BIT_TX_FIFO_FULL)
				*data32 = 0;
			else
				*data32 = 1;
			return errNone;
		
		case VDRV_CTL_OP_START_BREAK:
			repalmUartTx(prt->which, NULL, 1, true);
			return errNone;
		
		case VDRV_CTL_OP_STOP_BREAK:
			repalmUartTx(prt->which, NULL, 0, true);
			return errNone;
		
		case VDRV_CTL_OP_FLUSH_TX_FIFO:
			while (!(repalmUartGetSta(prt->which) & UART_STA_BIT_TX_FIFO_EMPTY));
			return errNone;
		
		case VDRV_CTL_OP_FLUSH_RX_FIFO:
			//nothing
			return errNone;
		
		case VDRV_CTL_OP_SEND_QUEUED_DATA:
			t = HALTimeGetSystemTime();
			do {
				if (repalmUartGetSta(prt->which) & UART_STA_BIT_TX_FIFO_EMPTY)
					return errNone;
			} while (HALTimeGetSystemTime() - t <= prt->txTimeout);
			return serErrTimeOut;
		
		case VDRV_CTL_OP_GET_BEST_TX_BLOCK_SZ:
			*data32 = 64;
			return errNone;
		
		case VDRV_CTL_OP_GET_MAX_RX_BLOCK_SZ:
			*data32 = RX_BUF_SZ;
			return errNone;
		
		case VDRV_CTL_OP_SET_DTR:
			return *data8 ? errNone : serErrNotSupported;
		
		case VDRV_CTL_OP_GET_DTR:
			*data8 = 1;
			return errNone;
		
		case VDRV_CTL_OP_WAIT_FOR_CONFIG:
			return errNone;
		
		case VDRV_CTL_OP_USB_GET_DEVICE_DESCR:
		case VDRV_CTL_OP_USB_GET_CONFIG_DESCR:
			return serErrNotSupported;
		
		case VDRV_CTL_OP_IRDA_ENABLE:
			if (prt == g->sir) {
				
				prt->cfg.rxEn = 1;
				prt->cfg.txEn = 0;
				serPrvUpdateConfig(prt);
				return errNone;
			}
			return serErrNotSupported;
		
		case VDRV_CTL_OP_IRDA_DISABLE:
			if (prt == g->sir) {
				
				prt->cfg.rxEn = 0;
				prt->cfg.txEn = 0;
				serPrvUpdateConfig(prt);
			}
			return errNone;
		
		case VDRV_CTL_OP_IR_RX_ENA:
			if (prt != g->sir)
				return serErrNotSupported;
			prt->cfg.rxEn = 1;
			prt->cfg.txEn = 0;
			serPrvUpdateConfig(prt);
			return errNone;
			
		case VDRV_CTL_OP_IR_RX_DIS:
			if (prt != g->sir)
				return serErrNotSupported;
			prt->cfg.rxEn = 0;
			prt->cfg.txEn = 1;
			serPrvUpdateConfig(prt);
			return errNone;

		case VDRV_CTL_OP_IRQ_DISABLE:
			prt->cfg.rxEn = 0;
			serPrvUpdateConfig(prt);
			return errNone;
		
		case VDRV_CTL_OP_IRQ_ENABLE:
			prt->cfg.rxEn = 1;
			if (prt == g->sir)
				prt->cfg.txEn = 0;
			serPrvUpdateConfig(prt);
			return errNone;
		
		case VDRV_CTL_OP_SET_NEW_RECV_QUEUE:
			prt->recvQ = data;
			//fallthrough
			
		case VDRV_CTL_OP_NOTIF_RX_BYTES_CONSUMED:
			//portRxEnable(prt, prt->recvQ->RecvQueueGetQueueSpaceF(prt->recvQ->userData) != 0);
			return errNone;
		
		case VDRV_CTL_OP_GET_IR_SPEED_BITMAPS:
			//we could use higher speeds, but palms never did, so deal with it later
			*data16 = VDRV_IR_SPEED_2K2 | VDRV_IR_SPEED_9K6 | VDRV_IR_SPEED_19K2 | VDRV_IR_SPEED_38K4 | VDRV_IR_SPEED_57K6 | VDRV_IR_SPEED_115K;
			return errNone;
		
		default:
			return serErrBadParam;
	}
}

uint32_t portWrite(void *drvrData, const void *buf, uint32_t size, Err *errP)
{
	struct OpenPort *prt = portVerify(drvrData);
	const uint8_t *src = (const uint8_t*)buf;
	struct Globals *g = serGlobalsGet();
	bool prevRxEn = false;
	uint32_t time;
	
	if (!prt)
		return serErrBadPort;
	
	if (!errP || (size && !buf))
		return serErrBadParam;
	
	*errP = errNone;
	
	if (!size)
		return 0;
	
	if (prt == g->sir) {	//tx on
		
		prevRxEn = prt->cfg.rxEn;
		
		if (!prt->cfg.txEn) {
			
			prt->cfg.txEn = 1;
			prt->cfg.rxEn = 0;
			
			(void)repalmUartConfig(prt->which, &prt->cfg, portPrvRxfunc, prt);
		}
	}
	
	time = HALTimeGetSystemTime();
	do {
		
		uint32_t doneNow = repalmUartTx(prt->which, src, size, false);

		src += doneNow;
		size -= doneNow;
		
	} while (size && (HALTimeGetSystemTime() - time <= prt->txTimeout));
	
	if (size)
		*errP = serErrTimeOut;
	
	if (prt == g->sir) {	//tx off
		
		prt->cfg.txEn = 0;
		prt->cfg.rxEn = prevRxEn;
		
		(void)repalmUartConfig(prt->which, &prt->cfg, portPrvRxfunc, prt);
	}
	
	return src - ((const uint8_t*)buf);
}

Err portGetStatus(void *drvrData, uint16_t* statusP /* VDRV_STA_* bitfield */ )
{
	struct OpenPort *prt = portVerify(drvrData);
	struct Globals *g = serGlobalsGet();
	uint32_t sta;
	
	if (!prt)
		return serErrBadPort;
	
	if (!statusP)
		return serErrBadParam;
	
	if (prt == g->sir)		//IrDA has no status
		return serErrNotSupported;
	
	sta = repalmUartGetSta(prt->which);
	
	*statusP = VDRV_STA_CTS | ((sta & UART_STA_BIT_TX_FIFO_EMPTY) ? VDRV_STA_TX_FIFO_EMPTY : 0) | ((sta & UART_STA_BIT_TX_FIFO_FULL) ? 0 : VDRV_STA_TX_FIFO_FULL);	//always ok to send
	
	loge("sta = 0x%04x\n", *statusP);
	
	return errNone;
}

Err portCustomControl(void *drvrData, ...)
{
	return serErrNotSupported;
}

Err portGetInfo(struct VdrvPortInfo* info)
{
	static const char portNameSrcSer[] = "rePalm UART port";
	static const char portNameSrcIr[] = "rePalm IR port";
	const char* portNameSrc;
	char* portNameDst;
	
	if (!info)
		return serErrBadParam;
	
	switch (info->portIdx) {
		case 0:
			info->portName = PORT_NAME_IR;
			info->portFlags = VDRV_PORT_FLAG_VISIBLE_IN_CNC_MGR | VDRV_PORT_FLAG_IRDA_CAPABLE | VDRV_PORT_FLAG_CAN_RUN_IN_BACKGROUND;
			info->maxBaudrate = 115200;
			info->handshakeBaudrate = 19200;
			portNameSrc = portNameSrcIr;
			break;
		
		case 1:
			info->portName = PORT_NAME_UART;
			info->portFlags = VDRV_PORT_FLAG_VISIBLE_IN_CNC_MGR | VDRV_PORT_FLAG_RS232_CAPABLE | VDRV_PORT_FLAG_CAN_RUN_IN_BACKGROUND | VDRV_PORT_FLAG_CRADLE_PORT;
			info->maxBaudrate = 2000000;
			info->handshakeBaudrate = 19200;
			portNameSrc = portNameSrcSer;
			break;
		
		default:
			return serErrBadPort;
	}
	
	info->driverVer = VDRV_VERSION_5;
	info->dbCrid = MY_CRID;
	portNameDst = info->portNameStr;
	if (portNameDst)
		StrCopy(portNameDst, portNameSrc);
	
	return errNone;
}

