#include <kernel/hw/RP2040/palmcard.h>
#include "common.h"
#include "printf.h"
#include "sdUtil.h"
#include "rp2040.h"
#include <boot.h>
#include "sdHw.h"
#include <dal.h>
#include <kal.h>


//TODO: insert/remove handling, error handling (retry on crc err, etc), power management (put card to deselected state on sleep, maybe)



struct HwData {
	
	uint32_t rdTimeoutTicks, wrTimeoutTicks;
	uint32_t rdTimeoutBits, wrTimeoutBits;
	uint32_t curClockRate;
	uint16_t timeoutBytes;
};


enum SdHwReadResult sdHwReadData(struct SdHwData *hwData, uint8_t *data, uint_fast16_t sz)
{
	struct HwData *hwd = (struct HwData*)hwData;
	
	switch (repalmSdioDataRx(data, 1, sz, hwd->rdTimeoutBits)) {
		case SdReadOK:
			return SdHwReadOK;
			
		case SdReadTimeout:
			return SdHwReadTimeout;
		
		case SdReadCrcErr:
			return SdHwReadCrcErr;
		
		case SdReadFramingError:
			return SdHwReadFramingError;
		
		default:
			return SdHwReadInternalError;
	}
}

enum SdHwWriteReply sdHwWriteData(struct SdHwData *hwData, const uint8_t *data, uint_fast16_t sz, bool isMultiblock)
{
	struct HwData *hwd = (struct HwData*)hwData;

	switch (repalmSdioDataTx(data, 1, sz, hwd->wrTimeoutBits)) {
		case SdWriteOK:
			return SdHwWriteAccepted;
		
		case SdWriteCrcError:
			return SdHwWriteCrcErr;
		
		case SdWriteCardError:
			return SdHwWriteError;
		
		case SdWriteDataTxTimeout:
		case SdWriteFramingError:
		default:
			return SdHwCommErr;
		
		case SdWriteBusyTimeout:
			return SdHwTimeout;
	}
}

enum SdHwCmdResult sdHwCmd(struct SdHwData *hwData, uint_fast8_t cmd, uint32_t param, bool cmdCrcRequired, enum SdHwRespType respTyp, void *respBufOut, enum SdHwDataDir dataDir, uint_fast16_t blockSz, uint32_t numBlocks)
{
	uint8_t rawReplyBuf[17], *dst8 = (uint8_t*)respBufOut;
	struct HwData *hwd = (struct HwData*)hwData;
	uint_fast8_t i, nReplyBits;
	bool crcReply = true;
	enum SdCmdRet ret;
	
	switch (respTyp) {
		case SdRespTypeNone:
		default:
			nReplyBits = 0;
			break;
		
		case SdRespTypeR3:
			crcReply = false;
			//fallthrough
		
		case SdRespTypeR1:
		case SdRespTypeR1withBusy:
		case SdRespTypeR7:
		case SdRespTypeSdR6:
			nReplyBits = 6 * 8;
			break;
		
		case SdRespTypeSdR2:
			nReplyBits = 17 * 8;
			break;
	}
	
	ret = repalmSdioCmd(cmd, param, rawReplyBuf, nReplyBits, crcReply);

	//logi("CMD%u (%08xh) -> %u\n", cmd, param, ret);
	switch (ret) {
		case SdCmdOK:
			break;
		
		case SdCmdRespTimeout:
			return SdHwCmdResultRespTimeout;
		
		case SdCmdRespCrcErr:
		default:
			return SdCmdInternalError;
	}

	switch (respTyp) {
		case SdRespTypeNone:
			return SdHwCmdResultOK;
		
		case SdRespTypeR1:
		case SdRespTypeR1withBusy:
			if (rawReplyBuf[0] != cmd)
				return SdCmdInternalError;
				
			*dst8 = sdPrvR1toSpiR1((((uint32_t)rawReplyBuf[1]) << 24) | (((uint32_t)rawReplyBuf[2]) << 16) | (((uint32_t)rawReplyBuf[3]) << 8) | (((uint32_t)rawReplyBuf[4]) << 0));
			if (respTyp == SdRespTypeR1)
				return SdHwCmdResultOK;
			else
				return repalmSdioBusyWait(hwd->wrTimeoutBits) ? SdHwCmdResultOK : SdHwCmdResultRespTimeout;
		
		case SdRespTypeSdR2:
			if (rawReplyBuf[0] != 0x3f)
				return SdCmdInternalError;
			
			for (i = 0; i < 16; i++)
				*dst8++ = rawReplyBuf[i + 1];
			
			return SdHwCmdResultOK;
		
		case SdRespTypeR3:
			if (rawReplyBuf[0] != 0x3f || rawReplyBuf[5] != 0xff)
				return SdCmdInternalError;
			
			for (i = 0; i < 4; i++)
				*dst8++ = rawReplyBuf[i + 1];
			return SdHwCmdResultOK;
		
		case SdRespTypeSdR6:
			if (rawReplyBuf[0] != 0x03)
				return SdCmdInternalError;
			
			for (i = 0; i < 4; i++)
				*dst8++ = rawReplyBuf[i + 1];
			return SdHwCmdResultOK;
		
		case SdRespTypeR7:
			if (rawReplyBuf[0] != 0x08 || rawReplyBuf[1] || rawReplyBuf[2] || (rawReplyBuf[3] & 0xf0))
				return SdCmdInternalError;
			
			for (i = 0; i < 4; i++)
				*dst8++ = rawReplyBuf[i + 1];
			return SdHwCmdResultOK;
		
		default:
			return SdCmdInternalError;
	}
}

uint32_t sdHwGetMaxBlocksAtOnce(struct SdHwData *hwData)
{
	return 0x10000;	//arbitrary
}

uint32_t sdHwGetMaxBlockSize(struct SdHwData *hwData)
{
	return 0xf000;	//subject to our DMA unit count limit
}

bool sdHwPrgBusyWait(struct SdHwData *hwData)
{
	struct HwData *hwd = (struct HwData*)hwData;
	
	return repalmSdioBusyWait(hwd->wrTimeoutBits);
}

void sdHwChipDeselect(struct SdHwData *hwData)
{
	//nothing
}

uint32_t sdHwInit(struct SdHwData *hwData)		//should set speed to 400khz
{
	sdHwSetTimeouts(hwData, 1000, 0, 0);
	sdHwSetSpeed(hwData, 400000, false);
	
	return SD_HW_FLAG_INITED | SD_HW_FLAG_SDIO_IFACE;
}

void sdHwShutdown(struct SdHwData *hwData)
{
	//TODO
}

void sdHwGiveInitClocks(struct SdHwData *hwData)
{
	uint8_t dummy;
	
	//[ab]use the data RX code to send clocks (no data is coming, so this will just timeout)
	repalmSdioDataRx(&dummy, 1, 1, 128);
}

static bool sdHwPrvPowerCtl(bool on)	//return old state
{
	uint32_t newVal = on, oldVal;
		
	(void)repalmDalPwrCtl(PWR_SEL_SD_CARD, &newVal, &oldVal);
	
	return !!oldVal;
}

bool sdHwIsCardInserted(struct SdHwData *hwData)
{
	bool cardIn, wasOn = sdHwPrvPowerCtl(true);
	
	KALTaskDelay(1);
	cardIn = !!(sio_hw->gpio_in & (1 << 25));
	
	if (!wasOn)
		sdHwPrvPowerCtl(false);
	
	return cardIn;
}

bool sdHwIsCardLockSwitchOn(struct SdHwData *hwData)
{
	return false;
}

bool sdHwSetBusWidth(struct SdHwData *hwData, bool useFourWide)
{
	return !useFourWide;
}

void sdHwCardPower(struct SdHwData *hwData, bool on)
{
	(void)hwData;
	sdHwPrvPowerCtl(on);
}

void sdHwSleep(struct SdHwData *hwData)
{
	//TODO
}

void sdHwWake(struct SdHwData *hwData)
{
	//TODO
}

void sdHwRxRawBytes(struct SdHwData *hwData, void *dst /* can be NULL*/, uint_fast16_t numBytes)
{
	//not used for SDIO HWs
}

void sdHwNotifyRCA(struct SdHwData *hwData, uint_fast16_t rca)
{
	//we do not need it
}

bool sdHwMultiBlockWriteSignalEnd(struct SdHwData *hwData)
{
	//not needed
	return true;
}

bool sdHwMultiBlockReadSignalEnd(struct SdHwData *hwData)
{
	//not needed
	return true;
}

static void sdHwPrvRecalcTimeouts(struct HwData *hwd)
{
	hwd->rdTimeoutBits = 8 * hwd->timeoutBytes + ((uint64_t)hwd->rdTimeoutTicks * (uint64_t)hwd->curClockRate) / TICKS_PER_SECOND;
	hwd->wrTimeoutBits = 8 * hwd->timeoutBytes + ((uint64_t)hwd->wrTimeoutTicks * (uint64_t)hwd->curClockRate) / TICKS_PER_SECOND;
}

void sdHwSetSpeed(struct SdHwData *hwData, uint32_t maxHz, bool highSpeedSignalling)
{
	uint32_t actualRate = repalmSdioSetSpeed(maxHz);
	struct HwData *hwd = (struct HwData*)hwData;
	
	hwd->curClockRate = actualRate;
	sdHwPrvRecalcTimeouts(hwd);
}

void sdHwSetTimeouts(struct SdHwData *hwData, uint_fast16_t timeoutBytes, uint32_t rdTimeoutTicks, uint32_t wrTimeoutTicks)
{
	struct HwData *hwd = (struct HwData*)hwData;
	
	hwd->rdTimeoutTicks = rdTimeoutTicks;
	hwd->wrTimeoutTicks = wrTimeoutTicks;
	hwd->timeoutBytes = timeoutBytes;
	
	sdHwPrvRecalcTimeouts(hwd);
}