#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "halPenAndKeys.h"
#include "joystick.h"
#include "zodiac.h"
#include "printf.h"
#include "boot.h"
#include "dal.h"
#include "kal.h"
#include <KeyMgr.h>


#pragma GCC push_options
#pragma GCC optimize("Os")


///XXX: see .text:100081C4


enum TwInputTypes {
	twInputPenX = 0x100,
	twInputPenY,
	twInputPenZ,
	
	twInputNavX,
	twInputNavY,
	
	twInputKeyState, 
	twInput4Way,
	twInput8Way, 
	
	twInputLaunch = 0x10e,
	twInputPower, 
	twInputBluetooth, 
	twInputFunction,
	twInputActionA,
	twInputActionB,
	twInputActionC,
	twInputActionD,
	
	twInputTriggerA = 0x11A,
	twInputTriggerB,
	twInputNavUp, 
	twInputNavDown,
	twInputNavLeft, 
	twInputNavRight, 
	twInputNavSelect, 
	
	twInputHard1 = 0x122,
	twInputHard2,
	twInputHard3,
	twInputHard4,
	twInputTimeStamp,
	twInputSequence, 
	
	twInputNavR = 0x12A,
	twInputNavTheta,
	
	twInputMax = 0x7fffffff,
};

enum TwNavDirection {
	twNavigatorCenter,
	twNavigatorUp,
	twNavigatorUpRight,
	twNavigatorRight,
	twNavigatorDownRight,
	twNavigatorDown,
	twNavigatorDownLeft,
	twNavigatorLeft,
	twNavigatorUpLeft,
};

#define TW_INPUT_KEY_BIT_FUNCTION			0x00008000
#define TW_INPUT_KEY_BIT_SELECT				0x00100000
#define TW_INPUT_KEY_BIT_LAUNCH				0x00800000
#define TW_INPUT_KEY_BIT_BLUETOOTH			0x01000000
#define TW_INPUT_KEY_BIT_TRIGGER_A			0x04000000
#define TW_INPUT_KEY_BIT_TRIGGER_B			0x08000000
#define TW_INPUT_KEY_BIT_ACTION_A			0x10000000
#define TW_INPUT_KEY_BIT_ACTION_B			0x20000000
#define TW_INPUT_KEY_BIT_ACTION_C			0x40000000
#define TW_INPUT_KEY_BIT_ACTION_D			0x80000000




#define TW_INPUT_ERR_INVALID_HANDLE			0x0009
#define TW_INPUT_ERR_NO_EVENTS				0x000B
#define TW_INPUT_ERR_OUT_OF_MEM				0x000C
#define TW_INPUT_ERR_INVALID_PTR			0x000E
#define TW_INPUT_ERR_BUSY					0x0010	//someone else already has it activated
#define TW_INPUT_ERR_INVALID_ARG			0x0016
#define TW_INPUT_ERR_NOT_IMPL				0x005A

#define TW_INPUT_MAGIX						0xA5A5BEEFUL

#define TW_INPUT_FLAG_NEEDS_PEN				0x00000001UL
#define TW_INPUT_FLAG_NEEDS_KEYS			0x00000002UL
#define TW_INPUT_FLAG_NEEDS_JOY				0x00000004UL
#define TW_INPUT_FLAG_NEEDS_METADATA		0x00000008UL

struct TwInputOpenState {
	uint32_t magix;					//0x00
	uint32_t flags;					//0x04	TW_INPUT_FLAG_*
	int32_t pollInterval;			//0x08
	int32_t capacity;				//0x0c	in events
	enum TwInputTypes *formats;		//0x10	list of formats, i32 each
	int32_t evtSz;					//0x14	size of each event (one subsample of each format), always multiple of 4. thus also size of formats list
	int32_t *events;				//0x18	events in the queue
	int32_t readPtr;				//0x1c	read >= write means no events avail. multiply by 4 to get pointer added to "events"
	int32_t writePtr;				//0x20
	int32_t capInWords;				//0x24	set capacity sets this to evtSz * capacity / 4
};


static int16_t mTwInputPenX, mTwInputPenY, mTwInputPrevPenX, mTwInputPrevPenY, mTwInputJoyX, mTwInputJoyY, mTwInputPrevJoyX, mTwInputPrevJoyY;
static uint32_t mTwInputMutex, mGlobalEvtsDequeuedCounter, mAnalogTimer;
static bool mTwInputPenDown, mTwInputPrevPenDown;
static struct TwInputOpenState *mActiveQueue;
static bool mTwInputNeedsPen = false;




#define HANDLE_CHECK()											\
			if (!handle || handle->magix != TW_INPUT_MAGIX)		\
				return TW_INPUT_ERR_INVALID_HANDLE

#define READ_CHECK()											\
			if (!evtVals)										\
				return TW_INPUT_ERR_INVALID_PTR;				\
																\
			if (szInBytes != handle->evtSz)						\
				return TW_INPUT_ERR_INVALID_ARG





static enum TwNavDirection twInputPrvCalcPos4w(int32_t joyX, int32_t joyY)
{
	int32_t absX = joyX < 0 ? -joyX : joyX;
	int32_t absY = joyY < 0 ? -joyY : joyY;
	
	if (joyX * joyX + joyY * joyY < 0x24000000)	//	displacement of at least 3/4 of the full displacement is required to not be "center"
		return twNavigatorCenter;
	
	if (absY > absX)
		return joyY > 0 ? twNavigatorDown : twNavigatorUp;
	else
		return joyX > 0 ? twNavigatorRight : twNavigatorLeft;
}

static enum TwNavDirection twInputPrvCalcPos8w(int32_t joyX, int32_t joyY)
{
	uint32_t absX = joyX < 0 ? -joyX : joyX, absY = joyY < 0 ? -joyY : joyY;
	
	if (joyX * joyX + joyY * joyY < 0x24000000)	//	displacement of at least 3/4 of the full displacement is required to not be "center"
		return twNavigatorCenter;
	
	if (absY * 3135 > 8192 * absX)	//3135 / 8192 is approx sin(22.5 degrees)
		return joyY > 0 ? twNavigatorDown : twNavigatorUp;
	
	if (absX * 3135 > 8192 * absY)
		return joyX > 0 ? twNavigatorRight : twNavigatorLeft;
	
	if (joyX > 0 && joyY > 0)
		return twNavigatorDownRight;
	else if (joyX > 0)
		return twNavigatorUpRight;
	else if (joyY > 0)
		return twNavigatorDownLeft;
	else
		return twNavigatorUpLeft;
}

static uint32_t twInputPrvCalcRadius(int32_t x, int32_t y)
{
	uint32_t v = x * x + y * y, guess = 0x8000, step = guess / 2, attempt;
	
	while ((attempt = guess * guess) != v && step) {
		
		if (attempt > v)
			guess -= step;
		else
			guess += step;
		
		step >>= 1;
	}
	
	return guess;
}

static uint32_t twInputPrvAtan45(uint32_t x, uint32_t y)		//atan, in proper coordinates (counterclockwise). x & y are nonnegative, < 0x8000. angle <= 45 degrees, x != 0
{
	//only the first 46 elements are neeed, but, what the hell, let's have them all
	static const uint16_t tanTab[] = {0, 71, 143, 214, 286, 358, 430, 502, 575, 648, 722, 796, 870, 945, 1021, 1097, 1174, 1252, 1330, 1410, 1490, 1572, 1654, 1738, 1823, 1909, 1997, 2087, 2177, 2270, 2364, 2461, 2559, 2659, 2762, 2868, 2975, 3086, 3200, 3316, 3436, 3560, 3688, 3819, 3955, 4096, 4241, 4392, 4549, 4711, 4881, 5058, 5242, 5435, 5637, 5849, 6072, 6307, 6554, 6816, 7094, 7389, 7703, 8038};
	uint32_t div = (y << 12) / x, guess = sizeof(tanTab) / sizeof(*tanTab), step = guess / 2;
	
	while (step) {
		uint32_t now = tanTab[guess];
		if (now > div)
			guess -= step;
		else if (now == div)
			break;
		else
			guess += step;
		step >>= 1;
	}
	
	return guess;
}

static uint32_t twInputPrvAtan90(uint32_t x, uint32_t y)	//atan, in proper coordinates (counterclockwise). x & y are nonnegative, < 0x8000
{
	if (y > x)
		return 90 - twInputPrvAtan45(y, x);
	else
		return twInputPrvAtan45(x, y);
}

static uint32_t twInputPrvCalcTheta(int32_t x, int32_t y)	//atan2, basically, except Tapwave coordinates
{
	uint32_t degreesToAdd = 0;
	
	if (!x && !y)
		return 0;
	
	if (y > 0) {
		
		x = -x;
		y = -y;
		degreesToAdd += 180;
	}
	
	if (x < 0)
		return degreesToAdd + 180 - twInputPrvAtan90(-x, -y);
	else
		return degreesToAdd + twInputPrvAtan90(x, -y);
}

static bool twInputPrvEventPollGuts(int32_t *vals, const enum TwInputTypes *evts, int32_t evtSz, bool addEvt)	//return true if any events of value were provided
{
	int32_t i, val, joyX = mTwInputJoyX, joyY = mTwInputJoyY, dir4w = -1, numEvts = evtSz / sizeof(*evts);
	uint32_t keys = HALKeyGetState();
	bool anythingReported = false;
	enum TwInputTypes evt;
	
	for (i = 0; i < numEvts; i++) {
		
		switch (evt = *evts++) {
			case twInputPenX:
				if (mTwInputPenX != mTwInputPrevPenX)
					anythingReported = true;
				mTwInputPrevPenX = mTwInputPenX;
				val = mTwInputPenX;
				break;
			
			case twInputPenY:
				if (mTwInputPenY != mTwInputPrevPenY)
					anythingReported = true;
				mTwInputPrevPenY = mTwInputPenY;
				val = mTwInputPenY;
				break;
			
			case twInputPenZ:
				if (mTwInputPenDown  != mTwInputPrevPenDown)
					anythingReported = true;
				mTwInputPrevPenDown = mTwInputPenDown;
				val = mTwInputPenDown ? 0x7fff : 0;
				break;
			
			case twInputNavX:
				if (joyX != mTwInputPrevJoyX)
					anythingReported = true;
				mTwInputPrevJoyX = joyX;
				val = joyX;
				break;
			
			case twInputNavY:
				if (joyY != mTwInputPrevJoyY)
					anythingReported = true;
				mTwInputPrevJoyY = joyY;
				val = joyY;
				break;
			
			case twInputKeyState:
				val = keys;
				break;
			
			case twInput4Way:
				if (dir4w == -1)
					dir4w = twInputPrvCalcPos4w(joyX, joyY);
				val = dir4w;
				break;
			
			case twInput8Way:
				val = twInputPrvCalcPos8w(joyX, joyY);
				break;
			
			case twInputLaunch:
				val = !!(keys & TW_INPUT_KEY_BIT_LAUNCH);
				break;
			
			case twInputPower:
				val = !!(keys & keyBitPower);
				break;
			
			case twInputBluetooth:
				val = !!(keys & TW_INPUT_KEY_BIT_BLUETOOTH);
				break;
			
			case twInputFunction:
				val = !!(keys & TW_INPUT_KEY_BIT_FUNCTION);
				break;
			
			case twInputActionA:
				val = !!(keys & TW_INPUT_KEY_BIT_ACTION_A);
				break;
			
			case twInputActionB:
				val = !!(keys & TW_INPUT_KEY_BIT_ACTION_B);
				break;
			
			case twInputActionC:
				val = !!(keys & TW_INPUT_KEY_BIT_ACTION_C);
				break;
			
			case twInputActionD:
				val = !!(keys & TW_INPUT_KEY_BIT_ACTION_D);
				break;
			
			case twInputTriggerA:
				val = !!(keys & TW_INPUT_KEY_BIT_TRIGGER_A);
				break;
			
			case twInputTriggerB:
				val = !!(keys & TW_INPUT_KEY_BIT_TRIGGER_B);
				break;
			
			case twInputNavUp:
				if (dir4w == -1)
					dir4w = twInputPrvCalcPos4w(joyX, joyY);
				val = (dir4w == twNavigatorUp);
				break;
			
			case twInputNavDown:
				if (dir4w == -1)
					dir4w = twInputPrvCalcPos4w(joyX, joyY);
				val = (dir4w == twNavigatorDown);
				break;
			
			case twInputNavLeft:
				if (dir4w == -1)
					dir4w = twInputPrvCalcPos4w(joyX, joyY);
				val = (dir4w == twNavigatorLeft);
				break;
			
			case twInputNavRight:
				if (dir4w == -1)
					dir4w = twInputPrvCalcPos4w(joyX, joyY);
				val = (dir4w == twNavigatorRight);
				break;
			
			case twInputNavSelect:
				val = !!(keys & TW_INPUT_KEY_BIT_SELECT);
				break;
			
			case twInputHard1:
				val = !!(keys & keyBitHard1);
				break;
			
			case twInputHard2:
				val = !!(keys & keyBitHard2);
				break;
			
			case twInputHard3:
				val = !!(keys & keyBitHard3);
				break;
			
			case twInputHard4:
				val = !!(keys & keyBitHard4);
				break;
			
			case twInputTimeStamp:
				val = HALTimeGetSystemTime();
				break;
			
			case twInputSequence:
				val = mGlobalEvtsDequeuedCounter;
				break;
			
			case twInputNavR:
				val = twInputPrvCalcRadius(joyX, joyY);
				anythingReported = true;	//always
				break;
			
			case twInputNavTheta:
				val = twInputPrvCalcTheta(joyX, joyY);
				anythingReported = true;	//always
				break;
			
			default:
				logw("Unexpected TwInput selector %u\n", evt);
				val = 0;
		}
		*vals++ = val;
	}
	
	if (anythingReported && addEvt) {
		
		static const struct HalEvtKey k = {.ascii = 0x1b00, .modifiers = commandKeyMask, };
		
		HALEventPost(HAL_EVENT_TYPE_KEY, &k);
	}
	
	return anythingReported;
}

static bool twInputPrvFilterPenEvt(const struct HalEvtPen *pen)
{
	bool allowHalEvtPosting = true, sendTwEvt = false;
	uint32_t i;
	
	KALMutexReserve(mTwInputMutex, -1);
	
	if (mActiveQueue && mActiveQueue->writePtr < mActiveQueue->capInWords) {
		
		for (i = 0; i < mActiveQueue->evtSz / sizeof(*mActiveQueue->formats); i++) {
			
			if (mActiveQueue->formats[i] == twInputPenX || mActiveQueue->formats[i] == twInputPenY || mActiveQueue->formats[i] == twInputPenZ) {
				
				mTwInputPenX = pen->x;
				mTwInputPenY = pen->y;
				mTwInputPenDown = (pen->x == -1) ? 0 : 0x7fff;
				allowHalEvtPosting  = false;
				
				if (mTwInputPenX != mTwInputPrevPenX ||  twInputPenY != mTwInputPrevPenY || mTwInputPenDown != mTwInputPrevPenDown)
					sendTwEvt = true;
				break;
			}
		}
		
		if (sendTwEvt && twInputPrvEventPollGuts(mActiveQueue->events + mActiveQueue->writePtr, mActiveQueue->formats, mActiveQueue->evtSz, sendTwEvt))
			mGlobalEvtsDequeuedCounter++;
	}
	
	KALMutexRelease(mTwInputMutex);
	
	return allowHalEvtPosting;
}

static bool twInputPrvFilterKeyEvt(const struct HalEvtKey *key)
{
	loge("TODO: %s\n", __func__);
	
	return true;
}

bool zodFilterHalEvents(uint32_t type, const void* data)
{
	if (type == HAL_EVENT_TYPE_PEN)
		return twInputPrvFilterPenEvt((const struct HalEvtPen*)data);
	
	if (type == HAL_EVENT_TYPE_KEY)
		return twInputPrvFilterKeyEvt((const struct HalEvtKey*)data);
	
	return true;
}

Err DALEXPORT impl_TwInputOpen(struct TwInputOpenState **handleP, const char *name, const char *mode)
{
	struct TwInputOpenState *handle;
	
	logt("%s(..., '%s', '%s')\n", __func__, name, mode);
	
	if (strcmp(name, "twinput") || strcmp(mode, "r"))
		return TW_INPUT_ERR_INVALID_ARG;
	
	if (!handleP)
		return TW_INPUT_ERR_INVALID_PTR;
	
	handle = MemPtrNew(sizeof(*handle));
	if (!handle)
		return TW_INPUT_ERR_OUT_OF_MEM;
	
	memset(handle, 0, sizeof(*handle));
	handle->magix = TW_INPUT_MAGIX;
	
	if (handleP)
		*handleP = handle;
	
	return errNone;
}

static Err twInputPrvDeactivateLocked(struct TwInputOpenState *handle)
{
	HANDLE_CHECK();
	
	if (mActiveQueue != handle)
		return TW_INPUT_ERR_INVALID_ARG;
	
	mActiveQueue = NULL;
	
	return errNone;
}

static Err twInputPrvActivateLocked(struct TwInputOpenState *handle)
{
	HANDLE_CHECK();
	
	if (mActiveQueue)
		return TW_INPUT_ERR_BUSY;
	
	mActiveQueue = handle;
	mTwInputNeedsPen = !!(handle->flags & TW_INPUT_FLAG_NEEDS_PEN);
	
	return errNone;
}

Err DALEXPORT impl_TwInputGetPeriod(struct TwInputOpenState *handle, int32_t *msecP)
{
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	if (msecP)
		*msecP = handle->pollInterval;
	
	return errNone;
}

Err DALEXPORT impl_TwInputSetPeriod(struct TwInputOpenState *handle, int32_t msec)
{
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	handle->pollInterval = msec;
	
	return errNone;
}

Err DALEXPORT impl_TwInputGetCapacity(struct TwInputOpenState *handle, int32_t *capacityP)
{
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	if (capacityP)
		*capacityP = handle->capacity;
	
	return errNone;
}
Err DALEXPORT impl_TwInputSetCapacity(struct TwInputOpenState *handle, int32_t capacity)
{
	int32_t *newBuf = NULL;
	Err ret = errNone;
	uint32_t newSz;
	
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	newSz = capacity * handle->evtSz;
	
	KALMutexReserve(mTwInputMutex, -1);
	
	if (capacity <= 0) {
		
		if (handle->events)
			MemPtrFree(handle->events);
		
		handle->capacity = 0;
		handle->events = NULL;
		handle->readPtr = 0;
		handle->writePtr = 0;
		handle->capInWords = 0;
	}
	else if (!(newSz % handle->evtSz) && (!newSz || (newBuf = (int32_t*)MemPtrNew(newSz)) != NULL)) {
			
		handle->capInWords = newSz / sizeof(int32_t);
		
		if (newBuf && handle->readPtr < handle->writePtr) {
			
			memcpy(newBuf, handle->events + handle->readPtr, (handle->writePtr - handle->readPtr) * sizeof(int32_t));
			handle->writePtr = handle->writePtr - handle->readPtr;
		}
		else {
			
			handle->writePtr = 0;
		}
		
		handle->readPtr = 0;
		
		if (handle->events)
			MemPtrFree(handle->events);
		
		handle->events = newBuf;
		handle->capacity = capacity;
	}
	else
		ret = TW_INPUT_ERR_OUT_OF_MEM;
	
	KALMutexRelease(mTwInputMutex);
	
	return ret;
}

Err DALEXPORT impl_TwInputGetFormat(struct TwInputOpenState *handle, enum TwInputTypes *formats, int32_t *sizeInBytesP)
{
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	if (!formats || !sizeInBytesP || ((*sizeInBytesP) % sizeof(enum TwInputTypes)))
		return TW_INPUT_ERR_INVALID_ARG;
	
	if (!handle->formats)
		*sizeInBytesP = 0;
	else {
		
		int32_t now = *sizeInBytesP;
		
		if (now > handle->evtSz)
			now = handle->evtSz;
		
		memcpy(formats, handle->formats, now);
		
		*sizeInBytesP = handle->evtSz;
	}
	
	return errNone;
}

Err DALEXPORT impl_TwInputSetFormat(struct TwInputOpenState *handle, const enum TwInputTypes *formats, int32_t sizeInBytes)
{
	enum TwInputTypes *formatsCopy;
	int32_t i, *dataBuf = NULL;
	Err ret = errNone;
	
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	if (!formats || sizeInBytes < 0 || (sizeInBytes % sizeof(enum TwInputTypes))) {
		logw("%s: invalid args\n", __func__);
		return TW_INPUT_ERR_INVALID_ARG;
	}
	
	formatsCopy = (enum TwInputTypes*)MemPtrNew(sizeInBytes);
	if (!formatsCopy) {
		
		logw("Failed to allocate %u bytes for copy of formats\n", sizeInBytes);
		return TW_INPUT_ERR_OUT_OF_MEM;
	}
	
	if (handle->capacity) {
		dataBuf = (int32_t*)MemPtrNew(sizeInBytes * handle->capacity);
		if (!dataBuf) {
			
			logw("Failed to allocate %u bytes for data buffer\n", sizeInBytes * handle->capacity);
			MemPtrFree(formatsCopy);
			return TW_INPUT_ERR_OUT_OF_MEM;
		}
	}
	
	memcpy(formatsCopy, formats, sizeInBytes);
	
	KALMutexReserve(mTwInputMutex, -1);
	
	if (mActiveQueue == handle) {	//cannot be done to an active queue
		ret = TW_INPUT_ERR_BUSY;
		MemPtrFree(formatsCopy);
		if (dataBuf)
			MemPtrFree(dataBuf);
	}
	else {
		
		handle->readPtr = 0;
		handle->writePtr = 0;
		handle->capInWords = 0;
		
		if (handle->formats)
			MemChunkFree(handle->formats);
		
		handle->formats = formatsCopy;
		handle->evtSz = sizeInBytes;
		
		//see if we need pen input now
		handle->flags &=~ (TW_INPUT_FLAG_NEEDS_PEN | TW_INPUT_FLAG_NEEDS_KEYS | TW_INPUT_FLAG_NEEDS_JOY | TW_INPUT_FLAG_NEEDS_METADATA);
		
		for (i = 0; i < sizeInBytes / (int32_t)sizeof(*formats); i++) {
			
			switch (formats[i]) {
				case twInputPenX:
				case twInputPenY:
				case twInputPenZ:
					handle->flags |= TW_INPUT_FLAG_NEEDS_PEN;
					break;
				
				case twInputNavX:
				case twInputNavY:
				case twInputNavR:
				case twInputNavTheta:
					handle->flags |= TW_INPUT_FLAG_NEEDS_JOY;
					break;
				
				case twInputKeyState:
				case twInput4Way:
				case twInput8Way:
				case twInputLaunch:
				case twInputPower:
				case twInputBluetooth:
				case twInputFunction:
				case twInputActionA:
				case twInputActionB:
				case twInputActionC:
				case twInputActionD:
				case twInputTriggerA:
				case twInputTriggerB:
				case twInputNavUp:
				case twInputNavDown:
				case twInputNavLeft:
				case twInputNavRight:
				case twInputNavSelect:
				case twInputHard1:
				case twInputHard2:
				case twInputHard3:
				case twInputHard4:
					handle->flags |= TW_INPUT_FLAG_NEEDS_KEYS;
					break;
				
				case twInputTimeStamp:
				case twInputSequence:
					handle->flags |= TW_INPUT_FLAG_NEEDS_METADATA;
					break;
				
				default:
					break;
			}
		}
		
		if (handle->events)
			MemChunkFree(handle->events);
		
		handle->events = dataBuf;
		handle->capInWords = sizeInBytes * handle->capacity / sizeof(int32_t);
	}
	KALMutexRelease(mTwInputMutex);
	
	return ret;
}

static Err twInputPrvReadEvts(struct TwInputOpenState *handle, int32_t *evtVals, int32_t szInBytes, bool dequeue)
{
	Err ret = TW_INPUT_ERR_NO_EVENTS;
	
	HANDLE_CHECK();
	READ_CHECK();
	
	KALMutexReserve(mTwInputMutex, -1);
	
	if (handle->readPtr < handle->writePtr) {
		memcpy(evtVals, &handle->events[handle->readPtr], szInBytes);
		
		if (dequeue) {
			
			handle->readPtr += szInBytes / sizeof(int32_t);
			
			if (handle->readPtr == handle->writePtr) {
				
				handle->readPtr = 0;
				handle->writePtr = 0;
			}
		}
		ret = errNone;
	}
	KALMutexRelease(mTwInputMutex);
	
	return ret;
}

Err DALEXPORT impl_TwInputPeek(struct TwInputOpenState *handle, int32_t *evtVals, int32_t szInBytes)
{
	logt("%s\n", __func__);
	
	return twInputPrvReadEvts(handle, evtVals, szInBytes, false);
}

Err DALEXPORT impl_TwInputRead(struct TwInputOpenState *handle, int32_t *evtVals, int32_t szInBytes)
{
	logt("%s\n", __func__);
	
	return twInputPrvReadEvts(handle, evtVals, szInBytes, true);
}

Err DALEXPORT impl_TwInputPoll(struct TwInputOpenState *handle, int32_t *evtVals, int32_t szInBytes)
{
	int16_t x, y;
	
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	READ_CHECK();
	
	joystickRead(&x, &y);
	mTwInputJoyX = x;
	mTwInputJoyY = y;
	
	twInputPrvEventPollGuts(evtVals, handle->formats, handle->evtSz, false);
	mGlobalEvtsDequeuedCounter++;
	
	return errNone;
}

Err DALEXPORT impl_TwInputControl(struct TwInputOpenState *handle, int32_t cmd, void *data, int32_t szInBytes)
{
	logt("%s\n", __func__);
	
	HANDLE_CHECK();
	
	return TW_INPUT_ERR_NOT_IMPL;
}

Err DALEXPORT impl_TwInputActivate(struct TwInputOpenState *handle)
{
	Err e;
	
	logi("%s\n", __func__);
	
	KALMutexReserve(mTwInputMutex, -1);
	e = twInputPrvActivateLocked(handle);
	KALMutexRelease(mTwInputMutex);
	
	return e;
}

Err DALEXPORT impl_TwInputDeactivate(struct TwInputOpenState *handle)
{
	Err e;
	
	logi("%s\n", __func__);
	
	KALMutexReserve(mTwInputMutex, -1);
	e = twInputPrvDeactivateLocked(handle);
	KALMutexRelease(mTwInputMutex);
	
	return e;
}

Err DALEXPORT impl_TwInputClose(struct TwInputOpenState *handle)
{
	logi("%s\n", __func__);
	
	HANDLE_CHECK();
	
	(void)impl_TwInputDeactivate(handle);
	
	if (handle->formats)
		MemChunkFree(handle->formats);
	if (handle->events)
		MemChunkFree(handle->events);
	
	handle->magix = 0;
	
	MemChunkFree(handle);
	
	return errNone;
}

void zodPrvAppExitNotifTwInput(void)
{
	struct TwInputOpenState *handle;
	
	KALMutexReserve(mTwInputMutex, -1);
	handle = mActiveQueue;
	if (handle)
		(void)twInputPrvDeactivateLocked(mActiveQueue);
	KALMutexRelease(mTwInputMutex);
	if (handle)
		(void)impl_TwInputClose(handle);
}

static void twInputPrvAnalongTimerCbk(void *userData)
{
	//todo
	
	//XXX: For most analog devices, the system uses a minimal threshold to avoid spurious events: If the user touches the navigator only slightly, the system does not generate an event.
	
	//push polling unto analog hw driver :)
}

static void twInputPrvJoyUpdate(int16_t x, int16_t y)
{
	//may be called with bad R9...
		
	mTwInputJoyX = x;
	mTwInputJoyY = y;
}

void zodTwInputInit(void)
{
	if (errNone != KALMutexCreate(&mTwInputMutex, CREATE_4CC('_','T','w','I')))
		fatal("Failed to init TwInput mtx\n");
	
	if (errNone != KALTimerCreate(&mAnalogTimer, CREATE_4CC('_','T','w','I'), twInputPrvAnalongTimerCbk, NULL))
		fatal("Failed to init TwInput tmr\n");
	
	joystickInit(&twInputPrvJoyUpdate);
}

Err DALEXPORT impl_TwNavResetCalibration(void)
{
	logt("%s\n", __func__);
	
	return errNone;
}

Err DALEXPORT impl_TwNavCalibrate(int32_t p1, int32_t p2, int32_t p3, int32_t p4, int32_t p5, int32_t p6, int32_t p7, int32_t p8)
{
	logt("%s(%d, %d, %d, %d, %d, %d, %d, %d)\n", __func__, p1, p2, p3, p4, p5, p6, p7, p8);
	
	return errNone;
}