#include "printf.h"
#include "memmap.h"
#include "entry.h"
#include "mpu.h"
#include "cpu.h"

#ifdef __MPU_PRESENT
	
	static uint8_t mStackGuardMpuRegionNumber;
	
	static bool mpuRegConfig(uint32_t idx, uint32_t addr, uint32_t sz, uint32_t perm)		//assumes size is not too large
	{	
		uint32_t effSz = sz;
		uint32_t effAddr = addr;
		uint32_t i, srd = 0;
		
		//round up size to power of two
		if (effSz & (effSz - 1))
			effSz = 1 << (32 - __builtin_clz(effSz));
		
		#ifdef BUILD_FOR_THUMB_1
			if (effSz < 256)
				effSz = 256;
		#else
			if (effSz < 32)
				effSz = 32;
		#endif
		
		while (1) {
			
			//round address down to multiple of size
			effAddr &= ~(effSz - 1);
			
			//see if we match
			if (effAddr + effSz - addr >= sz)
				break;
			
			//we can only go a size higher if that is at all possible
			if (effSz == 0x80000000)
				return false;
			
			//go higher size
			effSz <<= 1;
		}
		
		//see if start lines up with an 8th of the region
		for (i = 0; i < 8; srd |= (1 << i), i++) {
			if (effAddr + (effSz / 8) * i == addr)
				break;
		}
		
		//see if end lines up with 8-th of region
		for (i = 8; i; i--, srd |= (1 << i)) {
			if (effAddr + (effSz / 8) * i == addr + sz)
				break;
		}
		
		if (srd == 0xff)
			return false;
		
		if (effSz < 32)
			return false;
		
		if (effSz < 256 && srd)
			return false;
		
		MPU->RNR = idx;
		MPU->RASR = 0;	//disable this region for now
		
		MPU->RBAR = effAddr;
		MPU->RASR = perm | MPU_RASR_ENABLE_Msk | (((30 - __builtin_clz(effSz)) << MPU_RASR_SIZE_Pos) & MPU_RASR_SIZE_Msk) | (srd << MPU_RASR_SRD_Pos);	//config and enable
		
		logvst("mpuRegConfig(0x%08x + 0x%08x) -> {0x%08x+0x%08X, 0x%02x} -> {0x%08x 0x%08x}\n", addr, sz, effAddr, effSz, srd, MPU->RBAR, MPU->RASR);
		return true;
	}
	
	static uint32_t mpuGetNumRgions(void)
	{
		uint32_t nregions;
		
		if (MPU->TYPE & 1) {
			loge("Separate I/D MPU not supported\n");
			return 0;
		}
		
		return (MPU->TYPE & MPU_TYPE_DREGION_Msk) >> MPU_TYPE_DREGION_Pos;
	}

#endif

void mpuInstrCacheClearDataCacheClean(uintptr_t addr, int32_t sz)
{
	#ifdef PLATFORM_MUST_DEAL_WITH_CACHE_COHERENCY
		if (sz < 0) {
			
			SCB_CleanDCache();
			SCB_InvalidateICache();
		}
		else {
			
			uint32_t i, end = (addr + sz + 31) &~ 31;
			
	    	__DSB();
	    	
			for (i = addr &~ 31; i != end; i += 32) {
				
				SCB->DCCMVAC = addr;
			}
			
		    __DSB();
		    __ISB();
			for (i = addr &~ 31; i != end; i += 32) {
				
				SCB->ICIMVAU = addr;
			}
			
		    __DSB();
		    __ISB();
		}
	#endif
}

void mpuSetStorageRamWriteable(bool allowWrites)
{
	#ifdef __MPU_PRESENT
		
		if (mpuGetNumRgions())
			machSetStorageAreaWriteable(allowWrites);
	#endif
}

bool mpuInit(void)
{
	#ifdef __MPU_PRESENT
	
		if (mpuGetNumRgions()) {
			
			mStackGuardMpuRegionNumber = mpuGetNumRgions() - 1;
			mpuSetStackGuard(0);
		}
	
	#endif
	
	return true;
}

void __attribute__((used)) mpuEarlyInit(void)
{
	#ifdef __MPU_PRESENT
	
		uint32_t i, nregions = mpuGetNumRgions();
		
		for (i = 0; i < nregions; i++) {
			MPU->RBAR = 0x10 + i;
			MPU->RASR = 0x0310003f;	//all perms, on, 4GB, no caching (else hardware gets affected and we boned)	
		}
		
		MPU->CTRL = MPU_CTRL_ENABLE_Msk | MPU_CTRL_HFNMIENA_Msk | MPU_CTRL_PRIVDEFENA_Msk;	//PRIVDEFENA is for devices, without HFNMIENA we cnanot execute form RAM on real STM hw
	#endif
}

bool mpuSetStackGuard(uint32_t addr)	//we could allow supervisor to write to stack guards so that we can take stack overflow exception properly (XXX: is this how HWR works??)
{
	#if defined(__MPU_PRESENT) && !defined(BUILD_FOR_THUMB_1)		//T1 minimum size is too big

		if (mpuGetNumRgions()) {
			if (!addr) {
				MPU->RNR = mStackGuardMpuRegionNumber;
				MPU->RASR = 0;	//disable this region for now
				return true;
			}
			
			//round address up 32 bytes
			if (addr & 31) {
				addr += 31;
				addr &=~ 31;
			}
			
			return mpuRegConfig(mStackGuardMpuRegionNumber, addr, MPU_STACK_GUARD_SIZE, MPU_PERM_U_XX_S_XX | MPU_MEM_TYPE_RAM | MPU_PERM_NX | MPU_FLAG_ENABLED);
		}
		else {
			
			//we did what we could
			return true;
		}
	#else
	
		return true;
	#endif
}




