#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include "remoteioCommsProto.h"
#include "ral_export.h"
#include "remoteioComms.h"
#include "memmap.h"
#include "printf.h"
#include "timers.h"	//for benchmarks
#include "irqs.h"
#include "heap.h"
#include "ral.h"
#include "kal.h"
#include "cpu.h"


static const uint16_t mVisorCode[] = {
	#include "firmware.inc"
};

static const uint16_t mSpringboardRom[] = {
	#include "romimg.inc"
};


static bool mWantBattInfo = true, mWantUpdate = false, mWantLedCtl = false, mLocksBroken = false;
static int16_t mDesiredBrightness = -1, mDesiredContrast = -1, mDesiredDepth = -1;
static struct MsgBatteryInfo mBatteryInfo = {};
static VisorCommsTxReplyHandler mSerTxReplyCbk;
static VisorCommsOpReplyHandler mSerOpReplyCbk;
static VisorCommsButtonStateCbk mInputBtnCbk;
static uint32_t mScreenOfst, mScreenLength;
static VisorCommsPenStateCbk mInputPenCbk;
static uint16_t mVisorCodeDownloadIdx = 0;
static uint32_t mSerialSem, mSerialMutex;
static struct MsgScreenData *mScreenMsg;
static VisorCommsSerRxHandler mSerRxCbk;
static struct MsgContinueBoot mCb = {};
static int8_t mDesiredBacklight = -1;
static struct MsgLedControl mLedCtl;
static struct MsgClut *mClutMsg;
static uint16_t mSerTxDataLen;
static uint8_t *mScreenPtr;
static bool mHaveSerTxData;
static uint16_t mSerTxMsg;
static void *mSerCbkData;
static void *mSerTxData;

static uint64_t mDispUpdateStartTime;




//base address
#define COMMS_SRAM_BASE 		0xc0000000
#define COMMS_RAM_HALFWORDS		1024


//mailbox sizes
#define TO_VISOR_MBX_START		382		//allows a 4bpp screen update in 10 messages
#define TO_VISOR_MBX_LEN		(COMMS_RAM_HALFWORDS - TO_VISOR_MBX_START)
#define TO_MODULE_MBX_START		(sizeof(mSpringboardRom) / sizeof(*mSpringboardRom))
#define TO_MODULE_MBX_LEN		(TO_VISOR_MBX_START - TO_MODULE_MBX_START)


static void copyFromSram(void *dstP, uint32_t srcAddrOfst, uint32_t bytes)
{
	volatile uint16_t *src = (volatile uint16_t*)(COMMS_SRAM_BASE + 2 * srcAddrOfst);
	uint32_t halfwords = bytes / 2;
	uint16_t *dst = (uint16_t*)dstP;
	
	while (halfwords--) {
		
		*dst++ = *src;
		src += 2;
	}
	if (bytes & 1)
		*((uint8_t*)dst) = (*src) >> 8;
}

//copies a halfword at a time
static void copyToSram(uint32_t dstAddrOfst, const void *srcP, uint32_t bytes)
{
	volatile uint16_t *dst = (volatile uint16_t*)(COMMS_SRAM_BASE + 2 * dstAddrOfst);
	const uint16_t *src = (const uint16_t*)srcP;
	uint32_t halfwords = bytes / 2;
	
	while (halfwords--) {
		
		*dst = *src++;
		dst += 2;
	}
	if (bytes & 1)
		*dst = ((uint32_t)(*(const uint8_t*)src)) << 8;
}

void remoteioCommsMsgCopyToSram(const void *srcP, uint32_t bytes, uint32_t dstOffsetHalfwords)
{
	copyToSram(2 * (TO_VISOR_MBX_START + dstOffsetHalfwords), srcP, bytes);
}

void remoteioCommsCopyMsgFromSram(void *dstP, uint32_t bytes, uint32_t skipHalfwordUpFront)
{
	copyFromSram(dstP, 2 * (TO_MODULE_MBX_START + skipHalfwordUpFront), bytes);
}

static void visorIrqSet(bool active)
{
	GPIOA->BSRR = active ? (1 << (4 + 16)) : (1 << 4);
}

static void visorIrqUpdateState(void)
{
	visorIrqSet(mWantBattInfo || mDesiredBacklight >= 0 || mDesiredBrightness >= 0 || mDesiredContrast >= 0 || mDesiredDepth >= 0 || (mClutMsg && mClutMsg->numEntries) || mScreenLength || mWantUpdate || mHaveSerTxData || mWantLedCtl);
}

static void fmcFlush(void)
{
	volatile uint16_t *dst = (volatile uint16_t*)COMMS_SRAM_BASE;
	uint_fast8_t i;
	
	//the fifo sucks for us in the fmc, but we cannot turn it off, so let's fill it with crap at odd addresses we do not use
	for (i = 0; i < 256; i++)
		dst[1 + i * 2] = i;
}

void __attribute__((used)) EXTI0_IRQHandler(void)	
{
	volatile uint16_t *dst = (volatile uint16_t*)COMMS_SRAM_BASE;
	bool reply = true, debug = false;		//debugger somehow interferes in AHB's access to the SRAM, so we need this special scary hack
	struct MsgScreenData msgDisp;
	struct MsgButtonEvt msgKeys;
	uint16_t msgType, msgMark;
	struct MsgSingleWord swm;
	struct MsgPenEvt msgPen;
	uint32_t i, r9;


	//ack the interrupt
	EXTI->PR = 1;

	//get message info
	do {
		msgMark = dst[6];
	} while (msgMark != dst[6]);
	
	do {
		msgType = dst[2];
	} while (msgType != dst[2]);
	
	if (debug)
		logi("RX %04x %04x\n", msgMark, msgType);
	
	r9 = ralSetSafeR9();	//irq runs with random r9
	
	//handle simple messages
	if (msgMark == MSG_MRK_REQUEST) {
		
		msgMark = MSG_MRK_REPLY;
		
		switch (msgType) {
			case MSG_S_SHABBAT:
				msgType = MSG_S_SHALOM;
				break;
			
			case MSG_S_CODE_SIZE:
				msgType = sizeof(mVisorCode) / sizeof(*mVisorCode);
				mVisorCodeDownloadIdx = 0;
				break;
			
			case MSG_S_CODE_DATA:
				if (debug)
					logi("dl %u / %u\n", mVisorCodeDownloadIdx, sizeof(mVisorCode) / sizeof(*mVisorCode));
				if (mVisorCodeDownloadIdx < sizeof(mVisorCode) / sizeof(*mVisorCode))
					msgType = mVisorCode[mVisorCodeDownloadIdx++];
				else
					msgType = MSG_S_NOT_UNDERSTOOD;
				break;
			
			case MSG_S_MAINBOX_INFO_0:
				msgType = TO_MODULE_MBX_START;
				break;
			
			case MSG_S_MAINBOX_INFO_1:
				msgType = TO_MODULE_MBX_LEN;
				break;
			
			case MSG_S_MAINBOX_INFO_2:
				msgType = TO_VISOR_MBX_START;
				break;
			
			case MSG_S_MAINBOX_INFO_3:
				msgType = TO_VISOR_MBX_LEN;
				break;
			
			case MSG_C_NOP:
		send_any_reply:
		
				if (mDesiredDepth >= 0) {
					
					swm.value = mDesiredDepth;
					mDesiredDepth = -1;
					remoteioCommsMsgCopyToSram(&swm, sizeof(swm), 0);
					msgType = MSG_C_SET_SCREEN_DEPTH;
				}
				else if (mClutMsg && mClutMsg->numEntries) {
					
					remoteioCommsMsgCopyToSram(mClutMsg, sizeof(struct MsgClut), 0);
					mClutMsg->numEntries = 0;
					msgType = MSG_C_SET_CLUT;
				}
				else if (mScreenLength) {
					
					uint32_t bytesMax = TO_VISOR_MBX_LEN * sizeof(uint16_t) - sizeof(struct MsgScreenData);
					uint32_t bytesLeft = mScreenLength - mScreenOfst;
					uint32_t bytesNow = bytesLeft;
					
					if (bytesNow > bytesMax)
						bytesNow = bytesMax;
					if (bytesNow > 0x7fff)		//protocol limit
						bytesNow = 0x7fff;
					
					msgDisp.sz = bytesNow + ((bytesNow == bytesLeft) ? 0x8000 : 0);
					msgDisp.ofst = mScreenOfst;
					
					remoteioCommsMsgCopyToSram(&msgDisp, sizeof(struct MsgScreenData), 0);
					remoteioCommsMsgCopyToSram(mScreenPtr + mScreenOfst, (bytesNow + 1) &~ 1 /* copy wordwise */, sizeof(struct MsgScreenData) / sizeof(uint16_t));
					
					if (bytesNow == bytesLeft) {
						
						mScreenOfst = 0;
						mScreenLength = 0;
					}
					else {
						
						mScreenOfst += bytesNow;
					}

					msgType = MSG_C_SCREEN_IMAGE;
					
		//			if (!mScreenLength)
		//				logi("Disp update took %llu cy\n", timerGetTime() - mDispUpdateStartTime);
				}
				else if (mDesiredBrightness >= 0) {
					
					swm.value = mDesiredBrightness;
					mDesiredBrightness = -1;
					remoteioCommsMsgCopyToSram(&swm, sizeof(swm), 0);
					msgType = MSG_C_SET_BRIGHTNESS;
				}
				else if (mDesiredBacklight >= 0) {
					
					swm.value = mDesiredBacklight;
					mDesiredBacklight = -1;
					remoteioCommsMsgCopyToSram(&swm, sizeof(swm), 0);
					msgType = MSG_C_SET_BACKLIGHT;
				}
				else if (mDesiredContrast >= 0) {
					
					swm.value = mDesiredContrast;
					mDesiredContrast = -1;
					remoteioCommsMsgCopyToSram(&swm, sizeof(swm), 0);
					msgType = MSG_C_SET_CONTRAST;
				}
				else if (mWantBattInfo) {
					
					msgType = MSG_C_GET_BATT_INFO;
				}
				else if (mHaveSerTxData) {
					
					remoteioCommsMsgCopyToSram(mSerTxData, mSerTxDataLen, 0);
					msgType = mSerTxMsg;
					mHaveSerTxData = false;
					KALSemaphoreSignal(mSerialSem);
				}
				else if (mWantLedCtl) {
					
					remoteioCommsMsgCopyToSram(&mLedCtl, sizeof(struct MsgLedControl), 0);
					mWantLedCtl = false;
					msgType = MSG_C_LED_CONTROL;
				}
				else if (mWantUpdate) {
					
					msgType = MSG_C_UPDATE;
					mWantUpdate = false;
				}
				else
					msgType = MSG_C_ACK;
				break;
			
			case MSG_C_CONTINUE_BOOT:
				remoteioCommsCopyMsgFromSram(&mCb, sizeof(mCb), 0);
				logi("got continue boot. screen is %ux%ux%ubpp, supported: 0x%04x\n", mCb.dispW, mCb.dispH, mCb.curDepth, mCb.supportedDepths);
				msgType = MSG_C_ACK;
				break;
			
			case MSG_C_BATTERY_INFO:
				remoteioCommsCopyMsgFromSram(&mBatteryInfo, sizeof(mBatteryInfo), 0);
				mWantBattInfo = false;
				
				logt("battery is %u%%, %u mV, %splugged in, type %d\n",
					(mBatteryInfo.flags & BATT_INFO_PERCENT_MASK) >> BATT_INFO_PERCENT_SHIFT,
					mBatteryInfo.centivolts * 10,
					(mBatteryInfo.flags & BATT_INFO_PLUGGED_IN_MASK) ? "" : "not ",
					(mBatteryInfo.flags & BATT_INFO_KIND_MASK) >> BATT_INFO_KIND_SHIFT);
				
				goto send_any_reply;
			
			case MSG_C_PEN_INFO:
				remoteioCommsCopyMsgFromSram(&msgPen, sizeof(msgPen), 0);
				if (mInputPenCbk && !mLocksBroken)
					mInputPenCbk(msgPen.x, msgPen.y);
				goto send_any_reply;
			
			case MSG_C_BUTTON_INFO:
				remoteioCommsCopyMsgFromSram(&msgKeys, sizeof(msgKeys), 0);
				if (mInputBtnCbk && !mLocksBroken)
					mInputBtnCbk(msgKeys.keysState);
				goto send_any_reply;
			
			case MSG_C_SER_DATA_RX:
				msgType = MSG_C_ACK;
				if (mSerRxCbk && !mLocksBroken && mSerRxCbk(mSerCbkData, &msgType))
					break;
				goto send_any_reply;
			
			case MSG_C_SER_OP_REPLY:
				msgType = MSG_C_ACK;
				if (mSerOpReplyCbk && !mLocksBroken && mSerOpReplyCbk(mSerCbkData, &msgType))
					break;
				goto send_any_reply;
			
			case MSG_C_SER_DATA_TX_REPLY:
				msgType = MSG_C_ACK;
				if (mSerTxReplyCbk && !mLocksBroken && mSerTxReplyCbk(mSerCbkData, &msgType))
					break;
				goto send_any_reply;
			
			default:
				msgType = MSG_S_NOT_UNDERSTOOD;
				break;
		}
	}
	else
		reply = false;

	if (reply) {
		uint16_t check;
		
		if (debug)
			logi("TX %04x %04x\n", msgMark, msgType);
		
		for (check = 0; check < 128; check++)
			dst[2] = msgType;
		
		asm volatile("dsb 0x0f" ::: "memory");
		
		fmcFlush();
		
		for (check = 0; check < 128; check++)
			dst[2] = msgType;
		
		asm volatile("dsb 0x0f" ::: "memory");
		
		fmcFlush();
		
		asm volatile("dsb 0x0f" ::: "memory");
		
		if ((check = dst[2]) != msgType)
			fatal("T1 wrote 0x%04x, got 0x%04x, now 0x%04x\n", msgType, check, dst[2]);
		
		asm volatile("dsb 0x0f" ::: "memory");
		
		for (check = 0; check < 128; check++)
			dst[6] = msgMark;
		
		
		asm volatile("dsb 0x0f" ::: "memory");
		
		//the fifo sucks for us in the fmc, but we cnanot turn it off, so let's fill it with crap at odd addresses we do not use
		fmcFlush();
	}
	
	visorIrqUpdateState();
	
	ralRestoreR9(r9);
	
	asm volatile("DSB 0x0f");		//c-m4f erratum
}

void remoteioCommsEarlyInit(void)
{
	//copy rom ASAP
	copyToSram(0, mSpringboardRom, sizeof(mSpringboardRom));
	
	visorIrqSet(false);
}

bool remoteioCommsSerInit(VisorCommsSerRxHandler rxF, VisorCommsOpReplyHandler replyF, VisorCommsTxReplyHandler txReplyF, void* cbkData)
{
	if (mSerRxCbk)
		return false;
	
	mSerOpReplyCbk = replyF;
	mSerTxReplyCbk = txReplyF;
	mSerCbkData = cbkData;
	asm volatile("":::"memory");
	//write this one last
	mSerRxCbk = rxF;
	
	return true;
}

void remoteioCommsSerDeinit(void)
{
	mSerRxCbk = NULL;
	//write that one first
	asm volatile("":::"memory");
	mSerOpReplyCbk = NULL;
	mSerTxReplyCbk = NULL;
	mSerCbkData = NULL;
}

static bool remoteioCommsSerPrvSend(void *data, uint16_t len, uint16_t message)
{
	Err e;
	
	if (len >= TO_VISOR_MBX_LEN * sizeof(uint16_t))
		return false;
	
	if (!mSerialSem) {			//racy but ok for now...
	
		//serial sem
		if (errNone != KALSemaphoreCreate(&mSerialSem, CREATE_4CC('s','s','e','m'), 0))
			fatal("Failed to init serial sem in visor comms\n");
		
		if (errNone != KALMutexCreate(&mSerialMutex, CREATE_4CC('s','s','e','m')))
			fatal("Failed to init serial mtx in visor comms\n");
	}
	
	//wait for comms
	KALMutexReserve(mSerialMutex, -1);
	
	mSerTxMsg = message;
	mSerTxData = data;
	mSerTxDataLen = len;
	asm volatile("":::"memory");	//make sure this this is written last
	mHaveSerTxData = true;
	visorIrqUpdateState();
	e = KALSemaphoreWait(mSerialSem, 1000);
	if (e)
		loge("Serial wait error: %04x\n", e);
	KALMutexRelease(mSerialMutex);
	
	return true;
}

bool remoteioCommsSerOp(void *msgToSend, uint16_t len)
{
	return remoteioCommsSerPrvSend(msgToSend, len, MSG_C_SER_OP);
}	

bool remoteioCommsSerTx(void *msgToSend, uint16_t len)
{
	return remoteioCommsSerPrvSend(msgToSend, len, MSG_C_SER_DATA_TX);
}

void remoteioCommsGetBufSzs(uint32_t *toVisorBytesP, uint32_t *fromVisorBytesP)
{
	if (toVisorBytesP)
		*toVisorBytesP = sizeof(uint16_t) * TO_VISOR_MBX_LEN;
	if (fromVisorBytesP)
		*fromVisorBytesP = sizeof(uint16_t) * TO_MODULE_MBX_LEN;
}

void remoteioCommsLateInit(void)
{
	//config irq to us and enable it
	SYSCFG->EXTICR[0] = (SYSCFG->EXTICR[0] &~ SYSCFG_EXTICR1_EXTI0_Msk) | (SYSCFG_EXTICR1_EXTI0_PA << SYSCFG_EXTICR1_EXTI0_Pos);
	EXTI->PR = 1;	//clear it
	EXTI->IMR |= 1;
	EXTI->FTSR |= 0x0001;	//int on falling edge
	NVIC_EnableIRQ(EXTI0_IRQn);
	
	//export funcs
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_UPDATE_START, remoteioCommsUpdateStart))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_COPY_TO_SRAM, remoteioCommsMsgCopyToSram))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_COPY_FROM_SRAM, remoteioCommsCopyMsgFromSram))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_INIT, remoteioCommsSerInit))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_OP, remoteioCommsSerOp))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_TX, remoteioCommsSerTx))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_SER_DEINIT, remoteioCommsSerDeinit))
		fatal("Failed to export func\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_REMOTEIO_GET_BUF_SZS, remoteioCommsGetBufSzs))
		fatal("Failed to export func\n");
}

const struct MsgContinueBoot* remoteioCommsWaitForContinueBooot(void)
{
	if (!mCb.supportedDepths) {
		logi("Waiting for comms init\n");
		while (!mCb.supportedDepths) {		//will never be zero
			asm volatile("":::"memory");	//IRQ handler will change values
		}
		logi("The wait is over\n");
	}
	
	return &mCb;
}

const struct MsgBatteryInfo* remoteioCommsGetBatteryInfo(void)
{
	mWantBattInfo = true;
	visorIrqUpdateState();
	return &mBatteryInfo;
}

void remoteioCommsSetBrightness(uint8_t val)
{
	mDesiredBrightness = (uint16_t)val;
	visorIrqUpdateState();
}

void remoteioCommsSetBacklight(bool on)
{
	mDesiredBacklight = on ? 1 : 0;
	visorIrqUpdateState();
}

void remoteioCommsSetContrast(uint8_t val)
{
	mDesiredContrast = (uint16_t)val;
	visorIrqUpdateState();
}

void remoteioCommsSetScreenDepth(uint8_t val)
{
	mDesiredDepth = (uint16_t)val;
	visorIrqUpdateState();
}

void remoteioCommsSetScreenClut(uint32_t nEntries, const struct PalmClutEntry *entries)
{
	uint32_t i;
	
	if (!mClutMsg) {
	
		struct MsgClut *t = kheapAlloc(sizeof(struct MsgClut));
		if (!t)
			fatal("Cannot allocate clut message\n");
		
		t->numEntries = 0;				//important to set before we set the pointer
		asm volatile("":::"memory");
		mClutMsg = t;
	}
	
	for (i = 0; i < nEntries; i++) {
		
		mClutMsg->colors[i].idx_r = (i << 8) + entries[i].r;
		mClutMsg->colors[i].g_b = (((uint32_t)entries[i].g) << 8) + entries[i].b;
	}
	
	//must finish all these write before setting the number of entries
	asm volatile("":::"memory");
	mClutMsg->numEntries = nEntries;
	
	visorIrqUpdateState();
}

//XXX: for future screen encoding for speed
static uint8_t* lzNumEncode(uint8_t *dst, uint32_t num)
{
	uint32_t t;
	
	do {
		t = num >> 7;
		*dst++ = (num & 0x7f) | (t ? 0x80 : 0);
		num = t;
	} while (num);
	
	return dst;
}

static uint32_t lzNumEncodedLength(uint32_t val)		// ipromise this works
{
	uint32_t t = 38 - __builtin_clz(val);
	
	t *= 0x2493;
	
	t >>= 16;
	
	return t;
}

static uint32_t screenCompress(uint8_t *dst, const uint8_t *src, uint32_t srcLen)		//may read up to 3 bytes past end of source buffer. 
{
	uint32_t i, srcPos, marker;
	uint8_t *dstInitial = dst;
	static uint32_t H[256];			//histogram AND hash table both
	
	//build histogram
	memset(H, 0, sizeof(H));
	for (i = 0; i < srcLen; i++)
		H[src[i]]++;
	
	//find least common byte to use as marker
	for (marker = 0, i = 1; i < 256; i++) {
		if (H[marker] > H[i])
			marker = i;
	}
	
	//output marker and first byte, init hash
	*dst++ = marker;
	*dst++ = src[0];
	srcPos = 1;
	memset(H, 0, sizeof(H));
	
	//compress (consider disabling alignment trapping for speed)
	while (srcPos < srcLen) {
		
		uint32_t hash, i, matchPos, matchLen, encodeLen;
		
		//hash current input
		hash = 0x177;
		for (i = 0; i < 4; i++) {
			hash ^= src[srcPos + i];
			hash *= 0x01000193;
		}
		hash %= sizeof(H) / sizeof*(H);
		
		//see what the entry is, and if it is nonzero, see if it is a match
		matchPos = H[hash];
		matchLen = 0;
		if (matchPos) {
			while(matchLen < srcLen - srcPos && src[matchPos + matchLen] == src[srcPos + matchLen])
				matchLen++;
		}
		
		//write ourselves into the hash table
		H[hash] = srcPos;
		
		//calculate encoded len of this match
		encodeLen = 1;	//marker
		i = matchLen - 3;	//constant offset
		encodeLen += lzNumEncodedLength(i);
		i = srcPos - matchPos;	//always nonzero!
		encodeLen += lzNumEncodedLength(i);
		
		//if it is worth it, encode as match
		if (encodeLen <= matchLen) {
			*dst++ = marker;
			dst = lzNumEncode(dst, srcPos - matchPos);
			dst = lzNumEncode(dst, matchLen - 3);
			srcPos += matchLen;
		}
		else if (src[srcPos] == marker) {
			*dst++ = marker;
			*dst++ = 0;
			srcPos++;
		}
		else {
			
			*dst++ = src[srcPos++];
		}
	}
	
	return dst - dstInitial;
}

void remoteioCommsBreakLocks(void)
{
	mLocksBroken = true;
}

void remoteioCommsRequestScreenRedraw(void *data, uint32_t len, bool is16bitData)	//data MUST remain valid until remoteioCommsIsScreenRedrawDone() returns true
{
	static uint8_t *compressedData;
	static uint32_t *swappedData;
	const uint32_t *src;
	uint32_t *dst;
	uint32_t i;
	
	if (!mCb.supportedDepths)	//we do not yet know enough
		return;
	
	if (!compressedData) {
		
		uint32_t screenBytesMax = mCb.dispW * mCb.dispH * (32 - __builtin_clz(mCb.supportedDepths)) / 8;
		uint32_t compressedMax = (257 * screenBytesMax + 255) / 256 + 1;
		
		swappedData = (uint32_t*)CPU_HARDWIRED_VTMP1_SPACE;
		compressedData = (uint8_t*)(CPU_HARDWIRED_VTMP1_SPACE + screenBytesMax);
		
		if (screenBytesMax + compressedMax > CPU_HARDWIRED_VTMP1_SIZE)
			fatal("Cannot allocate swapped screen buf/compressed buf - too little space\n");
	}
	
	asm volatile("":::"memory");
	
	if (mScreenLength)
		return;
	
	asm volatile("":::"memory");
	
//	mDispUpdateStartTime = timerGetTime();
	
	//16-bit data needs to be byteswapped
	if (is16bitData) {
		
		src = (const uint32_t*)data;
		dst = swappedData;
		
		for (i = 0; i < len / 4; i++)	//assume 4 byte salignment because we can
			asm("rev16 %0, %1":"=l"(*dst++):"l"(*src++):"memory");
		
		data = swappedData;
	}
	
	//compress
	len = screenCompress(compressedData, data, len);
	
	//swap again since this is now byte data
	dst = (uint32_t*)compressedData;
	for (i = 0; i < (len + 3) / 4; i++, dst++)	//may byteswap an extra halfword, this is safe
		asm("rev16 %0, %1":"=l"(*dst):"l"(*dst):"memory");
	
	//send it over
	
	NVIC_DisableIRQ(EXTI0_IRQn);
	
	mScreenPtr = compressedData;
	mScreenOfst = 0;
	mScreenLength = len;
	
	//write the above before irqs are on
	asm volatile("":::"memory");
	
	NVIC_EnableIRQ(EXTI0_IRQn);
	
	visorIrqUpdateState();
}

bool remoteioCommsIsScreenRedrawDone(void)
{
	asm volatile("":::"memory");
	
	return !mScreenLength;
}

void remoteioCommsSetInputHandlers(VisorCommsButtonStateCbk btnF, VisorCommsPenStateCbk penF)
{
	mInputBtnCbk = btnF;
	mInputPenCbk = penF;
}

bool* remoteioCommsUpdateStart(volatile void **stateP)
{
	mWantUpdate = true;
	visorIrqUpdateState();
	
	*stateP = (volatile void*)COMMS_SRAM_BASE;
	return &mWantUpdate;
}

void remoteioCommsLedControl(uint32_t pattern, uint16_t csecPerPiece, uint16_t csecBetween, uint16_t numTimes)
{
	struct MsgLedControl ctl = {.patternHi = pattern >> 16, .patternLo = pattern, .csecPerPiece = csecPerPiece, .csecBetween = csecBetween, .numTimes = numTimes, };
	
	mLedCtl = ctl;
	mWantLedCtl = true;
	visorIrqUpdateState();
}