#define WEAK __attribute__ ((weak))
#define ALIAS(f) __attribute__ ((weak, alias (#f)))

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE

#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "ral_export.h"
#include "cpu.h"
#include "printf.h"
#include "pinout.h"
#include "timers.h"
#include "memmap.h"
#include "entry.h"
#include "heap.h"
#include "irqs.h"
#include "boot.h"
#include "mpu.h"
#include "ral.h"

//PIO0 is used for the screen, PIO1 is shared between various things, driven here.

void __attribute__((used)) IntDefaultHandler(void)
{
	cpuIrqDefaultHandler();
}

WEAK void TIMER0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void TIMER1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void TIMER2_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void TIMER3_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void PWM_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void USB_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void XIP_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void PIO0_0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void PIO0_1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void PIO1_0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void PIO1_1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void DMA0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void DMA1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void IO_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void QSPI_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void SIO0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void SIO1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void CLOCKS_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void SPI0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void SPI1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void UART0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void UART1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void ADC_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void I2C0_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void I2C1_IRQHandler(void) ALIAS(IntDefaultHandler);
WEAK void RTC_IRQHandler(void) ALIAS(IntDefaultHandler);


__attribute__ ((aligned (256))) void (* const __ISR_VECTORS[])  (void) =
{
	0,							// unused: initial sp
	0,							// unused: reset handler

	NMI_Handler,				// The NMI handler
	HardFault_Handler,			// The hard fault handler

#ifdef BUILD_FOR_THUMB_1
	0,
	0,
	0,
#else
	MemManage_Handler,
	BusFault_Handler,
	UsageFault_Handler,
#endif
	0,
	0,
	0,
	0,
	
	SVC_Handler,				// SVCall handler
#ifdef BUILD_FOR_THUMB_1
	0,
#else
	DebugMonitor_Handler,		// Reserved
#endif
	0,
	PendSV_Handler,				// The PendSV handler
	SysTick_Handler,			// The SysTick handler
	
	// Chip Level interrupts here
	TIMER0_IRQHandler,
	TIMER1_IRQHandler,
	TIMER2_IRQHandler,
	TIMER3_IRQHandler,
	PWM_IRQHandler,
	USB_IRQHandler,
	XIP_IRQHandler,
	PIO0_0_IRQHandler,
	PIO0_1_IRQHandler,
	PIO1_0_IRQHandler,
	PIO1_1_IRQHandler,
	DMA0_IRQHandler,
	DMA1_IRQHandler,
	IO_IRQHandler,
	QSPI_IRQHandler,
	SIO0_IRQHandler,
	SIO1_IRQHandler,
	CLOCKS_IRQHandler,
	SPI0_IRQHandler,
	SPI1_IRQHandler,
	UART0_IRQHandler,
	UART1_IRQHandler,
	ADC_IRQHandler,
	I2C0_IRQHandler,
	I2C1_IRQHandler,
	RTC_IRQHandler,
};

void machIdle(void)
{
	asm volatile("wfi");
}

static void mpuRegCfg(uint32_t idx, uint32_t addr, uint32_t cfg)
{
	#ifdef __MPU_PRESENT
		if (!(MPU->TYPE & 1) && (MPU->TYPE & MPU_TYPE_DREGION_Msk)) {	//we do what we can
			
			MPU->RBAR = addr | 0x10 | idx;
			MPU->RASR = cfg;
		}
	#endif
}

//configures mmu for storage ram
void machSetStorageAreaWriteable(bool writeable)
{
	//64KB of storage heap
	mpuRegCfg(1, CPU_STORAGE_RAM_BASE, (writeable ? MPU_PERM_U_RW_S_RW : MPU_PERM_U_RO_S_RO) | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | MPU_REGION_SZ_64KB);
}

static void gpiosConfig(void)
{
	static struct {
		uint8_t pin;
		uint8_t padCfg;
		uint8_t funcSel;
	} const cfgs[] = {
		{PIN_LCD_DnC, 0, 5},
		{PIN_LCD_CS, PADS_BANK0_GPIO0_DRIVE_VALUE_12MA, 5},
		{PIN_SPI_CLK, PADS_BANK0_GPIO0_SLEWFAST_BITS, 5},
		{PIN_SPI_MOSI, PADS_BANK0_GPIO0_SLEWFAST_BITS, 5},
		{PIN_SPI_MISO, PADS_BANK0_GPIO0_OD_BITS | PADS_BANK0_GPIO0_IE_BITS | PADS_BANK0_GPIO0_PDE_BITS, 5},
		{PIN_LCD_BL, PADS_BANK0_GPIO0_DRIVE_VALUE_12MA, 4},	//we drive a BJT - current matters
		{PIN_LCD_RESET, 0, 5},
		{PIN_TOUCH_CS, PADS_BANK0_GPIO0_DRIVE_VALUE_12MA, 5},
		{PIN_TOUCH_IRQ, PADS_BANK0_GPIO0_OD_BITS | PADS_BANK0_GPIO0_IE_BITS | PADS_BANK0_GPIO0_SCHMITT_BITS, 5},
		{PIN_SDIO_D3_CS, 0, 5},
		{PIN_REG_MODE, 0, 5},
		{PIN_VBUS_SENSE, PADS_BANK0_GPIO0_OD_BITS | PADS_BANK0_GPIO0_IE_BITS | PADS_BANK0_GPIO0_SCHMITT_BITS, 5},
		{PIN_LED, PADS_BANK0_GPIO0_DRIVE_VALUE_12MA, 5},
		{PIN_VSYS_DIV3, PADS_BANK0_GPIO0_OD_BITS, 5},
		
		{PIN_SDIO_CLK, 0, IO_BANK0_GPIO0_CTRL_FUNCSEL_VALUE_PIO1_0 | PADS_BANK0_GPIO0_IE_BITS},
		{PIN_SDIO_CMD, 0, IO_BANK0_GPIO0_CTRL_FUNCSEL_VALUE_PIO1_0},
		{PIN_SDIO_D0, PADS_BANK0_GPIO0_OD_BITS | PADS_BANK0_GPIO0_IE_BITS, IO_BANK0_GPIO0_CTRL_FUNCSEL_VALUE_PIO1_0},
	};
	
	uint_fast8_t i;
	
	for (i = 0; i < 30; i++) {
		
		padsbank0_hw->io[i] = PADS_BANK0_GPIO0_OD_BITS | PADS_BANK0_GPIO0_PDE_BITS;
		iobank0_hw->io[i].ctrl |= IO_BANK0_GPIO0_CTRL_FUNCSEL_VALUE_NULL;
	}
	
	for (i = 0; i < sizeof(cfgs) / sizeof(*cfgs); i++) {
		
		uint_fast8_t pinNo = cfgs[i].pin;
		
		padsbank0_hw->io[pinNo] = cfgs[i].padCfg;
		iobank0_hw->io[pinNo].ctrl = (iobank0_hw->io[pinNo].ctrl &~ IO_BANK0_GPIO0_CTRL_FUNCSEL_BITS) | (cfgs[i].funcSel << IO_BANK0_GPIO0_CTRL_FUNCSEL_LSB);
	}
	
	sio_hw->gpio_oe = (1 << PIN_SDIO_CLK) | (1 << PIN_SDIO_CMD) | (1 << PIN_LCD_DnC) | (1 << PIN_LCD_CS) | (1 << PIN_SPI_CLK) | (1 << PIN_SPI_MOSI) | (1 << PIN_LCD_BL) | (1 << PIN_LCD_RESET) | (1 << PIN_TOUCH_CS) | (1 << PIN_SDIO_D3_CS) | (1 << PIN_REG_MODE) | (1 << PIN_LED);
	sio_hw->gpio_out = (1 << PIN_LCD_CS) | (1 << PIN_TOUCH_CS) | (1 << PIN_SDIO_D3_CS);
}


#define SIDE_SET_HAS_ENABLE_BIT				0
#define SIDE_SET_NUM_BITS					1
#define DEFINE_PIO_INSTRS
#include "pioAsm.h"

	static int8_t mSdPioNum = -1;

	static uint_fast8_t sdPioSetupProgram(uint_fast8_t pc)
	{
		uint32_t numInstrSlots = sizeof(pio1_hw->instr_mem) / sizeof(*pio1_hw->instr_mem);
		
		//verify we have the space
		if (numInstrSlots - pc < 2)
			return 0;
		
		pio1_hw->instr_mem[pc++] = I_OUT(0, 0, OUT_DST_PINS, 1);
		pio1_hw->instr_mem[pc++] = I_IN(0, 1, IN_SRC_PINS, 1);
		
		return pc;
	}
	
	static int8_t prvRepalmGetSdioPioInfo(void)	//return pio number
	{
		return mSdPioNum;
	}
	
	static void sdSetup(uint_fast8_t *pioPCP, uint_fast8_t *pioSmP)
	{
		static const uint8_t outPins[] = {PIN_SDIO_CLK, PIN_SDIO_CMD};
		uint_fast8_t start = *pioPCP, end, sm = (*pioSmP)++, i;
		
		if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_GET_SD_PIO_INFO, &prvRepalmGetSdioPioInfo))
			return;
		
		//are there SMs?
		if (sm >= NUM_PIO_STATE_MACHINES)
			return;
	
		//set pin directions
		for (i = 0; i < sizeof(outPins) / sizeof(*outPins); i++) {
			
			pio1_hw->sm[sm].pinctrl = (pio1_hw->sm[sm].pinctrl &~ (PIO_SM1_PINCTRL_SET_COUNT_BITS | PIO_SM1_PINCTRL_SET_BASE_BITS)) | (1 << PIO_SM1_PINCTRL_SET_COUNT_LSB) | (((uint32_t)outPins[i]) << PIO_SM1_PINCTRL_SET_BASE_LSB);
			pio1_hw->sm[sm].instr = I_SET(0, 0, SET_DST_PINDIRS, 1);
		}
		
		//input should skip sync
		pio1_hw->input_sync_bypass |= 1 << PIN_SDIO_D0;
	
		end = sdPioSetupProgram(start);
		
		//was there code space?
		if (!end)
			return;
		
		*pioPCP = end--;
		
		pio1_hw->sm[sm].clkdiv = (240 << PIO_SM0_CLKDIV_INT_LSB);
		pio1_hw->sm[sm].execctrl = (pio0_hw->sm[sm].execctrl &~ (PIO_SM0_EXECCTRL_WRAP_TOP_BITS | PIO_SM0_EXECCTRL_WRAP_BOTTOM_BITS | PIO_SM2_EXECCTRL_SIDE_EN_BITS)) | (end << PIO_SM0_EXECCTRL_WRAP_TOP_LSB) | (start << PIO_SM0_EXECCTRL_WRAP_BOTTOM_LSB);
		pio1_hw->sm[sm].shiftctrl = (pio0_hw->sm[sm].shiftctrl &~ (PIO_SM1_SHIFTCTRL_PULL_THRESH_BITS | PIO_SM1_SHIFTCTRL_PUSH_THRESH_BITS | PIO_SM0_SHIFTCTRL_IN_SHIFTDIR_BITS | PIO_SM0_SHIFTCTRL_OUT_SHIFTDIR_BITS)) | (8 << PIO_SM1_SHIFTCTRL_PULL_THRESH_LSB) | (8 << PIO_SM1_SHIFTCTRL_PUSH_THRESH_LSB) | PIO_SM0_SHIFTCTRL_AUTOPULL_BITS | PIO_SM0_SHIFTCTRL_AUTOPUSH_BITS;
		pio1_hw->sm[sm].pinctrl = (SIDE_SET_BITS_USED << PIO_SM1_PINCTRL_SIDESET_COUNT_LSB) | (PIN_SDIO_CLK << PIO_SM1_PINCTRL_SIDESET_BASE_LSB) | (1 << PIO_SM1_PINCTRL_OUT_COUNT_LSB) | (PIN_SDIO_CMD << PIO_SM1_PINCTRL_OUT_BASE_LSB) | (PIN_SDIO_D0 << PIO_SM1_PINCTRL_IN_BASE_LSB);
	
		//start it
		pio1_hw->sm[sm].instr = I_JMP(0, 0, JMP_ALWAYS, start);
	
		logi("SDIO used sm %u, code %u..%u\n", sm, start, end);
		mSdPioNum = sm;
	}

#define UNDEF_PIO_INSTRS
#include "pioAsm.h"

static void prvPio1setup(void)
{
	uint_fast8_t pioPC = 0, pioSm = 0;
	
	pio1_hw->ctrl = (pio1_hw->ctrl &~ PIO_CTRL_SM_ENABLE_BITS) | PIO_CTRL_SM_RESTART_BITS;	//disable and reset pio1 state machines
	
	//set up each SM for each client
	sdSetup(&pioPC, &pioSm);
	
	//enable all the SMs that were set up
	logi("%u SMs and %u instructions used in PIO1\n", pioSm, pioPC);
	pio1_hw->ctrl |= ((1 << pioSm) - 1) << PIO_CTRL_SM_ENABLE_LSB;
}

void __attribute__((used)) machInit(uint32_t stage, const void* data)
{
	if (stage == STAGE_INIT_EARLY) {	//no globals/vectors/anything yet!!!
		
		uint32_t unitsToReset;
		
		//tell refclock to use ROSC
		clocks_hw->clk[clk_ref].ctrl = (clocks_hw->clk[clk_ref].ctrl &~ CLOCKS_CLK_REF_CTRL_SRC_BITS) | (CLOCKS_CLK_REF_CTRL_SRC_VALUE_ROSC_CLKSRC_PH << CLOCKS_CLK_REF_CTRL_SRC_LSB);
		
		//use ref clock for cpu, use sys clock for periphs
		clocks_hw->clk[clk_peri].ctrl &=~ CLOCKS_CLK_PERI_CTRL_ENABLE_BITS;
		clocks_hw->clk[clk_sys].ctrl = (clocks_hw->clk[clk_sys].ctrl &~ CLOCKS_CLK_SYS_CTRL_SRC_BITS)| (CLOCKS_CLK_SYS_CTRL_SRC_VALUE_CLK_REF << CLOCKS_CLK_SYS_CTRL_SRC_LSB);
		clocks_hw->clk[clk_peri].ctrl = (clocks_hw->clk[clk_peri].ctrl &~ (CLOCKS_CLK_PERI_CTRL_KILL_BITS | CLOCKS_CLK_PERI_CTRL_AUXSRC_BITS)) | CLOCKS_CLK_PERI_CTRL_ENABLE_BITS | (CLOCKS_CLK_PERI_CTRL_AUXSRC_VALUE_CLK_SYS << CLOCKS_CLK_PERI_CTRL_AUXSRC_LSB);
		
		//start XOSC (by stopping it first...)
		xosc_hw->ctrl = (xosc_hw->ctrl &~ XOSC_CTRL_ENABLE_BITS) | (XOSC_CTRL_ENABLE_VALUE_DISABLE << XOSC_CTRL_ENABLE_LSB);
		while (xosc_hw->status & XOSC_STATUS_ENABLED_BITS);
		xosc_hw->startup = (xosc_hw->startup &~ XOSC_STARTUP_DELAY_BITS) | (8191 << XOSC_STARTUP_DELAY_LSB);
		xosc_hw->ctrl = (xosc_hw->ctrl &~ (XOSC_CTRL_FREQ_RANGE_BITS | XOSC_CTRL_ENABLE_BITS)) | (XOSC_CTRL_ENABLE_VALUE_ENABLE << XOSC_CTRL_ENABLE_LSB) | (XOSC_CTRL_FREQ_RANGE_VALUE_1_15MHZ << XOSC_CTRL_FREQ_RANGE_LSB);
		while ((xosc_hw->status & (XOSC_STATUS_STABLE_BITS | XOSC_STATUS_ENABLED_BITS)) != (XOSC_STATUS_STABLE_BITS | XOSC_STATUS_ENABLED_BITS));
		
		//tell refclock to use XOSC
		clocks_hw->clk[clk_ref].ctrl = (clocks_hw->clk[clk_ref].ctrl &~ CLOCKS_CLK_REF_CTRL_SRC_BITS) | (CLOCKS_CLK_REF_CTRL_SRC_VALUE_XOSC_CLKSRC << CLOCKS_CLK_REF_CTRL_SRC_LSB);
		
		//reset units
		unitsToReset = RESETS_RESET_PWM_BITS | RESETS_RESET_RTC_BITS | RESETS_RESET_PADS_BANK0_BITS | RESETS_RESET_IO_BANK0_BITS | RESETS_RESET_PIO0_BITS | RESETS_RESET_PIO1_BITS | RESETS_RESET_DMA_BITS | RESETS_RESET_PLL_SYS_BITS | RESETS_RESET_TIMER_BITS;
		resets_hw->reset |= unitsToReset;		//this is correct...
		resets_hw->reset |= unitsToReset;
		resets_hw->reset &=~ unitsToReset;
		resets_hw->reset &=~ unitsToReset;
		resets_hw->reset &=~ unitsToReset;
		while ((resets_hw->reset_done & unitsToReset) != unitsToReset);
			
		//many MHz please
		logt("pll setup\n");
		pll_sys_hw->pwr |= (PLL_PWR_VCOPD_BITS | PLL_PWR_POSTDIVPD_BITS | PLL_PWR_PD_BITS);		//dividers on
		pll_sys_hw->fbdiv_int = (pll_sys_hw->fbdiv_int &~ PLL_FBDIV_INT_BITS) | ((CPU_CLOCK_RATE / 1000000 / 2) << PLL_FBDIV_INT_LSB);
		pll_sys_hw->prim = (pll_sys_hw->prim &~ (PLL_PRIM_POSTDIV1_BITS | PLL_PRIM_POSTDIV2_BITS)) | (6 << PLL_PRIM_POSTDIV1_LSB) | (1 << PLL_PRIM_POSTDIV2_LSB);
		pll_sys_hw->pwr &=~ (PLL_PWR_VCOPD_BITS | PLL_PWR_POSTDIVPD_BITS | PLL_PWR_PD_BITS);		//dividers on
		while (!(pll_sys_hw->cs & PLL_CS_LOCK_BITS));
		pll_sys_hw->cs &=~PLL_CS_BYPASS_BITS;
		logt("pll is up\n");
	
		//switch sys.AUX to pll
		clocks_hw->clk[clk_sys].ctrl = (clocks_hw->clk[clk_sys].ctrl &~ CLOCKS_CLK_SYS_CTRL_AUXSRC_BITS) | (CLOCKS_CLK_SYS_CTRL_AUXSRC_VALUE_CLKSRC_PLL_SYS << CLOCKS_CLK_SYS_CTRL_AUXSRC_LSB);
		
		//switch sys to AUX and wait for it
		clocks_hw->clk[clk_sys].ctrl = (clocks_hw->clk[clk_sys].ctrl &~ CLOCKS_CLK_SYS_CTRL_SRC_BITS) | (CLOCKS_CLK_SYS_CTRL_SRC_VALUE_CLKSRC_CLK_SYS_AUX << CLOCKS_CLK_SYS_CTRL_SRC_LSB);
		while (((clocks_hw->clk[clk_sys].selected & CLOCKS_CLK_REF_SELECTED_BITS) >> CLOCKS_CLK_REF_SELECTED_LSB) != (1 << CLOCKS_CLK_SYS_CTRL_SRC_VALUE_CLKSRC_CLK_SYS_AUX));
		
		//DMA units get fucked up if they do not get bus cycles, to make them high priority
		bus_ctrl_hw->priority = (bus_ctrl_hw->priority &~ (BUSCTRL_BUS_PRIORITY_PROC1_BITS | BUSCTRL_BUS_PRIORITY_PROC0_BITS)) | BUSCTRL_BUS_PRIORITY_DMA_R_BITS | BUSCTRL_BUS_PRIORITY_DMA_W_BITS;
		
		gpiosConfig();
	}
	else if (stage == STAGE_SOMEWHAT_EARLY_INIT) {
		
		rp2040rtInit();
	}
	else if (stage == STAGE_INIT_SET_VTOR) {
		
		SCB->VTOR = (uint32_t)__ISR_VECTORS;
	}
	else if (stage == STAGE_SETUP_HEAPS) {
		
		kheapRegisterHeap(HAL_STATIC_MEM_BASE, HAL_STATIC_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_USABLE_FOR_DMA | MEM_USABLE_FOR_EXEC | MEM_FAST);
	}
	else if (stage == STAGE_INIT_MPU) {
		
		#ifdef __MPU_PRESENT
		
			uint32_t i;
		
			//4MB region for OS (round down to include 1st stage)
			mpuRegCfg(0, CPU_ROM_BASE &~ 0xffff, MPU_PERM_U_RO_S_RO | MPU_MEM_TYPE_ROM | MPU_FLAG_ENABLED | MPU_REGION_SZ_2MB);
			
			//storage RAM
			machSetStorageAreaWriteable(false);
			
			//dynamic ram and other ram
			mpuRegCfg(2, 0x20000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | MPU_REGION_SZ_512KB);
			
			//IOPORT & SCB (0xD0000000 + 0x20000000, device, nx)
			mpuRegCfg(3, 0x80000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | MPU_REGION_SZ_2GB | MPU_SRD_0th | MPU_SRD_1st | MPU_SRD_2nd | MPU_SRD_3rd | MPU_SRD_4th | MPU_SRD_7th);
			
			//Periphs (0x40000000 + 0x20000000, device, nx)
			mpuRegCfg(4, 0x40000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | MPU_REGION_SZ_512MB);
			
			//rp2040 ROM (for using rom funcs)
			mpuRegCfg(5, 0x00000000, MPU_PERM_U_RO_S_RO | MPU_MEM_TYPE_ROM | MPU_FLAG_ENABLED | MPU_REGION_SZ_16KB);
			
			//disable the other regions
			for (i = 6; i < (MPU->TYPE & MPU_TYPE_DREGION_Msk) >> MPU_TYPE_DREGION_Pos; i++)
				mpuRegCfg(i, 0, 0);

			//mpu on
			MPU->CTRL = MPU_CTRL_HFNMIENA_Msk | MPU_CTRL_ENABLE_Msk;	//HFNMIENA is needed for m0faultDispatch to properly detect accesses blocked by MPU
		#endif
	}
	else if (stage == STAGE_INIT_INTERRUPTS) {
		
		uint_fast16_t i;
		
		const struct MachInitDataInterrupts *info = (const struct MachInitDataInterrupts*)data;
		uint32_t mediumPrio = (info->lowestAllowablePrio + info->highestAllowablePrio) / 2;
		
		//set all HW ints to medium prio
		for (i = (unsigned)TIMER0_IRQn; i <= (unsigned)TIMER0_IRQn + CPU_NUM_IRQS; i++) {
			
			NVIC_SetPriority(i, mediumPrio);
		}
	
		//scheduler timer interrupt is high prio too so nobody else can interrupt it (and more importantly - it will not interrupt syscalls and vise-versa)
		NVIC_SetPriority(TIMER0_IRQn, info->schedulingTimerPrio);
		
		//set up shared PIO's programs
		prvPio1setup();
	}
}

bool hwMaybeGetRomToken(uint32_t name, const void **dataP, uint16_t *szP)
{
	return false;
}

void hwGetMiscFlags(uint16_t *miscFlagsP, uint16_t *extMiscFlagsP)
{
	if (miscFlagsP)
		*miscFlagsP = hwrMiscFlagHasMiscFlagExt | hwrMiscFlagHasCradleDetect | hwrMiscFlagNoRTCBug | hwrMiscFlagHasMbdIrDA;
	
	if (extMiscFlagsP)
		*extMiscFlagsP = hwrMiscFlagExt115KIrOK | hwrMiscFlagExtHasLiIon;
}

int32_t cpuGetClockRate(enum ClockRateDevice dev)
{
	switch (dev) {
		case CpuClockRate:
		case TimerClockRate:
			return CPU_CLOCK_RATE;
		
		case SdioUnitClockRate:
			return CPU_CLOCK_RATE / 2;
		
		default:
			return-1;
	}
}

bool hwPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{
	return false;
}

void machSleep(void)
{
	logi("pretending to sleep\n");
	SysTaskDelay(10000);
	logi("waking up\n");
	dalModifyWakeFlags(DAL_WAKE_FLAG_GENERAL, 0);
	//nothing yet
}

static void machBusyWaitDelay(uint64_t ticks)
{
	uint64_t start = timerGetTime();
	
	while (timerGetTime() - start < ticks);
}

void machBusyWaitDelayMsec(uint32_t msec)
{
	machBusyWaitDelay((uint64_t)msec * TIMER_TICKS_PER_MSEC);
}

void machBusyWaitDelayUsec(uint32_t usec)
{
	machBusyWaitDelay((uint64_t)usec * (TIMER_TICKS_PER_MSEC / 1000));
}

void deviceReset(bool doHardReset)
{
	if (doHardReset) {
		
		machSetStorageAreaWriteable(true);
		asm volatile("cpsid i");
		memset((void*)CPU_STORAGE_RAM_BASE, 0, CPU_STORAGE_RAM_SIZE);
	}
	NVIC_SystemReset();
}

Err machinePaceDispatch(EmulStateRef ref, uint16_t call, Err *ret68kP)
{
	return sysErrNotAllowed;
}


