#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "palmcardComms.h"
#include "palmcardProto.h"
#include "audiohw.h"
#include "printf.h"
#include "heap.h"
#include "irqs.h"
#include "ral.h"
#include "cpu.h"



#define SAMPLES_PER_BUFFER	512
	
static AudioOutHwReadyForMoreSamplesF mReadyForSamplesF;


//keep in mind that the data is sent as pairs of u16s so we need to swap adjacent samples. we thus process two at a time
static void audioPrvConvertData(void *dstPtr, const int32_t *src, uint32_t nSamples)
{
	uint32_t dummy1, dummy2, dummy3;
	/*
		basically:
	
			u8* dst = dstPtr
			u8 ofst = 1
	
			while(num--) {
				
				int32_t t = *src++;
				
				//saturate to signed 24-bit value
				if (t >= 0x800000)
					t = 0x7fffff;
				else if (t < -0x800000)
					t = -0x800000;
				
				//convert to unsigned
				t += 0x00800000;
				
				//scale to our range (unsigned div, it is faster)
				t = t >> 16
				
				dst[ofst] = t;
				dst++;
				ofst = -ofst;		//byteswaps as we go
			}
	*/
	
	
	asm volatile(
		"	movs	%2, #1				\n\t"
		"1:								\n\t"
		"	ldmia	%4!, {%0}			\n\t"	//get sample
		"	asrs	%1, %0, #23			\n\t"	//saturate
		"	beq		no_sat				\n\t"	//was positive, no sat needed
		"	adds	%1, #1				\n\t"
		"	beq		no_sat				\n\t"	//was negative, no sat needed
		"	bpl		sat_pos				\n\t"	//was positive and needs to be saturated
		"sat_neg:						\n\t"
		"	movs	%0, #0x81			\n\t"
		"	b		sat_done			\n\t"
		"sat_pos:						\n\t"
		"	movs	%0, #0x7f			\n\t"
		"	b		sat_done			\n\t"
		"no_sat:						\n\t"
		"	asrs	%0, %0, #16			\n\t"
		"	adds	%0, #0x80			\n\t"
		"sat_done:						\n\t"
		"	strb	%0, [%5, %2]		\n\t"
		"	adds	%5, #1				\n\t"
		"	negs	%2, %2				\n\t"
		"	subs	%3, #1				\n\t"
		"	bne		1b					\n\t"
		:"=&l"(dummy1), "=&l"(dummy2), "=&l"(dummy3), "+l"(nSamples), "+l"(src), "+l"(dstPtr)
		:
		:"memory", "cc"
	);
}

void palmcardCommsExtNeedAudioData(uint16_t *dst)
{
	uint32_t r9state = ralSetSafeR9();

	audioPrvConvertData(dst, mReadyForSamplesF(true), SAMPLES_PER_BUFFER);
	mReadyForSamplesF(false);

	ralRestoreR9(r9state);
}

bool audioOutHwInit(AudioOutHwReadyForMoreSamplesF readyForSamplesF, uint32_t *numSamplesPerBufP, enum AudioSampleRate* nativeRateP, bool *nativeStereoP)
{
	*nativeRateP = AudioRate32000;
	*nativeStereoP = false;
	*numSamplesPerBufP = SAMPLES_PER_BUFFER;
	mReadyForSamplesF = readyForSamplesF;

	return true;
}


void audioOutHwSetState(bool on)
{
	uint_fast16_t devState = palmcardCommsGetDeviceState(), devStateNew;
	
	if (on)
		devStateNew = devState | PCC_DEVSTATE_BIT_AUDIO_ON;
	else
		devStateNew = devState &~ PCC_DEVSTATE_BIT_AUDIO_ON;
	
	if (devStateNew != devState) {
		
		logt("audio %s %s\n", on ? "ON" : "OFF", "start");
		palmcardCommsSetDeviceState(devStateNew);
		palmcardCommsUpdateIrqSta(PCC_IRQ_BIT_HAVE_NEW_DEVICE_CONFIG, 0);
		while (palmcardCommsGetPendingIrqs() & PCC_IRQ_BIT_HAVE_NEW_DEVICE_CONFIG);
		logt("audio %s %s\n", on ? "ON" : "OFF", "done");
	}
}

//this func is not called if sampled oudio is available
bool audioOnlySimpleOutInit(void)
{
	return false;
}

//this func is not called if sampled oudio is available
void audioOnlySimpleTone(uint32_t freq, uint32_t amp)
{
	//nothing
}