#include "memoryHeap.h"
#include "printf.h"
#include "boot.h"
#include "heap.h"
#include "dal.h"

//#define GUARD_HEAP_ALLOCS

#define PTR_FOR_ZERO_SIZED_ALLOCS		((void*)0xf0f0f0ff)

#define MAX_HEAPS		8

struct HeapInfo {
	struct MemoryHeap *heap;
	uint32_t attrs;
};

static struct HeapInfo mHeaps[MAX_HEAPS] = {};

void kheapInit(void)
{
	//nothing, actually
}

void kheapRegisterHeap(uint32_t start, uint32_t size, uint32_t attributes)
{
	uint32_t i;
	
	for (i = 0; i < MAX_HEAPS; i++) {
		if (!mHeaps[i].heap) {
			
			mHeaps[i].heap = memoryheapInit(start, size);
			mHeaps[i].attrs = attributes;
			break;
		}
	}
}

static void* memAlloc(uint32_t sz, uint32_t intendedUse, void* from)
{
	if (!sz)
		return PTR_FOR_ZERO_SIZED_ALLOCS;
	else {
		
		bool systemHeapAvail = (dalGetInitStage() >= DAL_INIT_STAGE_MEM_AND_DM) && !(intendedUse & MEM_NO_OS_HEAP), systemTried = false;
		uint32_t i, j, requiredAttrs = intendedUse &~ MEM_NO_OS_HEAP;
		void *ret;
	
		if (systemHeapAvail && !(intendedUse & (MEM_FAST | MEM_UNCACHED))) {		//try system first if avail and allowed
			
			systemTried = true;
			ret = MemChunkNew(0, sz, 0x1200);
			if (ret) {
				logst("alloced %u bytes with request 0x%02x from heap %d: 0x%08x from 0x%08x\n", sz, intendedUse, -1, ret, from);
				return ret;
			}
		}
		
		for (j = 0; j < 2; j++) {
			for (i = 0; i < MAX_HEAPS; i++) {
				
				if (!mHeaps[i].heap)
					break;
				
				if ((mHeaps[i].attrs & requiredAttrs) == requiredAttrs) {
					
					ret = memoryheapAlloc(mHeaps[i].heap, sz);
					if (ret) {
						logst("alloced %u bytes with request 0x%02x from heap %d: 0x%08x from 0x%08x\n", sz, intendedUse, i, ret, from);
						return ret;
					}
				}
			}
			
			//try again without requiring "fast" if it was required
			if (requiredAttrs & MEM_FAST)
				requiredAttrs &=~ MEM_FAST;
			else
				break;
		}
		
		//try the systemheap again if we have not yet and it is available
		if (!systemTried && systemHeapAvail) {
			ret = MemChunkNew(0, sz, 0x1200);
			if (ret) {
				logst("alloced %u bytes with request 0x%02x from heap %d: 0x%08x from 0x%08x\n", sz, intendedUse, -1, ret, from);
				return ret;
			}
		}
		
		return NULL;
	}
}

static void memFree(void* ptr)
{
	if (!ptr || ptr == PTR_FOR_ZERO_SIZED_ALLOCS)
		return;
	else {
		
		uint32_t i;
		
		for (i = 0; i < MAX_HEAPS; i++) {
				
			if (!mHeaps[i].heap)
				break;
			
			if (memoryheapIsThisInThere(mHeaps[i].heap, ptr)) {
				
				memoryheapFree(mHeaps[i].heap, ptr);
				return;
			}
		}
	}
	
	MemChunkFree(ptr);
}

static struct MemoryHeap* kheapPrvFindHeapForChunk(void *ptr)
{
	if (ptr && ptr != PTR_FOR_ZERO_SIZED_ALLOCS) {
		
		uint32_t i;
		
		for (i = 0; i < MAX_HEAPS; i++) {
				
			if (!mHeaps[i].heap)
				break;
			
			if (memoryheapIsThisInThere(mHeaps[i].heap, ptr)) {
				
				return mHeaps[i].heap;
			}
		}
	}
	return NULL;
}

bool kheapIsInOurHeaps(void *ptr)
{
	return !!kheapPrvFindHeapForChunk(ptr);
}

uint32_t kheapGetChunkSize(void *ptr)
{
	struct MemoryHeap *heap = kheapPrvFindHeapForChunk(ptr);
	
	if (heap)
		return memoryheapGetChunkActualSize(heap, ptr);
	else
		return MemPtrSize(ptr);
}


#ifdef GUARD_HEAP_ALLOCS

	void* kheapAllocEx(uint32_t sz, uint32_t intendedUse)
	{
		uint32_t *mem;
		
		sz = (sz + 3) / 4 * 4;	//round to word size
		sz += 12;
		
		mem = memAlloc(sz, intendedUse, __builtin_return_address(0));
		mem[0] = 0xfeeff00f ^ (uintptr_t)mem;
		mem[1] = sz;
		mem[(sz / 4) - 1] = 0xcafeface ^ (uintptr_t)mem;
		
		logi("MEMDBG alloc 0x%08x..0x%08x from 0x%08x\n", mem + 2, mem + sz / 4, __builtin_return_address(0));
		
		return mem + 2;
	}
	
	void kheapFree(void* ptr)
	{
		uint32_t *mem = ((uint32_t*)ptr) - 2, sz = mem[1];
		
		logi("MEMDBG free 0x%08x from 0x%08x\n", ptr, __builtin_return_address(0));
		
		if (mem[0] == 0xfefefefe)
			fatal("Freeing chunk at 0x%08x whih has already been freed\n", ptr);
		
		if (mem[0] != (0xfeeff00f ^ (uintptr_t)mem))
			fatal("Chunk at 0x%08x had its header corrupted from 0x%08x to 0x%08x\n", ptr, 0xfeeff00f ^ (uintptr_t)mem, mem[0]);
		if ((mem[1] & 3) || mem[1] < 12)
			fatal("Chunk at 0x%08x has bad size %u\n", ptr, mem[1]);
		if (mem[sz / 4 - 1] != (0xcafeface ^ (uintptr_t)mem))
			fatal("Chunk at 0x%08x had its trailer corrupted from 0x%08x to 0x%08x\n", ptr, 0xcafeface ^ (uintptr_t)mem, mem[sz / 4 - 1]);
		
		mem[0] = 0xfefefefe;
		
		memFree(mem);
	}

#else

	void* kheapAllocEx(uint32_t sz, uint32_t intendedUse)
	{
		return memAlloc(sz, intendedUse, __builtin_return_address(0));
	}
	
	void kheapFree(void* ptr)
	{
		memFree(ptr);
	}

#endif
