#include "serial.h"
#include <SerialMgr.h>
#include <PalmOS.h>


#pragma GCC optimize ("Os")			//no need to waste space

#define SER_RX_BUF_SZ				16384


static void serialPrvWakeupHandler(UInt32 refCon)
{
	struct Globals *g = (struct Globals*)refCon;
	
	g->serRxPending = true;
	EvtWakeup();
}

void msioSerPeriodic(struct Globals *g, UInt16 maxBytesAcceptable)
{
	if (g->serRxPending) {
		
		UInt32 statisField, len;
		UInt16 lineErrors;
		UInt8 *buf;
		Err e;
		
		do {
			(void)SrmGetStatus (g->serPort, &statisField, &lineErrors);
			
			e = SrmReceiveWindowOpen(g->serPort, &buf, &len);
			if (e != errNone) {
				
				if (len > maxBytesAcceptable)
					len = maxBytesAcceptable;
				
				msioSerRxedData(g, buf, len, lineErrors);
				(void)SrmReceiveWindowClose(g->serPort, len);
			}
			else {
				
				len = 0;
				msioSerRxedData(g, NULL, 0, lineErrors);
			}
		} while (len);
		
		g->serRxPending = false;
		SrmPrimeWakeupHandler(g->serPort, 1);
	}
}

void msioSerHandleOpReq(struct Globals *g, const struct MsioPktSerOp *op, struct MsioPktSerOp *reply)
{
	UInt16 len = 0, lineErrs;
	void *ptr = NULL;
	UInt32 t;
	
	reply->op = SerOpSuccess;
	switch (op->op) {
		
		default:
			reply->op = SerOpFailure;
			break;
		
		case SerOpOpen:
			if (SrmOpen(op->data.open.port, op->data.open.baud, &g->serPort) != errNone)
				reply->op = SerOpFailure;
			else {
				
				g->serRxBuf = MemPtrNew(SER_RX_BUF_SZ + 32);
				if (!g->serRxBuf) {
					
					SrmClose(g->serPort);
					reply->op = SerOpFailure;
				}
				else {
					SrmSetReceiveBuffer(g->serPort, g->serRxBuf, SER_RX_BUF_SZ); 
					
					SrmSetWakeupHandler(g->serPort, &serialPrvWakeupHandler, (UInt32)g);
					SrmPrimeWakeupHandler(g->serPort, 1);
				}
			}
			break;
		
		case SerOpClose:
			SrmSetReceiveBuffer(g->serPort, NULL, 0); 
			MemPtrFree(g->serRxBuf);
			
			if (errNone != SrmClose(g->serPort))
				reply->op = SerOpFailure;
			break;
		
		case SerOpSetSettings:
			len = sizeof(op->data.settings.baud);
			if (errNone != SrmControl(g->serPort, srmCtlSetBaudRate, (void*)&op->data.settings.baud, &len))
				reply->op = SerOpFailure;
			len = sizeof(op->data.settings.ctsTimeout);
			if (errNone != SrmControl(g->serPort, srmCtlSetCtsTimeout, (void*)&op->data.settings.ctsTimeout, &len))
				reply->op = SerOpFailure;
			len = sizeof(t);
			t = op->data.settings.flags &~ (SER_FLAG_IR_MODE | SER_FLAG_IR_RX_ON);
			if (errNone != SrmControl(g->serPort, srmCtlSetFlags, &t, &len))
				reply->op = SerOpFailure;
			
			if (op->data.settings.flags & SER_FLAG_IR_MODE) {
				
				if (errNone != SrmControl(g->serPort, srmCtlIrDAEnable, NULL, NULL))
					reply->op = SerOpFailure;
				if (errNone != SrmControl(g->serPort, (op->data.settings.flags & SER_FLAG_IR_RX_ON) ? srmCtlRxEnable : srmCtlRxDisable, NULL, NULL))
					reply->op = SerOpFailure;
			}
			else {
				if (errNone != SrmControl(g->serPort, srmCtlIrDAEnable, NULL, NULL))
					reply->op = SerOpFailure;
			}
			break;
		
		case SerOpGetSettings:
			reply->op = SerOpCurSettings;
			len = sizeof(reply->data.settings.baud);
			if (errNone != SrmControl(g->serPort, srmCtlGetBaudRate, &reply->data.settings.baud, &len))
				reply->op = SerOpFailure;
			len = sizeof(reply->data.settings.ctsTimeout);
			if (errNone != SrmControl(g->serPort, srmCtlGetCtsTimeout, &reply->data.settings.ctsTimeout, &len))
				reply->op = SerOpFailure;
			len = sizeof(t);
			if (errNone != SrmControl(g->serPort, srmCtlGetFlags, &t, &len))
				reply->op = SerOpFailure;
			reply->data.settings.flags = t;
			
			break;
		
		case SerOpGetState:
			reply->op = SerOpCurState;
			if (errNone != SrmGetStatus(g->serPort, &t, &lineErrs))
				reply->op = SerOpFailure;
			else {
				
				reply->data.state.errors = lineErrs;
				reply->data.state.ctsOn = !!(t & srmStatusCtsOn);
				reply->data.state.dsrOn = !!(t & srmStatusDsrOn);
			}
			break;
		
		case SerOpControl:
			reply->data.control.err = errNone;
			switch (op->data.control.selector) {
				case srmCtlSetBaudRate:				//4byte value in/out
				case srmCtlGetBaudRate:
				case srmCtlSetFlags:
				case srmCtlGetFlags:
				case srmCtlSetCtsTimeout:
				case srmCtlGetCtsTimeout:
				case srmCtlGetOptimalTransmitSize:
					reply->data.control.val = op->data.control.val;
					ptr = &reply->data.control.val;
					len = 4;
					break;
				
				case srmCtlStartBreak:				//NULL, 0
				case srmCtlStopBreak:
				case srmCtlStartLocalLoopback:
				case srmCtlStopLocalLoopback:
				case srmCtlIrDAEnable:
				case srmCtlIrDADisable:
				case srmCtlRxEnable:
				case srmCtlRxDisable:
					ptr = NULL;
					len = 0;
					break;
				
				case srmCtlSetDTRAsserted:
				case srmCtlGetDTRAsserted:
					reply->data.control.val = !!op->data.control.val;
					ptr = ((char*)&reply->data.control.val) + 3;	//we are BE
					len = 1;
					break;
				
				default:
					reply->data.control.err  = sysErrParamErr;
					break;
			}
			if (reply->data.control.err != errNone)
				reply->data.control.err = SrmControl(g->serPort, op->data.control.selector, ptr, &len);
			reply->op = SerOpControlReply;
			break;
		
		case SerOpSendWait:
			if (errNone != SrmSendWait(g->serPort))
				reply->op = SerOpFailure;
			break;
	}
}

void msioSerHandleDataReq(struct Globals *g, const struct MsioSerialData *sd, struct MsioSerialDataTxReply *reply)		//asked to send data
{
	reply->hdr.pktTyp = MSIO_PKT_SER_TX_REPLY;
	reply->numSent = SrmSend(g->serPort, sd->data, (sd->bytesAndFlags & MSG_BYTES_AND_FLAGS_BYTES_MASK) >> MSG_BYTES_AND_FLAGS_BYTES_SHIFT, &reply->err);
}



