#include <PalmOS.h>
#include <SonyCLIE.h>
#include "driverless.h"
#include "deviceID.h"
#include "SonyMSIO.h"
#include "asm.h"

#pragma GCC optimize ("Os")			//no need to waste space


#define MS_GLOBALS_OS3	(((struct MsGlobals***)0x304)[0][0])
#define MS_GLOBALS_OS4	(((struct MsGlobals***)0x304)[0][9])


typedef void (*IrqHandler)(void);
typedef UInt8 (*FuncTpcR)(struct Mb86189 *chip, UInt8 tpc, Int16 nBytes, UInt16 *valOutP, UInt8 *staOutP);
typedef UInt8 (*FuncTpcW)(struct Mb86189 *chip, UInt8 tpc, Int16 nBytes, const UInt16 *valOutP, UInt8 *staOutP);

struct MsGlobals {
	UInt32 unk[2];
	IrqHandler insertInterrupt;
	IrqHandler msioInterrupt;
	UInt32 powerTimer;
};

struct DriverData {
	struct Mb86189 *chip;
	void (*irqHandler)(void* userata);
	void *irqHandlerData;
	
	void *prevIrqHandler;
	
	FuncTpcR fTpcR;
	FuncTpcW fTpcW;
	
	//code to serve as our low level irq handler
	UInt16 irqhCode[9];
	
	UInt32 regsRange;
	UInt8 curFunc;
	Boolean irqEnabled;
};


#define PDSEL		(*(volatile UInt8*) 0xfffff41b)
#define ISRH		(*(volatile UInt16*)0xfffff30c)
#define IMRH		(*(volatile UInt16*)0xfffff304)

static inline struct Mb86189* mb86189getBaseSony(void)
{
	UInt32 csSize = (0x00020000UL << (((*(volatile UInt16*)0xFFFFF110) >> 1) & 7));
	UInt32 csBase = ((UInt32)(*(volatile UInt16*)0xFFFFF100)) << 13;
	
	return (struct Mb86189*)(csSize + csBase);
}

static inline struct Mb86189* mb86189getBaseAcer(void)
{
	return (struct Mb86189*)0x10C00000;
}

static inline void msPrvIrqAck(void)
{
	ISRH = 4;		//clear state
}

static inline void msPrvIrqOn(void)
{
	IMRH &=~ 4;	//on
}

static inline void msPrvIrqOff(void)
{
	IMRH |= 4;	//off
}

static inline Boolean msPrvIrqIsOn(void)
{
	return !(IMRH & 4);
}

static inline void msPrvIrqCheck(struct DriverData *dd)
{
	if ((dd->chip->MSICSL & 0x40) && dd->irqEnabled)
		dd->irqHandler(dd->irqHandlerData);
}

static void msPrvLowLevelIrqh(struct DriverData *dd)
{
	msPrvIrqAck();
	msPrvIrqCheck(dd);
}

static Err msioPrvTpcR(struct DriverData *dd, UInt8 tpc, Int16 nBytes, UInt16 *valOutP)
{
	UInt8 ret, sta;
	Boolean hadIrq;
	
	hadIrq = msPrvIrqIsOn();
	if (hadIrq) {
		
		msPrvIrqAck();
		msPrvIrqOff();
	}
	
	ret = dd->fTpcR(dd->chip, tpc, nBytes, valOutP, &sta);
	
	if (hadIrq) {
		msPrvIrqOn();
		msPrvIrqCheck(dd);
	}
	
	return (!ret && (sta & 0x80)) ? errNone : 0x8000 + sta;
}

static Err msioPrvTpcW(struct DriverData *dd, UInt8 tpc, Int16 nBytes, const UInt16 *valInP)
{
	UInt8 ret, sta;
	Boolean hadIrq;
	
	hadIrq = msPrvIrqIsOn();
	if (hadIrq) {
		
		msPrvIrqAck();
		msPrvIrqOff();
	}
	
	ret = dd->fTpcW(dd->chip, tpc, nBytes, valInP, &sta);
	
	if (hadIrq) {
		msPrvIrqOn();
		msPrvIrqCheck(dd);
	}
	
	return (!ret && (sta & 0x80)) ? errNone : 0x8000 + sta;
}

static Err msioPrvSelectRegsRange(struct DriverData *dd, UInt8 rStart, UInt8 rLen, UInt8 wStart, UInt8 wLen)
{
	Err e;
	union {
		struct {
			UInt8 rStart, rLen, wStart, wLen;
		};
		UInt32 val32;
	} rr = {.rStart = rStart, .rLen = rLen, .wStart = wStart, .wLen = wLen, };
	
	if (rr.val32 == dd->regsRange)
		return errNone;
	
	e = msioPrvTpcW(dd, MS_SET_RW_REG_ADRS, 4, (const UInt16*)&rr.val32);
	if (e != errNone)
		return e;
	
	dd->regsRange = rr.val32;
	return errNone;
}

static Err msioPrvSelectFunc(struct DriverData *dd, UInt8 func)
{
	UInt16 func16 = ((UInt16)func) << 8;
	Err e;
	
	if (dd->curFunc == func)
		return errNone;
	
	msPrvIrqOff();
	msPrvIrqAck();
	
	e = msioPrvSelectRegsRange(dd, 6, 1, 6, 1);
	if (e != errNone)
		return e;
	
	e = msioPrvTpcW(dd, MS_WR_REG, 1, (const UInt16*)&func16);
	if (e != errNone)
		return e;
	
	e = msioPrvTpcR(dd, MS_RD_REG, 1, (UInt16*)&func16);
	if (e != errNone)
		return e;

	if ((func16 >> 8) == func) {
		dd->curFunc = func;
		return errNone;
	}
	
	return 2;
}

static Err driverless_CustomControl(void *driverData, UInt16 selector, void *apiData)
{
	struct DriverData *dd = (struct DriverData*)driverData;
	Err e;
	
	switch (selector) {
		case SONY_MSIO_OP_API_VERSION: {
			struct SonyMsioApiVersion *d = (struct SonyMsioApiVersion*)apiData;
			
			d->ver = SONY_MSIO_API_VER_1;
			return errNone;
		}
		case SONY_MSIO_OP_FUNCTION_ENABLE: {
			struct SonyMsioFunctionEnable *d = (struct SonyMsioFunctionEnable*)apiData;
			
			if (dd->curFunc == d->functionID)
				return errNone;
			
			msPrvIrqOff();
			msPrvIrqAck();
			
			dd->irqHandler = d->irqHandler;
			dd->irqHandlerData = d->irqHandlerData;
			
			return msioPrvSelectFunc(dd, d->functionID);
		}
		case SONY_MSIO_OP_FUNCTION_DISABLE: {
			struct SonyMsioFunctionDisable *d = (struct SonyMsioFunctionDisable*)apiData;
			
			if (dd->curFunc != d->functionID)
				return errNone;
			
			return msioPrvSelectFunc(dd, 0xff);
		}
		case SONY_MSIO_OP_INT_CONTROL: {
			struct SonyMsioIntControl *d = (struct SonyMsioIntControl*)apiData;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			if (d->irqOn) {
				
				dd->irqEnabled = true;
				msPrvIrqAck();
				msPrvIrqOn();
			}
			else {
				
				dd->irqEnabled = false;
				msPrvIrqOff();
				msPrvIrqAck();
			}
			return errNone;
		}
		case SONY_MSIO_OP_CHECK_STICK_TYPE: {
			struct SonyMsioCheckStickType *d = (struct SonyMsioCheckStickType*)apiData;
			UInt8 i;
			
			*d->isMsioStickOutP = false;
			
			for (i = 1; i < 0xff; i++) {
				
				if (errNone != msioPrvSelectFunc(dd, i))
					continue;
				
				*d->isMsioStickOutP = true;
				break;
			}
			
			(void)msioPrvSelectFunc(dd, 0xff);
			return errNone;
		}
		case SONY_MSIO_OP_IO_GET_INT: {
			struct SonyMsioIoGetInt *d = (struct SonyMsioIoGetInt*)apiData;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			return msioPrvTpcR(dd, MS_GET_INT, d->numBytesToRead, (UInt16*)d->intRegValOutP);
		}
		case SONY_MSIO_OP_IO_READ_DATA: {
			struct SonyMsioIoReadData *d = (struct SonyMsioIoReadData*)apiData;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			return msioPrvTpcR(dd, MS_RD_LDATA, d->dataLen, (UInt16*)d->dataOutP);
		}
		case SONY_MSIO_OP_IO_WRITE_DATA: {
			struct SonyMsioIoWriteData *d = (struct SonyMsioIoWriteData*)apiData;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			return msioPrvTpcW(dd, MS_WR_LDATA, d->dataLen, (const UInt16*)d->dataInP);
		}
		case SONY_MSIO_OP_IO_SET_CMD: {
			struct SonyMsioIoSetCmd *d = (struct SonyMsioIoSetCmd*)apiData;
			UInt16 cmd = ((UInt16)d->cmd) << 8;		//must be 16 bit aligned;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			return msioPrvTpcW(dd, MS_SET_CMD, 1, &cmd);
		}
		case SONY_MSIO_OP_IO_TPC_W: {
			struct SonyMsioIoTpcW *d = (struct SonyMsioIoTpcW*)apiData;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			return msioPrvTpcW(dd, d->tpc, d->dataLen, (const UInt16*)d->dataToSendP);
		}
		case SONY_MSIO_OP_IO_TPC_R: {
			struct SonyMsioIoTpcR *d = (struct SonyMsioIoTpcR*)apiData;
			
			e = msioPrvSelectFunc(dd, d->functionID);
			if (e != errNone)
				return e;
			
			return msioPrvTpcR(dd, d->tpc, d->dataLen, (UInt16*)d->dataOutP);
		}
		default:
			SysFatalAlert("Unknown ioctl");
			return sysErrParamErr;
	}
}

static Err driverless_CommonEnableControl(void **driverDataP, Boolean on, struct Mb86189 *chip, UInt8 ppcVal, UInt8 ppdVal, FuncTpcR fTpcR, FuncTpcW fTpcW, void **irqHandlerP)
{
	struct DriverData *dd;

	if (on) {
		
		UInt16 maxWait = 0;
				
		dd = MemPtrNew(sizeof(*dd));
		MemPtrSetOwner(dd, 0);
		MemSet(dd, sizeof(*dd), 0);
		
		//set pointers
		dd->fTpcR = fTpcR;
		dd->fTpcW = fTpcW;
		
		//init chip
		chip->MSPPC = 0;
		chip->MSPPD &=~ 0x0E;
		chip->MSPPC = ppcVal;
		
		
		//reset chip
		chip->MSCSH &=~ 0x40;
		chip->MSICSH |= 0x80;
		chip->MSCSH |= 0x80;
		chip->MSCSH &=~ 0x80;
		
		//wait
		while (!(chip->MSCSL & 0x80) && ++maxWait);
		(void)chip->MSICSL;
		SysTaskDelay(1);
		
		//card power
		chip->MSPPD |= ppdVal;
		chip->MSCSH |= 0x20;
		SysTaskDelay(2);
		
		msPrvIrqOff();
		msPrvIrqAck();
		
		dd->irqhCode[0] = 0x2f3c;							//move.l    dd, -(ap)
		*(void**)(dd->irqhCode + 1) = dd;
		dd->irqhCode[3] = 0x207c;							//movea.l   msPrvLowLevelIrqh, a0
		*(void**)(dd->irqhCode + 4) = msPrvLowLevelIrqh;
		dd->irqhCode[6] = 0x4e90;							//jsr       (a0)
		dd->irqhCode[7] = 0x584f;							//addq.w    #4,a7
		dd->irqhCode[8] = 0x4e75;							//rts
		
		dd->chip = chip;
		dd->prevIrqHandler = *irqHandlerP;
		*irqHandlerP = (void*)dd->irqhCode;
				
		dd->regsRange = 0xffffffff;
		dd->curFunc = 0xff;
		dd->irqEnabled = false;
		PDSEL &=~ 0x40;
		
		*driverDataP = dd;
	}
	else {
		dd = (struct DriverData*)*driverDataP;
		PDSEL |= 0x40;
		
		msPrvIrqOff();
		msPrvIrqAck();
		*irqHandlerP = dd->prevIrqHandler;
		
		chip->MSCSH &=~ 0x20;
		chip->MSPPD &=~ 0x02;
		 
		MemPtrFree(dd);
		*driverDataP = NULL;
	}
	
	return errNone;
}

static Err driverless_s300_500c_EnableControl(void **driverDataP, Boolean on)
{
	return driverless_CommonEnableControl(driverDataP, on, mb86189getBaseSony(), 0x0E, 0x02, msioTpcR_20MHz, msioTpcW_20MHz, (void**)&MS_GLOBALS_OS3->msioInterrupt);
}

static Err driverless_s320_s360_EnableControl(void **driverDataP, Boolean on)
{
	return driverless_CommonEnableControl(driverDataP, on, mb86189getBaseSony(), 0x0E, 0x02, msioTpcR_33MHz, msioTpcW_33MHz, &((void***)0x122)[0][595]);
}

static Err driverless_s10_EnableControl(void **driverDataP, Boolean on)
{
	return driverless_CommonEnableControl(driverDataP, on, mb86189getBaseAcer(), 0xFF, 0x04, msioTpcR_33MHz, msioTpcW_33MHz, &((void***)0x122)[0][595]);
}



DRIVERLESS_DEFINE(sonyHwrOEMDeviceIDPda1Color, driverless_s300_500c_EnableControl, driverless_CustomControl);
DRIVERLESS_DEFINE(sonyHwrOEMDeviceIDPda1Mono, driverless_s300_500c_EnableControl, driverless_CustomControl);
DRIVERLESS_DEFINE(sonyHwrOEMDeviceIDNasca, driverless_s320_s360_EnableControl, driverless_CustomControl);
DRIVERLESS_DEFINE(sonyHwrOEMDeviceIDNasca2, driverless_s320_s360_EnableControl, driverless_CustomControl);
DRIVERLESS_DEFINE(acerHwrOEMDeviceIDMP300, driverless_s10_EnableControl, driverless_CustomControl);