#ifndef _CORTEX_M_EMIT_H_
#define _CORTEX_M_EMIT_H_

#include <stdbool.h>
#include <stdint.h>
#include "printf.h"
#include "mpu.h"

#define EMIT_REG_NO_SP		13
#define EMIT_REG_NO_LR		14
#define EMIT_REG_NO_PC		15


#define EMIT_IS_LOREG(regNo)	((regNo) < 8)


enum EmitCc {
	EmitCcEq = 0,
	EmitCcNe = 1,
	EmitCcCs = 2,
	EmitCcCc = 3,
	EmitCcMi = 4,
	EmitCcPl = 5,
	EmitCcVs = 6,
	EmitCcVc = 7,
	EmitCcHi = 8,
	EmitCcLs = 9,
	EmitCcGe = 10,
	EmitCcLt = 11,
	EmitCcGt = 12,
	EmitCcLe = 13,
	EmitCcAl = 14,
	EmitCcNv = 15,
};

struct EmitBuf {
	#ifdef BUILDING_FOR_BIG_ARM
		enum EmitCc curCc;
	#endif
	
	void *buf;			//current pointer for output
	void *bufStart;		//where buffer starts
	void *bufEnd;		//just past buffer end
};

struct EmitTableBranchState {
	union {
		uint8_t *ptrB;
		uint16_t *ptrH;
		uint16_t *srcPtr;
		uintptr_t srcAddr;
	};
	uint32_t numCases	: 31;
	uint32_t isHalfword	: 1;
};

enum EmitStatus {
	EmitErrNone = 0,
	EmitErrNoSpace,
	EmitErrNotEncodeable,
	EmitErrInvalidInput,		//only for invalid input valus (not for invalid requests of the backend)
	EmitErrInternalErr,			//some other internal error
};

enum EmitFlagSettingPrefs {
	EmitLeaveFlags,
	EmitSetFlags,
	EmitFlagsDoNotCare,
};

enum EmitShiftType {
	EmitShiftLsl,
	EmitShiftLsr,
	EmitShiftAsr,
	EmitShiftRor,
};

enum EmitMemOpSz {
	EmitSzByte,
	EmitSzHalfword,
	EmitSzWord,
};

enum EmitAddrMode {
	EmitAdrModeIndex,		//[Rx, #y]	OR	[Rx, Ry]
	EmitAdrModePostindex,	//[Rx], #y	OR	[Rx], Ry
	EmitAdrModeIndexWbak,	//[Rx, #y]!	OR	[Rx, Ry]!
};

#define EMIT_SYSM_APSR					0
#define EMIT_MSR_APSR_MASK_NZCVQ		0b10


//convenience (no neither of these can be expressed in terms of the other!)
#define EMIT_TO(nm, _dst, ...)	do { enum EmitStatus sta = emit ## nm(_dst, __VA_ARGS__); if (sta != EmitErrNone) return sta; } while (0)
#define EMIT(nm, ...)			do { enum EmitStatus sta = emit ## nm(dest, __VA_ARGS__); if (sta != EmitErrNone) return sta; } while (0)


static inline enum EmitCc emitCcInvert(enum EmitCc cc)
{
	switch (cc) {
		case EmitCcEq:
		case EmitCcNe:
		case EmitCcCs:
		case EmitCcCc:
		case EmitCcMi:
		case EmitCcPl:
		case EmitCcVs:
		case EmitCcVc:
		case EmitCcHi:
		case EmitCcLs:
		case EmitCcGe:
		case EmitCcLt:
		case EmitCcGt:
		case EmitCcLe:
			return (enum EmitCc)(((uint32_t)cc) ^ 1);
		
		default:
			fatal("CC %u is not invertible\n", (unsigned)cc);
			return EmitCcNv;
	}
}

//buffer management
void emitBufferInit(struct EmitBuf *dest, void *buf, uint32_t len);
enum EmitStatus emitSaveSpace(struct EmitBuf *dest, struct EmitBuf *spaceOutP, uint32_t numHalfwords);
uintptr_t emitGetPtrToJumpHere(const struct EmitBuf *dest);

void* emitBufferBackup(const struct EmitBuf *dest);	//backup is only valid with the same buffer	
void emitBufferRestore(struct EmitBuf *dest, void* backup);

void* emitBufPtrToFuncPtr(void *buf);

#ifdef PLATFORM_MUST_DEAL_WITH_CACHE_COHERENCY			//done this way to make absolute sure we do not call an empty func on the jit hot path... gcc LTO i snot the smartest...

	static inline void emitBufCacheCoherencyManage(const struct EmitBuf *dest)
	{
		mpuInstrCacheClearDataCacheClean((uintptr_t)dest->bufStart, (char*)dest->buf - (char*)dest->bufStart);
	}

#else

	static inline void emitBufCacheCoherencyManage(const struct EmitBuf *dest)
	{
		//nothing
	}
	
#endif


//all funcs with shift expect it in processed form! "ROR #0" is RRX
//all of these emit a single instr, they do not do anything clever. All invalid requests are denied
enum EmitStatus emitLLrawHalfword(struct EmitBuf *dest, uint32_t halfword);
enum EmitStatus emitLLrawWord(struct EmitBuf *dest, uint32_t word);
enum EmitStatus emitLLcps(struct EmitBuf *dest, bool i, bool f, bool enable);
enum EmitStatus emitLLit(struct EmitBuf *dest, enum EmitCc cc);
enum EmitStatus emitLLitt(struct EmitBuf *dest, enum EmitCc cc);
enum EmitStatus emitLLittt(struct EmitBuf *dest, enum EmitCc cc);
enum EmitStatus emitLLitttt(struct EmitBuf *dest, enum EmitCc cc);
enum EmitStatus emitLLite(struct EmitBuf *dest, enum EmitCc cc);
enum EmitStatus emitSetCond(struct EmitBuf *dest, enum EmitCc cc);				//like "IT" but for grown-ups
enum EmitStatus emitLLnop(struct EmitBuf *dest, bool w);
enum EmitStatus emitLLbx(struct EmitBuf *dest, uint32_t regNo);
enum EmitStatus emitLLblx(struct EmitBuf *dest, uint32_t regNo);
enum EmitStatus emitLLudf(struct EmitBuf *dest, uint32_t imm, bool w);
enum EmitStatus emitLLbranch(struct EmitBuf *dest, uintptr_t dstAddr, enum EmitCc cc);	//range is limited as per ISA. if it is in IT, you must pass condition as AL, may generate more than one instr
enum EmitStatus emitLLbl(struct EmitBuf *dest, uintptr_t dstAddr);
enum EmitStatus emitLLcbz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr);
enum EmitStatus emitLLcbnz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr);
enum EmitStatus emitLLextend(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, uint32_t rotateBy, bool byte, bool unsign);
enum EmitStatus emitLLrbit(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo);
enum EmitStatus emitLLrev(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo);
enum EmitStatus emitLLrev16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo);
enum EmitStatus emitLLrevsh(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo);
enum EmitStatus emitLLbfi(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t lsb, uint32_t width);
enum EmitStatus emitLLbfx(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t lsb, uint32_t width, bool unsign, bool mayCorruptFlags);
enum EmitStatus emitLLmov(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLmvnReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLcmpReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt);
enum EmitStatus emitLLcmnReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt);
enum EmitStatus emitLLtstReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt);
enum EmitStatus emitLLteqReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt);
enum EmitStatus emitLLaddReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLsubReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLsbcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLadcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLrsbReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLandReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLorrReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLeorReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLbicReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLmulReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);	//can sometimes set flags
enum EmitStatus emitLLmlaReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo);	//never sets flags
enum EmitStatus emitLLmlsReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo);	//never sets flags
enum EmitStatus emitLLabsJump(struct EmitBuf *dest, uintptr_t dstAddr);	//to ARM or thumb. watch that low bit!
enum EmitStatus emitLLloadSp(struct EmitBuf *dest, uint32_t val);

enum EmitStatus emitLLshiftByReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);

//these always set carry based on result so how the imm is calculated does not matter
enum EmitStatus emitLLaddImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLsubImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLrsbImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLadcImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLsbcImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLmovImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLmvnImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt);
enum EmitStatus emitLLcmpImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm);
enum EmitStatus emitLLcmnImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm);

//these care how you calculate the imm
enum EmitStatus emitLLandImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs);
enum EmitStatus emitLLeorImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs);
enum EmitStatus emitLLorrImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs);
enum EmitStatus emitLLbicImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs);
enum EmitStatus emitLLtstImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t val, uint32_t rorBy);
enum EmitStatus emitLLteqImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t val, uint32_t rorBy);

enum EmitStatus emitLLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus emitLLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus emitLLldmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus emitLLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);

enum EmitStatus emitLLstoreImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, enum EmitAddrMode adrMode);
enum EmitStatus emitLLloadImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode);

enum EmitStatus emitLLstoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz);
enum EmitStatus emitLLloadRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, bool sext);

enum EmitStatus emitLLstrex(struct EmitBuf *dest, uint32_t rdNo, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz);
enum EmitStatus emitLLldrex(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz);

enum EmitStatus emitLLldrdImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode);
enum EmitStatus emitLLstrdImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode);

enum EmitStatus emitLLlongMul(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool isUnsigned, bool accumulate);
enum EmitStatus emitLLumull(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo);	//convenience func for emitLLlongMul
enum EmitStatus emitLLumlal(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo);	//convenience func for emitLLlongMul
enum EmitStatus emitLLsmull(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo);	//convenience func for emitLLlongMul
enum EmitStatus emitLLsmlal(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo);	//convenience func for emitLLlongMul

enum EmitStatus emitLLpkhbt(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt);
enum EmitStatus emitLLpkhtb(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt);

enum EmitStatus emitLLmrs(struct EmitBuf *dest, uint32_t rdNo, uint32_t sysm);
enum EmitStatus emitLLmsr(struct EmitBuf *dest, uint32_t sysm, uint32_t mask, uint32_t rnNo);

enum EmitStatus emitLLsvc(struct EmitBuf *dest, uint32_t imm);

enum EmitStatus emitLLclz(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo);

enum EmitStatus emitLLsdiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLudiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);

enum EmitStatus emitLLssat(struct EmitBuf *dest, uint32_t rdNo, uint32_t immTo, uint32_t rnNo, enum EmitShiftType shiftType, uint32_t shiftAmt);
enum EmitStatus emitLLusat(struct EmitBuf *dest, uint32_t rdNo, uint32_t immTo, uint32_t rnNo, enum EmitShiftType shiftType, uint32_t shiftAmt);

//table branch management
enum EmitStatus emitLLtableBranch(struct EmitBuf *dest, struct EmitTableBranchState *state, uint32_t rmNo, bool halfwordSized, uint32_t numCases);
enum EmitStatus emitLLtableBranchSetCase(struct EmitTableBranchState *state, uint32_t caseIdx, uintptr_t dstAddr);

//DSP (v7E instrs)
enum EmitStatus emitLLqadd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqsub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqdadd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqdsub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);

enum EmitStatus emitLLsmulxy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop);
enum EmitStatus emitLLsmlaxy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool nTop, bool mTop);
enum EmitStatus emitLLsmulwy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool mTop);
enum EmitStatus emitLLsmlawy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool mTop);
enum EmitStatus emitLLsmlalxy(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop);

enum EmitStatus emitLLuhadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuhadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuhsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuhsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLshadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLshadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLshsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLshsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuqadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuqadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuqsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuqsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLqsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLsadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLsadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLssub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLssub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLuadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLusub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus emitLLusub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);


//VFP instrs
enum EmitStatus emitLLvldmiaSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak);
enum EmitStatus emitLLvstmiaSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak);
enum EmitStatus emitLLvldmdbSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak);
enum EmitStatus emitLLvstmdbSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak);
enum EmitStatus emitLLvmovArmToVfpSP(struct EmitBuf *dest, uint32_t dstReg, uint32_t srcReg);
enum EmitStatus emitLLvmovVfpToArmSP(struct EmitBuf *dest, uint32_t dstReg, uint32_t srcReg);
enum EmitStatus emitLLvmovArmToVfp2xSP(struct EmitBuf *dest, uint32_t dstReg, uint32_t srcReg1, uint32_t srcReg2);	//dst is contuguous regs
enum EmitStatus emitLLvmovVfpToArm2xSP(struct EmitBuf *dest, uint32_t dstReg1, uint32_t dstReg2, uint32_t srcReg);	//src is contuguous regs
enum EmitStatus emitLLvcvtToFixedPtSP(struct EmitBuf *dest, uint32_t reg, uint32_t size, uint32_t fracBits, bool unsign);
enum EmitStatus emitLLvcvtFromFixedPtSP(struct EmitBuf *dest, uint32_t reg, uint32_t size, uint32_t fracBits, bool unsign);

//you almost certainly do not want to use this and instead want to use jitEmitLoadImmToReg()
enum EmitStatus emitHLloadImmToReg(struct EmitBuf *dest, uint32_t regNo, uint32_t val, bool canCorruptNZ, bool canCorruptC, bool isInIt);	//GUARANTEED TO NEVER CORRUPT V flag!!!
enum EmitStatus emitHLjump(struct EmitBuf *dest, uintptr_t dstAddr);

//these allow an empty set and just do nothing
enum EmitStatus emitHLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus emitHLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus emitHLldmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus emitHLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak);

enum EmitStatus emitHLpush(struct EmitBuf *dest, uint32_t regsMask);
enum EmitStatus emitHLpop(struct EmitBuf *dest, uint32_t regsMask);


#endif
