#ifndef _AUDIO_H_
#define _AUDIO_H_

#include <stdbool.h>
#include <stdint.h>
#include "emit.h"
#include "kal.h"


struct AudioOutputStrmHandle;

enum AudioSampleRate {	//these are compared numerically, and so should be in ascending order
	AudioRate8000,
	AudioRate11025,
	AudioRate16000,
	AudioRate22050,
	AudioRate24000,
	AudioRate32000,
	AudioRate44100,
	AudioRate48000,
	
	AudioRateNumRates,
};

enum AudioSampleType {
	AudioSampleU8,
	AudioSampleS8,
	AudioSampleU16LE,
	AudioSampleU16BE,
	AudioSampleS16LE,
	AudioSampleS16BE,
	AudioSampleU32LE,
	AudioSampleU32BE,
	AudioSampleS32LE,
	AudioSampleS32BE,
	AudioSampleFloatLE,
	AudioSampleFloatBE,
};

enum AudioChannelConfig {
	AudioMono,
	AudioStereo,
};

#define AUDIO_VOLUME_UNITY					1024

//common
bool audioOutInit(enum AudioSampleRate *nativeRateP, bool *nativeStereoP);
bool audioInInit(void);		// not *having* a MIC is not an error!


//input (provided buffers must be 4-byte aligned). number of samples will be a multiple of 8
bool audioMicSetup(enum AudioSampleRate rate, uint32_t samplesPerBuf, enum AudioSampleType sampTyp, enum AudioChannelConfig chCfg);
uint32_t audioMicGetSamples(void* samples, uint32_t size, uint32_t volL, uint32_t volR);			//return bytes given
void audioMicStop(void);

//output
struct AudioOutputStrmHandle* audiostreamCreateOutput(enum AudioSampleRate rate, enum AudioSampleType sampTyp, enum AudioChannelConfig chans, uint32_t samplesPerBuf, bool loopMode);
void audiostreamDestroyOutput(struct AudioOutputStrmHandle* strm);
void audiostreamSetVolumesOutput(struct AudioOutputStrmHandle* strm, uint32_t volumeL, uint32_t volumeR);	//0 ..64k. 1024 is unity gain
uint32_t audiostreamAddSamplesOutput(struct AudioOutputStrmHandle* strm, const void* samples, uint32_t size);	//return bytes taken
//sony-only
bool audiostreamOutputGetSonyStats(struct AudioOutputStrmHandle* strm, uint32_t *numOutSamplesProducedP, uint32_t *lastMixedTicksP);



//simple-only output (always called under a mutex, never re-entrant)
bool audioOnlySimpleOutInit(void);						//will not be attempted if sampled audio inits
void audioOnlySimpleTone(uint32_t freq, uint32_t amp);	//zero amplitude for "off". amplitude goes up to sndMaxAmp, inclusive. freq is in hertz

//internal (helpful info)
uint32_t audiostreamOutputGetOptimalBufSz(struct AudioOutputStrmHandle* strm);	//in bytes
uint32_t audioGetNumOutStreams(void);

//internal (for loop buffers)
void* audioOutputLoopStreamGetBuffer(struct AudioOutputStrmHandle* strm);
void audioOutputLoopStreamEnable(struct AudioOutputStrmHandle* strm);

//very internal (for audio backends)

#define MIXER_BE_RECKLESS_WITH_MIXING		0		//if set we'll just add samples ni the intermediate buffer. this allows 4+ max-volume streams to signed overflow. this is unlikely, but possible. setting this to zero avoid this by saturating in addition
#define MIXER_ALLOW_VFP_USE					0


static const uint8_t mSampleShifts[] = {
	[AudioSampleU8] = __builtin_ctz(sizeof(uint8_t)),
	[AudioSampleS8] = __builtin_ctz(sizeof(int8_t)),
	[AudioSampleU16LE] = __builtin_ctz(sizeof(uint16_t)),
	[AudioSampleU16BE] = __builtin_ctz(sizeof(uint16_t)),
	[AudioSampleS16LE] = __builtin_ctz(sizeof(int16_t)),
	[AudioSampleS16BE] = __builtin_ctz(sizeof(int16_t)),
	[AudioSampleU32LE] = __builtin_ctz(sizeof(uint32_t)),
	[AudioSampleU32BE] = __builtin_ctz(sizeof(uint32_t)),
	[AudioSampleS32LE] = __builtin_ctz(sizeof(int32_t)),
	[AudioSampleS32BE] = __builtin_ctz(sizeof(int32_t)),
	[AudioSampleFloatLE] = __builtin_ctz(sizeof(float)),
	[AudioSampleFloatBE] = __builtin_ctz(sizeof(float)),
};

enum EmitStatus audioPrvStreamCreateOutputMixFuncGutsNoResamp(struct EmitBuf *dest, enum AudioSampleType sampTyp, enum AudioChannelConfig chans, bool nativeFmtIsStereo);
enum EmitStatus audioPrvStreamCreateOutputMixFuncGutsUpsample(struct EmitBuf *dest, const uint16_t* resampTab, enum AudioSampleType sampTyp, enum AudioChannelConfig chans, bool nativeFmtIsStereo);
enum EmitStatus audioPrvStreamCreateOutputMixFuncGutsDownsample(struct EmitBuf *dest, const uint16_t* resampTab, enum AudioSampleType sampTyp, enum AudioChannelConfig chans, bool nativeFmtIsStereo);


//mic writer. accepts only groups of 2 samples. shouldnt be a problem
enum EmitStatus audioPrvMicCreateConvertFunc(struct EmitBuf *dest, enum AudioSampleType sampTyp, enum AudioChannelConfig chans);


#endif
