#include "util.h"


#define TITLE_OBJ_SURROGATE_ID_VAL			0xffff
#define TITLE_OBJ_SURROGATE_FAKE_WIN_VAL	((WinHandle)0x0f28c496)

#define SET_APPROPRIATE_TEXT_COLOR			1
#define SET_APPROPRIATE_FORE_COLOR			2
#define SET_APPROPRIATE_BACK_COLOR			3



static void PrvSetAppropriateColors(struct PalmForm *frm, uint32_t flags)
{
	UIColorTableEntries clrText, clrFore, clrBack;
	
	if (frm->formID == 12000) {
		clrText = UIAlertFrame;
		clrFore = UIAlertFrame;
		clrBack = UIAlertFill;
	}
	else if (frm->win.flags.modal) {
		clrText = UIDialogFrame;
		clrFore = UIDialogFrame;
		clrBack = UIDialogFill;
	}
	else {
		clrText = UIFormFrame;
		clrFore = UIFormFrame;
		clrBack = UIFormFill;
	}
	
	if (flags & SET_APPROPRIATE_TEXT_COLOR)
		WinSetTextColor(UIColorGetTableEntryIndex(clrText));
	
	if (flags & SET_APPROPRIATE_FORE_COLOR)
		WinSetForeColor(UIColorGetTableEntryIndex(clrFore));
	
	if (flags & SET_APPROPRIATE_BACK_COLOR)
		WinSetBackColor(UIColorGetTableEntryIndex(clrBack));
}

static void PrvDrawHelpIcon(Coord posRight, bool active)
{
	struct RectangleType rect = {.topLeft = {.x = posRight - 12, .y = 0}, .extent = {.x = 10, .y = 10}, };
	char helpIcon = 10;	//help icon in symbol font
	Coord pos;
	
	FntSetFont(symbolFont);
	pos = rect.topLeft.x + (10 - FntCharWidth(helpIcon)) / 2;
	
	if (active) {
		WinDrawRectangle(&rect, 5);
		WinDrawInvertedChars(&helpIcon, 1, pos, 0);
	}
	else {
		WinEraseRectangle(&rect, 5);
		WinDrawChars(&helpIcon, 1, pos, 0);
	}
}

static void PrvDrawTitle(struct PalmForm *frm, struct PalmTitleObj *obj)
{
	uint32_t dstDensity, titleLen = 0, titleWidth, prevCoordSys, screenDensity, titleAreaWidth = 0;
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	Coord winExtentX, winExtentY;
	const char *titleStr = NULL;
	struct RectangleType rect;
	
	AttnIndicatorAllow(false);
	WinPushDrawState();
	prevCoordSys = WinSetCoordinateSystem(kDensityLow);
	FntSetFont(boldFont);
	PrvSetAppropriateColors(frm, SET_APPROPRIATE_TEXT_COLOR | SET_APPROPRIATE_FORE_COLOR | SET_APPROPRIATE_BACK_COLOR);
	
	if (obj && obj->text)
		titleLen = StrLen(titleStr = obj->text);
	
	titleWidth = FntCharsWidth(titleStr, titleLen);
	dstDensity = BmpGetDensity(WinGetBitmap(WinGetDrawWindow()));
	screenDensity = BmpGetDensity(WinGetBitmap(WinGetDisplayWindow()));
	WinGetWindowExtent(&winExtentX, &winExtentY);
	
	if (frm->win.flags.modal) {
		
		rect.topLeft.x = 0;
		rect.extent.x = winExtentX;
		rect.extent.y = FntLineHeight();
		
		switch (dstDensity) {
			case kDensityOneAndAHalf:
			case kDensityTriple:
				rect.topLeft.y = 0;
				break;
			default:
				rect.topLeft.y = 1;
				break;
		}
		
		WinDrawLine(0, 0, winExtentX, 0);
		WinDrawRectangle(&rect, 0);
		WinDrawInvertedChars(titleStr, titleLen, (winExtentX - titleWidth + 1) / 2, 0);
		if (frm->helpRscId)
			PrvDrawHelpIcon(winExtentX, false);
	}
	else if (obj->rect.extent.x == winExtentX) {
		
		rect.topLeft.x = 0;
		rect.topLeft.y = 0;
		rect.extent.x = winExtentX;
		rect.extent.y = FntLineHeight() + 2;
		WinDrawRectangle(&rect, 3);
		
		WinDrawInvertedChars(titleStr, titleLen, (winExtentX - titleWidth + 1) / 2, 1);
	}
	else {
		
		struct RectangleType nativeWinClip, tmp;
		
		rect.topLeft.x = 0;
		rect.topLeft.y = 0;
		rect.extent.x = titleAreaWidth = titleWidth + 5;
		rect.extent.y = 15;


		//draw the left curved part. curvature could show up over the line, so we clip and then extend the rectangle down
		WinSetCoordinateSystem(kCoordinatesNative);
		scaleRect(&rect, w->drawState->scaling.preGarnet.standardToActive);
		WinGetClip(&nativeWinClip);
		rect.extent.y -= 1;
		
		//we want to set clip to rect but we shoudl also respect eixsting clip
		//if we do not, we'll draw the title tab an dline but no text (since
		// we draw that after resetting clip back)
		RctGetIntersection(&nativeWinClip, &rect, &tmp);

		WinSetClip(&tmp);
		rect.extent.y += 10;
		WinDrawRectangle(&rect, (4248 * BmpGetDensity(WinGetBitmap(WinGetDisplayWindow()))) >> 16);
		WinSetClip(&nativeWinClip);
		
		//draw the line
		rect.topLeft.x = 0;
		rect.topLeft.y = scaleCoord(13, w->drawState->scaling.preGarnet.standardToActive, false);
		rect.extent.x = scaleCoord(winExtentX, w->drawState->scaling.preGarnet.standardToActive, false);
		rect.extent.y = (screenDensity == kDensityOneAndAHalf) ? 2 : 2 * screenDensity / 72;
		WinDrawRectangle(&rect, 0);
		
		WinSetCoordinateSystem(kDensityLow);
		
		//text
		WinDrawInvertedChars(titleStr, titleLen, 3, 2);
	}
	
	WinSetCoordinateSystem(prevCoordSys);
	WinPopDrawState();
	
	frm->attr.attnIndSupported = !frm->win.flags.modal && titleAreaWidth >= 5;
	if (frm->attr.attnIndSupported)
		AttnIndicatorAllow(true);
}

static void PrvRedrawTitle(struct PalmForm *frm, struct PalmTitleObj *obj, const char * newTitle)
{
	uint32_t curStrWidth = 0, newStrLen = 0, newStrWidth = 0;
	char *curTitle = obj->text;
	
	AttnIndicatorAllow(false);
	WinPushDrawState();
	FntSetFont(boldFont);
	PrvSetAppropriateColors(frm, SET_APPROPRIATE_TEXT_COLOR | SET_APPROPRIATE_FORE_COLOR | SET_APPROPRIATE_BACK_COLOR);
	
	if (curTitle)
		curStrWidth = FntCharsWidth(curTitle, StrLen(curTitle));
	if (newTitle)
		newStrWidth = FntCharsWidth(newTitle, newStrLen = StrLen(newTitle));
	
	if (newStrWidth != curStrWidth) {
	
		struct RectangleType rect;
		uint32_t prevCoordSys;
		
		rect.topLeft.y = 0;
		if (newStrWidth > curStrWidth) {
			
			rect.topLeft.x = 0;
			rect.extent.x = newStrWidth + 5;
		}
		else {
			rect.topLeft.x = newStrWidth + 2;
			rect.extent.x = curStrWidth - newStrWidth + 3;
			rect.extent.y = 13;
			WinEraseRectangle(&rect, 0);
			rect.topLeft.x -= 3;
			rect.extent.x = 6;
		}
		rect.extent.y = 15;
		//it seems to me like the 1.5DD logic from PrvDrawTitle never made it here, like it seemingly should have.
		// That being said, the results look correct and this is what garnet would do, so i am keeping this as is
		prevCoordSys = WinSetCoordinateSystem(kCoordinatesDouble);
		scaleRect(&rect, ((struct PalmWindow*)WinGetDrawWindow())->drawState->scaling.preGarnet.standardToActive);
		WinDrawRectangle(&rect, 7);
		WinSetCoordinateSystem(prevCoordSys);
	}
	
	if (newTitle)
		WinDrawInvertedChars(newTitle, newStrLen, 3, 2);
	
	WinPopDrawState();
	
	frm->attr.attnIndSupported = !frm->win.flags.modal && newStrWidth >= 5;
	if (frm->attr.attnIndSupported)
		AttnIndicatorAllow(true);
}

static void pLstDrawList(struct PalmListObj *list)
{
	struct Globals *g = globalsGet();
	
	if (list->id == TITLE_OBJ_SURROGATE_ID_VAL && list->popupWin == TITLE_OBJ_SURROGATE_FAKE_WIN_VAL && !list->attr.usable) {
		
		struct PalmTitleObj *ttlObj = (struct PalmTitleObj*)list->ListItemDrawF;
		struct PalmForm *frm = (struct PalmForm*)list->itemsText;
		
		PrvDrawTitle(frm, ttlObj);
	}
	else
		g->ot_pLstDrawList(list);
}
DEF_UI_PATCH_PARTIAL(pLstDrawList, 0x360);

static void pFrmDrawForm(struct PalmForm *frm)
{
	struct Globals *g = globalsGet();
	uint32_t i;
	
	//if it exists, hide the titlebar obj by converting it into a disabled list with some special easy-to-identify-properties
	for (i = 0; i < frm->numObjects; i++) {
		
		if (frm->objs[i].objKind == frmTitleObj) {
			
			struct PalmListObj *newList = MemChunkNew(0, sizeof(struct PalmListObj), 0x200);
			if (!newList)
				continue;
			
			newList->id = TITLE_OBJ_SURROGATE_ID_VAL;
			newList->attr.usable = false;
			newList->popupWin = TITLE_OBJ_SURROGATE_FAKE_WIN_VAL;
			newList->ListItemDrawF = (void*)frm->objs[i].obj;
			newList->itemsText = (char**)frm;
			frm->objs[i].obj = newList;
			frm->objs[i].objKind = frmListObj;
		}
	}
	
	//call old trap, which will call LstDrawList() on our fake title(s) objs
	g->ot_pFrmDrawForm(frm);
	
	//restore what I had done
	for (i = 0; i < frm->numObjects; i++) {
		
		struct PalmListObj *list = frm->objs[i].obj;
		
		if (frm->objs[i].objKind != frmListObj)
			continue;
		
		if (list->id == TITLE_OBJ_SURROGATE_ID_VAL && list->popupWin == TITLE_OBJ_SURROGATE_FAKE_WIN_VAL && !list->attr.usable) {
			
			frm->objs[i].objKind = frmTitleObj;
			frm->objs[i].obj = list->ListItemDrawF;
			
			MemChunkFree(list);
		}
	}
}
DEF_UI_PATCH_PARTIAL(pFrmDrawForm, 0x218);

static void FrmTitleOp(struct PalmForm *frm, char* title,void (*setterF)(char** curTxtP, char* newTxt))
{
	uint32_t i;
	
	//if it exists, hide the titlebar obj by converting it into a disabled list with some special easy-to-identify-properties
	for (i = 0; i < frm->numObjects; i++) {
		
		if (frm->objs[i].objKind == frmTitleObj) {
			
			struct PalmTitleObj *ttlObj = (struct PalmTitleObj*)frm->objs[i].obj;
			
			if (!frm->attr.visible){
				
				setterF(&ttlObj->text, title);
			}
			else if (frm->win.flags.modal) {
			
				setterF(&ttlObj->text, title);
				PrvDrawTitle(frm, ttlObj);
			}
			else {
				
				PrvRedrawTitle(frm, ttlObj, title);
				setterF(&ttlObj->text, title);
			}
		}
	}
}

static void FrmSetTitleWork(char** curTxtP, char* newTxt)
{
	*curTxtP = newTxt;
}

static void pFrmSetTitle(struct PalmForm *frm, char* title)
{
	FrmTitleOp(frm, title, &FrmSetTitleWork);
}
DEF_UI_PATCH(pFrmSetTitle, 0x2E8);

static void FrmCopyTitleWork(char** curTxtP, char* newTxt)
{
	StrCopy(*curTxtP,  newTxt);
}

static void pFrmCopyTitle(struct PalmForm *frm, char* title)
{
	FrmTitleOp(frm, title, &FrmCopyTitleWork);
}
DEF_UI_PATCH(pFrmCopyTitle, 0x200);

static bool PrvPointInIndicator(struct PalmForm *frm, Coord x, Coord y)
{
	if (!AttnIndicatorGetBlinkPattern())
		return false;
	
	if (!AttnIndicatorAllowed())
		return false;
	
	if (!AttnIndicatorEnabled())
		return false;
	
	return (x >= 0 && x < 16 && y>=0 && y < 15);
}

static int32_t PrvPointInObject(struct PalmForm *frm, Coord x, Coord y)		//return index of object or negative if nonw
{
	struct RectangleType bounds;
	uint32_t i;
	
	for (i = 0; i < frm->numObjects; i++) {
		
		FrmGetObjectBounds((struct FormType*)frm, i, &bounds);
		if (RctPtInRectangle(x, y, &bounds))
			return i;
	}
	
	return -1;
}

static bool PrvHelpHandleEvent(struct PalmForm *frm, EventType *evt)
{
	RectangleType winBounds, helpButtonBounds = {.topLeft = {.y = 0}, .extent = {.x = 10, .y = 10}};
	Boolean penDown = true;	//to make compiler happy about types
	bool active = false;
	int16_t penX, penY;
	
	//we do not need to handl ehelp button if there isn't one
	if (!frm->helpRscId)
		return false;
	
	//sort out if the stylus is even in it
	WinGetBounds((WinHandle)&frm->win, &winBounds);
	helpButtonBounds.topLeft.x = winBounds.extent.x - 12;
	if (!RctPtInRectangle(penX = evt->screenX, penY = evt->screenY, &helpButtonBounds))
		return false;
	
	WinPushDrawState();
	PrvSetAppropriateColors(frm, SET_APPROPRIATE_TEXT_COLOR | SET_APPROPRIATE_FORE_COLOR | SET_APPROPRIATE_BACK_COLOR);
	
	while (penDown) {
		
		if (RctPtInRectangle(penX, penY, &helpButtonBounds)) {
			if (!active)
				PrvDrawHelpIcon(winBounds.extent.x, true);
			active = true;
		}
		else {
			if (active)
				PrvDrawHelpIcon(winBounds.extent.x, false);
			active = false;
		}
		EvtGetPen(&penX, &penY, &penDown);
	}
	
	//if we got out with active == true, then user clicked the help button
	//but first, redraw the help button as inactive
	if (active)
		PrvDrawHelpIcon(winBounds.extent.x, false);

	WinPopDrawState();
	
	if (active)
		FrmHelp(frm->helpRscId);
	
	return true;
}

static bool pFrmHandleEvent(struct PalmForm *frm, EventType *evt)
{
	//we replicate the logic UP to where PrvHelpHandleEvent is called to make sure the logic WOULD get that far, after that, we let it go
	if (evt->eType == penDownEvent && !PrvPointInIndicator(frm, evt->screenX, evt->screenY) && PrvPointInObject(frm, evt->screenX, evt->screenY) < 0 && PrvHelpHandleEvent(frm, evt))
		return true;
	
	//if we got this far, call the proper handler
	return globalsGet()->ot_pFrmHandleEvent(frm, evt);
}
DEF_UI_PATCH_PARTIAL(pFrmHandleEvent, 0x280);

static uint16_t buttonFrameToFrameType(const struct PalmControl *ctl)
{
	switch (ctl->style) {
		case PALM_CONTROL_STYLE_BUTTON:
		case PALM_CONTROL_STYLE_REPEATING_BUTTON:
			switch (ctl->attr.frame) {
				case PALM_CTL_FRAME_STD_BTN:	return WinFrameLE_roundFrame;
				case PALM_CTL_FRAME_BOLD_BTN:	return WinFrameLE_boldRoundFrame;
				case PALM_CTL_FRAME_RECTANGLE:	return WinFrameLE_rectangleFrame;
				default:						return WinFrameLE_none;
			}
			break;
		
		case PALM_CONTROL_STYLE_PUSHBUTTON:
		case PALM_CONTROL_STYLE_SELECTOR_TRIGGER:
			return WinFrameLE_rectangleFrame;
		
		default:
			return WinFrameLE_none;
	}
}

static void pCtlEraseControl(struct PalmControl *ctl)
{
	if (!ctl->attr.usable || ! ctl->attr.visible)
		return;
	
	if (ctl->style == PALM_CONTROL_STYLE_CHECKBOX || (ctl->style != PALM_CONTROL_STYLE_SELECTOR_TRIGGER && ctl->attr.frame == PALM_CTL_FRAME_NONE))
		WinEraseRectangle(&ctl->bounds, 0);
	else {
		union PalmFrameFlags frame = {.frameType = buttonFrameToFrameType(ctl), };
		
		if (frame.frameType == WinFrameLE_roundFrame && BmpGetDensity(WinGetBitmap(WinGetDrawWindow())) == kDensityOneAndAHalf) {
			
			struct RectangleType r = ctl->bounds;
			
			WinSetCoordinateSystem(kCoordinatesNative);
			WinScaleRectangle(&r);
			RctInsetRectangle(&r, -1);
			WinEraseRectangle(&r, 4);
			WinSetCoordinateSystem(kCoordinatesStandard);
		}
		else {
			
			WinEraseRectangle(&ctl->bounds, frame.cornerDiam);
			WinEraseRectangleFrame(frame.frameType, &ctl->bounds);
		}
	}
	
	ctl->attr.visible = false;
	ctl->attr.drawnAsSelected = false;
}
DEF_UI_PATCH(pCtlEraseControl, 0xAC);

static void oldTrapPrvDrawControl(struct PalmControl *ctl, bool on)	//calls ROM's PrvDrawControl() due to a little bit of cleverness
{
	bool prevOn = ctl->attr.on;
	ctl->attr.on = on;
	globalsGet()->ot_pCtlDrawControl(ctl);	//directly calls PrvDrawControl() with ctl->on
	ctl->attr.on = prevOn;
}

static void PrvDrawControl(struct PalmControl *ctl, bool on)
{
	bool specialDensity = BmpGetDensity(WinGetBitmap(WinGetDrawWindow())) == kDensityOneAndAHalf;
	union PalmFrameFlags frm = {.frameType = buttonFrameToFrameType(ctl), };
	struct UiGlobals *uig = UIGetGlobalsPtr();
	struct Globals *g = globalsGet();
	
	if (!ctl->attr.usable)
		return;
	
	if (specialDensity) {
		
		bool any = false;
		
		if (ctl->style != PALM_CONTROL_STYLE_SELECTOR_TRIGGER && frm.frameType == WinFrameLE_roundFrame && uig->uiConfigs.alwaysRectaBorders) {
			
			any = true;
			g->uiHacksEnabled.replaceOneRectangleFrame = true;
		}
		
		if (ctl->style != PALM_CONTROL_STYLE_CHECKBOX && ctl->style != PALM_CONTROL_STYLE_SLIDER && ctl->style != PALM_CONTROL_STYLE_FEEDBACK_SLIDER && !ctl->attr.graphical && frm.frameType == WinFrameLE_roundFrame && (ctl->style == PALM_CONTROL_STYLE_POPUP_TRIGGER || (ctl->text && StrLen(ctl->text) && StrCompare(ctl->text, " ")))) {
			
			any = true;
			g->uiHacksEnabled.replaceOneEraseRectangle = true;
		}
		if (any)
			g->ctl = ctl;
	}
	
	oldTrapPrvDrawControl(ctl, on);
	
	if (specialDensity) {
		
		g->ctl = NULL;
		g->uiHacksEnabled.replaceOneRectangleFrame = false;
		g->uiHacksEnabled.replaceOneEraseRectangle = true;
	}
}

static void pCtlDrawControl(struct PalmControl *ctl)
{
	PrvDrawControl(ctl, ctl->attr.on);
}
DEF_UI_PATCH_PARTIAL(pCtlDrawControl, 0xA4);

static void pCtlSetValue(struct PalmControl *ctl, int16_t val)
{
	if (ctl->style == PALM_CONTROL_STYLE_SLIDER || ctl->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER) {
		
		struct PalmSliderControl *sld = (struct PalmSliderControl*)ctl;
		
		if (val > sld->maxValue)
			val = sld->maxValue;
		if (val < sld->minValue)
			val = sld->minValue;
		
		if (val != sld->value) {
			
			sld->value = val;
			if (ctl->attr.visible)
				PrvDrawControl(ctl, false);
		}
	}
	else {
		
		val = !!val;
		if (ctl->attr.visible && !ctl->attr.on != !val)
			PrvDrawControl(ctl, val);
		ctl->attr.on = val;
	}
}
DEF_UI_PATCH(pCtlSetValue, 0xE8);

static void pCtlSetGraphics(struct PalmControl *ctl, uint16_t newBitmapID, uint16_t newSelectedBitmapID)
{
	if (ctl->attr.graphical) {
		
		struct PalmGraphicControl *gctl = (struct PalmGraphicControl*)ctl;
		
		if (newBitmapID)
			gctl->bitmapId = newBitmapID;
		
		if (newSelectedBitmapID)
			gctl->selectedBitmapId = newSelectedBitmapID;
		
		if (ctl->attr.visible)
			PrvDrawControl(ctl, ctl->attr.on);
	}
}
DEF_UI_PATCH(pCtlSetGraphics, 0xD8);

static void pCtlSetSliderValues(struct PalmControl *ctl, const uint16_t *minValueP, const uint16_t *maxValueP, const uint16_t *pageSizeP, const uint16_t *valueP)
{
	if (ctl->style == PALM_CONTROL_STYLE_SLIDER || ctl->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER) {
		
		struct PalmSliderControl *sld = (struct PalmSliderControl*)ctl;
		
		if (!sld->activeSliderP) {
			
			if (minValueP)
				sld->minValue = *minValueP;
			
			if (maxValueP)
				sld->maxValue = *maxValueP;
			
			if (pageSizeP)
				sld->pageSize = *pageSizeP;
			
			if (valueP)
				sld->value = *valueP;
		
			if (ctl->attr.visible)
				PrvDrawControl(ctl, false);
		}
	}
}
DEF_UI_PATCH(pCtlSetSliderValues, 0xE0);

static void PrvSliderMakeActive(struct PalmSliderControl *sld)		//ours is idempotent
{
	struct PalmActiveSlider *as;
	Coord bodyWidth;
	int32_t t;
	Err e;
	
	if (sld->activeSliderP)
		return;
	
	as = MemChunkNew(0, sizeof(struct PalmActiveSlider), 0x200);
	
	//offscreen window for smoothness (we CAN work without it)
	as->renderWinH = WinCreateOffscreenWindow(sld->bounds.extent.x, sld->bounds.extent.y, nativeFormat, &e);
	if (e != errNone)
		as->renderWinH = NULL;
	
	//get resources
	as->thumbBmpH = DmGetResource('abmp', sld->thumbBmpId ? sld->thumbBmpId : 13350);
	as->backgroundBmpH = DmGetResource('abmp', sld->backgroundBmpId ? sld->backgroundBmpId : 13351);
	
	//lock them
	as->thumbBmpPtr = MemHandleLock(as->thumbBmpH);
	as->backgroundBmpPtr = MemHandleLock(as->backgroundBmpH);

	//other misc init
	as->trackingPen = false;
	as->newValue = sld->value;

	//work out the slider body position
	WinGetBitmapDimensions(as->thumbBmpPtr, &bodyWidth, NULL);
	t = sld->maxValue - sld->minValue;
	as->thumbPosX = (t / 2 + (sld->value - sld->minValue) * (sld->bounds.extent.x - bodyWidth)) / t;
	
	sld->activeSliderP = as;
}

static void PrvSliderMakeInactive(struct PalmSliderControl *sld)		//ours is idempotent
{
	struct PalmActiveSlider *as = sld->activeSliderP;
	
	if (!as)
		return;
	
	sld->activeSliderP = NULL;
	
	if (as->renderWinH)
		WinDeleteWindow(as->renderWinH, false);
	
	MemHandleUnlock(as->thumbBmpH);
	MemHandleUnlock(as->backgroundBmpH);
	DmReleaseResource(as->thumbBmpH);
	DmReleaseResource(as->backgroundBmpH);
	
	MemChunkFree(as);
}

static bool PrvSliderTrackPen(struct PalmSliderControl *sld, EventType *evt, const struct RectangleType *frameRect, EventType *tmpEvt)
{
	struct PalmActiveSlider *as;
	uint32_t prevTimestamp = 0;
	int32_t newThumbPosX, t, q;
	int16_t bodyWidth;
	
	PrvSliderMakeActive(sld);
	as = sld->activeSliderP;
	
	*tmpEvt = *evt;
	
	if (evt->eType == ctlEnterEvent)
		as->newValue = sld->value;		//nexcessary since slider could have been made active by drawing it, and so as->newValue could be stale
	else {
		as->newValue = tmpEvt->data.ctlRepeat.value;
		EvtGetPen(&tmpEvt->screenX, &tmpEvt->screenY, &tmpEvt->penDown);
	}
	
	WinGetBitmapDimensions(as->thumbBmpPtr, &bodyWidth, NULL);
	
	while (tmpEvt->penDown) {
		
		if (tmpEvt->screenY >= sld->bounds.topLeft.y && tmpEvt->screenY <= sld->bounds.topLeft.y + sld->bounds.extent.y) {	//is the pen even in the slider height-wise?
			
			int32_t pixInSlider = tmpEvt->screenX - sld->bounds.topLeft.x;
			
			//pen over the body means we are dragging it -> handle that
			if ((pixInSlider >= as->thumbPosX && pixInSlider <= as->thumbPosX + bodyWidth) || as->trackingPen) {
				
				if (!as->trackingPen)
					SndPlaySystemSound(sndClick);
				as->trackingPen = true;
				
				//sort out slider body placement
				newThumbPosX = pixInSlider - (bodyWidth / 2);
				if (newThumbPosX < 0)
					newThumbPosX = 0;
				if (newThumbPosX > sld->bounds.extent.x - bodyWidth)
					newThumbPosX = sld->bounds.extent.x - bodyWidth;
				
				//did the slider body move? if so, we have some work to do
				if (newThumbPosX != as->thumbPosX) {
						
					//record the move
					as->thumbPosX = newThumbPosX;
					
					//map position back into slider value
					q = sld->bounds.extent.x - bodyWidth;
					t = newThumbPosX * (sld->maxValue - sld->minValue);
					t = (t + q / 2) / q + sld->minValue;
					
					//did the VALUE change? (it is possible for slider to move without changing the value: imagine a 1..3 slider that is 100px wide
					if (t != as->newValue) {
						
						//record the new value and draw the control in the new state
						as->newValue = t;
						PrvDrawControl((struct PalmControl*)sld, false);
						
						//feedback sliders send an event anytime the value changes -> send it
						
						if (sld->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER) {
							tmpEvt->eType = ctlRepeatEvent;
							tmpEvt->data.ctlRepeat.time = HALTimeGetSystemTime();
							tmpEvt->data.ctlRepeat.value = as->newValue;
							SysEventAddToQueue(tmpEvt);
							return true;
						}
					}
				}
			}
			else if (pixInSlider >= 0 && pixInSlider <= sld->bounds.extent.x) { //pen is not over the body. it could be over the slider background -> handle that
				
				//for feedback sliders, we need to recall the last time we sent a repeat event
				if (sld->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER && evt->eType == ctlRepeatEvent)
					prevTimestamp = evt->data.ctlRepeat.time;
				
				//if it is a ctl enter (first time we're here) or if the proper amount of time has elapsed, we have work to do
				if (evt->eType == ctlEnterEvent || HALTimeGetSystemTime() >= prevTimestamp + 150) {
				
					//move a big step in the requested direction
					if (pixInSlider < as->thumbPosX)
						as->newValue -= sld->pageSize;
					else
						as->newValue += sld->pageSize;
				
					//range check (sorry, oracle)
					if (as->newValue < sld->minValue) {
						as->newValue = sld->minValue;
						SndPlaySystemSound (sndWarning);
					}
					else if (as->newValue > sld->maxValue) {
						as->newValue = sld->maxValue;
						SndPlaySystemSound (sndWarning);
					}
					
					//propagate value to position of body
					q = sld->maxValue - sld->minValue;
					t = ((as->newValue - sld->minValue) * (sld->bounds.extent.x - bodyWidth) + q / 2) / q;
					
					//do not jump over the stylus while doing a big jump. if that happens, pretend user clicked on slider body
					if ((t < pixInSlider && as->thumbPosX > pixInSlider) || (t > pixInSlider && as->thumbPosX < pixInSlider))
						as->trackingPen = true;
					else {
						
						as->thumbPosX = t;
						prevTimestamp = HALTimeGetSystemTime() + ((!prevTimestamp && evt->eType == ctlEnterEvent) ? 250 : 0);
						SndPlaySystemSound(sndClick);
						PrvDrawControl((struct PalmControl*)sld, false);
						
						if (sld->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER) {	//feedback sliders always send events
							
							tmpEvt->eType = ctlRepeatEvent;
							tmpEvt->data.ctlRepeat.time = prevTimestamp;
							tmpEvt->data.ctlRepeat.value = as->newValue;
							SysEventAddToQueue(tmpEvt);
							return true;
						}
					}
				}
			}
		}
		else if (as->newValue != sld->value) {		//not in slider height wise? reset value to original if needed and bail
			
			as->newValue = sld->value;
			PrvDrawControl((struct PalmControl*)sld, false);

			if (sld->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER) {	//send one last event
				tmpEvt->eType = ctlRepeatEvent;
				tmpEvt->data.ctlRepeat.time = prevTimestamp;
				tmpEvt->data.ctlRepeat.value = as->newValue;
				SysEventAddToQueue(tmpEvt);
				return true;
			}
		}
		
		EvtGetPen(&tmpEvt->screenX, &tmpEvt->screenY, &tmpEvt->penDown);
	}
	
	//pen is up and we're still here? send an exist event if it was raised outside the control,e else a select event
	if (tmpEvt->screenY < sld->bounds.topLeft.y || tmpEvt->screenY > sld->bounds.topLeft.y + sld->bounds.extent.y)
		tmpEvt->eType = ctlExitEvent;
	else {

		sld->value = t = as->newValue;
		tmpEvt->eType = ctlSelectEvent;
		tmpEvt->data.ctlSelect.value = t;
	}
	SysEventAddToQueue(tmpEvt);
	
	PrvSliderMakeInactive(sld);
	return true;
}

static bool PrvOtherControlsTrackPen(struct PalmControl *ctl, EventType *evt, const struct RectangleType *frameRect, EventType *tmpEvt)
{
	bool hilighted = true, initState = ctl->attr.on;
	Boolean penDown = true;	//to make compiler happy about types
	int16_t penX, penY;
	
	while (penDown) {
		EvtGetPen(&penX, &penY, &penDown);
		if (RctPtInRectangle(penX, penY, frameRect)) {			//track pen inside out bouds
			
			if (penDown) {										//dragging on our control - keep tracking
				
				if (!hilighted) {
					PrvDrawControl(ctl, !initState);
					hilighted = true;
					evt->data.ctlRepeat.time = HALTimeGetSystemTime();		//yes, modify incoming event
				}
				
				if (ctl->style == PALM_CONTROL_STYLE_REPEATING_BUTTON) {		//handle repeating controls
					
					//is it time to send another repeat event?
					//yes this code is correct despite not making sense. behaviour replicted ot keep look & feel same as stock
					if (evt->eType == ctlEnterEvent || (HALTimeGetSystemTime() > evt->data.ctlRepeat.time && HALTimeGetSystemTime() + 150 >= evt->data.ctlRepeat.time)) {
						
						tmpEvt->eType = ctlRepeatEvent;
						tmpEvt->penDown = true;
						tmpEvt->screenX = penX;
						tmpEvt->screenY = penY;
						tmpEvt->data.ctlRepeat.controlID = ctl->id;
						tmpEvt->data.ctlRepeat.pControl = (void*)ctl;
						tmpEvt->data.ctlRepeat.time = HALTimeGetSystemTime() + ((evt->eType == ctlEnterEvent) ? 350 : 0);
						SysEventAddToQueue (tmpEvt);
						SndPlaySystemSound(sndClick);
						return true;
					}
				}
			}
			else {												//pen released insid econtorl: notify that user finished with the control
				
				if (ctl->style == PALM_CONTROL_STYLE_CHECKBOX || ctl->style == PALM_CONTROL_STYLE_PUSHBUTTON)
					ctl->attr.on = !ctl->attr.on;
				else
					PrvDrawControl(ctl, ctl->attr.on);
				
				if (ctl->style != PALM_CONTROL_STYLE_REPEATING_BUTTON) {
					tmpEvt->eType = ctlSelectEvent;
					tmpEvt->screenX = penX;
					tmpEvt->screenY = penY;
					tmpEvt->data.ctlSelect.controlID = ctl->id;
					tmpEvt->data.ctlSelect.pControl = (void*)ctl;
					tmpEvt->data.ctlSelect.on = ctl->attr.on;
					SysEventAddToQueue(tmpEvt);
					SndPlaySystemSound(sndClick);
					return true;
				}
			}
		}
		else {													//outside of our bounds
			
			if (hilighted)
				PrvDrawControl(ctl, initState);
			hilighted = false;
			
			if (!penDown) {										//pen released outside control: notify that user finished with the control
				
				tmpEvt->eType = ctlExitEvent;
				tmpEvt->screenX = penX;
				tmpEvt->screenY = penY;
				tmpEvt->data.ctlExit.controlID = ctl->id;
				tmpEvt->data.ctlExit.pControl = (void*)ctl;
				SysEventAddToQueue(tmpEvt);
			}
		}
	}
	
	return false;
}

static bool pCtlHandleEvent(struct PalmControl *ctl, EventType *evt)
{
	struct RectangleType frameRect;
	EventType sentEvt = {0, };
	
	if (!ctl->attr.usable|| !ctl->attr.visible || !ctl->attr.enabled)
		return false;
	
	WinGetFramesRectangle(buttonFrameToFrameType(ctl), &ctl->bounds, &frameRect);
	
	switch (evt->eType) {
		
		case penDownEvent:
			if (!RctPtInRectangle(evt->screenX, evt->screenY, &frameRect))
				break;
			sentEvt = *evt;
			sentEvt.eType = ctlEnterEvent;
			sentEvt.data.ctlEnter.controlID = ctl->id;
			sentEvt.data.ctlEnter.pControl = (void*)ctl;
			SysEventAddToQueue(&sentEvt);
			return true;
		
		case ctlEnterEvent:
			if (evt->data.ctlEnter.controlID != ctl->id)
				break;
			PrvDrawControl(ctl, !ctl->attr.on);
			goto ctl_enter_and_repeat_common;
		
		case ctlRepeatEvent:
			if (evt->data.ctlRepeat.controlID != ctl->id)
				break;
				
	ctl_enter_and_repeat_common:
			if (ctl->style == PALM_CONTROL_STYLE_SLIDER || ctl->style == PALM_CONTROL_STYLE_FEEDBACK_SLIDER)
				return PrvSliderTrackPen((struct PalmSliderControl*)ctl, evt, &frameRect, &sentEvt);
			else
				return PrvOtherControlsTrackPen(ctl, evt, &frameRect, &sentEvt);
		
		default:
			return false;
	}
	
	return false;
}
DEF_UI_PATCH(pCtlHandleEvent, 0xBC);



