/******************************************************************************
 * Copyright (c) 1995-2005 palmOne, Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/
/**
 * @ingroup PIM
 */

/**
 * @file MemoDB.h
 *
 * @brief Contains database record type and constants for Memos application.
 *
 * This file contains the structure of the record in Memos DB and can be used
 * to access the database directly. One way to utilize this header file is to
 * combine it with the old Memo source code so that the record packing
 * and unpacking routines are adjusted for the new structure.
 *
 * Please note that accessing the database directly is generally not recommended
 * and this is offered as a temporary solution for 3rd party developers. The
 * structure might change again in the future.
 *
 * <hr>
 */

#ifndef __MEMODB_H__
#define __MEMODB_H__

#include <PalmTypes.h>
#include <DataMgr.h>

/** The app info string resource ID for localized strings to replace the
    strings in the application info block depending on the current
    system locale state (e.g.: Spanish). */
#define LocalizedAppInfoStr         1000

#define memoDBVersionNum            1

/** Used when the version of a database is not compatible with this
    version of the application. */
#ifndef appErrVersionIncompatible
#define appErrVersionIncompatible   (appErrorClass | 1)
#endif

/** @brief Application Info Block
 *
 * This structure is used to store info applicable to all records
 * in the database, specific to the application, inter-session (like
 * preferences), etc.
 */
typedef struct
{
    /** Bitfield of categories with a different name. */
    UInt16 renamedCategories;

    char   categoryLabels[dmRecNumCategories][dmCategoryLength];

    UInt8  categoryUniqIDs[dmRecNumCategories];

    /**
     * Unique IDs generated by the device are between 0 - 127.
     * Those from the PC are 128 - 255.
     */
    UInt8  lastUniqID;

    /** From the compiler word aligning things. */
    UInt8  reserved1;

    /**
     * Whether category colors were edited since last sync.
     * Least significant bit first.
     */
    UInt16 categoryColorsEdited;

    /**
     * Added as part of the Mullet version of this application,
     * so that we can later support color categories without
     * breaking the conduits.
     */
    UInt8  categoryColors[dmRecNumCategories];

    UInt16  reserved2;

    UInt16  reserved3;

    /** The List View sort order. */
    UInt8   sortOrder;

    UInt8   reserved;

} MemoAppInfoType;

typedef MemoAppInfoType * MemoAppInfoPtr;


/**
 * @name Database sort orders: seen in List View.
 */
/*@{*/
#define soUnsorted          0
#define soAlphabetic        1
/*@}*/

/** @brief Record format
 *
 * This is the record format as used by the app.  The Memo
 * record is stored in the database as a string.  The
 * start of the string is represented by the note member
 * of this struct.  Thus, the address of the note member
 * is the address of the memo string.
 */
typedef struct {
    char    note;       /**< null terminated */
    UInt8   reserved;
} MemoDBRecordType;

typedef MemoDBRecordType * MemoDBRecordPtr;

/** @brief A memo string
 *
 * This is essentially just a container for a pointer to a
 * memo string.  It is used by MemoNewNote() as the source
 * for the memo data to copy into a new memo record.
 */
typedef struct {
    Char *              note;
} MemoItemType;

typedef MemoItemType * MemoItemPtr;


#ifdef __cplusplus
extern "C" {
#endif


//-------------------------------------------------------------------
// Routines
//-------------------------------------------------------------------

/**
 * Create an app info chunk if missing and initialize with defaults.
 *
 * @param dbP               Open database ptr.
 * @return                  0 if successful, error code if not.
 */
extern Err      MemoAppInfoInit(DmOpenRef dbP);

/**
 * Create a new record at the given position in the database.
 *
 * @param dbP               Open database ptr.
 * @param item              Ptr to the container of the memo string
 *                              to copy into the new record.
 * @param index             Ptr to the record index to place the
 *                              new record.
 * @return                  0 if successful, error code if not.
 */
extern Err      MemoNewRecord(DmOpenRef dbP, MemoItemPtr item, UInt16 *index);

/**
 * Change the sort order of the database.
 *
 * @param dbP               Open database ptr.
 * @param sortOrder         The sort order to which the database
 *                              should be changed. e.g.:
 *                              soAlphabetic or soUnsorted
 * @return                  0 if successful, error code if not.
 */
extern Err      MemoChangeSortOrder(DmOpenRef dbP, UInt8 sortOrder);

/**
 * Get the database's current sort order.
 *
 * @param dbP               Open database ptr.
 * @return                  The sort order of the database. e.g.:
 *                              soAlphabetic or soUnsorted
 */
extern UInt8    MemoGetSortOrder (DmOpenRef dbP);

/**
 * Sort the database according to the current sort order.
 *
 * @param dbP               Open database ptr.
 * @return                  Nothing.
 */
extern void     MemoSort (DmOpenRef dbP);

/**
 * Move the record corresponding to the given index to a position
 *  in the database in accordance with the current sort order.
 *
 * @param dbP               Open database ptr.
 * @param indexP            Ptr to the index of the record to move,
 *                              and set to the new index of the record.
 * @return                  0 if successful, error code if not.
 */
extern Err      MemoSortRecord (DmOpenRef dbP, UInt16 * indexP);

/**
 * Get the application's database, opening or creating it as neccessary.
 *
 * @param dbPP              Ptr to a database ref var (DmOpenRef) to set.
 * @param mode              How to open the database (e.g.: dmModeReadWrite).
 * @return                  0 if successful, error code if not.
 */
extern Err      MemoGetDatabase (DmOpenRef *dbPP, UInt16 mode);

#ifdef __cplusplus
}
#endif


#endif


