#define F_CPU	8000000
#include <avr/wdt.h>
#include <stdio.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/delay.h>
#include <avr/sleep.h>
#include "../nRF.h"
#include "../protocol.h"

#define sbi(var, mask)   ((var) |= (uint8_t)(1 << mask))
#define cbi(var, mask)   ((var) &= (uint8_t)~(1 << mask))


#define TX_SCK	4 //Output
#define TX_MISO 5 //Input
#define TX_MOSI	6 //Output

#define TX_CE	1 //Output
#define TX_CSN	2 //Output

#define BTN_UP	0
#define BTN_DN	1
#define BTN_LT	2
#define BTN_RT	3
#define BTN_CT	7


#define ADDR_ME		0x11223345
#define ADDR_LIGHTS	0x3A50B211

ISR(PCINT0_vect){;}	//This vector is only here to wake unit up from sleep mode
ISR(PCINT1_vect){;}	//This vector is only here to wake unit up from sleep mode

void nrfCE(char active)
{
	if(active) sbi(PORTB, TX_CE);
	else cbi(PORTB, TX_CE);
}

void nrfCS(char active)
{
	if(active) cbi(PORTB, TX_CSN);
	else sbi(PORTB, TX_CSN);
}

void nrfDelay200us(void)
{
	asm volatile(
		"1:			\n\t"
		"	dec  r1		\n\t"
		"	brne 1b		\n\t"
		"1:			\n\t"
		"	dec  r1		\n\t"
		"	brne 1b		\n\t"
	);
}

static void delay_ms(uint8_t x){
	while(x--){
		nrfDelay200us();
		nrfDelay200us();
		nrfDelay200us();
		nrfDelay200us();
		nrfDelay200us();
	}
}


uint8_t __attribute__((naked,noinline)) nrfSpiByte(uint8_t byte){

	asm volatile(
		"	ldi  r31, 0x10	\n\t"
		"	ldi  r30, 8	\n\t"	//i = 8
		"spi_loop:		\n\t"
		"	cbi  0x1B, 6	\n\t"	//MOSI = 0
		"	sbrc r24, 7	\n\t"	//if(byte & 0x80)
		"	sbi  0x1B, 6	\n\t"	//  MOSI = 1
		"	out  0x19, r31	\n\t"	//CLK = 1
		"	lsl  r24	\n\t"	//byte <<= 1
		"	sbic 0x19, 5	\n\t"	//if(PINA & (1 << 5))
		"	inc  r24	\n\t"	//  byte++
		"	dec  r30	\n\t"
		"	out  0x19, r31	\n\t"	//CLK = 0
		"	brne spi_loop	\n\t"
		"	sbi  0x1B, 6	\n\t"	//  MOSI = 1 (we prefer it to idle high)
		"	ret		\n\t"
	);
}

void nrfWaitForIrq(void){
	asm volatile("sleep");
}

void ioinit (void)
{
	wdt_reset();
	wdt_disable();

	//1 = Output, 0 = Input
	DDRA = (1 << TX_MOSI) | (1 << TX_SCK);
	DDRB = 0b00000110; //(CE on PB1) (CS on PB2)
	
	//Enable pull-up resistors (page 74)
	PORTA = 0b10001111 | (1 << TX_MOSI); //Pulling up a pin that is grounded will cause 90uA current leak, mosi is high
	
	cbi(PORTB, TX_CE); //Stand by mode
	
	GIFR = (1<<PCIF0) | (1<<PCIF1); //Enable the Pin Change interrupts to monitor button presses
	GIMSK = (1<<PCIE0) | (1<<PCIE1); //Enable Pin Change Interrupt Request
	PCMSK0 = (1<<BTN_UP)|(1<<BTN_DN)|(1<<BTN_LT)|(1<<BTN_RT)|(1<<BTN_CT);
	PCMSK1 = 1;	//wake on NRF irq change
	MCUCR = (1<<SM1)|(1<<SE); //Setup Power-down mode and enable sleep
	
	sei(); //Enable interrupts
}

uint32_t get_remote_ver(void)
{
	struct rf_packet p;
	uint32_t ret = 0;
	uint8_t i;
		
	p.hdr.addr = ADDR_ME;
	p.hdr.type = PKT_GET_FW_VER;
	
	if (!nrfSend(&p, sizeof(struct rf_hdr), NUM_RETRIES)) return -1;
	nrfRxEnable(true);
	for(i = 0; i < NUM_RETRIES; i++){
		
		if (nrfRecv(&p, sizeof(struct rf_packet)) && p.hdr.type == PKT_FW_VER){
			ret = p.ver;
			break;
		}
		delay_ms(30);
	}
	nrfRxEnable(false);
	return ret;
}

static void sendPkt(uint8_t typ, int8_t val)
{
	struct rf_packet p;
	
	p.hdr.addr = ADDR_ME;
	p.hdr.type = typ;
	p.change = val;
	
	nrfSend(&p, sizeof(struct rf_hdr) + sizeof(uint8_t), NUM_RETRIES);
}

static void led(char on){	//do not call in middle of transactions...
	
	if (on) PORTA &=~ (1 << TX_MOSI);
	else PORTA |= (1 << TX_MOSI);
}

int main (void)
{
	uint8_t prev;
	
	ioinit();
	nrfInit(RF_CHANNEL, RF_PWR_3);
	nrfSetAddr(NRF_ADDR_MINE, ADDR_ME);
	nrfSetAddr(NRF_ADDR_TO, ADDR_LIGHTS);
	nrfRxEnable(false);
	nrfSetPowerState(false);

	
	prev = PINA;
	
	while(1){
		
		uint8_t cur = PINA;
		uint8_t act = (~cur) & prev & 0x8F;
		prev = cur;
		uint8_t delay = 250;
		
		while(act){
			nrfSetPowerState(true);
			if(act & (1 << BTN_UP)) sendPkt(PKT_CHG_BRI, 1);
			if(act & (1 << BTN_DN)) sendPkt(PKT_CHG_BRI, -1);
			if(act & (1 << BTN_RT)) sendPkt(PKT_CHG_SPEED, 1);
			if(act & (1 << BTN_LT)) sendPkt(PKT_CHG_SPEED, -1);
			if(act & (1 << BTN_CT)) sendPkt(PKT_CHG_PATTERN, 1);
			nrfSetPowerState(false);
			led(1);
			while(delay--) delay_ms(2);
			led(0);
			delay = 30;
			
			act &= ~(prev = PINA);
		}
		
		nrfRxEnable(false);
		nrfSetPowerState(false);
		PORTA |= 1 << TX_MOSI;
		ACSR = (1<<ACD); //Turn off Analog Comparator - this removes about 1uA
		PRR = 0x0F; //Reduce all power right before sleep
		delay = PORTA;
		asm volatile ("sleep");
		PORTA &=~ (1 << TX_MOSI);
		
		//Sleep until a button wakes us up on interrupt
	}
	
	return 0;
}
