#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include "partitions.h"
#include "printf.h"
#include "entry.h"
#include "util.h"
#include "load.h"


#define MY_SYS_TYPE			0x82010000	//cpu=0x82 (i think this is r3000), machine is DS2100/3100
#define STRINGIFY2(x)		#x
#define STRINGIFY(x)		STRINGIFY2(x)
#define MY_SYS_TYPE_STR		STRINGIFY(MY_SYS_TYPE)

struct DecMemBitmap {
	uint32_t pgSz;
	uint8_t bmp[];
};

struct DecPromVectors {
	uint32_t padding0[9];
	int (*getchar)(void);								//0x24
	uint32_t padding1[2];
	void (*printf)(const char * fmt, ...);				//0x30
	uint32_t padding2[12];
	const char* (*getenv)(const char *);				//0x64
	uint32_t padding3[1];
	uint32_t* (*slotaddr)(int32_t slot);				//0x6c
	uint32_t padding4[3];
	void (*clearcache)(void);							//0x7c
	uint32_t (*getsysid)(void);							//0x80
	uint32_t (*getmembitmap)(struct DecMemBitmap *out);	//0x84	returns bmp size
	uint32_t padding5[7];
	const void* (*gettcinfo)(void);						//0xa4
	uint32_t padding6[1];
	void (*rex)(char cmd);								//0xac
};

static uint8_t mRamSizes[2];							//in units of 128K

void prPutchar(char chr)
{
	consoleWrite(chr);
}

static void* v2p(void* addr)
{
	return (void*)(((uintptr_t)addr) & 0x1fffffff);	//assumes a lot of things :D
}

static void probeTlbs(void)
{
	uint32_t i, lastSuccess = 0;

	for (i = 1; i <= 64; i++) {
		uint32_t wrVal = (i - 1) << 8, readVal;

		asm volatile(
			"mtc0 %1, $0	\n\t"		//reg 0 is index
			"mfc0 %0, $0	\n\t"
			:"=r"(readVal)
			:"r"(wrVal)
		);

		if ((readVal & 0x00003f00) == wrVal)
			lastSuccess = i;
	}

	pr("TLB: %u entries detected\r\n", lastSuccess);
}

static uint_fast8_t getRamSize(uint32_t base)		//assumes we do not corrupt anything with our probe writes
{
	volatile char *mem = (volatile char*)base;
	uint32_t size;
	uint8_t bkp0, backups[7];
	int_fast8_t i;
	uint_fast8_t ret;


	bkp0 = mem[0];
	mem[0] = 0xbb;
	for (i = sizeof(backups) - 1, size = (1 << 17) << (sizeof(backups) - 1); i >= 0; i--, size >>= 1) {
		backups[i] = mem[size];
		mem[size] = 0xA0 + i;
	}
	//write something at an even lower boundary to make sure we do not misdetect something smaller as our smallest size
	mem[size] = 0xCC;
	ret = mem[0];
	mem[0] = bkp0;
	for (i = sizeof(backups) - 1, size = (1 << 17) << (sizeof(backups) - 1); i >= 0; i--, size >>= 1) {
		mem[size] = backups[i];
	}
	if (ret == 0xbb)	//no aliasing but write sticks = 16Mbyte
		return 128;

	if ((ret >> 4) != 0x0A)
		return 0;

	return 1 << (ret - 0xA0);
}

static void probeRams(void)
{
	uint32_t base = 0x80000000, spacing = 0x02000000;
	unsigned i;

	//we more or less require the first RAM to fit the kernel (4Mbyte) the second can be any size
	//this behaviour assumes that no bus faults are taken!
	//we probe each RAM for size anyways

	for (i = 0; i < 2; i++, base += spacing) {

		pr("RAM %u @ 0x%08x...", i, base);

		mRamSizes[i] = getRamSize(base);
		pr(" %u KB\r\n", 128 * mRamSizes[i]);
	}
}


static uint32_t promGetMemBitmap(struct DecMemBitmap *out)
{
	unsigned i, j;


	//linux reads this map weirdly, only considering the bytes that are 0xFF, so we just set up 0xFF to mean a 128K
	for (i = 0; i < mRamSizes[0]; i++)
		out->bmp[i] = 0xff;
	for (;i < 0x100;i++)
		out->bmp[i] = 0x00;	//hole
	for (j = 0; j < mRamSizes[1]; j++)
		out->bmp[i++] = 0xff;
	out->pgSz = 1 << 14;

	return i;		//return is in bytes of bitmap
}

static const char* promGetenv(const char *envVar)
{
	if (!strcmp(envVar, "systype"))
		return MY_SYS_TYPE_STR;
	
	fatal("got asked for unknown env type '%s', not sure how to reply", envVar);
}

static int promGetchar(void)
{
	return -1;
}

static uint32_t* promSlotAddr(int32_t slot)
{
	return NULL;	//no turbochannel slots here....
}

static void promClearCache(void)
{
	//nothing
}

static uint32_t promGetSysId(void)
{
	return MY_SYS_TYPE;
}

static const void* promGetTcInfo(void)
{
	return NULL;	//no turbochannel slots here....
}

static void promRex(char action)
{
	if (action == 'b')		//reboot
		((void (*)(void))0xbfc00000)();
	else if (action == 'h') {
		pr("\r\n\r\nHALTED\r\n\r\n");
		while(1);
	}
	else {
		pr("Not sure how to do rex action '%c'\r\n", action);
		while(1);
	}
}

static bool diskReadForFs(void *userData, uint32_t sec, void *dst)
{
	uint32_t startSec = (uint32_t)userData;
	
	return readblock(sec + startSec, v2p(dst));
}

static bool diskReadForPartMgr(void *userData, void *dst, uint32_t sector)
{
	(void)userData;
	
	return readblock(sector, v2p(dst));
}

static const struct DecPromVectors mVecs = {
	.printf = pr,
	.getchar = promGetchar,
	.getenv = promGetenv,
	.slotaddr = promSlotAddr,
	.clearcache = promClearCache,
	.getsysid = promGetSysId,
	.getmembitmap = promGetMemBitmap,
	.gettcinfo = promGetTcInfo,
	.rex = promRex,
};

void __attribute__((noreturn)) start(void)
{
	static uint8_t __attribute__((aligned(512))) NOBSSCLEAR mMiscDiscBuf[VFS_SECTOR_SIZE];	//align guarantees no kilobyte crossings for sector read
	struct VfsFileOpenInfoOpaque foi;
	char fName[VFS_MAX_NAME_LEN + 1];
	uint32_t startSec, numSec;
	struct VfsFile *dir, *fil;
	struct VfsVolume *vol;
	
	
	pr("\rLOADER 1.2.1\r\n");

	probeTlbs();

	if (!partMgrFindBootPartition(diskReadForPartMgr, NULL, mMiscDiscBuf, &startSec, &numSec))
		fatal("no boot partition\r\n");
	
	probeRams();

	pr("Boot partition: %u + %u sec\r\n", startSec, numSec);
	
	vfsInit(mMiscDiscBuf);
	
	vol = vfsMount(diskReadForFs, (void*)startSec);
	if (!vol)
		fatal("mount failed\r\n");
	
	dir = vfsOpenRoot(vol);
	if (!dir)
		fatal("failed to open root dir\r\n");

	while (vfsDirEnum(dir, fName, &foi)) {
		
		uint32_t fileSize;
		uint8_t type;
		
		pr(" > '%s'\r\n", fName);
		
		if (!isValidKernelName(fName) || !vfsFileInfo(vol, &foi, &type, &fileSize) || type != VFS_TYPE_FILE || fileSize < (1 << 20))
			continue;
		
		fil = vfsFileOpen(vol, &foi);
		if (!fil)
			pr("cannot open file\r\n");
		else {
			
			loadOS(fil, fileSize, &mVecs);
			pr("failed to load this kernel\r\n");
			vfsFileClose(fil);
		}
	}
	vfsFileClose(dir);
	
	fatal("No loadable kernels found\r\n");
}
