require "rexml/document"

###################################################	import code

def decode_string(str)
	
	r = []
	
	while str.length > 0 do
		
		r << str[0..1].to_i(16)
		str = str[2..-1]
	end
	
	r.pack("c*")
end

def importXML(file)

	$stderr.puts "\tReading...\n"
	xml = REXML::Document.new File.read file
	
	items = []
	smsi = 0
	mmsi = 0
	smso = 0
	mmso = 0
	
	$stderr.puts "\tParsing...\n"
	xml.elements.each("messages/item") do |item|
		
		message = {}
		dir_in = FALSE
		
		message[:type] = item.attributes["type"]
		message[:address] = item.attributes["address"]
		message[:time] = item.attributes["time"]
		message[:direction] = item.attributes["direction"]
		
		if message[:direction] == "in" then
			dir_in = TRUE
		elsif message[:direction] != "out" then
			abort "Unknown message direction \"#{message[:direction]}\"\n"
		end
		
		if message[:type] == "sms" then 
			if dir_in then
				smsi = smsi + 1
			else
				smso = smso + 1
			end
			
			message[:body] = decode_string item.attributes["body"]
			
		elsif message[:type] == "mms" then
			if dir_in then
				mmsi = mmsi + 1
			else
				mmso = mmso + 1
			end
			
			message[:subject] = decode_string item.attributes["subject"]
			message[:files] = []
			
			item.elements.each("file") do |attachement|
				
				file = {}
				
				file[:name] = attachement.attributes["name"]
				file[:mime] = attachement.attributes["mime"]
				file[:data] = attachement.attributes["data"]
				
				message[:files] << file
			end
		else
			abort "Unknown message type \"#{message[:type]}\"\n"
		end
		
		items << message
	end
	
	return [items, smsi, mmsi, smso,mmso]
end

###################################################	processing code (incl generating groups)

def addr2country(addr)
	
	#emails are in the us
	if addr.include? "@" then return "us" end
	
	#non-plus numbers are shortSMS numbers in the us
	if addr[0..0] != "+" then return "us" end
	
	# +1 are us numbers
	if addr[1..1] == "1" then return "us" end
	
	# +7 are russia numbers
	if addr[1..1] == "7" then return "ru" end
	
	# +33 are france numbers
	if addr[1..2] == "33" then return "fr" end
	
	# +34 are spain numbers
	if addr[1..2] == "34" then return "es" end
	
	# +49 are germany numbers
	if addr[1..2] == "49" then return "de" end
	
	# +91 are india numbers
	if addr[1..2] == "91" then return "in" end
	
	# +972 are israel numbers
	if addr[1..3] == "972" then return "il" end
	
	# +380 are ukraine numbers
	if addr[1..3] == "380" then return "ua" end
	
	abort "DO NOT KNOW WHAT COUNTRY THE NUMBER '" + addr + "' belongs to."

end

def addr2hash(addr)
	
	hashable = if addr.include? "@" or addr.length < 4 then addr else addr[-4..-1] end
	
	hash = 0x811C9DC5
	
	hashable.each_byte do |b|
		
		hash = ((hash ^ b) * 0x1000193) & 0xFFFFFFFF
	end
	
	if hash >= 0x80000000 then
		
		hash = ("0x" + hash.to_s(16)).to_i(16) - 0x100000000
		
	end

	return hash
end

def generateSmil(files)
	
	smil = ""
	cid = 100
	layout = ""
	data = ""
	
	files.each_index do |i|
		
		if files[i][:mime].include?("image/jpeg") or files[i][:mime].include?("image/jpg") or files[i][:mime].include?("image/gif") then
			
			layout = layout + "      <region fit=\"meet\" height=\"70%\" id=\"Image#{cid}\" left=\"0%\" top=\"0%\" width=\"100%\"/>\n"
			data = data + "      <img region=\"Image#{cid}\" src=\"cid:#{cid}\"/>\n"
			
		elsif files[i][:mime].include?("text/plain") then
			
			layout = layout + "      <region fit=\"scroll\" height=\"30%\" id=\"Text#{cid}\" left=\"0%\" top=\"70%\" width=\"100%\"/>\n"
			data = data + "      <text region=\"Text#{cid}\" src=\"cid:#{cid}\"/>\n"
		elsif files[i][:mime].include?("audio/amr") then
			
			layout = layout + "      <region fit=\"scroll\" height=\"30%\" id=\"Audio#{cid}\" left=\"0%\" top=\"70%\" width=\"100%\"/>\n"
			data = data + "      <audio region=\"Audio#{cid}\" src=\"cid:#{cid}\"/>\n"
		else
			
			abort "file \"#{files[i][:name]}\" has mimetype \"#{files[i][:mime]}\" which is strange\n"
		end
		
		files[i][:cid] = cid
		cid = cid + 1
	end
	
	smil = smil + "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<smil>\n  <head>\n    <layout>\n      <root-layout height=\"320\" width=\"240\"/>\n"
	smil = smil + layout
	smil = smil + "    </layout>\n  </head>\n  <body>\n    <par dur=\"8000ms\">\n"
	smil = smil + data
	smil = smil + "    </par>\n  </body>\n</smil>\n"
	
	return [smil, files]
end

def process(posts)
	
	posts.sort! {|a, b| a[:time] <=> b[:time]}
	groups = []
	
	posts.each_index do |i|
		
		found = FALSE
		
		groups.each_index do |gi|
			
			if groups[gi][:address].casecmp(posts[i][:address]) == 0 then
				found = TRUE
				
				if posts[groups[gi][:latest]][:time] < posts[i][:time] then
					groups[gi][:latest] = i
				end
				
				posts[i][:group] = gi
			end
		end
		
		if !found then
			
			posts[i][:group] = groups.length
			groups << {:address => posts[i][:address], :latest=>i, :hash=>addr2hash(posts[i][:address]), :country=>addr2country(posts[i][:address])}
		end
		
		if posts[i][:type] == "mms" then
			
			posts[i][:smil], posts[i][:files] = generateSmil(posts[i][:files])
		end
	end
	
	[posts, groups]
end

###################################################	export code

def quoted(str)
	
	str.gsub(/'/, "''")
end

def hexencode(str)
	
	r = ""
	
	str.each_byte do |i|
		
		r = r + sprintf("%02x", i)
	end
	
	return r.upcase
end

def insertPiece(piece_id, msg_id, piece_idx, preview_id, data, mime, name, cid)
	
	print "INSERT INTO \"msg_pieces\" VALUES(#{piece_id},#{msg_id + 1},X'#{data}',#{piece_idx},#{preview_id},'#{mime}',0,"
	print "0,0,'<#{cid}>',#{if name then '"'+name+'"' else "NULL" end},NULL);\n"					
end

def exportXML(posts, groups)	#posts are sorted by time now from earliest to latest, groups are sorted somehow...
	
	piece_id = 1
	
	### export groups
	
	groups.each_index do |gi|
		
		g = groups[gi]
		
		print "INSERT INTO \"msg_group\" VALUES(#{gi + 1},0,#{g[:latest] + 1},0,#{g[:hash]});\n"
		print "INSERT INTO \"group_member\" VALUES(#{gi + 1},#{gi + 1},'#{g[:address]}','#{g[:country]}');\n"
	end
	
	##export messages
	
	posts.each_index do |i|
		
		p = posts[i]
		
		if p[:type] == "sms" then
			
			#insert message
			
			print "INSERT INTO \"message\" VALUES(#{i + 1},'#{p[:address]}',#{p[:time]},'#{quoted(p[:body])}',"
			print  "#{if p[:direction] == 'in' then 2 else 3 end},0,NULL,#{p[:group] + 1},0,0,4,0,NULL,"
			print  "'#{groups[p[:group]][:country]}',NULL,NULL,1);\n"
			
		else
			#generate the XML address field
			
			xml = ""
			xml = xml + "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
			xml = xml + "<!DOCTYPE plist PUBLIC \"-//Apple//DTD PLIST 1.0//EN\" \"http://www.apple.com/DTDs/PropertyList-1.0.dtd\">"
			xml = xml + "<plist version=\"1.0\">\n<array>\n\t<string>#{p[:address]}</string>\n</array>\n</plist>\n"
			
			
			#insert message
			
			print "INSERT INTO \"message\" VALUES(#{i + 1},'#{p[:address]}',#{p[:time]},NULL,"
			print  "#{if p[:direction] == 'in' then 2 else 3 end},0,NULL,#{p[:group] + 1},0,0,4,0,'#{quoted(p[:subject])}',"
			print  "'#{groups[p[:group]][:country]}',NULL,X'#{hexencode(xml)}',1);\n"
			
			
			#insert SMIL
			
			insertPiece(piece_id, i, 0, 0, hexencode(p[:smil]), "application/smil", nil, "0000")
			piece_id = piece_id + 1
			
			#insert pieces
			
			p[:files].each_index do |fi|
				
				insertPiece(piece_id, i, fi + 1, -1, p[:files][fi][:data], p[:files][fi][:mime], p[:files][fi][:name], p[:files][fi][:cid])
				piece_id = piece_id + 1
			end
		end
	end
end


###################################################	main code


posts = [];
smsi = 0
mmsi = 0
smso = 0
mmso = 0

ARGV.each do |a|

	$stderr.puts "Parsing input file \"#{a}\"\n"
	newPosts, newSmsi, newMmsi, newSmso, newMmso = importXML(a)

	posts = posts + newPosts
	smsi = smsi + newSmsi
	mmsi = mmsi + newMmsi
	smso = smso + newSmso
	mmso = mmso + newMmso
end

$stderr.puts "Messages data\n"
$stderr.puts "\tSMS in : #{smsi}\n"
$stderr.puts "\tSMS out: #{smso}\n"
$stderr.puts "\tMMS in : #{mmsi}\n"
$stderr.puts "\tMMS out: #{mmso}\n"


$stderr.puts "Processing...\n"
posts, groups = process(posts)

$stderr.puts "Outputting...\n"

#send header

print "CREATE TABLE message (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, address TEXT, date INTEGER, text TEXT, "
print  "flags INTEGER, replace INTEGER, svc_center TEXT, group_id INTEGER, association_id INTEGER, height INTEGER,"
print  " UIFlags INTEGER, version INTEGER, subject TEXT, country TEXT, headers BLOB, recipients BLOB, read INTEGER);\n"
print "CREATE TABLE msg_group (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, type INTEGER, newest_message INTEGER, "
print  "unread_count INTEGER, hash INTEGER);\n"
print "CREATE TABLE group_member (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, group_id INTEGER, address TEXT, country TEXT);\n"
print "CREATE TABLE msg_pieces (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, message_id INTEGER, data BLOB, part_id INTEGER, "
print  "preview_part INTEGER, content_type TEXT, height INTEGER, version INTEGER, flags INTEGER, content_id TEXT, "
print  "content_loc TEXT, headers BLOB);\n"


#send data

exportXML(posts, groups)


#send footer

print "CREATE INDEX message_group_index ON message(group_id, ROWID);\n"
print "CREATE INDEX message_flags_index ON message(flags);\n"
print "CREATE INDEX pieces_message_index ON msg_pieces(message_id);\n"
print "CREATE TRIGGER insert_unread_message AFTER INSERT ON message "
print  "WHEN NOT read(new.flags) BEGIN UPDATE msg_group SET unread_count "
print  "= (SELECT unread_count FROM msg_group WHERE ROWID = new.group_id) "
print  "+ 1 WHERE ROWID = new.group_id; END;\n"
print "CREATE TRIGGER mark_message_unread AFTER UPDATE ON message WHEN "
print  "read(old.flags) AND NOT read(new.flags) BEGIN UPDATE msg_group SET "
print  "unread_count = (SELECT unread_count FROM msg_group WHERE ROWID = "
print  "new.group_id) + 1 WHERE ROWID = new.group_id; END;\n"
print "CREATE TRIGGER mark_message_read AFTER UPDATE ON message WHEN NOT "
print  "read(old.flags) AND read(new.flags) BEGIN UPDATE msg_group SET "
print  "unread_count = (SELECT unread_count FROM msg_group WHERE ROWID = "
print  "new.group_id) - 1 WHERE ROWID = new.group_id; END;\n"
print "CREATE TRIGGER delete_message AFTER DELETE ON message WHEN NOT "
print  "read(old.flags) BEGIN UPDATE msg_group SET unread_count = (SELECT "
print  "unread_count FROM msg_group WHERE ROWID = old.group_id) - 1 WHERE "
print  "ROWID = old.group_id; END;\n"
print "CREATE TRIGGER insert_newest_message AFTER INSERT ON message WHEN "
print  "new.ROWID >= IFNULL((SELECT MAX(ROWID) FROM message WHERE "
print  "message.group_id = new.group_id), 0) BEGIN UPDATE msg_group SET "
print  "newest_message = new.ROWID WHERE ROWID = new.group_id; END;\n"
print "CREATE TRIGGER delete_newest_message AFTER DELETE ON message WHEN "
print  "old.ROWID = (SELECT newest_message FROM msg_group WHERE ROWID = "
print  "old.group_id) BEGIN UPDATE msg_group SET newest_message = (SELECT "
print  "ROWID FROM message WHERE group_id = old.group_id AND ROWID = (SELECT "
print  "max(ROWID) FROM message WHERE group_id = old.group_id)) WHERE "
print  "ROWID = old.group_id; END;\n"
print "CREATE TRIGGER delete_pieces AFTER DELETE ON message BEGIN DELETE from "
print  "msg_pieces where old.ROWID == msg_pieces.message_id; END;\n"