#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>




static const debug_info = 0;

#define MSG(...)	fprintf(stderr, __VA_ARGS__)
#define DBG(...)	if (debug_info) fprintf(stderr, __VA_ARGS__)
#define ERR(...)	do{ fprintf(stderr, __VA_ARGS__); exit(-1); } while(0)
#define NFO(...)	fprintf(stdout, __VA_ARGS__)


#ifndef O_BINARY
#define O_BINARY			0
#endif

#define true				1
#define false				0

#define dmHdrAttrResDB		0x0001
#define	dmHdrAttrBackup		0x0008
#define	dmHdrAttrStream		0x0080


#pragma pack(push,1)

typedef unsigned long UInt32;
typedef unsigned short UInt16;
typedef unsigned char UInt8;
typedef signed long Int32;
typedef signed short Int16;
typedef signed char Int8;
typedef UInt8 Boolean;
typedef UInt8 bool;

typedef struct { 
	UInt32	type; 
	UInt16	id; 
	UInt32	localChunkID_offset; 
} RsrcEntryType;

typedef struct { 
	UInt32	localChunkID_offset;
	UInt8	attributes; 
	UInt8	uniqueID[3];
} RecordEntryType; 

typedef struct { 
	UInt32		nextRecordListID_offset;	//72 - 75
	UInt16		numRecords;					//76 - 77
	
	//RsrcEntryType res[];
	
} RecordListType; 


typedef struct { 
	UInt8	name[32];				//0  - 31 
	UInt16	attributes;				//32 - 33
	UInt16	version;				//34 - 35
	UInt32	creationDate;			//36 - 39
	UInt32	modificationDate;		//40 - 43
	UInt32	lastBackupDate;			//44 - 47
	UInt32	modificationNumber;		//48 - 51
	UInt32	appInfoID_offset;		//52 - 55
	UInt32	sortInfoID_offset;		//56 - 59
	UInt32	type;					//60 - 63
	UInt32	creator;				//64 - 67
	UInt32	uniqueIDSeed;			//68 - 71
	RecordListType	recordList;		//72 - ...
} DatabaseHdrType;

typedef struct{
	
	UInt32 type;
	UInt16 id;
	char* data;
	UInt32 startOffset;
	UInt32 sz;

}resType;



typedef struct{
	
	UInt8 uniqID[3];
	UInt8 attr;
	char* data;
	UInt32 startOffset;
	UInt32 sz;

}recType;


#pragma pack(pop)

#define MAKE_4_CC(h,mh,ml,l) ((h << 0) | (mh << 8) | (ml << 16) | (l << 24))
#define FMT4CC	"%c%c%c%c"
#define PRM4CC(v)	(int)((v >> 0) & 0xFF), (int)((v >> 8) & 0xFF), (int)((v >> 16) & 0xFF), (int)((v >> 24) & 0xFF)



#define DIRECTION_UNKNOWN		0
#define DIRECTION_IN			1
#define DIRECTION_OUT			2

#define TYPE_UNKNOWN			0
#define TYPE_SMS				1
#define TYPE_MMS				2
#define TYPE_FILE				3
#define TYPE_FILE_USED			6


typedef struct MsgType{
	
	time_t	time;
	UInt8	type		: 3;
	UInt8	direction	: 2;
	UInt8	reserved	: 3;
	char*	otherPartyNum;
	char*	otherPartyName;
	UInt32	curMsgID;
	UInt32	groupID;
	int dbMsgID;		//message id in new iphone database
	
	union{
		
		struct{
			
			char*	body;
			
		}sms;
		
		struct{
			
			bool	outbound;		//for MMS, MsgType->direction cannot be trusted, we use this instead
			UInt32	dataMsgID;
			char*	subject;
			
			
			struct MsgType* files;
		}mms;
		
		struct{
			
			UInt32	len;
			UInt8*	data;
			char*	name;
			char*	mimeType;
			
		}file;
	
	}data;
	
}MsgType;


bool processFile(void* ptr_, int size, UInt32 curMsgID, MsgType* m, bool* extraData);


//////////////////////////////////////////////////////// DATABASE PARSING CODE //////////////////////////////////////////////


unsigned long swap32(unsigned long x){
	
	return ((x & 0xFF000000) >> 24) | ((x & 0x00FF0000) >> 8) | ((x & 0x0000FF00) << 8) | ((x & 0x000000FF) << 24);
}

unsigned short swap16(unsigned short x){
	
	return ((x & 0xFF00) >> 8) | ((x & 0x00FF) <<8);
}

void swap24(void* a){		//0xAABBCC -> 0xCCBBAA
	
	char* c=(char*)a;
	char x;
	
	x=c[0];
	c[0]=c[2];
	c[2]=x;
}

long getFileSize(int f){
	
	long oldStreamPos,ret;
	
	oldStreamPos=lseek(f,0,SEEK_CUR);		//save current position
	
	ret=lseek(f,0,SEEK_END);				//seek to the end
	
	lseek(f,oldStreamPos,SEEK_SET);			//seek back to where we were at the beginning

	return ret;								//return file size
}




recType* readPdb(char* name,int* numRecP,char* dbName,UInt32* type,UInt32* crid){
	
	int fromFile;
	DatabaseHdrType hdr;
	int numRec;
	long fileSz;

	recType* rec=NULL;


	fromFile=open(name,O_RDONLY | O_BINARY,0);					//open source file
	
	
	if(fromFile){
		fileSz=getFileSize(fromFile);
		
		read(fromFile,&hdr,sizeof(hdr));//from now on we assume the file is ok...
		
		if(dbName)
			memmove(dbName,&hdr,32);
		
		if(type)
			*type=hdr.type;

		if(crid)
			*crid=hdr.creator;

		if(!hdr.recordList.nextRecordListID_offset){
			
				
			long i,j,k;
			
			numRec=swap16(hdr.recordList.numRecords);
			
			//printf("File has %d records\n",numRec);

			rec=(recType*)malloc(sizeof(recType)*numRec);
			
			
			//read info on resource types,IDs and start offsets

			
			for(i=0;i<numRec;i++){
				
				read(fromFile,&rec[i].startOffset,4);
				read(fromFile,&rec[i].attr,1);
				read(fromFile,&rec[i].uniqID,3);
				
				swap24(&(rec[i].uniqID));
				rec[i].startOffset=swap32(rec[i].startOffset);

				//printf("record #%d (uniqId=%d, attr=%d) is at offset %d\n",i,UInt24toUInt32(rec[i].uniqID),rec[i].attr,rec[i].startOffset);
				
			}
			
			//DBG("dst file open");
			
			//now find sizes
			
			for(i=0;i<numRec;i++){
				
				bool found=false;
				k=-1;

				for(j=0;j<numRec;j++){

					if(rec[j].startOffset>=rec[i].startOffset && (!found || rec[j].startOffset<rec[k].startOffset) && j!=i){
						
						k=j;
						found=true;

					}
				}

				if(k==-1){	//this is the last resource in the file
					
					rec[i].sz=fileSz-rec[i].startOffset;
					
				}
				else{

					rec[i].sz=rec[k].startOffset-rec[i].startOffset;

				}

				
				//printf("record #%d (uniqId=%d, attr=%d, sz=%d)\n",i,UInt24toUInt32(rec[i].uniqID),rec[i].attr,rec[i].sz);
				
			}

			//now read the actual data

			for(i=0;i<numRec;i++){
				
				lseek(fromFile,rec[i].startOffset,SEEK_SET);

				rec[i].data=(char*)malloc(rec[i].sz+1);	//the "+1" is for 0-sized resoruces

				read(fromFile,rec[i].data,rec[i].sz);
			}
			
			//now we're done reading the file
			
			*numRecP=numRec;
			
		}
		else{
			
			ERR("more then one record list in a PRC file is unsupported");
			
		
		}

		close(fromFile);
	}
	else{
		
		ERR("cannot open source file");
		
	}
	
	return rec;
}

//////////////////////////////////////////////////////// output buffering code //////////////////////////////////////////////


void outs(const char* buf){
	
	NFO(buf);
}

void outn(int val){
	
	char num[64];
	
	sprintf(num, "%d", val);
	
	outs(num);
}

void outsq(const char* str){
	
	char* buf = malloc(strlen(str) * 2 + 1);
	char c;
	int i = 0;
	
	if(!buf) ERR("cannot allocate buf to quote a string \"%s\"", str);
	
	do{
		
		c = *str++;
		buf[i++] = c;
		if(c == '\'') buf[i++] = c;
	
	}while(c);
	
	outs(buf);
	
	free(buf);
}

void outh(void* buf_, unsigned long numBytes){
	
	const char* hexBytes = "0123456789ABCDEF";
	UInt8* buf = buf_;
	unsigned long i;
	char x[3];
	
	x[2] = 0;
	
	for(i = 0; i < numBytes; i++){
		
		x[0] = hexBytes[buf[i] >> 4];
		x[1] = hexBytes[buf[i] & 15];
		
		outs(x);
	}
}

void ouths(char* str){
	
	outh(str, strlen(str));
}


//////////////////////////////////////////////////////// MAIN APP CODE //////////////////////////////////////////////


void* alloc(UInt32 sz){
	
	void* p = malloc(sz);
	
	if(!p) ERR("cannot allocate %d bytes\n", sz);
	
	return p;
}

int main(int argc, char** argv){
	
	if(argc < 2){
		
		ERR("USAGE: %s MessagesDatabase1.pdb [MessagesDatabase2.pdb [...]]\n", argv[0]);
	}
	else{
		
		int i,j, k;
		UInt32 type, crid;
		char dbName[32];
		recType* recs;
		int numRec;
		MsgType* msgs;
		int numMsgs;
		bool extraData;
		UInt8* tmp;
		MsgType tm;
		
		MsgType* allMsgs = NULL;
		int numAllMsgs = 0;
		
		MsgType** groupNewestMsg = NULL;
		char** groups = NULL;
		int numGroups = 0;
		
		while(argc >= 2){
			
			numMsgs = 0;
			
			MSG("Operating on \"%s\"\n", argv[1]);
			
			recs = readPdb(argv[1], &numRec, dbName, &type, &crid);
			
			argv++;
			argc--;
			
			if(!recs) ERR("Cannot read PDB file\n");
			
			MSG("Database is \"%s\" {'" FMT4CC "', '" FMT4CC "'} and has %d records\n", dbName, PRM4CC(type), PRM4CC(crid), numRec);
			
			msgs = alloc(sizeof(MsgType) * numRec);
			
			//writeFiles(recs, numRec, "FF");
			
			MSG("Processing records\n");
			for(i = 0; i < numRec; i++){
				
				extraData = false;
				
				
				DBG("Processing file 0x%lx/0x%lx (%d bytes long) [%02x%02x%02x]\n", i, numRec, recs[i].sz, (int)recs[i].uniqID[2], (int)recs[i].uniqID[1], (int)recs[i].uniqID[0]);
				if(processFile(recs[i].data, recs[i].sz, (((UInt32)recs[i].uniqID[2]) << 16) | (((UInt32)recs[i].uniqID[1]) << 8) | ((UInt32)recs[i].uniqID[0]), msgs + numMsgs, &extraData)){
					
		//			if(msgs[numMsgs].type == TYPE_SMS) continue;
					
					if(extraData){
						
						if(numMsgs == 0 || msgs[numMsgs - 1].type != TYPE_FILE) ERR("cannot have extradata here");
						tmp = realloc(msgs[numMsgs - 1].data.file.data, msgs[numMsgs - 1].data.file.len + msgs[numMsgs].data.file.len);
						if(!tmp) ERR("cannot resize file for extradata");
						msgs[numMsgs - 1].data.file.data = tmp;
						memcpy(tmp + msgs[numMsgs - 1].data.file.len, msgs[numMsgs].data.file.data, msgs[numMsgs].data.file.len);
						msgs[numMsgs - 1].data.file.len += msgs[numMsgs].data.file.len;
					}
					else{
						
						numMsgs++;
					}
				}
			}
			
			
			MSG("%d messages created from %d records\n", numMsgs, numRec);
			MSG("Reassembling MMS data collections\n");
		
			//attach files to their MMSs
			{
				for(i = 0; i < numMsgs; i++){
					
					if(msgs[i].type == TYPE_MMS){
						
						msgs[i].direction = msgs[i].data.mms.outbound ? DIRECTION_OUT : DIRECTION_IN;
						
						j = i + 1;
						
						DBG("Searching for %x from %x\n", msgs[i].data.mms.dataMsgID, msgs[i].curMsgID);
						
						if(msgs[i].data.mms.dataMsgID){
							
							while(j < numMsgs && msgs[j].curMsgID < msgs[i].data.mms.dataMsgID && msgs[j].type == TYPE_FILE){
								
								DBG("msgs[j].curMsgID=%x msgs[i].data.mms.dataMsgID=%x\n", msgs[j].curMsgID, msgs[i].data.mms.dataMsgID);
								j++;
							}
							
							if(j == numMsgs) ERR("Cannot find last msg %x for mms %x\n", msgs[i].data.mms.dataMsgID, msgs[i].curMsgID);
							if(msgs[j].type != TYPE_FILE) ERR("Non-file(%d) found in data chain %x for mms %x\n", msgs[j].type, msgs[j].curMsgID, msgs[i].curMsgID);
						}
						else{
							
							j = i;
						}
						
						msgs[i].data.mms.files = alloc(sizeof(MsgType) * (j + 1 - i));
						//DBG("> allocated %d entries for files for [%x]\n", j + 1 - i, msgs[i].curMsgID);
		
						msgs[i].data.mms.files[j - i].type = TYPE_UNKNOWN;
						//DBG("> file[%d].type=unnown\n", j - i);
						
						while(j > i){
							
							DBG("Data file for SMS %x at %x named \"%s\"\n", msgs[i].curMsgID, msgs[j].curMsgID, msgs[j].data.file.name);
							
							//DBG("> file[%d].type=%d, copied from [%x]\n", j - i - 1, msgs[j].type, msgs[j].curMsgID);
							
							msgs[j].type = TYPE_FILE_USED;
							msgs[i].data.mms.files[j - i -1] = msgs[j];
							
							j--;
						}
					}
				}
			}
			
			allMsgs = realloc(allMsgs, sizeof(MsgType) * (numAllMsgs + numMsgs));
			if(!allMsgs) ERR("cannot resize all messages buffer");
			
			memcpy(allMsgs + numAllMsgs, msgs, numMsgs * sizeof(MsgType));
			numAllMsgs += numMsgs;
		}
		
		msgs = allMsgs;
		numMsgs = numAllMsgs;
		
		//count some statistics
		{
			UInt32 numSmsIn = 0, numSmsOut = 0, numMmsIn = 0, numMmsOut = 0, numFiles = 0;
			unsigned long long fileSizes = 0;
			
			for(i = 0; i < numMsgs; i++){
				
				if(msgs[i].type == TYPE_FILE || msgs[i].type == TYPE_FILE_USED){
					
					numFiles++;
					fileSizes += msgs[i].data.file.len;
				}
				else if(msgs[i].type == TYPE_SMS){
					
					if(msgs[i].direction == DIRECTION_IN) numSmsIn++;
					else if(msgs[i].direction == DIRECTION_OUT) numSmsOut++;
					else ERR("strange direction %d for sms %x\n",msgs[i].direction, msgs[i].curMsgID);
				}
				else if(msgs[i].type == TYPE_MMS){
					
					if(msgs[i].direction == DIRECTION_IN) numMmsIn++;
					else if(msgs[i].direction == DIRECTION_OUT) numMmsOut++;
					else ERR("strange direction %d for mms %x\n",msgs[i].direction, msgs[i].curMsgID);
				}
				else ERR("strange type %d for msg [%x]\n", msgs[i].type, msgs[i].curMsgID);
			}
			
			MSG("Stats:\n\t%d SMS in\n\t%d SMS out\n\t%d MMS in\n\t%d MMS out\n\t%d files with total size %d average size %d bytes\n",
				numSmsIn, numSmsOut, numMmsIn, numMmsOut, numFiles,  (UInt32)fileSizes, numFiles ? ((UInt32)(fileSizes / numFiles)) : 0);
		}
		
		//remove files form main message list
		{
			
			i = 0;
			
			while(i < numMsgs){
				
				if(msgs[i].type == TYPE_FILE_USED || msgs[i].type == TYPE_FILE){
					
					
					if(msgs[i].type == TYPE_FILE){
						
						DBG("unexpected unattached file found at %x named \"%s\"\n", msgs[i].curMsgID, msgs[i].data.file.name);
					}
					
					numMsgs--;
					for(j = i; j < numMsgs; j++){
						
						msgs[j] = msgs[j + 1];
					}
					
					i--;	//counter the upcoming i++
				}
				else if(msgs[i].type != TYPE_SMS && msgs[i].type != TYPE_MMS){
					
					ERR("unexpected type %d record found at %x named \"%s\"\n", msgs[i].type, msgs[i].curMsgID, msgs[i].data.file.name);
				}
				
				i++;
			}
			
			MSG("%d message records left\n", numMsgs);
		}
		
		//sort messages by timestamp
		{
			
			for(i = 0; i < numMsgs; i++){		//selection sort
				
				if(msgs[i].time == 0){
					
					ERR("untimed message [%s] found\n", msgs[i].curMsgID);
				}
				
				k = i;
				
				for(j = i + 1; j < numMsgs; j++){
					
					if(msgs[j].time < msgs[k].time){
						
						k = j;
					}
				}
				
				tm = msgs[i];
				msgs[i] = msgs[k];
				msgs[k] = tm;
			}
		}
		
		//canonicalaize addresses
		{
			for(i = 0; i < numMsgs; i++){		//selection sort
				
				if(!msgs[i].otherPartyNum) ERR("message without number at [%x]\n", msgs[i].curMsgID);
				
					
				//remove all non-phone-numeric parts, remove names from emails
				{
					char* from = msgs[i].otherPartyNum;
					char* to = msgs[i].otherPartyNum;
					char c;
					
					DBG("PNUM[%x]='%s'\n", msgs[i].curMsgID, from);
					
					if(!strchr(msgs[i].otherPartyNum,'@')){		//allow emails
						
						//strip non-numeric things
						do{
							
							c = *from++;
							if((c >= '0' && c <= '9') || c == '+') *to++ = c;
							
						}while(c);
						
						*to = 0;
					}
					
					if(strchr(msgs[i].otherPartyNum,'@')){
						
						char* open = strchr(msgs[i].otherPartyNum,'<');
						int bad = 1;
						
						if(open){
							
							char* close = strchr(open,'>');
							
							if(close){
								
								msgs[i].otherPartyNum = open + 1;
								*close = NULL;
								bad = 0;
							}
						}
						else if(!strchr(msgs[i].otherPartyNum,'>')){
							
							bad = 0;
						}
						
						if(bad){
							
							ERR("invalid email address <<%s>> for msg [%x]\n", msgs[i].otherPartyNum, msgs[i].curMsgID);
						}
					}
					else if(strlen(msgs[i].otherPartyNum) >= 12 && msgs[i].otherPartyNum[0] == '+'){
						
						//ok number
					}
					else if(msgs[i].otherPartyNum[0] == '0' && msgs[i].otherPartyNum[1] == '1' && msgs[i].otherPartyNum[2] == '1'){
						
						char* canonicalNumber = malloc(16);
						
						sprintf(canonicalNumber, "+%s", msgs[i].otherPartyNum + 3);
						
						msgs[i].otherPartyNum = canonicalNumber;
					}
					else if(strchr(msgs[i].otherPartyNum,'+')){
						
						ERR("Too short of a string to be a PLUS-number \"%s\" in msg [%x]\n", msgs[i].otherPartyNum, msgs[i].curMsgID);
					}
					else if(strlen(msgs[i].otherPartyNum) == 11){
						
						char* canonicalNumber = malloc(16);
						
						sprintf(canonicalNumber, "+%s", msgs[i].otherPartyNum);
						
						msgs[i].otherPartyNum = canonicalNumber;
					}
					else if(strlen(msgs[i].otherPartyNum) == 10){
						
						char* canonicalNumber = malloc(16);
						
						sprintf(canonicalNumber, "+1%s", msgs[i].otherPartyNum);
						
						msgs[i].otherPartyNum = canonicalNumber;
					}
					else if(strlen(msgs[i].otherPartyNum) == 7){
						
						char* canonicalNumber = malloc(16);
						
						sprintf(canonicalNumber, "+1847%s", msgs[i].otherPartyNum);
						
						msgs[i].otherPartyNum = canonicalNumber;
					}
					else if((strlen(msgs[i].otherPartyNum) <= 6) && (strlen(msgs[i].otherPartyNum) >= 3)){
						
						DBG("%d-digit number '%s' in msg [%x] considered valid\n", strlen(msgs[i].otherPartyNum), msgs[i].otherPartyNum, msgs[i].curMsgID);
					}
					else{
						
						DBG("Too short of a string to be a number \"%s\" in msg [%x], deleting msg\n", msgs[i].otherPartyNum, msgs[i].curMsgID);
						
						numMsgs--;
						for(j = i; j < numMsgs; j++){
							
							msgs[j] = msgs[j + 1];
						}
						
						i--;	//counter the upcoming i++
					}
					
					DBG("NNUM[%x]='%s'\n", msgs[i].curMsgID, msgs[i].otherPartyNum);
					
				}
				
			}
		}
		
		//export XML
		{
			
			outs("<messages>\n");
			
			for(i = 0; i < numMsgs; i++){		//selection sort
				
				for(i = 0; i < numMsgs; i++){
					
					if(msgs[i].type == TYPE_SMS){
					
						outs("\t<item type=\"sms\" address=\"");
						outs(msgs[i].otherPartyNum);
						outs("\" time=\"");
						outn(msgs[i].time);
						outs("\" direction=\"");
						outs(msgs[i].direction == DIRECTION_IN ? "in" : "out");
						outs("\" body=\"");
						ouths(msgs[i].data.sms.body);
						outs("\" />\n");
					}
					else if(msgs[i].type == TYPE_MMS){
						
						outs("\t<item type=\"mms\" address=\"");
						outs(msgs[i].otherPartyNum);
						outs("\" time=\"");
						outn(msgs[i].time);
						outs("\" direction=\"");
						outs(msgs[i].direction == DIRECTION_IN ? "in" : "out");
						outs("\" subject=\"");
						ouths(msgs[i].data.mms.subject ? msgs[i].data.mms.subject : "--");
						outs("\">\n");
						{
							MsgType* m;
							
							m = &msgs[i].data.mms.files[0];
							
							while(m->type != TYPE_UNKNOWN){
								
								outs("\t\t<file name=\"");
								outs(m->data.file.name);
								outs("\" mime=\"");
								outs(m->data.file.mimeType);
								outs("\" data=\"");
								outh(m->data.file.data, m->data.file.len);
								outs("\" />\n");
								
								m++;
							}
						}
						outs("\t</item>\n");
					}
					else{
						
						ERR("strange message type");
					}
				}
			}
			
			outs("</messages>");
		}
	}
}



//////////////////////////////////////////////////////// MESSAGE PARSING CODE //////////////////////////////////////////////



#define time_diff (2082853800+5400-8*60*60) //64 years (1970-1904) +1,5 hour 

static const int gmtoffset = 8*60*60; // TODO.  this is right for EDT



UInt16 read16(UInt8* buf){
	
	return (((UInt32)buf[0]) << 8) | (((UInt32)buf[1]) << 0);
}

UInt32 read32(UInt8* buf){
	
	return (((UInt32)buf[0]) << 24) | (((UInt32)buf[1]) << 16) | (((UInt32)buf[2]) << 8) | (((UInt32)buf[3]) << 0);
}



#define ATTR_FLAG_READONLY	0x8000
#define ATTR_MASK_TYPE		0x6000

#define ATTR_TYPE_UINT32	0x0000		//always 4 bytes
#define ATTR_TYPE_CSTRING	0x2000		//includes length field
#define ATTR_TYPE_BINARY	0x4000		//includes length field

#define ATTR_MASK_IDX		0x0FFF		//XXX: should be 0x1FFF



UInt16 getAttributeTypeBits(UInt16 tag){
	
	return tag & 0xE000;
}

void setMsgType(MsgType* t, UInt8 mst){
	
	if(t->type != TYPE_UNKNOWN && t->type != mst){
		
		ERR("Trying to set message type %d from %d\n", mst, t->type);
	}
	
	t->type = mst;
}

bool processFile(void* ptr_, int size, UInt32 curMsgID, MsgType* m, bool* extraData){	//return if a msg was created
	
	UInt8* ptr = ptr_;
	UInt8* data;
	UInt16 type, len;
	UInt32 i, j;
	UInt8* tmp;
	bool ret = true;
	bool hasdata = false;
	
	i = 4;
	
	memset(m ,0, sizeof(MsgType));
	m->curMsgID = curMsgID;
	
	DBG("CUR MSG ID: %x\n", curMsgID);
	
	while(i + 1 < size){
		
		type = read16(ptr + i);
		i += 2;
		
		if((getAttributeTypeBits(type) & ATTR_MASK_TYPE) == ATTR_TYPE_UINT32){
			
			UInt32 val = read32(ptr + i);
			
			switch(type){
				
				case 0x0001:
					
					if(val == 0 || val == 1){
						DBG("VAL = %d\n", val);
						ret = false;
						goto out;
					}
					else if(val != 2 & val != 7){
						DBG("VAL = %d\n", val);
					}
					break;
				
				case 0x0006:
				case 0x0007:
				case 0x8000:
					
					m->time = val - time_diff - gmtoffset;
					
					break;
			
				default:;
					
		//			ERR("unknown UInt32 = 0x%08lx (type 0x%04x)\n", val, type);
			}
			
			i += 4;
		}
		else if((getAttributeTypeBits(type) & ATTR_MASK_TYPE) == ATTR_TYPE_CSTRING){
			
			len = read16(ptr + i);
			i += 2;
			data = ptr + i;
			
			switch(type){
				
				case 0x2000:
					
					DBG("MMS send URL: <<%s>>\n", data);
					setMsgType(m, TYPE_MMS);
					m->data.mms.outbound = true;
					break;
				
				case 0x2001:
					
					DBG("Folder: <<%s>>\n", data);
					setMsgType(m, TYPE_FILE);
					break;
				
				case 0x2002:
					
					setMsgType(m, TYPE_SMS);
					DBG("Body: <<%s>>\n", data);
					m->data.sms.body = data;
					break;
				
				case 0x2005:
					
					if(m->type == TYPE_UNKNOWN){
						
						DBG("MMS download URL: <<%s>>\n", data);
						m->data.mms.outbound = false;
						setMsgType(m, TYPE_MMS);
					}
					break;
				
				case 0x2006:
					
					DBG("Subject: <<%s>>\n", data);
					setMsgType(m, TYPE_MMS);
					m->data.mms.subject = data;
					break;
				
				case 0x2008:
					
					DBG("MIME type: <<%s>>\n", data);
					setMsgType(m, TYPE_FILE);
					m->data.file.mimeType = data;
					break;
				
				case 0x200e:
				case 0x4000:
					
					DBG("Error msg: <<%s>>\n", data);
					ret = false;
					goto out;								//no errors in output please
					break;
				
				case 0x2009:
				case 0x3000:
					
					DBG("File name: <<%s>>\n", data);
					setMsgType(m, TYPE_FILE);
					m->data.file.name = data;
					break;
				
				case 0x3002:
					
					DBG("MMS ID: <<%s>>\n", data);
					setMsgType(m, TYPE_MMS);
					break;
				
				case 0x3003:
					
					DBG("MMS Set ID: <<%s>>\n", data);
					setMsgType(m, TYPE_MMS);
					break;
				
				case 0x3004:
					
					DBG("STRANGE IMAGE NUMBER: <<%s>>\n", data);
					break;
				
				case 0x3005:
					
					DBG("SMS gateway: <<%s>>\n", data);
					break;
				
				case 0x3006:
					
					DBG("SMS center: <<%s>>\n", data);
					break;
				
				default:;
				
					ERR("unknown STR(@0x%08lx, %d b) = <<%s>>  (type 0x%04x)\n", i, len, data, type);
			}
			
			i += len;
		}
		else if((getAttributeTypeBits(type) & ATTR_MASK_TYPE) == ATTR_TYPE_BINARY){
			
			len = read16(ptr + i);
			i += 2;
			data = ptr + i;
			
			switch(type){
				
				case 0x4006:
					
					m->data.mms.dataMsgID = read32(data + len - 4);
					
					DBG(" MMS info found, %d bytes big\n", len);
					DBG(" MMS data at records 0x%06x<-0x%06x\n", m->data.mms.dataMsgID, curMsgID);
					
					break;
				
				case 0x4007:
					
					DBG(" Data chunk @ 0x%08lx, %d bytes big\n", i, len);
					tmp = realloc(m->data.file.data, m->data.file.len + len);
					if(!tmp) ERR("Cannot resize chunk for file\n");
					memcpy(tmp + m->data.file.len, data, len);
					m->data.file.len += len;
					m->data.file.data = tmp;
					hasdata = true;
					break;
				
				case 0x4009:
				case 0x400C:
					
					DBG("msg dir: %s\n",  (type == 0x4009) ? "out" : "in");
					
					m->direction =  (type == 0x4009) ? DIRECTION_OUT : DIRECTION_IN;
					
					j = 0x14;		//XXX: was 0
					
					for(; j < len && !data[j]; j++);
					
					if(j < len){
						
						m->otherPartyNum = data + j;
						DBG(" NUM: <<%s>> (@0x%x = 0x%x + 0x%x)\n", m->otherPartyNum, j + i, i, j);
						
						j += strlen(data + j) + 1;
						
						if(j < len){
							
							m->otherPartyName = data + j;
							DBG(" NAME: <<%s>> (@0x%x = 0x%x + 0x%x)\n", m->otherPartyName, j + i, i , j);
						}
					}
					
					break;
			
				default:;
					
	//				ERR("unknown BIN(@0x%08lx, %d b)  (type 0x%04x)\n", i, len, type);
			}
			i += len;
		}
		else{
			
			ERR("WTF\n");
		}
	}
	
	if(m->type == TYPE_UNKNOWN){
	
		if(hasdata){
			
			*extraData = true;
		}
		else{
			
			DBG("discarding msg %x due to lack of type\n", curMsgID);
			ret = false;
		}
	}
	
	if(m->direction == DIRECTION_UNKNOWN && m->type != TYPE_FILE){
		DBG("discarding msg %x due to lack of direction\n", curMsgID);
		ret = false;
	}
	
out:
	
	DBG("  => %s (%d)\n", ret ? "true" : "false", m->type);
	
	return ret;
}


/*

CREATE TABLE message (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, address TEXT, date INTEGER, text TEXT, flags INTEGER, replace INTEGER, svc_center TEXT, group_id INTEGER, association_id INTEGER, height INTEGER, UIFlags INTEGER, version INTEGER);
CREATE TABLE msg_group (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, type INTEGER, newest_message INTEGER, unread_count INTEGER);
CREATE TABLE group_member (ROWID INTEGER PRIMARY KEY AUTOINCREMENT, group_id INTEGER, address TEXT);


*/