/*
 * sender.c -- sender main routines
 *
 * This file is part of
 * 
 * rmdp -- Reliable Multicast data Distribution Protocol
 * 
 * (C) 1996, 1997 Luigi Rizzo and Lorenzo Vicisano
 *     (luigi@iet.unipi.it, vicisano@cs.ucl.ac.uk)
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Luigi Rizzo,
 *      Lorenzo Vicisano and other contributors.
 * 4. Neither the name of the Authors nor the names of other contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

#include "rmdp.h"
#include "sessions.h"

ui32 dumm_addr;
ui16 dumm_port;
int k_glob = 32,
    n_glob = 256;
int ttl_glob = TTL;
int data_pkt_len = DATA_SIZE;
EVLIST *events;	/* callback attach. and timers */
ui32 source_id;

void *no_ar;

/* 
 * various auxiliary functions **
 */

void
badcmdline(char *arg)
{
    fprintf(stderr,
	"Bad argument %s\nCommand line:\n\t rmdpd [options]\n"
	"options:\n"
	"\t-d <M-cast data address/port>\n"
	"\t-c <M-cast control address/port>\n"
	"\t-k <data packets per block>\n"
	"\t-n <total (data+redundancy) packets per block>\n"
	"\t-l <packet size (bytes)>\n"
	"\t-t <sender ttl>\n", arg);
    exit(1);
}

/*** I must find a better place for these two... ***/
n32	reply_addr;
n16	reply_port;

int
handle_v1req(u_char *rq, int len, u_char *resp)
{
    u_char *p = rq, *tx = resp;
    char *thepath = NULL ;
    n32 s_id, c_id, r_id, r_addr, m_addr ;
    n16 m_port, r_port ;
    ui32 req_rate = 0;
    int unicast_reply = 0 ;

    RMDP_TAG type = 0;

    c_id = 0;
    r_id = 0;
    s_id = 0;
    r_addr = 0;
    r_port = 0;
    m_addr = 0;
    m_port = 0;

    if (!check_magic(rq, len))
	return 0;
    p = rq + 4;
    len -= 4;
    fprintf(stderr,"REQ: ");
    while (len > 0) {
	u_char opt = *p;
	i16 l = getoptlen(p, len) ;
	if (l == 0) return 0;
	len -= l;

	switch(opt) {
	case RMDP_T_PAD :
	    fprintf(stderr, " pad,");
	    break;
	case RMDP_T_END :
	    fprintf(stderr, " <end>");
	    len = 0;
	    break ;
	case RMDP_T_NEW_REQ :	/* a new request */
	    type = opt;
	    fprintf(stderr, " new,");
	    break;
	case RMDP_T_R_CONT :
	    type = opt;
	    fprintf(stderr, " continue,");
	    break;
	case RMDP_T_R_RESUME :
	    type = opt;
	    fprintf(stderr, " resume,");
	    break;
	case RMDP_T_R_UNICAST :
	    unicast_reply = 1;
	    fprintf(stderr, " unicast,");
	    break;
	case RMDP_T_C_ID :
	    bcopy(p+2, &c_id , 4);
	    fprintf(stderr, " c_id 0x%08x,", c_id);
	    break;
	case RMDP_T_R_ID :
	    bcopy(p+2, &r_id , 4);
	    fprintf(stderr, " r_id 0x%08x,", r_id);
	    break;
	case RMDP_T_S_ID :
	    bcopy(p+2, &s_id , 4);
	    fprintf(stderr, " s_id 0x%08x", s_id);
	    break;
	case RMDP_T_R_RATE :
	    bcopy(p+2, &req_rate , 4);
	    req_rate = ntohl(req_rate);
	    fprintf(stderr, " rate %d b/s,", req_rate);
	    break;
	case RMDP_T_R_ADDR :
	    bcopy(p+2, &r_addr , 4);
	    fprintf(stderr, "\n\treply addr %s,",
		inet_ntoa(*(struct in_addr *)&r_addr) );
	    break;
	case RMDP_T_R_PORT :
	    bcopy(p+2, &r_port , 2);
	    fprintf(stderr, " reply port %d,", ntohs(r_port)  );
	    break;
	case RMDP_T_R_FILE :
	    thepath = p+3;
	    fprintf(stderr, "\n\tfile \"%s\",", thepath);
	    break;
	default :
	    fprintf(stderr, "\noption 0x%02x len %d unknown\n", opt, l);
	    break ;
	}
	p += l;
    }
    fprintf(stderr, "\n");
    switch (type) {
    case RMDP_T_NEW_REQ : /* need file, r_id, c_id, r_addr, r_port */
	if (!(thepath && r_id && c_id && r_addr && r_port)) {
	    fprintf(stderr, "missing parameters... \n");
	    if (thepath == NULL) fprintf(stderr,"no path\n");
	    if (r_id == 0) fprintf(stderr,"no r_id\n");
	    if (c_id == 0) fprintf(stderr,"no c_id\n");
	    if (r_addr == 0) fprintf(stderr,"no r_addr\n");
	    if (r_port == 0) fprintf(stderr,"no r_port\n");
	    return 0;
	}
	if (unicast_reply) {
	    m_addr = r_addr;
	    m_port = r_port;
	}
	/***
	 *** check if the path is acceptable, otherwise do not even respond
	 ***/
	thepath = validpath(thepath);
	if (thepath == NULL) {
	    fprintf(stderr,"+++ validpath %s returns null\n", thepath);
	    return 0 ;
	}
	schedule(thepath, c_id, r_id, &req_rate, &s_id, &m_addr, &m_port);
	tx = add_magic(resp);
	tx = add_opt1(tx, RMDP_T_RESP);
	tx = add_opt4(tx, RMDP_T_C_ID, c_id);
	tx = add_opt4(tx, RMDP_T_R_ID, r_id);
	if (s_id == 0) { /* file not found ... */
	    char buf[256];
	    sprintf(buf, "xx %s not found xx", thepath) ;
	    tx = add_opts( tx, RMDP_T_ERRMSG, buf );
	} else {
	    tx = add_opt4(tx, RMDP_T_S_ID, s_id);
	    tx = add_opt4(tx, RMDP_T_D_ADDR, m_addr);
	    tx = add_opt2(tx, RMDP_T_D_PORT, m_port);
	    tx = add_opt4(tx, RMDP_T_R_RATE, htonl(req_rate));
	}
	tx = add_opt1(tx, RMDP_T_END);

	reply_addr = r_addr ;
	reply_port = r_port ;
	break;
    case RMDP_T_R_CONT : /* continue transmission */
	if (!s_id) {
	    fprintf(stderr, "missing parameters... \n");
	    return 0;
	}
	prolong(s_id);
	/* XXX just accept... */
    }
    return tx - resp ;	/* XXX */
}

/*
 * handle an incoming request
 */
int
handle_req(void *p)
{
  struct sockaddr_in srcaddr;
  int ch = showselected(events);
  int srcaddrlen = sizeof(srcaddr);
  int len;
    char request[1024];
    u_char response[1024];

    if ((len = recvfrom(ch, (char *)&request, sizeof(request), 0,
	(struct sockaddr *)&srcaddr, &srcaddrlen))<=0) {
	perror("sender: recvfrom");
	return(-1);
    }
fprintf(stderr, "\n=== got packet of size %d on sock %d\n", len, ch); 
    len = handle_v1req(request, len, response);

    if (len) {
	int risu = -1;
	int rsock = openssock(reply_addr, reply_port, 0 /* XXX TTL */ );
	if (rsock) {
	    risu = send(rsock, response, len, 0) ;
	    close(rsock);
	}
	if (risu < 0 )
	    fprintf(stderr, "rmdpd: error sending response\n");
    }
    return 0;
}

int ctrls, datas;

int
main(int argc, char *argv[])
{
  int i;

  char dataaddr[ADDRLEN],
      ctrladdr[ADDRLEN];
			/* data & control m-cast addresses */
  n16 dport = 0 , cport = 0;
  ui8 ttl=TTL;

    extern char *optarg;
    extern int optind;

    /* parse command line... */
    while ((i = getopt(argc,argv,"t:d:c:k:n:l:")) != EOF) {
	switch(i) {
	case 'd': /* data addr/port */
	    if(set_link_addr(optarg, dataaddr, ADDRLEN, &dport, "data")<0)
		badcmdline(optarg);
	    break;
	case 'c': /* command addr/port */
	    if(set_link_addr(optarg, ctrladdr, ADDRLEN, &cport, "control")<0)
		badcmdline(optarg);
	    break;
	case 't': /* TTL */
	    ttl = atoi(optarg);
	    if (ttl <= 0 || ttl > 255) badcmdline(optarg);
	    break;
	case 'k': /* k */
	    k_glob = atoi(optarg);
	    fprintf(stderr,"rmdpd: k = %d\n", k_glob);
	    if (k_glob <= 0 || k_glob > 64) badcmdline(optarg);
	    break;
	case 'n': /* n */
	    n_glob = atoi(optarg);
	    if (n_glob < 0) badcmdline(optarg);
	    break;
	case 'l': /* data packet size */
	    data_pkt_len = atoi(optarg);
	    if (data_pkt_len < 16 /* XXX */ ||
		data_pkt_len > MAX_DATA_PKT_LEN) badcmdline(optarg);
	    break;
	}
    }
    if (cport == 0)
	set_link_addr(DEFAULT_CPORT, ctrladdr, ADDRLEN, &cport, "control");
    if (dport == 0)
	set_link_addr(DEFAULT_DPORT, dataaddr, ADDRLEN, &dport, "data");
    fprintf(stderr, "using ttl %d, k %d, n %d\n", ttl, k_glob, n_glob);
    argc -= optind;
    argv += optind;

    /* init encoding functions */
    if (n_glob<=k_glob) {
	fprintf(stderr,"Sorry, you have chosen no redundancy\n");
	exit(1);
    }

    if (n_glob>256) {
	fprintf(stderr,"Sorry, cannot handle %d FEC packets\n"
		"please reduce n\n",
		n_glob);
	exit(1);
    }

    fec_code = fec_new(k_glob, n_glob);

    events = createevlist();

    dumm_addr = inet_addr(dataaddr) ;
    dumm_port = dport;

    datas = openssock(dumm_addr, dport, ttl /* TTL */);
    /*** open the multicast channel ***/
    ctrls = openrsock(inet_addr(ctrladdr), &cport);
    insertchan(events, ctrls, 1 /* XXX */, handle_req, no_ar);
#if 0
    /*** open the unicast channel ***/
    ctrls = openrsock(0, &cport);
    insertchan(events, ctrls, 1 /* XXX */, handle_req, no_ar);
#endif
    while (getevent(events, 1)!=EV_NO_PENDING);

    fec_free(fec_code);
    exit(0);
}
