/*
 * PWCBSD - Philips USB webcam driver for FreeBSD 5.4 and higher
 *
 *  Copyright (C) 2006 Raaf 
 *
 * Based on the Linux pwc driver.
 *  
 *  Copyright (C) 1999-2003 Nemosoft Unv.
 *  Copyright (C) 2004-2006 Luc Saillard
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301  USA
 */

#ifndef PWC_H
#define PWC_H
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/bus.h>
#include <sys/conf.h>
#include <sys/fcntl.h>
#include <sys/tty.h>
#include <sys/selinfo.h>
#include <sys/poll.h>
#include <sys/uio.h>
#include <sys/sysctl.h>
#include <sys/malloc.h>
#include <sys/mman.h>
#include <vm/vm.h>
#include <vm/pmap.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include <dev/usb/usbdi_util.h>

#include "linux_usbif.h"

#include "pwc-info.h"

#define pwc_device pwc_softc
/** various forward declarations */
struct pwc_softc;
struct usb_device;
#include "spca5xx.h"

#include "pwc-uncompress.h"
#include "pwc-ioctl.h"
#include "videodev.h"

#include "spca_globals.h"

#define PWC_ASLEEP	1
#define PWC_INIT	2
#define PWC_POLL	4

/* Trace certain actions in the driver */
#define TRACE_MODULE	0x0001
#define TRACE_PROBE	0x0002
#define TRACE_OPEN	0x0004
#define TRACE_READ	0x0008
#define TRACE_MEMORY	0x0010
#define TRACE_FLOW	0x0020
#define TRACE_SIZE	0x0040
#define TRACE_PWCX	0x0080
#define TRACE_IOCTL	0x0100
#define TRACE_ISOC	0x0200
#define TRACE_ISOC_VERBOSE 0x0400
#define TRACE_READ_VERBOSE 0x0800
#define TRACE_SEQUENCE	0x1000

#ifdef USB_DEBUG
#define Trace(R, fmt , A...) if (pwcdebug & R) printf(PWC_NAME " %d " fmt , __LINE__ , ## A)
#define Debug(fmt , A...) if(pwcdebug) printf("%s:%d " PWC_NAME " " fmt , __FUNCTION__, __LINE__ , ##A)
// extern int  pwcdebug;
#else
#define Trace(R, A...)
#define Debug(A...)
#endif
#define Info(A...)  printf(PWC_NAME " " A)
#define Err(A...)   printf(PWC_NAME " " A)


/* Defines for ToUCam cameras */
#define TOUCAM_HEADER_SIZE		8
#define TOUCAM_TRAILER_SIZE		4

#define FEATURE_MOTOR_PANTILT		0x0001

#define PWC_NAME 	"pwc"

/* Ignore errors in the first N frames, to allow for startup delays */
#define FRAME_LOWMARK 5

/* Size and number of buffers for the isochronous pipe. */
#define MAX_ISOC_TRANSFERS		3

/* !!!!!!!!!! IMPORTANT DO NOT SET THIS HIGHER THAN 8 !!!!!!!!!!!!!!! 
 * !!!!!!!!!! OHCI IS BUGGY IF YOU DO SO              !!!!!!!!!!!!!!!*/
#define ISO_FRAMES_PER_DESC	8 /* XXX was 8 */

#define ISO_MAX_FRAME_SIZE	(1024)	/* XXX was 960 */
#define ISO_BUFFER_SIZE 	(ISO_FRAMES_PER_DESC * ISO_MAX_FRAME_SIZE)

/* Frame buffers: contains compressed or uncompressed video data. */
#define MAX_FRAMES		5
/* Maximum size after decompression is 640x480 YUV data, 1.5 * 640 * 480 */
/* XXX update to 640x480x4... */
#define PWC_FRAME_SIZE 		(640*480*4 + TOUCAM_HEADER_SIZE + TOUCAM_TRAILER_SIZE)

/* Absolute maximum number of buffers available for mmap() */
#define MAX_IMAGES 		10

#ifndef V4L2_PIX_FMT_PWC1
#define V4L2_PIX_FMT_PWC1	v4l2_fourcc('P','W','C','1')
#define V4L2_PIX_FMT_PWC2	v4l2_fourcc('P','W','C','2')
#endif


struct pwc_iso_buf {
	unsigned char *data;
 	u_int16_t sizes[ISO_FRAMES_PER_DESC];
	struct pwc_softc *sc;
};

/*
 * Callbacks for the various cameras
 */
/*
 * Things to do on attach i.e. after a probe has been successful.
 * Typically, fill in the supported modes, talk to the hardware to
 * detect camera features, reset leds, power save and so on.
 * Also, set a default video mode (or is this generic ?)
 */
typedef int (attach_cb_t)(struct pwc_softc *sc);

/*
 * Things to do on open.
 * Typically exit from power save, turn on leds, set a fallback format,
 * allocate decompress data etc.
 * Remember, the video format is not reset among opens(?)
 */
typedef int (open_cb_t)(struct pwc_softc *sc);

typedef int (setmode_cb_t)(struct pwc_softc *sc);
typedef int (start_cb_t)(struct pwc_softc *sc);
typedef int (stop_cb_t)(struct pwc_softc *sc);
typedef int (ioctl_cb_t)(struct pwc_softc *sc);
typedef int (close_cb_t)(struct pwc_softc *sc);
typedef int (consume_cb_t)(struct pwc_softc *sc, unsigned char *buf, int flen);
typedef int (decompress_cb_t)(struct pwc_softc *sc, unsigned char *dst, unsigned char *src, int srclen);

struct camera_callbacks {
	attach_cb_t	*cb_attach;
	open_cb_t	*cb_open;
	setmode_cb_t	*cb_setmode;
	start_cb_t	*cb_start;
	stop_cb_t	*cb_stop;
	ioctl_cb_t	*cb_ioctl;
	close_cb_t	*cb_close;
	consume_cb_t	*cb_consume;
	decompress_cb_t	*cb_decompress;
};

/* camera descriptor - all info we might need.
 */
struct pwc_info {
	struct usb_devno	devno;
	int			type;
	const char		*name;
	struct camera_callbacks *cb;
	int			bridge;
	int			sensor;
	int			dataformat;
	int			altinterface;
	int			videoendpoint;
};

/* intermediate buffers with raw data from the USB cam */
struct pwc_frame_buf {
   char *data;
   volatile int filled;		/* number of bytes filled */
   struct pwc_frame_buf *next;	/* list */

   /* spca fields, not really used but necessary for compiling */
   int last_packet;
   char *highwater;
   int totlength;	/* XXX camera-specific */
   int scanstate;
   int grabstate;
   int format;
};

/* additionnal informations used when dealing image between kernel and userland */
struct pwc_imgbuf {
	char  *bufmem;		/* addr of this buffer in kernel space */
	int   vma_use_count;	/* number of times this memory is mapped */
};

/*
 * State machine for vsync. NOCOPY means we are hunting for the
 * initial SYNC, which happens when we detect an end-of-frame
 * (in the original pwc driver, a transfer shorter than the previous one).
 * SYNCHUNT is set after the end of frame, and means we can start
 * copying the data, and enter the SYNCHED state as soon as we have
 * a non-zero frame. At the end of the frame we move to SYNCHUNT.
 */
enum {
	VSYNC_NOCOPY,
	VSYNC_SYNCHUNT,
	VSYNC_SYNCHED,
};


struct pwc_softc {
   device_t sc_dev; 
   usbd_device_handle udev;
   usbd_interface_handle sc_iface;
   usbd_pipe_handle sc_videopipe;
   struct cdev *sc_dev_t;
   
   struct selinfo rsel;
   // const char *name;
   //int type;                    /* type of cam (645, 646, 675, 680, 690, 720, 730, 740, 750) */
   int release;			/* release number */
   int features;		/* feature bits */
   char serial[USB_MAX_STRING_LEN]; /* serial number (string) */
   int error_status;		/* set when something goes wrong with the cam (unplugged, USB errors) */
   int usb_init;		/* set when the cam has been initialized over USB */

   struct pwc_info pwc_info;	/* local copy of pwc_info, with fixes if needed */
   struct camera_callbacks camera_cb;	/* local copy of pwc_info, with fixes if needed */

   int frames;			/* frames read so far */
   int start_ticks;			/* frames read so far */
   int byte_count;		/* total bytes read */
   int bytes_skipped;		/* bytes skipped sync-hunting */
   int power_save;
   int led_on;
   int led_off;
   int stats;
   int pwc_mbufs;
   int pwc_fbufs;
   int pwc_pad;

   // consume_cb_t *consume;	/* data consumer routine */

   /*** Video data ***/
   int vopen;			/* flag */
   int vendpoint;		/* video isoc endpoint */
   int vcinterface;		/* video control interface */
   int valternate;		/* alternate interface needed */
   int vframes, vsize;		/* frames-per-second & size (see PSZ_*) */
   int vpalette;		/* palette: 420P, RAW or RGBBAYER */
   int vframe_count;		/* received frames */
   int vframes_dumped; 		/* counter for dumped frames */
   int vframes_error;		/* frames received in error */
   int vmax_packet_size;	/* USB maxpacket size */
   u_int16_t vlast_packet_size;	/* for frame synchronisation */
   int visoc_errors;		/* number of contiguous ISOC errors */
   int vcompression;		/* desired compression factor */
   int vbandlength;		/* compressed band length; 0 is uncompressed */
   char vsnapshot;		/* snapshot mode */
   char vsync;			/* used by isoc handler */
   char vmirror;		/* for ToUCaM series */
   
   int cmd_len;
   unsigned char cmd_buf[13];

   /* The image acquisition requires 3 to 4 steps:
      1. data is gathered in short packets from the USB controller.
	We have up to MAX_ISOC_TRANSFERS pending usb transfers, each one
	transferring up to ISO_FRAMES_PER_DESC * ISO_MAX_FRAME_SIZE
	(FRAME here refers to usb frames).
	One transfer is not necessarily one video frame, it depends on image
	size and compression mode and a few more issues. However, we know
	that one (usb) frame is transferred in 1ms (full speed) or 123us
	(high speed) so we can tune the transfer size (i.e. the number of
	frames) so that interrupt occur every 30ms i.e. at least once per
	video frame. Faster than that is useless, slower means unnecessary
	frame delay.
	
      2. data is synchronized and packed into a frame buffer.
	We have MAX_FRAME video frame buffers, filled in the interrupt
	service routine with the data from the ISOC buffers.
	In most cases we need to identify synchronization markers
	indicating the beginning of the (video) frame, and do an initial
	decoding of the data stream to identify when a full video frame
	is available.

      3a. in case data is compressed, decompress it directly into image buffer
      3b. in case data is uncompressed, copy into image buffer with viewport
      4. data is transferred to the user process

      Note that MAX_ISOC_TRANSFERS != MAX_FRAMES != MAX_IMAGES....
      We have in effect a back-to-back-double-buffer system.
    */
   /* 1: isoc */
   usbd_xfer_handle xfer[MAX_ISOC_TRANSFERS];	/* XXX NEW_USB uses this */
#ifndef NEW_USB
   struct pwc_iso_buf sbuf[MAX_ISOC_TRANSFERS];
#endif
 
   char iso_init;

   /* 2: frame */
   struct pwc_frame_buf *fbuf;	/* all frames */
   struct pwc_frame_buf *empty_frames, *empty_frames_tail;	/* all empty frames */
   struct pwc_frame_buf *full_frames, *full_frames_tail;	/* all filled frames */
   struct pwc_frame_buf *fill_frame;	/* frame currently being filled */
   struct pwc_frame_buf *read_frame;	/* frame currently read by user process */
   int frame_header_size, frame_trailer_size;
   int frame_size;
   int frame_total_size; /* including header & trailer */
   int drop_frames;

   /* 3: decompression */
   void *decompress_data;		/* private data for decompression engine */

   /* 4: image */
   /* We have an 'image' and a 'view', where 'image' is the fixed-size image
      as delivered by the camera, and 'view' is the size requested by the
      program. The camera image is centered in this viewport, laced with
      a gray or black border. view_min <= image <= view <= view_max;
      In other word, the driver does padding, not truncation.
    */
   int image_mask;			/* bitmask of supported sizes */
   struct pwc_coord view_min, view_max;	/* minimum and maximum viewable sizes */
   struct pwc_coord abs_max;            /* maximum supported size with compression */
   struct pwc_coord image, view;	/* image and viewport size */
   struct pwc_coord offset;		/* offset within the viewport */

   void *image_data;			/* total buffer, which is subdivided into ... */
   struct pwc_imgbuf images[MAX_IMAGES];/* ...several images... */
   int fill_image;			/* ...which are rotated. */
   int len_per_image;			/* length per image */
   int image_read_pos;			/* In case we read data in pieces, keep track of were we are in the imagebuffer */
   int image_used[MAX_IMAGES];		/* For MCAPTURE and SYNC */

   struct mtx ptrlock;			/* for manipulating the buffer pointers */
   
   /*** motorized pan/tilt feature */
   struct pwc_mpt_range angle_range;
   int pan_angle;			/* in degrees * 100 */
   int tilt_angle;			/* absolute angle; 0,0 is home position */
   int snapshot_button_status;		/* set to 1 when the user push the button, reset to 0 when this value is read */

   /*** Misc. data ***/
   int state;				/* When waiting for a frame to finish... */

   /* XXX Finally, spwc support */
   struct usb_spca50x spca50x;
   struct sensorctrl sensor_ctrl;
};


#ifdef __cplusplus
extern "C" {
#endif

/** functions in pwc.c */
int pwc_next_fill_frame(struct pwc_softc *sc);


/** functions in pwc-if.c */
int pwc_try_video_mode(struct pwc_device *pdev, int width, int height, int new_fps, int new_compression, int new_snapshot);
int pwc_handle_frame(struct pwc_device *pdev);

/** Functions in pwc-misc.c */
/* sizes in pixels */
struct pwc_format {
	struct pwc_coord xy;	/* width * height */
	int		id;	/* format identifier, PSZ_* */
	const char *name;	/* format name */
};
extern struct pwc_format pwc_image_sizes[];

int set_alt_interface(struct usbd_device *udev, usbd_interface_handle iface, int alt);
int usb_control_msg(usbd_device_handle udev,
	u_int pipe,u_int8_t request,u_int8_t requesttype,
        u_int16_t value, u_int16_t index, void *data,
	u_int16_t size,int timeout);


struct pwc_format * pwc_decode_size(struct pwc_device *pdev, int width, int height);
/*
 * show existing configuration.
 * If alt and ep are != -1, return the endpoint for that configuration.
 */
usb_endpoint_descriptor_t *
pwc_show_configs(struct pwc_softc *sc, int alt, int ep);


/* callbacks for the various devices */
extern struct camera_callbacks pwc_callbacks;
extern struct camera_callbacks spca_callbacks;

/** Functions in pwc-ctrl.c */
/* Request a certain video mode. Returns < 0 if not possible */
extern int pwc_set_video_mode(struct pwc_device *pdev, int width, int height, int frames, int compression, int snapshot);
/* Calculate the number of bytes per image (not frame) */
extern int pwc_mpt_reset(struct pwc_device *pdev, int flags);
extern int pwc_mpt_set_angle(struct pwc_device *pdev, int pan, int tilt);

/* Various controls; should be obvious. Value 0..65535, or < 0 on error */
extern int pwc_get_brightness(struct pwc_device *pdev);
extern int pwc_set_brightness(struct pwc_device *pdev, int value);
extern int pwc_get_contrast(struct pwc_device *pdev);
extern int pwc_set_contrast(struct pwc_device *pdev, int value);
extern int pwc_get_gamma(struct pwc_device *pdev);
extern int pwc_set_gamma(struct pwc_device *pdev, int value);
extern int pwc_get_saturation(struct pwc_device *pdev);
extern int pwc_set_saturation(struct pwc_device *pdev, int value);
extern int pwc_set_leds(struct pwc_device *pdev, int on_value, int off_value);
extern int pwc_get_leds(struct pwc_device *pdev, int *on_value, int *off_value);
extern int pwc_get_cmos_sensor(struct pwc_device *pdev, int *sensor);
extern int pwc_restore_user(struct pwc_device *pdev);
extern int pwc_save_user(struct pwc_device *pdev);
extern int pwc_restore_factory(struct pwc_device *pdev);

/* exported for use by v4l2 controls */
extern int pwc_get_red_gain(struct pwc_device *pdev, int *value);
extern int pwc_set_red_gain(struct pwc_device *pdev, int value);
extern int pwc_get_blue_gain(struct pwc_device *pdev, int *value);
extern int pwc_set_blue_gain(struct pwc_device *pdev, int value);
extern int pwc_get_awb(struct pwc_device *pdev);
extern int pwc_set_awb(struct pwc_device *pdev, int mode);
extern int pwc_set_agc(struct pwc_device *pdev, int mode, int value);
extern int pwc_get_agc(struct pwc_device *pdev, int *value);
extern int pwc_set_shutter_speed(struct pwc_device *pdev, int mode, int value);
extern int pwc_get_shutter_speed(struct pwc_device *pdev,int *value);
extern int pwc_set_colour_mode(struct pwc_device *pdev, int colour);
extern int pwc_get_colour_mode(struct pwc_device *pdev, int *colour);

/* Power down or up the camera; not supported by all models */
extern int pwc_camera_power(struct pwc_device *pdev, int power);

/* Private ioctl()s; see pwc-ioctl.h */
extern int pwc_do_ioctl(struct pwc_device *pdev, unsigned int cmd, void *arg);

/** Functions in pwc-v4l.c */
extern int pwc_video_do_ioctl(struct pwc_device *pdev, unsigned int cmd, void *arg, int unit);
/** pwc-uncompress.c */
/* Expand frame to image, possibly including decompression. Uses read_frame and fill_image */
decompress_cb_t pwc_decompress;
decompress_cb_t spca_decompress;

/** in pwc-addons.c */
int zc3xx_consume(struct pwc_softc *sc, unsigned char *buf, int flen);
int spca561_consume(struct pwc_softc *sc, unsigned char *buf, int flen);
int qc_consume(struct pwc_softc *sc, unsigned char *buf, int flen);
int qc_probe_sensor(struct pwc_softc *sc);


/** in pwc-spca.c */
int pac207_consume(struct pwc_softc *sc, unsigned char *buf, int flen);

#ifdef __cplusplus
}
#endif

#endif
