/*
 * PWCBSD - Philips USB webcam driver for FreeBSD 5.4 and higher
 *
 *  Copyright (C) 2006 Raaf 
 *
 * Based on the Linux pwc driver.
 *  
 *  Copyright (C) 1999-2003 Nemosoft Unv.
 *  Copyright (C) 2004-2006 Luc Saillard
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301  USA
 */

#include "pwc.h"

/* used for the decoders - this is pwc-specific. */
#include "pwc-dec1.h"
#include "pwc-dec23.h"

struct pwc_format pwc_image_sizes[] = {
	{ { 128,  96, 0 }, PSZ_SQCIF,	"sqcif" },
	{ { 160, 120, 0 }, PSZ_QSIF,	"qsif" },
	{ { 176, 144, 0 }, PSZ_QCIF,	"qcif" },
	{ { 192, 144, 0 }, PSZ_QPAL,	"qpal" },
	{ { 320, 240, 0 }, PSZ_SIF,	"sif" },
	{ { 352, 288, 0 }, PSZ_CIF,	"cif" },
	{ { 384, 288, 0 }, PSZ_PAL,	"pal" },
	{ { 640, 480, 0 }, PSZ_VGA,	"vga" },
	{ { 1024, 768, 0 }, PSZ_XGA,	"xga" },
	{ { 0, 0, 0 }, 0,	"none" },
};

/* x,y -> PSZ_ */
struct pwc_format *pwc_decode_size(struct pwc_softc *pdev, int width, int height)
{
	int i;
	struct pwc_format *find = NULL;

	/* Make sure we don't go beyond our max size.
           NB: we have different limits for RAW and normal modes. In case
           you don't have the decompressor loaded or use RAW mode, 
           the maximum viewable size is smaller.
        */
	if (pdev->vpalette == VIDEO_PALETTE_RAW) {
		if (width > pdev->abs_max.x || height > pdev->abs_max.y) {
			Debug("VIDEO_PALETTE_RAW: going beyond abs_max.\n");
                	return NULL;
                }
	} else {
		if (width > pdev->view_max.x || height > pdev->view_max.y) {
			Debug("VIDEO_PALETTE_ not RAW: going beyond view_max.\n");
			return NULL;
		}
	}

	/* Find the largest size supported by the camera that fits into the
	   requested size.
	 */
	for (i = 0; pwc_image_sizes[i].xy.x != 0; i++) {
		struct pwc_coord *c = &pwc_image_sizes[i].xy;
		int id = pwc_image_sizes[i].id;
		if ( (pdev->image_mask & (1 << id)) == 0)
			continue;

		Trace(TRACE_SIZE, "pwc_decode_size i=%d wanted=%dx%d "\
		    " got=%dx%d\n", i, width, height, c->x, c->y);
		if(pdev->pwc_pad && c->x <= width && c->y <= height) {
			if (1 || find == NULL || find->xy.x * find->xy.y > c->x * c->y)
				find = &pwc_image_sizes[i];
		} else if(c->x == width && c->y == height) {
			find = &pwc_image_sizes[i];
			break;
		}
	}
	return find;
}

/* attach_cb for pwc */
static int pwc_construct(struct pwc_softc *sc)
{
	int i;
        if(sc->pwc_info.devno.ud_vendor == 0x046D) {
                if(sc->pwc_info.devno.ud_product == 0x08B4) {
                        sc->power_save = 1;
                } else if(sc->pwc_info.devno.ud_product == 0x08B5) {

                        /* Logitech QuickCam Orbit */
                        sc->features |= FEATURE_MOTOR_PANTILT;
                        sc->angle_range.pan_min  = -7000;
                        sc->angle_range.pan_max  =  7000;
                        sc->angle_range.tilt_min = -3000;
                        sc->angle_range.tilt_max =  2500;
                }
        }
        

	if (pwc_get_cmos_sensor(sc, &i) >= 0) {
                int sensor;
                
                switch(i) {
                case 0x00:
                        sensor = SENSOR_PWC_HYUNDAI;
                        break;

                case 0x20:
                        sensor = SENSOR_PWC_TDA8787;
                        break;
        
                case 0x2E:
                        sensor = SENSOR_PWC_SONY_Exas98L59;
                        break;

                case 0x2F:
                        sensor = SENSOR_PWC_SONY_ADI9804;
                        break;

                case 0x30:
                        sensor = SENSOR_PWC_SHARP_TDA8787;
                        break;
         
                case 0x3E:
                        sensor = SENSOR_PWC_SHARP_Exas98L59;
                        break;
 
                case 0x3F:
                        sensor = SENSOR_PWC_SHARP_ADI9804;
                        break;
  
                case 0x40:
                        sensor = SENSOR_PWC_UPA1021;
                        break;
 
                case 0x100:
                        sensor = SENSOR_PWC_VGA;
                        break;

                case 0x101:
                        sensor = SENSOR_PWC_PAL_MR;
                        break;

                default:
                        sensor = SENSOR_PWC_UNKNOWN;
                        break;
                }
                sc->pwc_info.sensor = sensor;
	}

	switch(sc->pwc_info.type) {
	case 645:
	case 646:
		sc->view_min.x = 128;
		sc->view_min.y =  96;
		sc->view_max.x = 352;
		sc->view_max.y = 288;
                sc->abs_max.x  = 352;
                sc->abs_max.y  = 288;
		sc->image_mask = 1 << PSZ_SQCIF | 1 << PSZ_QCIF | 1 << PSZ_CIF;
		sc->vcinterface = 2;
		sc->vendpoint = 4;
		sc->frame_header_size = 0;
		sc->frame_trailer_size = 0;
		break;
	case 675:
	case 680:
	case 690:
		sc->view_min.x = 128;
		sc->view_min.y =  96;
		/* Anthill bug #38: PWC always reports max size, even without PWCX */
		sc->view_max.x = 640;
		sc->view_max.y = 480;
		sc->image_mask = 1 << PSZ_SQCIF | 1 << PSZ_QSIF | 1 << PSZ_QCIF | 1 << PSZ_SIF | 1 << PSZ_CIF | 1 << PSZ_VGA;
                sc->abs_max.x = 640;
                sc->abs_max.y = 480;
		sc->vcinterface = 3;
		sc->vendpoint = 4;
		sc->frame_header_size = 0;
		sc->frame_trailer_size = 0;
		break;
	case 720:
	case 730:
	case 740:
	case 750:
		sc->view_min.x = 160;
		sc->view_min.y = 120;
		sc->view_max.x = 640;
		sc->view_max.y = 480;
		sc->image_mask = 1 << PSZ_QSIF | 1 << PSZ_SIF | 1 << PSZ_VGA;
                sc->abs_max.x = 640;
                sc->abs_max.y = 480;
		sc->vcinterface = 3;
		sc->vendpoint = 5;
		sc->frame_header_size = TOUCAM_HEADER_SIZE;
		sc->frame_trailer_size = TOUCAM_TRAILER_SIZE;
		break;
	}
	sc->vpalette = VIDEO_PALETTE_YUV420P; /* default */
	sc->view_min.size = sc->view_min.x * sc->view_min.y;
	sc->view_max.size = sc->view_max.x * sc->view_max.y;
	/* length of image, in YUV format; always allocate enough memory. */
	sc->len_per_image = (sc->abs_max.x * sc->abs_max.y * 3) / 2;

        pwc_set_leds(sc, 0, 0);

        if(sc->power_save)
                pwc_camera_power(sc, 0);
	return 0;
}

/* open callback for pwc
 * Do the malloc last so we don't have to undo it on failure.
 */
static int pwc_open_cb(struct pwc_softc *sc)
{
	int err = 0;

	/* set fallback mode */
        sc->vframes = 10;

        if(sc->pwc_info.type > 730) {
                sc->view.x = 160;
                sc->view.y = 120;
        } else {
                sc->view.x = 176;
                sc->view.y = 144;
        }

        /* Turn on camera */
        if (sc->power_save) {
                err = pwc_camera_power(sc, 1);
                if (err < 0) {
                        device_printf(sc->sc_dev, 
			"Failed to restore power to the camera! (%d)\n", -err);
			return -err;
		}
        }
	pwc_set_leds(sc, sc->led_on, sc->led_off);
        /* Allocate decompressor table space */
        if(sc->pwc_info.type == 645 || sc->pwc_info.type == 646)
                sc->decompress_data = malloc(sizeof(struct pwc_dec23_private), M_USBDEV, M_WAITOK);/* TOD
O & FIXME */
        else
                sc->decompress_data = malloc(sizeof(struct pwc_dec23_private), M_USBDEV, M_WAITOK);/* Tim
on & Kiara */
   
        if(sc->decompress_data == NULL) {
                device_printf(sc->sc_dev, "Failed to allocate decompress table.\n");
		return ENOMEM;
        }
	return 0;
}

/* close routine for pwc */
static int pwc_close_cb(struct pwc_softc *sc)
{
        if(sc->pwc_info.type == 645 || sc->pwc_info.type == 646)
                pwc_dec1_exit();
        else
                pwc_dec23_exit();       /* Timon & Kiara */
	set_alt_interface(sc->udev, sc->sc_iface, 0);
	pwc_set_leds(sc,0,0);
	if(sc->power_save) {
		if(pwc_camera_power(sc, 0) < 0)
			device_printf(sc->sc_dev, "Failed to power down the camera\n");
        }
	return 0;
}


static int pwc_consume(struct pwc_softc *sc, unsigned char *iso_buf, int flen)
{
	int awake = 0;	/* need to awake the caller at the end */
	struct pwc_frame_buf *fbuf = sc->fill_frame;
	unsigned char *fillptr = fbuf->data + fbuf->filled;

	/* XXX there is no individual framestatus in FreeBSD usbstack
	 * so just assume all frames are good
	 */

	if (flen > 0) { /* if valid data... */
		if (sc->vsync > VSYNC_NOCOPY) { /* ...and we are not sync-hunting... */
			sc->vsync = VSYNC_SYNCHED;
			/* ...copy data to frame buffer, if possible */
			if (flen + fbuf->filled > sc->frame_total_size) {
				Trace(TRACE_ISOC, "Frame buffer overflow (flen = %d,frame_total_size = %d).\n",flen, sc->frame_total_size);
				sc->vsync = VSYNC_NOCOPY; /* Hmm, let's wait for an EOF (end-of-frame) */
				sc->vframes_error++;
			}
			else {
				memcpy(fillptr, iso_buf, flen);
				fillptr += flen;
			}
		}
		fbuf->filled += flen;
	}

	/* now find end of packet, if any */
	if(flen >= sc->vlast_packet_size)
		goto eof_done;
	
	/* Shorter packet... We probably have the end of an image-frame; 
	   wake up read() process and let select()/poll() do something.
	   Decompression is done in user time over there.
	 */
	
	if (sc->vsync != VSYNC_SYNCHED)
		goto start_new_frame;
	
	/* The ToUCam Fun CMOS sensor causes the firmware to send 2 or 3 bogus 
	   frames on the USB wire after an exposure change. This conditition is 
	   however detected  in the cam and a bit is set in the header.
	 */
	if (sc->pwc_info.type == 730) {
		unsigned char *ptr = (unsigned char *)fbuf->data;
			
		if (ptr[1] == 1 && ptr[0] & 0x10) {
			sc->drop_frames += 2;
			sc->vframes_error++;
		}
		if ((ptr[0] ^ sc->vmirror) & 0x01) {
			if (ptr[0] & 0x01) {
				sc->snapshot_button_status = 1;
				Debug("pwc: Snapshot button pressed.\n");
			}
			else {
				Debug("pwc: Snapshot button released.\n");
			}
		}
		if ((ptr[0] ^ sc->vmirror) & 0x02) {
			if (ptr[0] & 0x02) {
				Debug("pwc: Image is mirrored.\n");
			} else {
				Debug("pwc: Image is normal.\n");
			}
		}
		sc->vmirror = ptr[0] & 0x03;
		/* Sometimes the trailer of the 730 is still sent as a 4 byte packet 
		   after a short frame; this condition is filtered out specifically. A 4 byte
		   frame doesn't make sense anyway.
		   So we get either this sequence: 
			drop_bit set -> 4 byte frame -> short frame -> good frame
		   Or this one:
			drop_bit set -> short frame -> good frame
		   So we drop either 3 or 2 frames in all!
		 */
		if (fbuf->filled == 4)
			sc->drop_frames++;
	} else if (sc->pwc_info.type == 740 || sc->pwc_info.type == 720) {
		unsigned char *ptr = (unsigned char *)fbuf->data;
		if ((ptr[0] ^ sc->vmirror) & 0x01) {
			if (ptr[0] & 0x01) {
				sc->snapshot_button_status = 1;
				Debug("pwc: Snapshot button pressed.\n");
			}
			else {
				Debug("pwc: Snapshot button released.\n");
			}
		}
		sc->vmirror = ptr[0] & 0x03;
	}

	/* In case we were instructed to drop the frame, do so silently.
	   The buffer pointers are not updated either (but the counters are reset below).
	 */
	if(sc->drop_frames > 0) {
		sc->drop_frames--;
	}
	/* Check for underflow first */
	else if(sc->pwc_info.type != 0 && fbuf->filled < sc->frame_total_size) {
		Trace(TRACE_ISOC,"Frame buffer underflow (have %d bytes need %d); discarded.\n", fbuf->filled, sc->frame_total_size);
		sc->vframes_error++;
	}
	else {
		/* Send only once per EOF */
		awake = 1; /* delay wake_ups */
		/* Find our next frame to fill. This will always succeed, since we
		 * nick a frame from either empty or full list, but if we had to
		 * take it from the full list, it means a frame got dropped.
		 */
		pwc_next_fill_frame(sc);
		fbuf = sc->fill_frame;
	}
	sc->vframe_count++;
start_new_frame:
	fbuf->filled = 0;
	fillptr = fbuf->data;
	sc->vsync = VSYNC_SYNCHUNT;
eof_done:
	sc->vlast_packet_size = flen;
	return awake;
}

struct camera_callbacks pwc_callbacks = {
        .cb_attach = pwc_construct,
        .cb_open = pwc_open_cb,
        .cb_close = pwc_close_cb,
        .cb_consume = pwc_consume,
        .cb_decompress = pwc_decompress,
};

/*
 * This is only for debugging.
 * Show the full configuration. If alt/ep are presented, select those
 * and do not print anything.
 */
usb_endpoint_descriptor_t *
pwc_show_configs(struct pwc_softc *sc, int alt, int ep)
{
	int i, l, ialt = (alt < 0 ? 0 : alt);
	usb_endpoint_descriptor_t *found = NULL, *edesc = NULL;

        if(sc->sc_iface == NULL || sc->sc_iface->idesc == NULL) {
		device_printf(sc->sc_dev, "Failed to get endpoint count\n");
		return NULL;
	}
	printf("have %d configs\n", sc->udev->ddesc.bNumConfigurations);
	for (;;) {
		if (set_alt_interface(sc->udev, sc->sc_iface, ialt) != USBD_NORMAL_COMPLETION) {
			if (alt >= 0)
				printf("Set packet size: set alt config %d error\n", ialt);
			break;
		}
		for (i = 0; i < sc->sc_iface->idesc->bNumEndpoints; i++) {

			edesc = usbd_interface2endpoint_descriptor(sc->sc_iface, i);
			if (!edesc)
				continue;
			l = UGETW(edesc->wMaxPacketSize);
			printf("alt %d [%d] endp %d size %d:%4d attr 0x%x interval %d\n",
				ialt, i, UE_GET_ADDR(edesc->bEndpointAddress),
				(l >> 11) & 3, l & 0x7ff,
				edesc->bmAttributes,
				edesc->bInterval);
			if(UE_GET_ADDR(edesc->bEndpointAddress) == ep)
				found = edesc;
		}
		if (alt >= 0)
			break;
		ialt++;
	}
	return found;
}
