/***
 *** pcbridge 3.03
 *** File: bridge.c
 ***
 ***	(C)opyright 1996 Luigi Rizzo (luigi@iet.unipi.it)
 ***	main routines for the brige.
 ***/

#define EXTERN
#include "bridge.h"
#include "ns8390.h"	/* needed for the test in pkt_send */

struct board_list {
    u_short ioport;
    u_long  membase;
} boards[] = {
    {	0x240,	0xd0000	},
    {	0x280,	0xd4000	},
    {	0x2c0,	0	},
    {	0x2e0,	0	},
    {	0x300,	0xc8000	},
    {	0x320,	0xcc000	},
    {	0x340,	0xe0000	},
#if 0
    {	0xff80,	0	},
    {	0xff40,	0	},
    {	0xff00,	0	},
    {	0xfec0,	0	},
    {	0xfe80,	0	},
    {	0xefc0,	0	},	/* robertini ? */
    {	0xef80,	0	},	/* robertini ? */
    {	0xef40,	0	},	/* robertini ? */
    {	0xef00,	0	},	/* robertini ? */
#endif
    {	0,	0	},
};

#define	BDG_PASS	1
#define	BDG_DISCARD	2
#define	BDG_NEXT	3

struct filter {
    u_short inside, outside, first_in, first_out;
} flt[] = {
    /*  inside_action   outside_action [beg , ] end+1 
    {	BDG_PASS,	BDG_DISCARD,	IP,	IP+1	}, /* IP */
    {	BDG_PASS,	BDG_DISCARD,	ARP,	ARP+1	}, /* ARP */
    {	BDG_DISCARD,	BDG_DISCARD,	0,	0	}, /* sink */
    {	0,		0,		0,	0	} /* terminator */
};

u_short eth_bcast[3]= { 0xffff, 0xffff, 0xffff };

static pkt_send(struct if_vars *ifp);

ask(char *s)
{
    int c;
    while (1) {
	printf(s);
	c = getchar();
	if ((c >= 'a') && (c <= 'z')) c &= 0x5F;
	if (c == '\r') break;
	putchar(c);
	if (c == 'N')
	    longjmp(jmp_restart);
	if (c == 'Y')
	    break;
    }
}

/***
 *** IPCHKSUM - Checksum IP Header
 ***/
ipchksum(u_short *ip, int len)
{
    u_long sum = 0;
    len >>= 1;
    while (len--) {
	sum += *(ip++);
	if (sum > 0xFFFF)
	    sum -= 0xFFFF;
    }
    return((~sum) & 0x0000FFFF);
}

flush_table()
{
    int i;
    for (i=0; i< HASH_SIZE; i++) {
	bdg_table[i].eth[0]= 0xffff; /* clear table */
	bdg_table[i].eth[1]= 0xffff; /* clear table */
	bdg_table[i].eth[2]= 0xffff; /* clear table */
	bdg_table[i].name= 0xffff; /* clear table */
    }
}

#ifdef DEBUG
void
buf_check()
{
    static checks=0;
    struct pkt_buf *newp=buf_free_list;
    /*** some consistency checks first ***/
    checks++;
    if (n_free <0 || n_free > n_buffers ||
	    (n_free >0 && !newp) || (n_free==0 && newp) ) {
	printf("\rPanic (%d) buf_check: n_free 0x%X n_buf %d newp 0x%X\r",
	    checks, n_free, n_buffers, newp);
	ask("");
    }
}
#else
#define buf_check()
#endif

struct pkt_buf *
buf_alloc()
{
    static buf_overflow=0; 
    struct pkt_buf *newp=buf_free_list;

    buf_check();
    if (newp) {
	buf_free_list= newp->next[0];
	newp->count=0;
	n_free--;
    } else {
        printf("Out of buffers %d  \r", buf_overflow++);
    }
    return newp;
}

int
buf_free(struct pkt_buf *b)
{
    buf_check();
    if (b->count !=0) {
	printf("\rPanic! freeing non-empty buffer, count=%d",b->count);
	ask("");
    }
    b->next[0]=buf_free_list;
    buf_free_list=b;
    n_free++;
}


/***
 *** main - Kick off routine
 ***/
main()
{
    int i, big_loops;
    char *p;
    extern char edata[], end[];
    for (p=edata; p<end; p++) *p = 0;	/* Zero BSS */

    gateA20();
    setjmp(jmp_restart);

    use_filters=0;

    for (i=0; i<MAX_CTRS; i++) { /*** reset counters ***/
	shadow_counters[i]=0;
	counters[i].low= 0;
	counters[i].high=0;
    }
    /***
     *** allocate the hash table (64K) and the packet buffers
     ***/
    bdg_table= HASH_TABLE_BASE;
    buf_free_list = buffers= (struct pkt_buf *)
	((char *)(HASH_TABLE_BASE)+HASH_SIZE*sizeof(struct hash_table));

#define	BUFFER_SIZE	0x1f000	/* should be RELOCADDR - buffers */
    n_free=n_buffers= BUFFER_SIZE / sizeof(struct pkt_buf);
    bzero(buf_free_list, n_buffers*sizeof(struct pkt_buf));
    /***
     *** build the linked list of buffers
     ***/
    for (i=0; i< n_buffers-1; i++) {
	buffers[i].next[0] = &(buffers[i+1]);
    }
    buffers[n_buffers-1].next[0]=NULL;

    flush_table();
    bdg_flush_ptr=0;

    /***
     *** For each interface, fill in appropriate data structs
     ***/
    for (i=0, curr_if=0; curr_if < MAX_CARDS && boards[i].ioport ; i++) {
	if (eth_probe(&(c_if[curr_if]),
		boards[i].ioport, boards[i].membase)) curr_if++;
    }

    /***
     *** print the initial screen
     ***/
    print_mask() ; /* print mask */
    print_addr(4,12, &(n_buffers),-2);

    /***
     *** start looking for packets
     *** This is the brigde main loop, we want to go FAST!
     ***/

again:
    for (big_loops=0; big_loops < 2048 ; big_loops++) {
	struct if_vars *sc;
	for (sc=&c_if[0], i=0; i< curr_if; i++, sc++) {
	    if (sc->tx_head) pkt_send(sc);
	    eth_poll(sc);
	}
    }

    /***
     *** every now and then, flush entries in the routing
     *** table. This way we avoid possible inconsistencies
     ***
     *** XXX It would be nicer if entries were aged instead.
     ***/
    bdg_table[bdg_flush_ptr++].name = 0xffff;
    if (bdg_flush_ptr>= HASH_SIZE)
	bdg_flush_ptr=0;
    COUNT(BIGLOOP_CTR);
    print_counters();
    goto again;
}

/***
 *** bdg_drop is invoked with the packet header and the if pointer.
 *** It must decide if the packet must be forwarded, and sets the
 *** dest field in the if_vars struct accordingly.
 *** Returns 1 if the packet must be dropped, 0 otherwise;
 *** 
 ***  Source and Destination address are searched
 ***  in the table and compared with the bridge's port address, causing
 ***  the following behaviour:
 *** 
 *** 	Dest. port	Resent to:
 *** 	==============================================
 *** 	bridge		Upper layer
 *** 	this port	Dropped
 *** 	other port	Destination port
 *** 	not found	All ports
 *** 	multi/broadcast	All ports and upper layer
 *** 
 ***  After processing, the packet is dropped and space freed.
 ***  If the packet must be used more than once, or can be got
 ***  via programmed I/O, it is copied to main RAM, which is usually
 ***  faster. IF_IN_CORE signals this.
 ***  Only one buffer is used.
 ***
 ***/

int
bdg_drop(struct if_vars *sc, struct eth_packet *pkt, int len)
{
    int index;
    u_short dest;
    struct if_vars *dc;

    COUNT(sc->name + RXPKT_CTR);
    COUNTN(sc->name + RXBYTES_CTR, len);

    if (eth_compare( sc->eth_node_addr, pkt->eth_dst) ) {
	/*** this is for the bridge only ***/
	COUNT(sc->name + RXBDG_CTR);
	sc->dest= DEST_BDG;
	return 0;
    }

    if (use_filters) {
	struct filter *f;
	u_short tag;
	u_short type=ntohs(pkt->eth_type);

	for ( f=flt; tag=f->inside; f++ ) {
	    if (type < f->first_in || type >= f->first_out) /* outside */
		tag=f->outside;
	    if (tag==BDG_PASS) goto flt_pass;
	    else if (tag==BDG_DISCARD) break;
	}
	COUNT(sc->name + FILTERED_CTR);
	return 1; /* drop */
    }
flt_pass:	;

    /***
     *** hash the source address
     ***/
    index= HASH_FN(pkt->eth_src);

    if ( bdg_table[index].name != 0xffff) { /*** the entry is valid. ***/
	if (!eth_compare( pkt->eth_src, bdg_table[index].eth)) {
	    COUNT(COLLSN_CTR); /* collision, count it */
	} else if (bdg_table[index].name != sc->name) { /* found a loop! */
	    COUNT(sc->name + LOOP_CTR);
	    COUNT(bdg_table[index].name + LOOP_CTR);
	    sc->flags |= IF_DISABLE; /* Temporarily disable tx on this if */
	    WRITERC(8, 14 +14*sc->name, 0x0e00 | 'X');
        }
    }

    /***
     *** now write the source address into the table
     ***/
    bdg_table[index].eth[0]=pkt->eth_src[0];
    bdg_table[index].eth[1]=pkt->eth_src[1];
    bdg_table[index].eth[2]=pkt->eth_src[2];
    bdg_table[index].name=sc->name;

    if (sc->flags & IF_DISABLE) {
	COUNT(sc->name + DROP_CTR);
	return 1; /* drop packet if disabled. */
    }
    if (pkt->eth_dst[0] & 1) {
	sc->dest=DEST_BDG | DEST_ALL;
	COUNT(sc->name + RXALL_CTR);
	if (eth_compare(pkt->eth_dst, eth_bcast)) {
	    COUNT(sc->name + BROADCAST_CTR);
	} else {
	    COUNT(sc->name + MULTICAST_CTR);
        }
	return 0;
    } else {
	/***
	 *** not a multicast. Look for a possible destination in table
	 ***/
	index= HASH_FN( pkt->eth_dst );
	dest=bdg_table[index].name;
	if ( (dest!=0xffff) &&
	     eth_compare( bdg_table[index].eth, pkt->eth_dst) ) {
	    /***
	     *** found a valid destination
	     ***/
	    if (dest == sc->name) { /* drop because local destination */
		COUNT(sc->name + LOCAL_DROP_CTR);
		return 1;
	    } else {
		sc->dest = dest;
		return 0;
	    }
	} else {
	    /***
	     *** destination not found
	     ***/
	    COUNT(sc->name + UNKNOWN_CTR);
	    COUNT(sc->name + RXALL_CTR);
	    sc->dest= DEST_ALL;
	    return 0;
	}
    }
    ask("should not reach this point!!!\r\n");
}

static short limits[] = {
	1, 2, 4, 8, 16, 32, 64, 128, 256,
	512, 1024, 2048, 4096, 8192
};

/***
 *** pkt_send - Transmit a frame.
 *** Assume ifp->tx_head != NULL
 ***/
static pkt_send(struct if_vars *ifp)
{   
    struct pkt_buf *b = ifp->tx_head;
    int	tx_busy;
    
    /***    
     *** this test is the only board-dependent thing.
     ***/
    switch (ifp->eth_vendor) {
    case VENDOR_SINK:
    default:
	tx_busy = 1;
	break;
    case VENDOR_WD:
    case VENDOR_NOVELL:
    case VENDOR_3COM:
	tx_busy = inb(ifp->eth_nic_base+ED_P0_CR) & ED_CR_TXP ;
	break;
    }
    
    if ( tx_busy ) {
        /*** 
         *** still busy sending previous packet. Must wait, but for
         *** a limited time. The timeout gets shorter as the number of
         *** free buffers decreases.
         ***/
	int limit;

	if (n_free>13) limit=10000;
	else limit= limits[n_free];

        if ( ifp->tx_delay++ < limit)
            return;
        else
            COUNT(ifp->name + DROP_CTR);
    } else {
        eth_transmit(ifp, (u_short)(b->len), (char *)&(b->pkt));
    }  
    /***
     *** A buffer is gone, either sent or dropped. Remove from queue
     ***/
    ifp->tx_delay = 0;  /* reset... */
    ifp->queued--;
    ifp->tx_head=b->next[ifp->name];
    b->count--;
    if (b->count==0) buf_free(b);
}    

static inline bdg_enqueue(struct if_vars *dc, struct pkt_buf *b)
{
    int i=dc->name;
    if (dc->tx_head) {
	dc->tx_tail->next[i]=b;
    } else {
	dc->tx_head=b;
    }
    dc->tx_tail=b;
    dc->queued++;
    b->next[i]=NULL;
    b->count++;     /* increase reference count */
    pkt_send(dc);
}

int bdg_forward (struct if_vars *sc, struct pkt_buf *b)
{
    struct if_vars *dc;
    int dest, fwd_last;

    if (sc->dest & DEST_ALL) {
	dest=0;
	fwd_last=curr_if-1;
    } else if ( (sc->dest & DEST_NONE) != DEST_NONE ) {
	dest= fwd_last = sc->dest & DEST_NONE;
	if (dest == sc->name) goto bdg_only;
    } else {
	goto bdg_only;
    }
    b->count++;     /* increase count to avoid premature flushing... */
    for (; dest<=fwd_last; dest++) {
        dc=&c_if[dest];
	if (sc != dc) {
	/***
	 *** if the interface is disabled, drop the packet.
	 *** Otherwise, queue onto the IF queue for subsequent transmit
	 ***/
	    if (dc->flags & IF_DISABLE) {
		COUNT(dest + DROP_CTR);
	    } else {
		bdg_enqueue(dc, b);
	    }
	}
    }
    b->count--; /* restore reference count */

bdg_only:
    /* deal with possible broadcasts for the bridge ------*/
    if (sc->dest & DEST_BDG) {
	process_pkt(&(b->pkt), b->len, sc->name);
    }
}

int print_counters()
{
    static int i=0;
    unsigned long j;
    static unsigned long lastsec=0;
    struct if_vars *sc;

    curr_time=timeofday();
    if (i==0 && lastsec == curr_time) return;
    lastsec = curr_time;

    if (++i>=curr_if) i=0;
    sc= &c_if[i];

    WRITERC(8, 14 +14*i, 0x0e00 |
	    (c_if[i].flags & IF_DISABLE ? 'x' : 'Y') );
    sc->flags &= ~IF_DISABLE; /* re-enable */

    if (1) {
	/*--- management code, send info on the network ---------------*/
	struct pkt_buf *b=buf_alloc();
	/*** no need to change the reference count ***/
	if (b) {
	    int j;
	    struct pcbridge_data *p=
		(struct pcbridge_data *)
			&( b->pkt.body[sizeof(struct iphdr) +
				sizeof(struct udphdr)] );
	    int len=sizeof (*p);

	    bzero(p, len); /* should be optimized */
	    sprintf(p->magic, "PCBridge 3.00");
	    for (j=0; j<curr_if; j++) {
		p->if_found[j+1]='1';
		p->if_addr[j+1][0]=c_if[j].eth_node_addr[0];
		p->if_addr[j+1][1]=c_if[j].eth_node_addr[1];
		p->if_addr[j+1][2]=c_if[j].eth_node_addr[2];
	    }
	    p->if_found[0]=i;
	    p->if_addr[0][0]=c_if[i].eth_node_addr[0];
	    p->if_addr[0][1]=c_if[i].eth_node_addr[1];
	    p->if_addr[0][2]=c_if[i].eth_node_addr[2];
	    lwcopy(counters, p->counters, (3+ sizeof(counters))/4 );
	    b->len = len + ETHER_HDR_SIZE;
	    sc->last_stat_sent=j;
	    make_pkt(sc, &(b->pkt), len);
	    bdg_enqueue(sc, b);
	}
    }
    print_addr(23, 12+i*14, &sc->queued, -2);
    for (j=0;j<NUM_COUNTERS;j++)
	print_stat(RXPKT_CTR + j*MAX_CARDS + i);
    if (i != 0) return;
    print_stat(BIGLOOP_CTR);
    print_stat(COLLSN_CTR);
    print_addr(5,26, &curr_time, -4);
    print_addr(4,18, &n_free, -2);
}
/* --------- end of file -------------- */
