/*
 *    **************************************************
 *    *                                                *
 *    *                   BIGMISC.C                    *
 *    *                                                *
 *    *          Extended Precision Calculator         *
 *    *                                                *
 *    *        Extended Precision Misc Routines        *
 *    *                                                *
 *    *              Version 4.4 05-31-89              *
 *    *                                                *
 *    *              Judson D. McClendon               *
 *    *              329 37th Court N.E.               *
 *    *              Birmingham, AL 35215              *
 *    *                 205-853-8440                   *
 *    *            Compuserve [74415,1003]             *
 *    *                                                *
 *    **************************************************
 */



/*
 *    **************************************************
 *    *                                                *
 *    *                   Includes                     *
 *    *                                                *
 *    **************************************************
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "bigcalc.h"
#include "biggvar.h"



/*
 *    **************************************************
 *    *                                                *
 *    *             Source Local Variables             *
 *    *                                                *
 *    **************************************************
 */

                     /* pi to 1100 digits */

static char piA[] = {   /* digits 1 - 500 */         /* Max string 512 */
   "31415926535897932384626433832795028841971693993751"
   "05820974944592307816406286208998628034825342117067"
   "98214808651328230664709384460955058223172535940812"
   "84811174502841027019385211055596446229489549303819"
   "64428810975665933446128475648233786783165271201909"
   "14564856692346034861045432664821339360726024914127"
   "37245870066063155881748815209209628292540917153643"
   "67892590360011330530548820466521384146951941511609"
   "43305727036575959195309218611738193261179310511854"
   "80744623799627495673518857527248912279381830119491"};

static char piB[] = {   /* digits 501 - 1000 */
   "29833673362440656643086021394946395224737190702179"
   "86094370277053921717629317675238467481846766940513"
   "20005681271452635608277857713427577896091736371787"
   "21468440901224953430146549585371050792279689258923"
   "54201995611212902196086403441815981362977477130996"
   "05187072113499999983729780499510597317328160963185"
   "95024459455346908302642522308253344685035261931188"
   "17101000313783875288658753320838142061717766914730"
   "35982534904287554687311595628638823537875937519577"
   "81857780532171226806613001927876611195909216420198"};

static char piC[] = {   /* digits 1001 - 1100 */
   "93809525720106548586327886593615338182796823030195"
   "20353018529689957736225994138912497217752834791315"};


                     /* pi/2 to 1100 digits */

static char halfpiA[] = {   /* digits 1 - 500 */         /* Max string 512 */
   "15707963267948966192313216916397514420985846996875"
   "52910487472296153908203143104499314017412671058533"
   "99107404325664115332354692230477529111586267970406"
   "42405587251420513509692605527798223114744774651909"
   "82214405487832966723064237824116893391582635600954"
   "57282428346173017430522716332410669680363012457063"
   "68622935033031577940874407604604814146270458576821"
   "83946295180005665265274410233260692073475970755804"
   "71652863518287979597654609305869096630589655255927"
   "40372311899813747836759428763624456139690915059745"};

static char halfpiB[] = {   /* digits 501 - 1000 */
   "64916836681220328321543010697473197612368595351089"
   "93047185138526960858814658837619233740923383470256"
   "60002840635726317804138928856713788948045868185893"
   "60734220450612476715073274792685525396139844629461"
   "77100997805606451098043201720907990681488738565498"
   "02593536056749999991864890249755298658664080481592"
   "97512229727673454151321261154126672342517630965594"
   "08550500156891937644329376660419071030858883457365"
   "17991267452143777343655797814319411768937968759788"
   "90928890266085613403306500963938305597954608210099"};

static char halfpiC[] = {   /* digits 1001 - 1100 */
   "46904762860053274293163943296807669091398411515097"
   "60176509264844978868112997069456248608876417395657"};


                     /* e to 1100 digits */

static char eA[] = {    /* digits 1 - 500 */         /* Max string 512 */
   "27182818284590452353602874713526624977572470936999"
   "59574966967627724076630353547594571382178525166427"
   "42746639193200305992181741359662904357290033429526"
   "05956307381323286279434907632338298807531952510190"
   "11573834187930702154089149934884167509244761460668"
   "08226480016847741185374234544243710753907774499206"
   "95517027618386062613313845830007520449338265602976"
   "06737113200709328709127443747047230696977209310141"
   "69283681902551510865746377211125238978442505695369"
   "67707854499699679468644549059879316368892300987931"};

static char eB[] = {   /* digits 501 - 1000 */
   "27736178215424999229576351482208269895193668033182"
   "52886939849646510582093923982948879332036250944311"
   "73012381970684161403970198376793206832823764648042"
   "95311802328782509819455815301756717361332069811250"
   "99618188159304169035159888851934580727386673858942"
   "28792284998920868058257492796104841984443634632449"
   "68487560233624827041978623209002160990235304369941"
   "84914631409343173814364054625315209618369088870701"
   "67683964243781405927145635490613031072085103837505"
   "10115747704171898610687396965521267154688957035035"};

static char eC[] = {   /* digits 1001 - 1100 */
   "40212340784981933432106817012100562788023519303322"
   "47450158539047304199577770935036604169973297250886"};


                     /* LN(10) to 1100 digits */

static char ln10A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "23025850929940456840179914546843642076011014886287"
   "72976033327900967572609677352480235997205089598298"
   "34196778404228624863340952546508280675666628736909"
   "87816894829072083255546808437998948262331985283935"
   "05308965377732628846163366222287698219886746543667"
   "47440424327436515504893431493939147961940440022210"
   "51017141748003688084012647080685567743216228355220"
   "11480466371565912137345074785694768346361679210180"
   "64450706480002775026849167465505868569356734206705"
   "81136429224554405758925724208241314695689016758940"};

static char ln10B[] = {   /* digits 501 - 1000 */
   "25677631135691929203337658714166023010570308963457"
   "20754403708474699401682692828084811842893148485249"
   "48644871927809676271275775397027668605952496716674"
   "18348570442250719796500471495105049221477656763693"
   "86629769795221107182645497347726624257094293225827"
   "98502585509785265383207606726317164309505995087807"
   "52371033310119785754733154142180842754386359177811"
   "70543098274823850456480190956102992918243182375253"
   "57709750539565187697510374970888692180205189339507"
   "23853920514463419726528728696511086257149219884997"};

static char ln10C[] = {   /* digits 1001 - 1100 */
   "87488737713456862091670584980782805975119385444500"
   "99781311469159346662410718466923101075984383191912"};


                     /* LN(.9) to 1100 digits */

static char lnP9A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "10536051565782630122750098083931279830612037298327"
   "40725639392336925840232401345464887656954621341207"
   "66027725910370517148673517013221767114560683627564"
   "22686827658166995879194648505249713751127872090836"
   "46753735546903376623278648795935883395531953832230"
   "68063737380570033668721271928105743343331765475507"
   "94614944081705922413762841002849656667436259102470"
   "59840699195837172323632818695878343286697490386837"
   "56269027085852455278757515428887026870363009824690"
   "11082327142665667835478506112527101842259460995695"};

static char lnP9B[] = {   /* digits 501 - 1000 */
   "66185808438491539044004722016071396991819933089951"
   "36510263864323300038738939668220467371784191475929"
   "86685982822084079834542016105247557309320878134190"
   "59615756458273314793523644227532690264532100041464"
   "72360147081252044683858538073749994146472651096247"
   "19576543319107662533692362403955709410367009490530"
   "49205022937133700879475429636102133797673326565200"
   "43594103248874058499648816054367564578501968701015"
   "94453148368584072225444462918836000317135394701735"
   "06295442271581066406201751744099906980646330903331"};

static char lnP9C[] = {   /* digits 1001 - 1100 */
   "94652628495773704947302137898129247855523065676807"
   "85754499445748903999957398542340330254289178206586"};


                     /* LN(.99) to 1100 digits */

static char lnP99A[] = {    /* digits 1 - 500 */     /* Max string 512 */
   "10050335853501441183548857558547706085515007674629"
   "87337869942552958300900425031364970819254909762769"
   "32797840645187672598274733197087864758716686058619"
   "00492707846971603671530739391724025806230580247206"
   "17865164542756934233059819918112542243687424360292"
   "94359501230410485470793917231588591444383104701161"
   "20194111146550799330396389404157201875299396788203"
   "91242742232546836953230479552424940087259011291507"
   "11097329723568570319116949818306487594953146830538"
   "58949466184487933084960707117093071364092020898927"};

static char lnP99B[] = {   /* digits 501 - 1000 */
   "46571581749210222077326391470369579846414072584662"
   "70622135437767791553808614377634738280145680995701"
   "14098531940607556356640812036549296319388571598176"
   "34914522270033364757602563930318393165484920865699"
   "57938163983079446395136128626914344078280087657471"
   "70813368990741450398620050723168375717304208495706"
   "12244300402114761596088119720156643590742443931854"
   "51292371472186701728655428637782051386601321849323"
   "40826115991832347875890716515729766415949966245448"
   "45650342903775968860019267495936954354893354067924"};

static char lnP99C[] = {   /* digits 1001 - 1100 */
   "90160703930353459813638976965902058951405992684709"
   "70214207155550604221069402972934195305038224526597"};


                     /* LN(.999) to 1100 digits */

static char lnP999A[] = {    /* digits 1 - 500 */    /* Max string 512 */
   "10005003335835335001429822540683449607552052504344"
   "09250988020797245202385869474688122817155429967880"
   "18431263808948128734298441901290606033024442610348"
   "67216410916952622895602909696647828915385026162729"
   "90966857436107999633073875363731189280341360848092"
   "42861809767373852152072711712258171935667719010311"
   "98785227466688711970147707618953119200323582469506"
   "80702121129678403553115532120870325308900430876320"
   "36205135523160659495542184955853957659625541740971"
   "16120644478003085119176624680867508202764157106383"};

static char lnP999B[] = {   /* digits 501 - 1000 */
   "85806603642277780720840689168603147843996398837817"
   "74033196613802472806044088160895550885582268739289"
   "46594689634546711578683599263658391173169865869075"
   "50975886539322579431218324124582670086518357002365"
   "98571467710848536080216559412771105222673205056669"
   "27928455781691616659095238819620795448199291152372"
   "05177936726180624823074512391052022279042533170246"
   "35226700182848502595292825433935019822407282109016"
   "63090431863480335220764664263842447084219339864518"
   "00157591216447352592671370111115695904518548209425"};

static char lnP999C[] = {   /* digits 1001 - 1100 */
   "80038344213262124282867835305335884269597591230726"
   "24180091824496360668174730633777881065863110897552"};


                     /* LN(.9999) to 1100 digits */

static char lnP9999A[] = {    /* digits 1 - 500 */   /* Max string 512 */
   "10000500033335833533350001428696439683539773457107"
   "55140898657627163447566114026170224959863374665287"
   "54178890672282471756725796221085164623132615895337"
   "63624630010418454030532657958553284926717668097503"
   "66453704508673105225036878099574184881241574306996"
   "02479139727577905513361598848420107930824747441437"
   "54400830277810747707877495945779058901263690675134"
   "98037695644449900901115316744551606509966123071296"
   "88068855412833938665195041607396824594548669758181"
   "47650673879414850807705004282597804431695048336440"};

static char lnP9999B[] = {   /* digits 501 - 1000 */
   "51341669719446708287045980753336064953371154124525"
   "43969544144000509695980555748266241648312175993302"
   "86004286961351471338434468807142558675846853433592"
   "01133468589300135711332289741845605445324420769019"
   "89660962248719449094195827815660278178392895843357"
   "11749457489199186888625740989474588482868154035234"
   "27611235084119816869945805045377327611916374688516"
   "75793669600645841920312034875470307723048715146253"
   "04916169286326959921025986124168216800441039763139"
   "99052892492800418297405135618198065311253654270327"};

static char lnP9999C[] = {   /* digits 1001 - 1100 */
   "84547932637240272006485629251179632981627821688537"
   "75442316425794682879847792138524769351974240715952"};


                     /* LN(.99999) to 1100 digits */

static char lnP99999A[] = {    /* digits 1 - 500 */  /* Max string 512 */
   "10000050000333335833353333500001428583928682540682"
   "54877353210755134469426136148636674376935861920829"
   "41735142472586415616762000865651052083312464588495"
   "54495151203930299334874604081394054793288978304832"
   "92521621950356253006582459265972679520764316774273"
   "22016519281453471095402151993532105172989789744976"
   "22066402850096243627773688938481873580069130249068"
   "29733169971932568868127449780969482064148415835627"
   "37747483172270701211278092878513458138924701406988"
   "52929332960599165101293873794775769729589078945746"};

static char lnP99999B[] = {   /* digits 501 - 1000 */
   "22419600304886879697329204816604909174563763403423"
   "97441084700412699714514592514059896896548737860637"
   "09985729928698697924025859747180251198268924078728"
   "80474829830423420241822171568268742976929128204744"
   "52944263357764317258624477956959906520290017627071"
   "02008975585191243248194559453916515458753605714095"
   "91073779167380196157384165762672909062692499458545"
   "49690097410815917984692554283693231564346999225481"
   "84027863823665494586561466897465546086698230819997"
   "64609854924845515597109326152706205350599560259781"};

static char lnP99999C[] = {   /* digits 1001 - 1100 */
   "98189548976873118670610057556227124517542175868536"
   "10727083087527560328597909809459043940209712870752"};


                     /* SIN(.1) to 1100 digits */

static char sinP1A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "99833416646828152306814198410622026989915388017982"
   "25999276686156165174428329242760966244380406303626"
   "78325031809359890354508072374704593788733561019849"
   "18410496834773050632832494359789005222424086081422"
   "72966743717361429690723851578245480827955359733878"
   "53463387923233361678867424907329845348005814497396"
   "04751862203867889743928811306547167725769195052793"
   "92262607481083336918563790338248847268301618655568"
   "89346698245411612735032192221340049562846009665223"
   "32207461112790826388892485997626004867369921168342"};

static char sinP1B[] = {   /* digits 501 - 1000 */
   "25822007849594489662342102170190915305175072701438"
   "37487428538739455042219898916646878947958333218732"
   "14892824314744574417630614483190418835012604416772"
   "12938344509031562810353361999685976614314713635237"
   "64713327576072485320419858982114221706486033745284"
   "22580789056073829983391226983813224628397600812369"
   "46677420279490476881259137535692930734542900571839"
   "93496383084481238029370793073519896487416427411495"
   "84626654721689231998075453149763755566049310161914"
   "74334017680553444002723416260117444319296284058354"};

static char sinP1C[] = {   /* digits 1001 - 1100 */
   "93872130849632187431309907771896446078938126363160"
   "25302131030949984668064192296841269237431597178125"};


                     /* SIN(.01) to 1100 digits */

static char sinP01A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "99998333341666646825424382690997290389643853601691"
   "51033879112479409734509063915965942636792961498990"
   "15251825689376067380711439147810183436799250452237"
   "48779233463339566295770428847517522816055835711010"
   "50774395187168606155330709987206369875092696688424"
   "90541364204623753507681641521959391575397060962515"
   "50071497343436501401260107564729605078738729840429"
   "87441343463278494770994371567032171767528027135974"
   "43546195233602035011214651999637411734890519279205"
   "51271878089079939686142777005076491985889067867722"};

static char sinP01B[] = {   /* digits 501 - 1000 */
   "05719520903185961474603095939933973363416265221714"
   "52145068273393341771117937273335459009509995988896"
   "19642913891756006434429991163737255940473661874082"
   "63088683497634340598889406453290876806826352554400"
   "42113324598097733014879809073704311558595748511922"
   "35000645606400377343263883770265172440601125789584"
   "28359074103973263511493912142496450758739296953977"
   "92073094118840236450685817485260602109928276704622"
   "51142999073649170506864819471418128310313128822546"
   "60611456524573907422800164140884130285721050703575"};

static char sinP01C[] = {   /* digits 1001 - 1100 */
   "01704080647353880794074342640290256620194547926244"
   "32167051360577584360724962828391639445978410560370"};


                     /* SIN(.001) to 1100 digits */

static char sinP001A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "99999983333334166666646825397100970015131473480865"
   "84190048145102714673516376365515440749327845858917"
   "04474712856562550569392029872924855755789892276667"
   "97704298110290998087807641517064252910574631431748"
   "50936016773958856266451298761230731759749305717989"
   "10550235392610771541261279604759660199348173407365"
   "78865897288333921120991800160521891039126169054427"
   "18873604879508582086968301228229164797145805071732"
   "32965551300106206405990035941144707201082513739520"
   "99779861547109162678316534591085861322505841086473"};

static char sinP001B[] = {   /* digits 501 - 1000 */
   "56906685531600061984773110968070961480863492393604"
   "72293539702984411547089341142580671593896263957801"
   "22101863893450590685505103137730021243817064332684"
   "05431094335900793630598856176174172417575590465793"
   "43221377047160432487483154447711767284980805828786"
   "52431950713139688921450934014473432877421098049187"
   "32029087150791132360397908105695331673661315874270"
   "94877607403626956830788134769606457740114267480351"
   "27361426130714955664817492955350563166004646658493"
   "24336279044509294780785285016881545351633507219491"};

static char sinP001C[] = {   /* digits 1001 - 1100 */
   "89915724145085050507523026051516345665181332046011"
   "69766743396337932246485484128386582874965066465440"};


                     /* COS(.1) to 1100 digits */

static char cosP1A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "99500416527802576609556198780387029483857622541508"
   "40359593527446852659102182404665296636185282629279"
   "10723685880836871860394139703522120459932582277765"
   "00233434126606061055341012689340392816994046794830"
   "27200020718214024442001160621592092042879890578727"
   "32689654637347621801752111388276881384838349480423"
   "27352219507885759748877880657414373095799340911315"
   "62646887269220290882659459060389392748252911983208"
   "12406203434702533302181978058808006060645784695402"
   "71540051162673225582298306135155856782760981045191"};

static char cosP1B[] = {   /* digits 501 - 1000 */
   "24203173805729717623413015685802692277952963048196"
   "95581708516886329798393454090838147538580719507353"
   "13037892361905672825085458775049615611448809472866"
   "51444413797913099178779125485426718947353264138239"
   "83422957335548076407296212010538573742696707806432"
   "60823341438366608860758765805498800986051276520656"
   "29924448402660404844961705157284628032560262233808"
   "28186943192824678771527190172705574611019968372833"
   "12566832594648414562425094985051732771234883152945"
   "22559346346243309348290864439511452419756927497513"};

static char cosP1C[] = {   /* digits 1001 - 1100 */
   "10676179622308536285252492870485071281885521678162"
   "67671288288109196720249973012691907856008716728411"};


                     /* COS(.01) to 1100 digits */

static char cosP01A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "99995000041666527778025793375220667321247058398027"
   "71111222757686464600881296434916351255437612973726"
   "56230561941513041834582815792993754862629509448536"
   "97292970784647512006914462382928612539466441550354"
   "57728832357826385724224649308096742936829445083538"
   "21781577471965569249697689958734005837731211373574"
   "85440439994355796196334951328177702604116145694127"
   "77336706911719098301564097003497342506438297965173"
   "28645325451458151717946847802184718372450569655622"
   "94195929079226212048522789105590903088120484847138"};

static char cosP01B[] = {   /* digits 501 - 1000 */
   "63004428493379141767370577843821480598708855134609"
   "79719812757116434617171373175337815552442269723199"
   "64508407408278929107116776376155434135436356633941"
   "04394842072117950771805396158667707823886239953368"
   "16615279174372882038310905003913887439032841067328"
   "90633020510872425101633363501039131638739757401409"
   "47370566536586394830489728404482917699629638057431"
   "76970309397338418277344406921271069049407460221888"
   "82531404254516877147340665076026618342561313147348"
   "04565874027443210391025721851648531160591697947530"};

static char cosP01C[] = {   /* digits 1001 - 1100 */
   "74187809011492744588254659713760991403095506170861"
   "52453206389454214684364014710510381215807269952331"};


                     /* COS(.001) to 1100 digits */

static char cosP001A[] = {    /* digits 1 - 500 */      /* Max string 512 */
   "99999950000004166666527777780257936480379188921289"
   "61458698562351112779445040208392490288883237630819"
   "53753587256368091915064737486642441428960104047064"
   "14575541558200886886668325062711490527942294855805"
   "61309935194598368152180098217664359538071378723338"
   "48724657137763505429298381693899720849748347295033"
   "33851881159120363391976978301914217279962544324209"
   "20919226144917975951951624931344012451863074884072"
   "30519808024456644721427335334399889159691389661476"
   "13914737488728908041758774390583877706135494129017"};

static char cosP001B[] = {   /* digits 501 - 1000 */
   "80709933688089544713267497493364039864297043637250"
   "99419648927857948970763793295781977802881003783415"
   "93094020198488385311496652735002947572963379425232"
   "68170074707686253042988700770409652389603430028247"
   "14914415029406845579741788550580044688554342445144"
   "68381776832965554654138249210759958048135186871432"
   "61591290025975006100421285272523991369898316685958"
   "10292144029551510454328611951654099003028564671852"
   "78937833540314882686099870223050547927445043052351"
   "91181631090880884648623855543982693629368409486485"};

static char cosP001C[] = {   /* digits 1001 - 1100 */
   "51270934105012682396774558823974435796721473943028"
   "10495338833331883272210005264468059211235493020749"};




/*
 *    **************************************************
 *    *                                                *
 *    *               Extended Routines                *
 *    *                                                *
 *    **************************************************
 */




/*
 *    **************************************************
 *    *                                                *
 *    *          Normalize Work Register [w]           *
 *    *                                                *
 *    **************************************************
 */

extern int Normalize(int w)

{

   int digits, shift;
   WORKDIGIT *wl, *wr;          /* Work left & right pointers */

   if (! work[w].digits) {             /* No digits, return zero */
      work[w].exp = 0L;
      work[w].sign = ' ';
      work[w].digits = 0;
      memset(work[w].man, 0, (workprec * sizeof(WORKDIGIT)));
      return(TRUE);
      }

   wl = &work[w].man[0] - 1;
   wr = &work[w].man[work[w].digits];
                                       /* Non significant digit to */
   *(wr + 1) = 1;                      /*  simplify next statement */

   while (! *(++wl))                   /* Find leftmost non zero digit */
      ;

   if (wl >= wr) {                     /* All digits zero, return zero */
      work[w].exp = 0L;
      work[w].sign = ' ';
      work[w].digits = 0;
      memset(work[w].man, 0, (workprec * sizeof(WORKDIGIT)));
      return(TRUE);
      }

   if (wr > wl + compprec)             /* Truncate to max precision */
      wr = wl + compprec;

   while (! *(--wr))                   /* Find rightmost non zero digit */
      ;

   digits = (wr - wl) + 1;             /* Number of significant digits */
   work[w].digits = digits;

   shift = wl - &work[w].man[0];       /* Normalize mantissa if needed */
   if (shift) {
      memmove(work[w].man, wl, (digits * sizeof(WORKDIGIT)) );
      work[w].exp -= (long)shift;
      }
                                       /* Zero digits beyond mantissa */

   memset(&work[w].man[digits], 0, ((workprec - digits) * sizeof(WORKDIGIT)) );

   if(work[w].exp > (MAXEXP + 1L)) {
      Overflow();
      return(FALSE);
      }

   if (work[w].exp < (MINEXP + 1L)) {
      ClearWork(w);
      }

   return(TRUE);

}




/*
 *    **************************************************
 *    *                                                *
 *    *                Flip Sign (+/-)                 *
 *    *                                                *
 *    **************************************************
 */

extern int FlipSign(int sign)

{

   if (sign == '-')
      return('+');

   if (sign == '+')
      return('-');

   return(' ');

}




/*
 *    **************************************************
 *    *                                                *
 *    *               Extended Rounding                *
 *    *                                                *
 *    *            work[2] = round(work[w])            *
 *    *                                                *
 *    **************************************************
 */

extern int ExtendedRound(int w)

{

   if ( (work[w].digits > normprec)
         &&
        (work[w].man[normprec] >= 5) ) {
      if (w) {
         MoveWorkWork(w, 0);
         }
      ClearWork(1);
      work[1].exp = work[0].exp - normprec;
      work[1].sign = work[0].sign;
      work[1].digits = 1;
      work[1].man[0] = 5;
      if (! ExtendedAdd()) {
         return(FALSE);
         }
      work[2].digits = normprec;
      }

   if (w != 2) {
      MoveWorkWork(2, w);
      }

   return(TRUE);

}




/*
 *    **************************************************
 *    *                                                *
 *    *             Extended Integer Part              *
 *    *                                                *
 *    *             work[0] = int(work[0])             *
 *    *                                                *
 *    **************************************************
 */

extern int ExtendedIntegerPart(void)

{

   int digits;

   digits = work[0].digits;

   if ( (! digits)                           /* If it's zero,       */
         ||
        (work[0].exp >= (long)digits) ) {    /*  or already integer */
      return(TRUE);                          /*  nothing to do      */
      }

   if (work[0].exp <= 0L) {                  /* All decimal,  */
      ClearWork(0);                          /*  return zero  */
      return(TRUE);
      }
                                             /* Zero decimal part */
   digits = (int)work[0].exp;
   memset(&work[0].man[digits], 0, ((workprec - digits) * sizeof(WORKDIGIT)));
   work[0].digits = digits;

   return(Normalize(0));

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Fraction Part              *
 *    *                                                *
 *    *            work[0] = frac(work[0])             *
 *    *                                                *
 *    **************************************************
 */

extern int ExtendedFractionPart(void)

{

   int digits;

   digits = work[0].digits;

   if ( (! digits)                           /* If it's zero,   */
         ||
        (work[0].exp <= 0L) ) {              /*  or all decimal */
      return(TRUE);                          /*  nothing to do  */
      }

   if (work[0].exp >= (long)digits) {        /* All integer,  */
      ClearWork(0);                          /*  return zero  */
      return(TRUE);
      }
                                             /* Zero integer part */
   digits = (int)work[0].exp;
   memset(&work[0].man[0], 0, (digits * sizeof(WORKDIGIT)));

   return(Normalize(0));

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Extended Get X from KBD              *
 *    *                                                *
 *    **************************************************
 */

extern int ExtendedGetX(void)

{

   long exponent;
   int sign, expsign;
   int row, col, exprow, expcol;
   int digits, intdigits, decdigits, exdigits, digitval;
   enum {ININT, INDEC, INEX} mode;
   BOOLEAN decimal;


/*  ******  I N I T I A L I Z E   V A R I A B L E S  ******  */

   sign = '+';                      /* Default to positive numbers */
   expsign = '+';

   row = entrysignrow;              /* Starting point for entry */
   col = MINDISPCOL;
   CurPos(row, col);

   exponent = 0L;                   /* Initialize digit work areas */
   digits = 0;
   intdigits = 0;
   decdigits = 0;
   exdigits = 0;
   digitval = 0;

   mode = ININT;                    /* Begin in integer mode */

   decimal = FALSE;                 /* No decimal point entered */

   ClearWork(0);


/*  ******  C H A R A C T E R   W A I T   L O O P  ******  */

   do {  /* while (TRUE) */

      /* First digit already entered on first pass */

      switch (chr) {             /* Test for number terminating operations */
         case (ADD):
         case (SUBTRACT):
         case (MULTIPLY):
         case (DIVIDE):
         case (HELP):
         case (POWER):
         case (SQUAREROOT):
         case (SQUARE):
         case (RECIPROCAL):
         case (FACTORIAL):
         case (INTEGER):
         case (FRACTION):
         case (RECALLPI):
         case (RECALLE):
         case (SIN):
         case (ARCSIN):
         case (COS):
         case (ARCCOS):
         case (TAN):
         case (ARCTAN):
         case (LOG):
         case (EXP10):
         case (LN):
         case (EXPE):
         case (LASTX):
         case (PRINTDISK):
         case (GROUPSIZE):
         case (VIEWREG):
         case (ENTER):
         case (SCINOT):
         case (CLEAR):
         case (PRINT):
         case (STOREX):
         case (RECALLREG):
         case (XCHGXY1):
         case (XCHGXY2):
         case (XCHGXREG):
         case (ROLLDOWN):
         case (ROLLUP):
            charpresent = TRUE;  /* Pass terminating operator */
         }  /* switch */

      if (charpresent)           /* Break from while */
         break;

      if (! isascii(chr))        /* Ignore non ASCII characters */
         continue;

/*  ******  I N T E G E R   M O D E  ******  */

      if (mode == ININT) {

         if (isdigit(chr)) {                    /* Numeric digit */
            if (digits < normprec) {
               DisplayChar(&row, &col, chr);
               work[0].man[digits] = chr - '0';
               digitval += work[0].man[digits];
               intdigits++;
               digits++;
               }
            else
               Beep();
            }

         else if (chr == '.') {                 /* . invokes decimal mode */
            DisplayChar(&row, &col, chr);
            decimal = TRUE;
            mode = INDEC;
            }

         else if (chr == 'E') {                 /* E invokes exponent mode, */
            if (! digits) {
               DisplayChar(&row, &col, '1');    /*  if no digits, make it 1, */
               work[0].man[digits] = 1;
               digitval += 1;
               intdigits++;
               digits++;
               DisplayExpChar(&row, &col, ' ');
               DisplayExpChar(&row, &col, 'e');
               CurGet(&exprow, &expcol);
               mode = INEX;
               }
            else if (digitval > 0) {            /*  or if non zero already */
               DisplayExpChar(&row, &col, ' ');
               DisplayExpChar(&row, &col, 'e');
               CurGet(&exprow, &expcol);
               mode = INEX;
               }
            else
               Beep();
            }

         else if (chr == CHGSIGN) {             /* Reverse mantissa sign */
            sign = FlipSign(sign);
            CurPos(entrysignrow, SIGNDISPCOL);
            if (sign == '+')
               WChar(' ');
            else
               WChar('-');
            CurPos(row, col);
            }

         else if (chr == BACKSPACE) {           /* Backspace backs up char */
            if (digits) {
               BackSpace(&row, &col);
               digits--;
               intdigits--;
               digitval -= work[0].man[digits];
               work[0].man[digits] = 0;
               }
            else
               Beep();
            }

         else if (chr == ESCAPE)                /* Changed his/her mind */
            return(FALSE);

         else                                   /* Bad keystroke */
            Beep();

         }  /* End integer part */


/*  ******  D E C I M A L   M O D E  ******  */

      else if (mode == INDEC) {

         if (isdigit(chr)) {                    /* Numeric digit */
            if (digits < normprec) {
               DisplayChar(&row, &col, chr);
               work[0].man[digits] = chr - '0';
               digitval += work[0].man[digits];
               decdigits++;
               digits++;
               }
            else {
               Beep();
               }
            }

         else if (chr == 'E') {                 /* E invokes exponent mode     */
            if (digitval > 0) {                 /*  if non zero digits entered */
               DisplayExpChar(&row, &col, ' ');
               DisplayExpChar(&row, &col, 'e');
               CurGet(&exprow, &expcol);
               mode = INEX;
               }
            else
               Beep();
            }

         else if (chr == CHGSIGN) {             /* Reverse mantissa sign */
            sign = FlipSign(sign);
            CurPos(entrysignrow, SIGNDISPCOL);
            if (sign == '+')
               WChar(' ');
            else
               WChar('-');
            CurPos(row, col);
            }

         else if (chr == BACKSPACE) {           /* Backspace backs up char */
            BackSpace(&row, &col);

            if (decdigits) {           /* Decimal digits, stay decimal mode */
               digits--;
               decdigits--;
               digitval -= work[0].man[digits];
               work[0].man[digits] = 0;
               }

            else {
               decimal = FALSE;        /* Wiped out decimal point */
               mode = ININT;           /* No digits, back in integer mode */
               }
            }

         else if (chr == ESCAPE)                /* Changed his/her mind */
            return(FALSE);

         else                                   /* Bad keystroke */
            Beep();

         }  /* End decimal part  */


/*  ******  E X P O N E N T   M O D E  ******  */

      else {   /* mode == INEXP */

         if (isdigit(chr)) {                    /* numeric digit */
            if (exdigits < MAXEXDIGITS) {
               exponent = (exponent * 10L) + (long)(chr - '0');
               DisplayExpChar(&row, &col, chr);
               exdigits++;
               }
            else {
                Beep();
               }
            }

         else if (chr == CHGSIGN) {             /* Sign ok if first char */
               expsign = FlipSign(expsign);
               DisplayExp(&row, &col, exprow, expcol, expsign, exponent);
               }

         else if (chr == BACKSPACE) {           /* Backspace backs up char */
            BackSpaceExp(&row, &col);

            if (exdigits) {            /* Exdigits means more exp to left */
               exdigits--;             /* Back out latest digits */
               exponent /= 10L;
               }

            else if (expsign == '-')    /* If sign entered, its gone now */
               expsign = '+';

            else {
               BackSpaceExp(&row, &col);  /* Must have been " e", dump space */

               if (decdigits || decimal)  /* If decimal digits or decimal */
                  mode = INDEC;           /*  point entered, decimal mode */
               else
                  mode = ININT;           /*  else must be integer part */
               }
            }

         else if (chr == ESCAPE)                /* Changed his/her mind */
            return(FALSE);

         else                                   /* Bad keystroke */
            Beep();

         }  /* mode == INEX */

      chr = GetChar();                    /* Get next char from kbd */

      } while (TRUE);

   if (expsign == '-')                 /* Invert exponent if neg sign */
      exponent = - exponent;

   if (digitval) {                     /* Number not zero */
      work[0].exp    = exponent + (long)intdigits;
      work[0].sign   = sign;
      work[0].digits = digits;
      return(Normalize(0));
      }
   else {                              /* Null number == 0 */
      work[0].exp    = 0L;
      work[0].sign   = ' ';
      work[0].digits = 0;
      return(TRUE);
      }

}





/*
 *    **************************************************
 *    *                                                *
 *    *                 Data Movement                  *
 *    *                                                *
 *    **************************************************
 */




/*
 *    **************************************************
 *    *                                                *
 *    *              Extended Recall Pi                *
 *    *                                                *
 *    *                work[dest] = pi                 *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallPi(int dest)

{

   WORKDIGIT *workx, *workend;
   char *pix;


   work[dest].exp = 1L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   pix = piA;
   while ((*pix) && (workx < workend))
      *(workx++) = (int) *(pix++) - '0';

   pix = piB;
   while ((*pix) && (workx < workend))
      *(workx++) = (int) *(pix++) - '0';

   pix = piC;
   while ((*pix) && (workx < workend))
      *(workx++) = (int) *(pix++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *             Extended Recall Pi/2               *
 *    *                                                *
 *    *               work[dest] = pi/2                *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallHalfPi(int dest)

{

   WORKDIGIT *workx, *workend;
   char *halfpix;


   work[dest].exp = 1L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   halfpix = halfpiA;
   while ((*halfpix) && (workx < workend))
      *(workx++) = (int) *(halfpix++) - '0';

   halfpix = halfpiB;
   while ((*halfpix) && (workx < workend))
      *(workx++) = (int) *(halfpix++) - '0';

   halfpix = halfpiC;
   while ((*halfpix) && (workx < workend))
      *(workx++) = (int) *(halfpix++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *              Extended Recall e                 *
 *    *                                                *
 *    *                work[dest] = e                  *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallE(int dest)

{

   WORKDIGIT *workx, *workend;
   char *ex;


   work[dest].exp = 1L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   ex = eA;
   while ((*ex) && (workx < workend))
      *(workx++) = (int) *(ex++) - '0';

   ex = eB;
   while ((*ex) && (workx < workend))
      *(workx++) = (int) *(ex++) - '0';

   ex = eC;
   while ((*ex) && (workx < workend))
      *(workx++) = (int) *(ex++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *             Extended Recall ln10               *
 *    *                                                *
 *    *               work[dest] = ln10                *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallLn10(int dest)

{

   WORKDIGIT *workx, *workend;
   char *ln10x;


   work[dest].exp = 1L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   ln10x = ln10A;
   while ((*ln10x) && (workx < workend))
      *(workx++) = (int) *(ln10x++) - '0';

   ln10x = ln10B;
   while ((*ln10x) && (workx < workend))
      *(workx++) = (int) *(ln10x++) - '0';

   ln10x = ln10C;
   while ((*ln10x) && (workx < workend))
      *(workx++) = (int) *(ln10x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *             Extended Recall ln(.9)             *
 *    *                                                *
 *    *               work[dest] = ln(.9)              *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallLnP9(int dest)

{

   WORKDIGIT *workx, *workend;
   char *lnP9x;


   work[dest].exp = 0L;
   work[dest].sign = '-';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   lnP9x = lnP9A;
   while ((*lnP9x) && (workx < workend))
      *(workx++) = (int) *(lnP9x++) - '0';

   lnP9x = lnP9B;
   while ((*lnP9x) && (workx < workend))
      *(workx++) = (int) *(lnP9x++) - '0';

   lnP9x = lnP9C;
   while ((*lnP9x) && (workx < workend))
      *(workx++) = (int) *(lnP9x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Recall ln(.99)             *
 *    *                                                *
 *    *              work[dest] = ln(.99)              *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallLnP99(int dest)

{

   WORKDIGIT *workx, *workend;
   char *lnP99x;


   work[dest].exp = -1L;
   work[dest].sign = '-';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   lnP99x = lnP99A;
   while ((*lnP99x) && (workx < workend))
      *(workx++) = (int) *(lnP99x++) - '0';

   lnP99x = lnP99B;
   while ((*lnP99x) && (workx < workend))
      *(workx++) = (int) *(lnP99x++) - '0';

   lnP99x = lnP99C;
   while ((*lnP99x) && (workx < workend))
      *(workx++) = (int) *(lnP99x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Recall ln(.999)            *
 *    *                                                *
 *    *              work[dest] = ln(.999)             *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallLnP999(int dest)

{

   WORKDIGIT *workx, *workend;
   char *lnP999x;


   work[dest].exp = -2L;
   work[dest].sign = '-';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   lnP999x = lnP999A;
   while ((*lnP999x) && (workx < workend))
      *(workx++) = (int) *(lnP999x++) - '0';

   lnP999x = lnP999B;
   while ((*lnP999x) && (workx < workend))
      *(workx++) = (int) *(lnP999x++) - '0';

   lnP999x = lnP999C;
   while ((*lnP999x) && (workx < workend))
      *(workx++) = (int) *(lnP999x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Extended Recall ln(.9999)            *
 *    *                                                *
 *    *             work[dest] = ln(.9999)             *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallLnP9999(int dest)

{

   WORKDIGIT *workx, *workend;
   char *lnP9999x;


   work[dest].exp = -3L;
   work[dest].sign = '-';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   lnP9999x = lnP9999A;
   while ((*lnP9999x) && (workx < workend))
      *(workx++) = (int) *(lnP9999x++) - '0';

   lnP9999x = lnP9999B;
   while ((*lnP9999x) && (workx < workend))
      *(workx++) = (int) *(lnP9999x++) - '0';

   lnP9999x = lnP9999C;
   while ((*lnP9999x) && (workx < workend))
      *(workx++) = (int) *(lnP9999x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Extended Recall ln(.99999)           *
 *    *                                                *
 *    *             work[dest] = ln(.99999)            *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallLnP99999(int dest)

{

   WORKDIGIT *workx, *workend;
   char *lnP99999x;


   work[dest].exp = -4L;
   work[dest].sign = '-';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   lnP99999x = lnP99999A;
   while ((*lnP99999x) && (workx < workend))
      *(workx++) = (int) *(lnP99999x++) - '0';

   lnP99999x = lnP99999B;
   while ((*lnP99999x) && (workx < workend))
      *(workx++) = (int) *(lnP99999x++) - '0';

   lnP99999x = lnP99999C;
   while ((*lnP99999x) && (workx < workend))
      *(workx++) = (int) *(lnP99999x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Recall sin(.1)             *
 *    *                                                *
 *    *              work[dest] = sin(.1)              *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallSinP1(int dest)

{

   WORKDIGIT *workx, *workend;
   char *sinP1x;


   work[dest].exp = -1L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   sinP1x = sinP1A;
   while ((*sinP1x) && (workx < workend))
      *(workx++) = (int) *(sinP1x++) - '0';

   sinP1x = sinP1B;
   while ((*sinP1x) && (workx < workend))
      *(workx++) = (int) *(sinP1x++) - '0';

   sinP1x = sinP1C;
   while ((*sinP1x) && (workx < workend))
      *(workx++) = (int) *(sinP1x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Recall sin(.01)            *
 *    *                                                *
 *    *              work[dest] = sin(.01)             *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallSinP01(int dest)

{

   WORKDIGIT *workx, *workend;
   char *sinP01x;


   work[dest].exp = -2L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   sinP01x = sinP01A;
   while ((*sinP01x) && (workx < workend))
      *(workx++) = (int) *(sinP01x++) - '0';

   sinP01x = sinP01B;
   while ((*sinP01x) && (workx < workend))
      *(workx++) = (int) *(sinP01x++) - '0';

   sinP01x = sinP01C;
   while ((*sinP01x) && (workx < workend))
      *(workx++) = (int) *(sinP01x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Extended Recall sin(.001)            *
 *    *                                                *
 *    *             work[dest] = sin(.001)             *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallSinP001(int dest)

{

   WORKDIGIT *workx, *workend;
   char *sinP001x;


   work[dest].exp = -3L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   sinP001x = sinP001A;
   while ((*sinP001x) && (workx < workend))
      *(workx++) = (int) *(sinP001x++) - '0';

   sinP001x = sinP001B;
   while ((*sinP001x) && (workx < workend))
      *(workx++) = (int) *(sinP001x++) - '0';

   sinP001x = sinP001C;
   while ((*sinP001x) && (workx < workend))
      *(workx++) = (int) *(sinP001x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Recall cos(.1)             *
 *    *                                                *
 *    *              work[dest] = cos(.1)              *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallCosP1(int dest)

{

   WORKDIGIT *workx, *workend;
   char *cosP1x;


   work[dest].exp = 0L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   cosP1x = cosP1A;
   while ((*cosP1x) && (workx < workend))
      *(workx++) = (int) *(cosP1x++) - '0';

   cosP1x = cosP1B;
   while ((*cosP1x) && (workx < workend))
      *(workx++) = (int) *(cosP1x++) - '0';

   cosP1x = cosP1C;
   while ((*cosP1x) && (workx < workend))
      *(workx++) = (int) *(cosP1x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *            Extended Recall cos(.01)            *
 *    *                                                *
 *    *              work[dest] = cos(.01)             *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallCosP01(int dest)

{

   WORKDIGIT *workx, *workend;
   char *cosP01x;


   work[dest].exp = 0L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   cosP01x = cosP01A;
   while ((*cosP01x) && (workx < workend))
      *(workx++) = (int) *(cosP01x++) - '0';

   cosP01x = cosP01B;
   while ((*cosP01x) && (workx < workend))
      *(workx++) = (int) *(cosP01x++) - '0';

   cosP01x = cosP01C;
   while ((*cosP01x) && (workx < workend))
      *(workx++) = (int) *(cosP01x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Extended Recall cos(.001)            *
 *    *                                                *
 *    *             work[dest] = cos(.001)             *
 *    *                                                *
 *    **************************************************
 */

extern void ExtendedRecallCosP001(int dest)

{

   WORKDIGIT *workx, *workend;
   char *cosP001x;


   work[dest].exp = 0L;
   work[dest].sign = '+';
   work[dest].digits = compprec;

   workx = work[dest].man;
   workend = &work[dest].man[compprec];

   cosP001x = cosP001A;
   while ((*cosP001x) && (workx < workend))
      *(workx++) = (int) *(cosP001x++) - '0';

   cosP001x = cosP001B;
   while ((*cosP001x) && (workx < workend))
      *(workx++) = (int) *(cosP001x++) - '0';

   cosP001x = cosP001C;
   while ((*cosP001x) && (workx < workend))
      *(workx++) = (int) *(cosP001x++) - '0';

   Normalize(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *       Clear Stack Registers [lo to hi]         *
 *    *                                                *
 *    **************************************************
 */

extern void ClearStack(int lo, int hi)

{

   int s;

   for (s = lo; s <= hi; s++) {
      stack[s].exp = 0L;
      stack[s].sign = ' ';
      stack[s].digits = 0;
      memset(stack[s].man, 0, (normprec * sizeof(NORMDIGIT)));
      }

}




/*
 *    **************************************************
 *    *                                                *
 *    *          Clear Registers [lo to hi]            *
 *    *                                                *
 *    **************************************************
 */

extern void ClearReg(int lo, int hi)

{

   int r;

   for (r = lo; r <= hi; r++) {
      reg[r].exp = 0L;
      reg[r].sign = ' ';
      reg[r].digits = 0;
      memset(reg[r].man, 0, (normprec * sizeof(NORMDIGIT)));
      }

}




/*
 *    **************************************************
 *    *                                                *
 *    *        Clear Work Registers [lo to hi]         *
 *    *                                                *
 *    **************************************************
 */

extern void ClearWork(int w)

{

   work[w].exp = 0L;
   work[w].sign = ' ';
   work[w].digits = 0;
   memset(work[w].man, 0, (workprec * sizeof(WORKDIGIT)));

}




/*
 *    **************************************************
 *    *                                                *
 *    *          Clear Temp Register [*temp]           *
 *    *                                                *
 *    **************************************************
 */

extern void ClearTemp(COMPTYPE *temp)

{

   temp->exp = 0L;
   temp->sign = ' ';
   temp->digits = 0;
   memset(temp->man, 0, (compprec * sizeof(WORKDIGIT)));

}




/*
 *    **************************************************
 *    *                                                *
 *    *        Move Stack[source] to Work[dest]        *
 *    *                                                *
 *    **************************************************
 */

extern void MoveStackWork(int source, int dest)

{

   int size, i;

   if (size = stack[source].digits) {
      if (size > workprec)
         size = workprec;
      work[dest].exp    = stack[source].exp;
      work[dest].sign   = stack[source].sign;
      work[dest].digits = size;

      for (i = 0; i < size; i++)
         work[dest].man[i] = (WORKDIGIT)stack[source].man[i];

      if (size < workprec)
         memset(&work[dest].man[size], 0, ((workprec - size) * sizeof(WORKDIGIT)));
      }
   else
      ClearWork(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *      Move Register[source] to Work[dest]       *
 *    *                                                *
 *    **************************************************
 */

extern void MoveRegWork(int source, int dest)

{

   int size, i;

   if (size = reg[source].digits) {
      if (size > workprec)
         size = workprec;
      work[dest].exp    = reg[source].exp;
      work[dest].sign   = reg[source].sign;
      work[dest].digits = size;

      for (i = 0; i < size; i++)
         work[dest].man[i] = (WORKDIGIT)reg[source].man[i];

      if (size < workprec)
         memset(&work[dest].man[size], 0, ((workprec - size) * sizeof(WORKDIGIT)));
      }
   else
      ClearWork(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *        Move Work[source] to Stack[dest]        *
 *    *                                                *
 *    **************************************************
 */

extern void MoveWorkStack(int source, int dest)

{

   int size, i;

   if (size = work[source].digits) {
      if (size > normprec)
         size = normprec;
      stack[dest].exp  = work[source].exp;
      stack[dest].sign = work[source].sign;

      for (i = 0; i < size; i++)
         stack[dest].man[i] = (NORMDIGIT)work[source].man[i];

      if (size < normprec)
         memset(&stack[dest].man[size], 0, ((normprec - size) * sizeof(NORMDIGIT)));

      while (!stack[dest].man[size - 1])
         size--;
      stack[dest].digits = size;
      }
   else
      ClearStack(dest, dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *         Move Work[source] to Reg[dest]         *
 *    *                                                *
 *    **************************************************
 */

extern void MoveWorkReg(int source, int dest)

{

   int size, i;

   if (size = work[source].digits) {
      if (size > normprec)
         size = normprec;
      reg[dest].exp  = work[source].exp;
      reg[dest].sign = work[source].sign;

      for (i = 0; i < size; i++)
         reg[dest].man[i] = (NORMDIGIT)work[source].man[i];

      if (size < normprec)
         memset(&reg[dest].man[size], 0, ((normprec - size) * sizeof(NORMDIGIT)));

      while (!reg[dest].man[size - 1])
         size--;
      reg[dest].digits = size;
      }
   else
      ClearReg(dest, dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *        Move Work[source] to Work[*dest]        *
 *    *                                                *
 *    **************************************************
 */

extern void MoveWorkWork(int source, int dest)

{

   int size;

   if (size = work[source].digits) {
      if (size > compprec)
         size = compprec;
      work[dest].exp    = work[source].exp;
      work[dest].sign   = work[source].sign;
      work[dest].digits = size;

      memcpy(work[dest].man, work[source].man, (size * sizeof(WORKDIGIT)));

      if (size < compprec)
         memset(&work[dest].man[size], 0, ((compprec - size) * sizeof(WORKDIGIT)));
      }
   else
      ClearWork(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *        Move Work[source] to Temp[*dest]        *
 *    *                                                *
 *    **************************************************
 */

extern void MoveWorkTemp(int source, COMPTYPE *dest)

{

   int size;

   if (size = work[source].digits) {
      if (size > compprec)
         size = compprec;
      dest->exp    = work[source].exp;
      dest->sign   = work[source].sign;
      dest->digits = size;

      memcpy(dest->man, work[source].man, (size * sizeof(WORKDIGIT)));

      if (size < compprec)
         memset(&dest->man[size], 0, ((compprec - size) * sizeof(WORKDIGIT)));
      }
   else
      ClearTemp(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *        Move Temp[*source] to Work[dest]        *
 *    *                                                *
 *    **************************************************
 */

extern void MoveTempWork(COMPTYPE *source, int dest)

{

   int size;

   if (size = source->digits) {
      if (size > workprec)
         size = workprec;
      work[dest].exp    = source->exp;
      work[dest].sign   = source->sign;
      work[dest].digits = size;

      memcpy(work[dest].man, source->man, (size * sizeof(WORKDIGIT)));

      if (size < workprec)
         memset(&work[dest].man[size], 0, ((workprec - size) * sizeof(WORKDIGIT)));
      }
   else
      ClearWork(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *       Move Temp[*source] to Temp[*dest]        *
 *    *                                                *
 *    **************************************************
 */

extern void MoveTempTemp(COMPTYPE *source, COMPTYPE *dest)

{

   int size;

   if (size = source->digits) {
      if (size > compprec)
         size = compprec;
      dest->exp    = source->exp;
      dest->sign   = source->sign;
      dest->digits = size;

      memcpy(dest->man, source->man, (size * sizeof(WORKDIGIT)));

      if (size < compprec)
         memset(&dest->man[size], 0, ((compprec - size) * sizeof(WORKDIGIT)));
      }
   else
      ClearTemp(dest);

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Set Work[dest] to Integer            *
 *    *                                                *
 *    **************************************************
 */

extern void SetWorkInteger(int dest, long integer)

{

   int i;
   long order;

   if (integer < 0L) {
      work[dest].sign = '-';
      integer = -integer;
      }
   else if (integer > 0L) {
      work[dest].sign = '+';
      }
   else {
      return;
      }

   order = 1000000000L;
   while (order > integer)
      order /= 10L;
   i = 0;
   while (order) {
      work[dest].man[i] = (WORKDIGIT) (integer / order);
      integer %= order;
      order /= 10L;
      i++;
      }
   memset(&work[dest].man[i], 0, ((workprec - i) * sizeof(WORKDIGIT)));
   work[dest].digits = i;
   work[dest].exp = (long) i;

}




/*
 *    **************************************************
 *    *                                                *
 *    *           Set Temp[*dest] to Integer           *
 *    *                                                *
 *    **************************************************
 */

extern void SetTempInteger(COMPTYPE *dest, long integer)

{

   int i;
   long order;

   if (integer < 0L) {
      dest->sign = '-';
      integer = -integer;
      }
   else if (integer > 0L) {
      dest->sign = '+';
      }
   else {
      return;
      }

   order = 1000000000L;
   while (order > integer)
      order /= 10L;
   i = 0;
   while (order) {
      dest->man[i] = (WORKDIGIT) (integer / order);
      integer %= order;
      order /= 10L;
      i++;
      }
   memset(&dest->man[i], 0, ((compprec - i) * sizeof(WORKDIGIT)));
   dest->digits = i;
   dest->exp = (long) i;

}
