/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * INCLUDES
 *
 */

#include "mylink.h"
#include "types.h"
#include "symbol.h"
#include "asm.h"
#include "main.h"
#include	"rpn.h"
#include "output.h"


#include	<stdio.h>
#include	<string.h>

void mergetwoexpressions( struct Expression *expr, struct Expression *src1, struct Expression *src2 )
{
   *expr = *src1;
   memcpy( &(expr->tRPN[expr->nRPNLength]), src2->tRPN, src2->nRPNLength );

   expr->nRPNLength += src2->nRPNLength;
   expr->isReloc |= src2->isReloc;
}

#define	joinexpr() mergetwoexpressions(expr,src1,src2)

/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * VARIABLES
 *
 */

/*
UBYTE   rpnexpr[2048];
ULONG   rpnptr = 0;
ULONG   rpnoutptr = 0;
ULONG   reloc = 0;
ULONG   pcrel = 0;
*/

/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * Add a byte to the RPN expression
 *
 */

void    pushbyte (struct Expression *expr, int b)
{
    expr->tRPN[expr->nRPNLength++] = b & 0xFF;
}

/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * Reset the RPN module
 *
 */

void    rpn_Reset (struct Expression *expr)
{
    expr->nRPNLength = expr->nRPNOut = expr->isReloc = 0;
    expr->fixSectionAddress = -1;
    expr->fixBank = -1;
    expr->intraSection = 1;
}

/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * Returns the next rpn byte in expression
 *
 */

UWORD   rpn_PopByte (struct Expression *expr)
{
    if (expr->nRPNOut == expr->nRPNLength)
    {
	return (0xDEAD);
    }
    else
    	return (expr->tRPN[expr->nRPNOut++]);
}

/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * Determine if the current expression is relocatable
 *
 */

ULONG   rpn_isReloc (struct Expression *expr)
{
    return (expr->isReloc);
}

/*
 * RGBAsm - RPN.C - Controls RPN expressions for objectfiles
 *
 * Add symbols, constants and operators to expression
 *
 */

void    rpn_Number (struct Expression *expr, ULONG i)
{
    rpn_Reset (expr);
    pushbyte (expr, RPN_CONST);
    pushbyte (expr, i);
    pushbyte (expr, i >> 8);
    pushbyte (expr, i >> 16);
    pushbyte (expr, i >> 24);
    expr->nVal = i;
    expr->fixBank = ANY_BANK;
}

void    rpn_Symbol (struct Expression *expr, char *tzSym)
{
    if (!sym_isConstant (tzSym))
    {
       	struct sSymbol *psym;

   	rpn_Reset(expr);

   	psym = sym_FindSymbol (tzSym);

	expr->fixSectionAddress = -1;
	expr->fixBank = -1;

	if(psym!=NULL)
	{
		if((psym->pSection != NULL) && (pCurrentSection != NULL))
		{
			expr->intraSection = (psym->pSection == pCurrentSection);

			if ( ((psym->pSection)->nOrg != (ULONG)-1) )
			{

				expr->fixSectionAddress = (psym->pSection)->nOrg;
				if( 	( ( expr->fixSectionAddress >= 0x0000 ) && ( expr->fixSectionAddress <= 0x3FFF ) ) ||
					( ( expr->fixSectionAddress >= 0xC000 ) && ( expr->fixSectionAddress <= 0xCFFF ) ) ||
					( ( expr->fixSectionAddress >= 0xFE00 ) && ( expr->fixSectionAddress <= 0xFFFF ) ))
					expr->fixBank = 0;
				else
					expr->fixBank = (psym->pSection)->nBank;
			}
		}
	}
	
	if (psym == NULL || psym->pSection == pCurrentSection || psym->pSection == NULL)
	{
	    expr->fixSectionAddress = -1;
	    if(pCurrentSection)
            	expr->fixBank = pCurrentSection->nBank;
	    else
	    	expr->fixBank = -1;
	}

	expr->isReloc = 1;
	pushbyte (expr,RPN_SYM);
	while (*tzSym)
	    pushbyte (expr,*tzSym++);
	pushbyte (expr,0);
    }
    else
    {
	rpn_Number (expr,sym_GetConstantValue (tzSym));
    }
}

void    rpn_Bank (struct Expression *expr,char *tzSym)
{
    if (!sym_isConstant (tzSym))
    {
	struct sSymbol *psym;

   	rpn_Reset( expr );

	psym = sym_FindSymbol (tzSym);
	if (nPass == 2 && psym == NULL)
	{
	    yyerror ("'%s' not defined", tzSym);
	}
	expr->isReloc = 1;
	pushbyte (expr,RPN_BANK);
	while (*tzSym)
	    pushbyte (expr,*tzSym++);
	pushbyte (expr,0);
    }
    else
	yyerror ("BANK argument must be a relocatable identifier");
}

int	rpn_RangeCheck( struct Expression *expr, struct Expression *src, SLONG low, SLONG high )
{
	*expr=*src;

	if( rpn_isReloc(src) )
	{
		pushbyte( expr, RPN_RANGECHECK );
		pushbyte( expr, low );
		pushbyte( expr, low>>8 );
		pushbyte( expr, low>>16 );
		pushbyte( expr, low>>24 );
		pushbyte( expr, high );
		pushbyte( expr, high>>8 );
		pushbyte( expr, high>>16 );
		pushbyte( expr, high>>24 );
		return( 1 );
	}
	else
	{
		return( expr->nVal>=low && expr->nVal<=high );
	}
}

#ifdef	GAMEBOY
void    rpn_CheckHRAM (struct Expression *expr, struct Expression *src)
{
   *expr = *src;
   pushbyte (expr, RPN_HRAM);
}
#endif

#ifdef	PCENGINE
void    rpn_CheckZP (struct Expression *expr, struct Expression *src)
{
   *expr = *src;
   pushbyte (expr, RPN_PCEZP);
}
#endif

void    rpn_LOGNOT (struct Expression *expr, struct Expression *src)
{
   *expr = *src;
   pushbyte (expr, RPN_LOGUNNOT);
}

void    rpn_LOGOR (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal||src2->nVal);
   pushbyte (expr,RPN_LOGOR);
}

void    rpn_LOGAND (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal&&src2->nVal);
   pushbyte (expr,RPN_LOGAND);
}

void    rpn_LOGEQU (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal==src2->nVal);
   pushbyte (expr,RPN_LOGEQ);
}

void    rpn_LOGGT (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal>src2->nVal);
   pushbyte (expr,RPN_LOGGT);
}

void    rpn_LOGLT (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal<src2->nVal);
   pushbyte (expr,RPN_LOGLT);
}

void    rpn_LOGGE (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal>=src2->nVal);
   pushbyte (expr,RPN_LOGGE);
}

void    rpn_LOGLE (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal<=src2->nVal);
   pushbyte (expr,RPN_LOGLE);
}

void    rpn_LOGNE (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal!=src2->nVal);
    pushbyte (expr,RPN_LOGNE);
}

void    rpn_ADD (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal+src2->nVal);
    pushbyte (expr,RPN_ADD);
}

void    rpn_SUB (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal-src2->nVal);
    pushbyte (expr,RPN_SUB);
}

void    rpn_XOR (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal^src2->nVal);
    pushbyte (expr,RPN_XOR);
}

void    rpn_OR (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal|src2->nVal);
    pushbyte (expr,RPN_OR);
}

void    rpn_AND (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal&src2->nVal);
    pushbyte (expr,RPN_AND);
}

void    rpn_SHL (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal<<src2->nVal);
    pushbyte (expr,RPN_SHL);
}

void    rpn_SHR (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal>>src2->nVal);
    pushbyte (expr,RPN_SHR);
}

void    rpn_MUL (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal*src2->nVal);
    pushbyte (expr,RPN_MUL);
}

void    rpn_DIV (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal/src2->nVal);
    pushbyte (expr,RPN_DIV);
}

void    rpn_MOD (struct Expression *expr, struct Expression *src1, struct Expression *src2)
{
   joinexpr();
   expr->nVal = (expr->nVal%src2->nVal);
    pushbyte (expr,RPN_MOD);
}

void    rpn_UNNEG (struct Expression *expr, struct Expression *src)
{
   *expr = *src;
   expr->nVal = -expr->nVal;
   pushbyte (expr,RPN_UNSUB);
}

void    rpn_UNNOT (struct Expression *expr, struct Expression *src)
{
   *expr = *src;
   expr->nVal = expr->nVal^0xFFFFFFFF;
    pushbyte (expr,RPN_UNNOT);
}


