//===========================================================
// DKICK - A program launcher
// Copyright (C) 1995 Douglas Boling
//
// Revision History:
//
// 1.0   Initial Release
//
//===========================================================

//-----------------------------------------------------------
// Include files
//-----------------------------------------------------------
#include "windows.h"
#include "commdlg.h"
#include "shellapi.h"

#include "stdlib.h"
#include "string.h"
#include "direct.h"

#include "dkick.h"
#include "dkdll.h"
//-----------------------------------------------------------
// Global data
//-----------------------------------------------------------
// Message dispatch table for MainWindowProc
struct decodeUINT MainMessages[] = {
	WM_CREATE, DoCreateMain,
	WM_SIZE, DoSizeMain,
	MYMSG_LAUNCHPROG, DoLaunchMain,
	WM_DROPFILES, DoDropFilesMain,
	WM_COMMAND, DoCommandMain,
	WM_CLOSE, DoCloseMain,
	WM_DESTROY, DoDestroyMain,
};
// Command Message dispatch for MainWindowProc
struct decodeCMD MainMenuItems[] = {
	IDD_PROGLIST, DoMainCtlProgList,
	IDD_ADD, DoMainCtlAdd,
	IDD_DEL, DoMainCtlDel,
	IDD_EDIT, DoMainCtlEdit,
	IDD_CONFIG, DoMainCtlConfig,
	IDD_ABOUT, DoMainCtlAbout,
	IDD_EXIT, DoMainCtlExit,
	IDOK, DoMainCtlExit,
};

HANDLE	hInst;
HWND		hMain;
HLOCAL	hLocal = 0;
INT 		sEntryCnt = 0;
PPROGENTRY paArray;
UINT		hTimer = 0;
char	szCmd[MAXCMDLINELEN];

char	szTitleText[] = "Drop Kick";         // Window title text
char	szAppName[] = "dkick";               // Application name
char	szIconName[] = "dkickIcon";          // Icon name
char	szProfileName[128];                  // INI file name
//============================================================
// WinMain -- entry point for this application from Windows.
//============================================================
INT APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, 
                     LPSTR lpCmdLine, INT nCmdShow) {
	MSG	msg;
	INT	rc;

	hInst = hInstance;

	// Don't allow more than one instance.
	if (hPrevInstance) 
	   return 1;
		
	// Init first instance of program.
	rc = InitApp(hInstance);
	if (rc) return rc;

	// Initialize this instance
	if((rc = InitInstance(hInstance, lpCmdLine, nCmdShow)) != 0)
		return rc;
	
	// Application message loop
	while (GetMessage (&msg, NULL, 0, 0)) {
		if (!IsDialogMessage (hMain, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}	
	}

	// Instance cleanup
	return TermInstance(hInstance, msg.wParam);
}
//-----------------------------------------------------------
// InitApp - Global initialization code for this application.
//-----------------------------------------------------------
INT InitApp(HANDLE hInstance) {
	WNDCLASS 	wc;
	//
	// Register App Main Window class
	//
	wc.style = 0;                             // Window style
	wc.lpfnWndProc = MainWndProc;             // Callback function
	wc.cbClsExtra = 0;                        // Extra class data
	wc.cbWndExtra = DLGWINDOWEXTRA;           // Extra window data
	wc.hInstance = hInstance;                 // Owner handle
	wc.hIcon = LoadIcon(hInst, szIconName);   // Application icon
	wc.hCursor = LoadCursor(NULL, IDC_ARROW); // Default cursor
	wc.hbrBackground = GetStockObject(LTGRAY_BRUSH); // Background color 
	wc.lpszMenuName =  NULL;                  // Menu name
	wc.lpszClassName = szAppName;             // Window class name

	if (RegisterClass(&wc) == 0)
		return 1;
	return 0;
}
//-----------------------------------------------------------
// InitInstance - Instance initialization code for this app.
//-----------------------------------------------------------
INT InitInstance(HANDLE hInstance, LPSTR lpCmdLine, INT nCmdShow) {
	INT	i, j;
	UINT fFlags;
	char szNum[33];
	PPROGENTRY paPtr;

	// Find program directory for local copy of INI file.
 	i = GetModuleFileName (hInstance, szProfileName, sizeof (szProfileName));
	while (szProfileName[i--] != '.');
	lstrcpy (&szProfileName[i+2], "INI");
	
	// Allocate memory for program array
	hLocal = LocalAlloc (GHND, sizeof (PROGENTRY) * MAXENTRIES);
	if (hLocal == 0)
		return 0x11;
	paArray = (PPROGENTRY) LocalLock (hLocal);
	
	if (*lpCmdLine != '\0') {
		lstrcpy (szCmd, lpCmdLine);
		i = -1;
		sEntryCnt = 0;
	} else {
		//
		// Read Drop Kick information from the INI file
		//
		sEntryCnt = GetPrivateProfileInt (szAppName, PRO_CNT, 0, 
		                                  szProfileName);
		if (sEntryCnt) {
			paPtr = paArray;
			for (i = 0; i < sEntryCnt; i++) {
				itoa (i, szNum, 10);
				paPtr->fFlags = GetPrivateProfileInt (PRO_FLAGS, szNum, 
				                                      0, szProfileName);
				GetPrivateProfileString (PRO_MENU, szNum, "", paPtr->szMenu,
				                         sizeof (paPtr->szMenu), 
				                         szProfileName);
				GetPrivateProfileString (PRO_CMD, szNum, "", paPtr->szCmd,
				                         sizeof (paPtr->szCmd), 
				                         szProfileName);
				GetPrivateProfileString (PRO_DIR, szNum, "", paPtr->szDir,
				                         sizeof (paPtr->szDir), 
				                         szProfileName);
				paPtr++;
			}	
		}
		i = GetPrivateProfileInt (szAppName, PRO_XPOS, 100,
		                          szProfileName);
		j = GetPrivateProfileInt (szAppName, PRO_YPOS, 100,
		                          szProfileName);
		fFlags = GetPrivateProfileInt (szAppName, PRO_FLAGS, 2, szProfileName);
	}	
	// Create main window
	hMain = CreateDialog (hInstance, szAppName, 0, NULL);
	if(!hMain) return 0x10;
	SetStatus (hMain, fFlags);
	
	if (i != -1) {
		SetWindowPos (hMain, NULL, i, j, 0, 0, SWP_NOSIZE);
		ShowWindow(hMain, nCmdShow | SW_SHOW);
		UpdateWindow(hMain);           // force WM_PAINT message
	} else {	
		ShowWindow(hMain, nCmdShow | SW_SHOW);
		ShowWindow(hMain, SW_MINIMIZE);
	}	
	return 0;                         // return success flag
}
//------------------------------------------------------------
// TermInstance - Instance termination code for this app.
//------------------------------------------------------------
INT TermInstance(HANDLE hinstance, int sDefRC) {
	
	SavetoINI();
	return sDefRC;
}
//============================================================
// Message handling procedures for MainWindow
//============================================================
//------------------------------------------------------------
// MainWndProc - Callback function for application window
//------------------------------------------------------------
LONG CALLBACK MainWndProc(HWND hWnd, UINT wMsg, UINT wParam, 
                          LONG lParam) {
	INT i;
	//
	// Search message list to see if we need to handle this
	// message.  If in list, call procedure.
	//
	for(i = 0; i < dim(MainMessages); i++) {
		if(wMsg == MainMessages[i].Code)
			return (*MainMessages[i].Fxn)(hWnd, wMsg, wParam, lParam);
	}
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoCreateMain - process WM_CREATE message for frame window.
//------------------------------------------------------------ 
LONG DoCreateMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	DragAcceptFiles (hWnd, TRUE);	
	return 0;
}
//------------------------------------------------------------
// DoSizeMain - process WM_SIZE message for frame window.
//------------------------------------------------------------ 
LONG DoSizeMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	SetButtons (hWnd);
	FillLB (hWnd);
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoLaunchMain - process MYMSG_PROGLAUNCH message.
//------------------------------------------------------------ 
LONG DoLaunchMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	PPROGENTRY paPtr;
	UINT hExecInst;
	INT sStartFlags;

	paPtr = paArray + wParam;

	if (strlen (paPtr->szDir) != 0) {
		if (paPtr->szDir[1] == ':') {
			//UCase drive letter & convert to num.
			_chdrive ((paPtr->szDir[0] & 0xdf) - 0x40);
			//Pt to dir after drive letter
			chdir ((char *)paPtr->szDir[2]);
		} else	
			chdir (paPtr->szDir);
	}
	if (paPtr->fFlags & SCH_ICON)
		sStartFlags = SW_SHOWMINNOACTIVE;
	else
		sStartFlags = SW_SHOWNOACTIVATE;

	hExecInst = WinExec (paPtr->szCmd, sStartFlags);
	if (hExecInst < 32) {
		paPtr->fFlags |= SCH_ERROR;
		PrintError (hWnd, ERR_LAUNCH + hExecInst);
	}
	return 0;
}	
//------------------------------------------------------------
// DoDropFilesMain - process WM_DROPFILES message for frame window.
//------------------------------------------------------------ 
LONG DoDropFilesMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	char szFileName[128], *pszPtr;
	INT i, sFiles, rc;
	PPROGENTRY paEntry;
	PROGENTRY prog;


	sFiles = DragQueryFile ((HDROP) wParam, -1, 0, 0);
	
	for (i = 0; i < sFiles; i++) {  
		if (sEntryCnt + 1 >= MAXENTRIES) {
			PrintError (hWnd, ERR_MAXSCH);
			break;
		}
		DragQueryFile ((HDROP) wParam, i, szFileName, sizeof (szFileName));

		memset (&prog, 0, sizeof prog);
		// Copy the filename
		lstrcpy (prog.szCmd, szFileName);
		lstrcpy (prog.szDir, szFileName);
		pszPtr = prog.szDir + lstrlen (prog.szDir);
		while (pszPtr > prog.szDir) {
			if (*pszPtr == '\\')
				break;
			pszPtr--;	
		}
		*pszPtr = '\0';
		// Save file name seperate from path.
		lstrcpy (prog.szMenu, pszPtr+1);
		
		// Find association for dropped file.
		rc = (INT)FindExecutable (prog.szCmd, prog.szDir, szFileName);
		if (rc <= 32) {
			PrintError (hWnd, rc);
			break;
		}
		lstrcpy (prog.szCmd, szFileName);
		lstrcat (prog.szCmd, " ");
		lstrcat (prog.szCmd, szFileName + lstrlen (szFileName)+1);
		
		// Create nice looking menu title from filenamew
		pszPtr = prog.szMenu + lstrlen (prog.szMenu);
		while (pszPtr > prog.szMenu) {
			if (*pszPtr == '.')
				break;
			pszPtr--;	
		}
		if (pszPtr != prog.szMenu)
			*pszPtr = '\0';
		AnsiLower (prog.szMenu);
		AnsiUpperBuff (prog.szMenu, 1);
		
		paEntry = paArray + sEntryCnt;
		*paEntry = prog;
		sEntryCnt++;
	}
	DragFinish ((HDROP) wParam);
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return TRUE;
}
//------------------------------------------------------------
// DoCommandMain - process WM_COMMAND message for frame window 
// by decoding the menubar item with the menuitems[] array, 
// then running the corresponding function to process the command.
//------------------------------------------------------------ 
LONG DoCommandMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	INT	i;
	UINT	idItem, wNotifyCode;
	HWND	hwndCtl;

	idItem = (UINT) wParam;                      // Parse Parameters
	hwndCtl = (HWND) LOWORD(lParam);
	wNotifyCode = (UINT) HIWORD(lParam);
	//
	// Call routine to handle control message
	//
	for(i = 0; i < dim(MainMenuItems); i++) {
		if(idItem == MainMenuItems[i].Code)
			return (*MainMenuItems[i].Fxn)(hWnd, idItem, hwndCtl, 
			                               wNotifyCode);
	}
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoCloseMain - process WM_CLOSE message for frame window.
//------------------------------------------------------------ 
LONG DoCloseMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	DestroyWindow (hMain);
	return TRUE;
}
//------------------------------------------------------------
// DoDestroyMain - process WM_DESTROY message for frame window.
//------------------------------------------------------------ 
LONG DoDestroyMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
   RECT	rect;

	DragAcceptFiles (hWnd, FALSE);

	//Save Window position
	if	(!IsIconic (hWnd) && !IsZoomed (hWnd)) {
		GetWindowRect (hWnd, &rect);
		MyWritePrivateProfileInt (szAppName, PRO_XPOS, rect.left,
		                          szProfileName);
		MyWritePrivateProfileInt (szAppName, PRO_YPOS, rect.top,
		                          szProfileName);
	}
	PostQuitMessage (0);
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//============================================================
// Control handling procedures for MainWindow
//============================================================
//------------------------------------------------------------
// DoMainCtlProgList - Process Listbox messages
//------------------------------------------------------------ 
LONG DoMainCtlProgList (HWND hWnd, UINT idItem, HWND hwndCtl, 
                        UINT wNotifyCode) {

	if (wNotifyCode == LBN_DBLCLK)
		SendMessage (hWnd, WM_COMMAND, IDD_EDIT, MAKELONG (0, BN_CLICKED));
	else if (wNotifyCode == LBN_SELCHANGE)
		SetButtons (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlAdd - Process Add button
//------------------------------------------------------------ 
LONG DoMainCtlAdd (HWND hWnd, UINT idItem, HWND hwndCtl, 
                   UINT wNotifyCode) {
	PPROGENTRY paEntry;
	PROGENTRY prog;

	if (wNotifyCode != BN_CLICKED)
		return 0;

	if (sEntryCnt + 1 >= MAXENTRIES) {
		PrintError (hWnd, ERR_MAXSCH);
		return 0;
	}
	memset (&prog, 0, sizeof prog);
	if (MyDisplayDialog(hInst, "AddBox", hWnd, (DLGPROC) AddDlgProc, 
	                    (LPARAM)MAKELONG (&prog, 0)) == 0) 
		return 0;
	paEntry = paArray + sEntryCnt;
	*paEntry = prog;
	sEntryCnt++;
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlDel - Process Del button
//------------------------------------------------------------ 
LONG DoMainCtlDel (HWND hWnd, UINT idItem, HWND hwndCtl, 
                   UINT wNotifyCode) {
	PPROGENTRY paEntry;
	INT i, sSel, sRC;
								
	if (wNotifyCode != BN_CLICKED)
		return 0;
		
	sSel = (INT) SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_GETCURSEL, 0, 0);
	if (sSel == LB_ERR) {
		PrintError (hWnd, ERR_NOSCHSEL);
		return 0;
	}
	sRC = MessageBox (hWnd, "Do you want to delete this item?", szTitleText,
	                  MB_YESNO);
	if (sRC != IDYES)
		return 0;
	//Delete item from list
	paEntry = paArray + sSel;
	for (i = sSel; i < sEntryCnt-1; i++) {
		*paEntry = *(paEntry+1);
		paEntry++;
	}	
	sEntryCnt--;
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlEdit - Process Edit button
//------------------------------------------------------------ 
LONG DoMainCtlEdit (HWND hWnd, UINT idItem, HWND hwndCtl, 
                    UINT wNotifyCode) {
	PPROGENTRY paEntry;
	PROGENTRY prog;
	INT sSel;
								
	if (wNotifyCode != BN_CLICKED)
		return 0;
		
	sSel = (INT) SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_GETCURSEL, 0, 0);

	if (sSel == LB_ERR) {
		PrintError (hWnd, ERR_NOSCHSEL);
		return 0;
	}
	paEntry = paArray + sSel;
	prog = *paEntry;
	if (MyDisplayDialog(hInst, "AddBox", hWnd, (DLGPROC) AddDlgProc, 
	                (LPARAM)MAKELONG (&prog, 1)) == 0)
		return 0;

	*paEntry = prog;
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlConfig - Process Config button
//------------------------------------------------------------ 
LONG DoMainCtlConfig (HWND hWnd, UINT idItem, HWND hwndCtl, 
                      UINT wNotifyCode) {
	UINT fFlags;

	if (wNotifyCode != BN_CLICKED)
		return 0;
	fFlags = GetStatus ();
	
	if (MyDisplayDialog(hInst, "ConfigBox", hWnd, 
   		             (DLGPROC) ConfigDlgProc, (LPARAM) (LPINT)&fFlags)) {
		SetStatus (hWnd, fFlags);
		SavetoINI();
	}
	return 0;
}
//------------------------------------------------------------
// DoMainCtlAbout - Process About button
//------------------------------------------------------------ 
LONG DoMainCtlAbout (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {
								
	if (wNotifyCode == BN_CLICKED)
		MyDisplayDialog(hInst, "AboutBox", hWnd, 
   		             (DLGPROC) AboutDlgProc, 0);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlExit - Process Exit button
//------------------------------------------------------------ 
LONG DoMainCtlExit (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {
	INT sRC;
	
	if (wNotifyCode != BN_CLICKED)
		return 0;

	sRC = MessageBox (hWnd, "DropKick menus are only available while DropKick"\
	                  " is running.\nAre you sure you want to close DropKick?", 
	                  szTitleText, MB_YESNO);
	if (sRC != IDYES)
		return 0;

	SendMessage (hWnd, WM_CLOSE, 0, 0);
	return 0;
}
//============================================================
// AddDlgProc - Add dialog box dialog procedure
//============================================================
BOOL CALLBACK AddDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                          LONG lParam) {
	static char *szFilter[] = {"Programs", "*.EXE;*.COM;*.PIF;*.BAT", 
	                           "All Files (*.*)", "*.*", 
										"" };
	static PPROGENTRY paEntry;
	char szTemp[128], *pszTemp;
	char szTemp1[128];
	INT i, rc;
                             
	switch (wMsg) {
		case WM_INITDIALOG:
			paEntry = (PPROGENTRY) LOWORD (lParam);
			if (HIWORD (lParam)) {

				SetDlgItemText (hWnd, IDD_MENUNAME, paEntry->szMenu);
				SetDlgItemText (hWnd, IDD_CMD, paEntry->szCmd);
				SetDlgItemText (hWnd, IDD_WDIR, paEntry->szDir);
				
				CheckDlgButton (hWnd, IDD_ICON, 
				                (paEntry->fFlags & SCH_ICON) ? 1 : 0);

				SetWindowText (hWnd, "Edit Program Details");
				SetDlgItemText (hWnd, IDOK, "&Save");
			} else {
				SetWindowText (hWnd, "Add Program");
			}	
			return TRUE;

		case WM_COMMAND:
			switch (wParam) {
		   	case IDD_ADDBROWSE:
					if (MyGetFilename (hWnd, szTemp, 
					                   sizeof (szTemp), *szFilter, 1))
					SetDlgItemText (hWnd, IDD_CMD, szTemp);
					pszTemp = strrchr (szTemp, '\\');
					if (pszTemp) {
						*pszTemp = '\0';
						SetDlgItemText (hWnd, IDD_WDIR, szTemp);
					} else	
						SetDlgItemText (hWnd, IDD_WDIR, "");
					return TRUE;

				case IDOK:
					//Get menu text
					GetDlgItemText (hWnd, IDD_MENUNAME, paEntry->szMenu, 
					                sizeof (paEntry->szMenu));
					if (strlen (paEntry->szMenu) == 0) {
						PrintError (hWnd, ERR_NOMENUNAME);
						return TRUE;
					}

					//Get and validate the working directory
					GetDlgItemText (hWnd, IDD_WDIR, szTemp, sizeof (szTemp));
					strcpy (paEntry->szDir, szTemp);
					if (strlen (szTemp) != 0) {
						getcwd (szTemp, sizeof (szTemp));
						i = chdir (paEntry->szDir);
						chdir (paEntry->szDir);
						
						if (i != 0) {
							strcpy (szTemp, "DropKick can not locate the working directory:\n\n");
							strcat (szTemp, paEntry->szDir);
							strcat (szTemp, "\n\nShould this directory be used anyway?");
							i = MessageBox (hWnd, szTemp, szTitleText, 
							                MB_ICONEXCLAMATION | MB_YESNO);
							if (i != IDYES)
								return TRUE;
						}
					}

					//Get and validate the command line
					GetDlgItemText (hWnd, IDD_CMD, szTemp, sizeof (szTemp));
					if (strlen (szTemp) == 0) {
						PrintError (hWnd, ERR_NOCMDLINE);
						return TRUE;
					}
					// Find association for dropped file.
					// First, strip off any cmd line params
					pszTemp = strpbrk (szTemp, " /\t");
					if (pszTemp)
						*pszTemp = '\0';
					// Call Shell.dll to get association	
					rc = (INT)FindExecutable (szTemp, paEntry->szDir, 
					                          szTemp1);
					if (rc <= 32) {
						PrintError (hWnd, rc);
						return TRUE;
					} 
					lstrcpy (paEntry->szCmd, szTemp1);
					lstrcat (paEntry->szCmd, " ");
					lstrcat (paEntry->szCmd, szTemp1 + lstrlen (szTemp1)+1);
					if (pszTemp) {
						lstrcat (paEntry->szCmd, " ");
						lstrcat (paEntry->szCmd, pszTemp+1);
					}
					//Get startup flags
					if (IsDlgButtonChecked (hWnd, IDD_ICON))
				      paEntry->fFlags |= SCH_ICON;
					else	
				      paEntry->fFlags &= ~SCH_ICON;
						
					EndDialog(hWnd, 1);
					return TRUE;
					
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
			}	
			return FALSE;
	}
	return FALSE;
}
//============================================================
// ConfigDlgProc - Configure dialog box dialog procedure
//============================================================
BOOL CALLBACK ConfigDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                             LONG lParam) {
	static UINT *pfFlags;
	                             
	switch (wMsg) {
		case WM_INITDIALOG:
			pfFlags = (UINT *) LOWORD (lParam);
			if (LOWORD (lParam)) {
				CheckRadioButton (hWnd, IDD_LEFT, IDD_RIGHT, 
				                  IDD_LEFT + (*pfFlags & SFT_BTN));

				CheckDlgButton (hWnd, IDD_ALT, 
				                (*pfFlags & SFT_ALT) ? 1 : 0);
				CheckDlgButton (hWnd, IDD_CTRL, 
				                (*pfFlags & SFT_CTRL) ? 1 : 0);
				CheckDlgButton (hWnd, IDD_SHIFT,
				                (*pfFlags & SFT_SHIFT) ? 1 : 0);
			} else {
				CheckRadioButton (hWnd, IDD_LEFT, IDD_RIGHT, IDD_LEFT);

			}	
			return TRUE;

		case WM_COMMAND:
			switch (wParam) {
				case IDOK:
			      *pfFlags &= ~SFT_BTN;
					if (IsDlgButtonChecked (hWnd, IDD_MID))
				      *pfFlags |= 1;
					else if (IsDlgButtonChecked (hWnd, IDD_RIGHT))
				      *pfFlags |= 2;
						
					if (IsDlgButtonChecked (hWnd, IDD_ALT))
				      *pfFlags |= SFT_ALT;
					else	
				      *pfFlags &= ~SFT_ALT;

					if (IsDlgButtonChecked (hWnd, IDD_CTRL))
				      *pfFlags |= SFT_CTRL;
					else	
				      *pfFlags &= ~SFT_CTRL;

					if (IsDlgButtonChecked (hWnd, IDD_SHIFT))
				      *pfFlags |= SFT_SHIFT;
					else	
				      *pfFlags &= ~SFT_SHIFT;
						
					EndDialog(hWnd, 1);
					return TRUE;
					
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
			}	
			return FALSE;
	}
	return FALSE;
}
//============================================================
// AboutDlgProc - About dialog box dialog procedure
//============================================================
BOOL CALLBACK AboutDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                            LONG lParam) {
	HDC hdc;
	HPEN hDPen, hLPen, hOldPen;
	PAINTSTRUCT ps;
	RECT rectAbout;
	INT xPos, yPos;
                             
	switch (wMsg) {
		case WM_INITDIALOG:
			GetWindowRect  (GetDesktopWindow(), &rectAbout);
			xPos = (rectAbout.right - rectAbout.left)/2;
			yPos = ((rectAbout.bottom - rectAbout.top))/2;
			GetWindowRect  (hWnd, &rectAbout);
			xPos = max (0, xPos - (rectAbout.right - rectAbout.left)/2);
			yPos = max (0, yPos - (rectAbout.bottom - rectAbout.top)/2);
			SetWindowPos (hWnd, 0, xPos, yPos, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
			ShowWindow (hWnd, SW_SHOW);
			return TRUE;

		case WM_PAINT:
			//
			// Draw the rectangle surrounding the dialog box text.
			//
			hdc = BeginPaint (hWnd, &ps);
			//Get text rectangle defined in RC file
			GetClientRect (GetDlgItem (hWnd, IDD_FRAME), &rectAbout);
			ClientToScreen (GetDlgItem (hWnd, IDD_FRAME), (LPPOINT)&rectAbout);
			ScreenToClient (hWnd, (LPPOINT)&rectAbout);
			rectAbout.left -= 2;
			rectAbout.top -= 2;
			rectAbout.right += rectAbout.left + 4;
			rectAbout.bottom += rectAbout.top + 4;
			hDPen = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNSHADOW));
			hLPen = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNHIGHLIGHT));

			//Start at bottom left, draw dark pen up and over.
			hOldPen = SelectObject (hdc, hDPen);
			MoveTo (hdc, rectAbout.left, rectAbout.bottom);
			LineTo (hdc, rectAbout.left, rectAbout.top);
			LineTo (hdc, rectAbout.right+1, rectAbout.top);
			//Start at bottom left, draw light pen over and up.
			SelectObject (hdc, hLPen);
			MoveTo (hdc, rectAbout.left+1, rectAbout.bottom);
			LineTo (hdc, rectAbout.right, rectAbout.bottom);
			LineTo (hdc, rectAbout.right, rectAbout.top);

			SelectObject (hdc, hOldPen);
			DeleteObject (hDPen);
			DeleteObject (hLPen);
			EndPaint (hWnd, &ps);
			return TRUE;

		case WM_COMMAND:
	   	if ((wParam == IDOK) || (wParam == IDCANCEL)) {
				EndDialog(hWnd, 0);
			}	
			return TRUE;
	}
	return FALSE;
}
//------------------------------------------------------------
// SavetoINI - Writes the data to the ini file.
//------------------------------------------------------------ 
void SavetoINI (void) {
	INT i;
	UINT fFlags;
	PPROGENTRY paPtr;
	char	szNum[8];
	
	WritePrivateProfileString (PRO_FLAGS, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_MENU, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_CMD, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_DIR, NULL, NULL, szProfileName);

	paPtr = paArray;
	for (i = 0; i < sEntryCnt; i++) {
		itoa (i, szNum, 10);
		MyWritePrivateProfileInt (PRO_FLAGS, szNum, paPtr->fFlags, szProfileName);
		WritePrivateProfileString (PRO_MENU, szNum, paPtr->szMenu, szProfileName);
		WritePrivateProfileString (PRO_CMD, szNum, paPtr->szCmd, szProfileName);
		WritePrivateProfileString (PRO_DIR, szNum, paPtr->szDir, szProfileName);
		paPtr++;
	}
	fFlags = GetStatus ();
	MyWritePrivateProfileInt (szAppName, PRO_CNT, sEntryCnt, szProfileName);
	MyWritePrivateProfileInt (szAppName, PRO_FLAGS, fFlags, szProfileName);
	return;
}	
//------------------------------------------------------------
// SetButtons - Enables and disables the buttons depending on
// the current state of the window.
//------------------------------------------------------------ 
void SetButtons (HWND hWnd) {
	if (SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_GETCURSEL, 0, 0)
	    == LB_ERR) {
		EnableWindow (GetDlgItem (hWnd, IDD_DEL), FALSE);
		EnableWindow (GetDlgItem (hWnd, IDD_EDIT), FALSE);
	} else {
		EnableWindow (GetDlgItem (hWnd, IDD_DEL), TRUE);
		EnableWindow (GetDlgItem (hWnd, IDD_EDIT), TRUE);
	}
	if (sEntryCnt == MAXENTRIES)
		EnableWindow (GetDlgItem (hWnd, IDD_ADD), FALSE);
	else		
		EnableWindow (GetDlgItem (hWnd, IDD_ADD), TRUE);
	return;		
}		
//------------------------------------------------------------
// FillLB - Clears, then refills the hot key listbox
//------------------------------------------------------------ 
void FillLB (HWND hWnd) {
	INT i;
	PPROGENTRY paPtr;

	// Clear list box and DLL menu list.	
	SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_RESETCONTENT, 0, 0);
	RemoveKickList (0);

	paPtr = paArray;
	for (i = 0; i < sEntryCnt; i++) {
		SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_ADDSTRING, 0, 
		                    (LPARAM) (LPSTR) paPtr->szMenu);
		SetKickList (paPtr->szMenu);
		paPtr++;
	}
	SetButtons (hWnd);
	return;
}	
//============================================================  
// General Helper Routines 
//============================================================  
//------------------------------------------------------------
// PrintError - Displays a message box with an error code
//------------------------------------------------------------ 
void PrintError (HWND hWnd, INT rc) {
	char szErrStr[80];
	char szTemp[12];
	
	if (rc > 0)
	   rc += ERR_LAUNCH;
	else 
	   rc = abs (rc);
	if (LoadString (hInst, rc, szErrStr, sizeof (szErrStr)) == 0) {
		itoa (rc, szTemp, 10);
		strcpy (szErrStr, "Error number: ");
		strcat (szErrStr, szTemp);
	}
	MessageBox (hWnd, szErrStr, szTitleText, MB_OK | MB_ICONHAND);
	return;
}
//------------------------------------------------------------
// MyDisplayDialog - Display a dialog box
//------------------------------------------------------------ 
INT MyDisplayDialog (HINSTANCE hInstance, LPCSTR szDlgName,
                     HWND hWnd, DLGPROC lpDialogProc, 
                     LPARAM lParam) {
    DLGPROC lpDlgProcInst;
    INT		rc;

    lpDlgProcInst = MakeProcInstance((FARPROC)lpDialogProc, hInstance);
    rc = DialogBoxParam (hInstance, szDlgName, hWnd, 
                         lpDlgProcInst, lParam);
    FreeProcInstance((FARPROC)lpDlgProcInst);
    return rc;                              
}
//------------------------------------------------------------
// MyWritePrivateProfileInt - Writes an integer to the profile
//------------------------------------------------------------
BOOL MyWritePrivateProfileInt (char *szSec, char *szEntry, 
                               int Num, char *szProfile) {
	char	szStr[33];
	                           
	itoa (Num, szStr, 10);
	return WritePrivateProfileString (szSec, szEntry, szStr, 
	                                  szProfile);
}
//------------------------------------------------------------
// MyGetFilename - Uses the common File Open dialog box to
// query the user for a filename
//------------------------------------------------------------
BOOL MyGetFilename (HWND hWnd, char *szFilename, INT sMaxLen, char *szFilter,
                    INT sFilterIndex) {

   OPENFILENAME ofn;

   *szFilename = '\0';

	ofn.lStructSize = sizeof (OPENFILENAME);
	ofn.hwndOwner = hWnd;
	ofn.hInstance = hInst;
	ofn.lpstrFilter = szFilter;
	ofn.lpstrCustomFilter = 0;
	ofn.nMaxCustFilter = 0;
	ofn.nFilterIndex = sFilterIndex;
	ofn.lpstrFile = szFilename;
	ofn.nMaxFile = sMaxLen;
	ofn.lpstrFileTitle = 0;
	ofn.nMaxFileTitle = 0;
	ofn.lpstrInitialDir = 0;
	ofn.lpstrTitle = "Browse";
	ofn.Flags = OFN_FILEMUSTEXIST;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;
	ofn.lpstrDefExt = "WBT";
	ofn.lCustData = 0;
	ofn.lpfnHook = 0;
	ofn.lpTemplateName = 0;

 	return (GetOpenFileName (&ofn));
}