/*
	RegTable.CPP version 1.0
	by Robert Schmidt of Ztiff Zox Softwear 1993

	Defines the member and friend functions of the RegisterTable class
		declared in RegTable.HPP.
*/

#include <conio.h>
#include <fstream.h>
#include "Screen.hpp"
#include "RegTable.hpp"



RegisterTable::RegisterTable(const char* fname)
	{
	// Open the file containing register descriptions
	ifstream ins(fname);
	// Read number of registers (first number in file)
	ins >> registers;
	// Allocate enough space
	reg = new NamedRegister[registers];
	// Read reg. descriptions
	for (int i=0; i<registers; i++)
		ins >> reg[i];
	// Get current register configuration from VGA, and use as default
	in();
	prevSel = select = 0;
	}

void RegisterTable::printCon(int r)
	{
	// This gotoxy divides the registers into two columns.
	gotoxy(40*(r / editHeight) +1, r % editHeight +1);
	// Optionally print the left cursor.
	textattr(CURSOR_COLOR);
	cprintf(r==select ? "\20" : " ");
	// Then put out the meat.
	reg[r].printCon();
	// And possibly the right cursor.
	textattr(CURSOR_COLOR);
	cprintf(r==select ? "\21" : " ");
	// This gotoxy just puts the hardware cursor where it won't distract you.
	gotoxy(40*(r / editHeight)+38, r % editHeight +1);
	}

void RegisterTable::printAllCon()
	{
	for (int r = 0; r < registers; r++)
		printCon(r);
	}

int RegisterTable::updateSelect()
	{
	if (select < 0)
		select = registers - 1;
	else
		if (select >= registers)
			select = 0;
	if (prevSel != select)
		{
		printCon(prevSel);
		prevSel = select;
		}
	printCon(select);
	return select;
	}

void RegisterTable::in()
	{
	for (int r = 0; r < registers; r++)
		reg[r].in();
	}

void RegisterTable::out()
	{
	// The next 5 lines are VGA specific
	outportb(0x3d4,0x11);				// Ensure CRT regs. 0-7 are writable!
	int v = inportb(0x3d5);				//	That is, clear bit 7 of port
	v &= 0x7f;							//	0x3D4, index 0x11.
	outportb(0x3d4,0x11);
	outportb(0x3d5,v);

	for (int r = 0; r < registers; r++)
		if (reg[r].isEnabled())
			reg[r].out();
	}

/*
	This istream operator >> reads an entire table of Register values
	into 'this' table.
	Notes:
		The stream is read until good() is no longer true.  Not good
			practice, but hey!
		If the read Register's port and index pair is not found in 'this'
			table, it is ignored.
		In effect, only the *values* in the table may change after >>,
			not port or index numbers.
*/

istream& operator>> (istream &in, RegisterTable &t)
	{
	int r = 0;
	t.doDisable();							// first disable all registers
	while (in.good())
		{
		Register temp;
		in >> temp;

		int prevr = r;

		//Search for the correct register position:
		while (temp.getPort() != t.reg[r].getPort() ||
			temp.getIndex() != t.reg[r].getIndex())
			{
			if (++r >= t.registers)
				r = 0;
			if (r == prevr)					// Have we looped around once?
				goto skip;					//	Register not supported!
			}
		// Correct register found, now store the value and enable it.
		t.reg[r].setValue(temp.getValue());
		t.reg[r].doEnable();				// enable this single register
skip:
		}
	return in;
	}


//	This operator << sends all enabled registers in t to the out stream.

ostream& operator<< (ostream &out, RegisterTable &t)
	{
	for (int r = 0; r < t.registers; r++)
		if (t.reg[r].isEnabled())
			out << Register(t.reg[r]);
	return out;
	}


// RegisterTable::showBitMask() updates the bit pattern display with
//	the value of selected register.  This is TWEAK specific.

void RegisterTable::showBitMask()
	{
	gotoxy(42,editHeight-4);
	textattr(BITHEADER_COLOR);
	cprintf("Bit mask: 7 6 5 4 3 2 1 0");
	gotoxy(51,editHeight-3);
	textattr(BITPATTERN_COLOR);
	unsigned char v = reg[select].getValue();
	for (int e=7; e>=0; e--)
		cprintf( v&(1<<e) ? " 1" : " 0");
	}

